;;; -*- Mode: LISP; Syntax: Ansi-common-lisp; Package: CL-LIB; Base: 10 -*-

(IN-PACKAGE CL-LIB)

;;;; Note the local extensions remain in cl-extensions. <miller>
;
;;;; ****************************************************************
;;;; Extensions to Common Lisp **************************************
;;;; ****************************************************************
;;;;
;;;; This file is a collection of extensions to Common Lisp. 
;;;;
;;;; It is a combination of the CL-LIB package copyleft by Brad Miller
;;;; <miller@cs.rochester.edu> and a similar collection by
;;;; Mark Kantrowitz <mkant+@cs.cmu.edu>.
;;;;
;;;; The following functions were originally from CL-LIB:
;;;;   let-if, factorial, update-alist, truncate-keywords, while,
;;;;   defclass-x, copy-hash-table, defflag, round-to, extract-keyword
;;;;   let*-non-null, mapc-dotted-list, 
;;;;   mapcar-dotted-list, mapcan-dotted-list, some-dotted-list, 
;;;;   every-dotted-list, msetq, mlet, dosequence, force-string, prefix?,
;;;;   elapsed-time-in-seconds, bit-length, flatten, 
;;;;   sum-of-powers-of-two-representation, 
;;;;   difference-of-powers-of-two-representation,
;;;;   ordinal-string, between, 
;;;;   cond-binding-predicate-to <quiroz@cs.rochester.edu>
;;;;   remove-keywords <baldwin@cs.rochester.edu>
;;;;
;;;; The following functions were contributed by Mark Kantrowitz:
;;;;   circular-list, dofile, seq-butlast, seq-last, firstn, in-order-union
;;;;   parse-with-delimiter, parse-with-delimiters, string-search-car,
;;;   string-search-cdr, parallel-substitute, lisp::nth-value,
;;;;   parse-with-string-delimiter, parse-with-string-delimiter*,
;;;;   member-or-eq, number-to-string, null-string, time-string.
;;;;   list-without-nulls, cartesian-product, cross-product, permutations
;;;;   powerset, occurs, split-string, format-justified-string, 
;;;;   eqmemb, neq, car-eq, dremove, displace, tailpush, explode,
;;;;   implode, crush, listify-string, and-list, or-list, lookup,
;;;;   make-variable, variablep, make-plist, make-keyword
;;;;   
;;;; The GNU Emacs distribution agreement is included by reference.
;;;; Share and Enjoy!
;;;;
;

;;; Uncomment this to make the extensions accessible from the Lisp package
;;; without the EXT prefix.
;(in-package "LISP")

;;; ********************************
;;; Sets ***************************
;;; ********************************
;;; list-without-nulls
;;; cross-product
;;; cartesian-product
;;; permutations

(defun list-without-nulls (list)
  "Returns a copy of list with all null elements removed."
  (let* ((head (list nil))
         (tail head))
    (loop
     (if (null list)
	 (return-from list-without-nulls (cdr head))
	 (when (car list)
	   (rplacd tail (list (car list)))
	   (setf tail (cdr tail))))
     (setf list (cdr list)))))

(defun cartesian-product (set1 set2)
  "Returns the cross product of two sets."
  (let ((result ()))
    (dolist (elt1 set1)
      (dolist (elt2 set2)
        (push (cons elt1 elt2) result)))
    result))

(defun cross-product (&rest lists)
  "Returns the cross product of a set of lists."
  (labels ((cross-product-internal (lists)
	     (if (null (cdr lists))
		 (mapcar #'list (car lists))
		 (let ((cross-product (cross-product-internal (cdr lists)))
		       (result '()))
		   (dolist (elt-1 (car lists))
		     (dolist (elt-2 cross-product)
		       (push (cons elt-1 elt-2) result)))
		   result))))
    (cross-product-internal lists)))

(defun permutations (items)
  "Given a list of items, returns all possible permutations of the list."
  (let ((result nil))
    (if (null items)
        '(nil)
        (dolist (item items result)
          (dolist (permutation (permutations (remove item items)))
            (push (cons item permutation) result))))))

(defun powerset (list)
  "Given a set, returns the set of all subsets of the set."
  (let ((result (list nil)))
    (dolist (item list result)
      (dolist (subset result)
	(push (cons item subset) result)))))

#-lispm
(defun circular-list (&rest list)
  "Creates a circular list of the arguments. Handy for use with 
   the list mapping functions. For example, 
     (mapcar #'+ '(1 2 3 4 5) (circular-list 3)) --> (4 5 6 7 8)
     (mapcar #'+ '(1 2 3 4 5) (circular-list 0 1)) --> (1 3 3 5 5)"
  (setf list (copy-list list))
  (setf (cdr (last list)) list)
  list)

(defun occurs (elt lst)
  "Returns T if ELT occurs somewhere in LST's tree structure."
  (cond ((null lst)
         nil)
        ((consp lst)
         ;; This walks down the tree structure of LST.
         (or (occurs elt (car lst))
             (occurs elt (cdr lst))))
        ((atom lst)
         ;; If we are at a leaf, test if ELT is the same as the leaf.
         (eq lst elt))))

(defun firstn (list &optional (n 1))
  "Returns a new list the same as List with only the first N elements."
  (cond ((> n (length list)) list)
	((< n 0) nil)
	(t (ldiff list (nthcdr n list)))))

(defun in-order-union (list1 list2)
  "Append and remove duplicates. Like union, but the objects are
   guarranteed to stay in order."
  (remove-duplicates (append list1 list2) :from-end t))

;;; ********************************
;;; Sequences **********************
;;; ********************************
(defun seq-butlast (sequence &optional (n 1))
  (let* ((length (length sequence))
	 (delta (- length n)))
    (when (plusp delta)
      (subseq sequence 0 delta))))

(defun seq-last (sequence &optional (n 1))
  (let* ((length (length sequence))
	 (delta (- length n)))
    (when (plusp delta)
      (subseq sequence delta length))))

;;; ********************************
;;; Strings ************************
;;; ********************************

(defun string-search-car (character-bag string)
  "Returns the part of the string before the first of the delimiters in 
   CHARACTER-BAG and the delimiter."
  (let* ((delimiter nil)
	 (delimiter-position (position-if #'(lambda (character)
					      (when (find character 
							  character-bag)
						(setq delimiter character)))
					  string)))
    (values (subseq string 0 delimiter-position)
	    delimiter)))

(defun string-search-cdr (character-bag string)
  "Returns the part of the string after the first of the delimiters in 
   CHARACTER-BAG, if any, and the delimiter. If none of the delimiters 
   are found, returns NIL and NIL."
  (let* ((delimiter nil)
	 (delimiter-position (position-if #'(lambda (character)
					      (when (find character 
							  character-bag)
						(setq delimiter character)))
					 string)))
    (if delimiter-position
	(values (subseq string (1+ delimiter-position))
		delimiter)
	;; Maybe this should be "" instead of NIL?
	(values nil delimiter))))


(defun parse-with-delimiter (line &optional (delim #\newline))
  "Breaks LINE into a list of strings, using DELIM as a 
   breaking point."
  ;; what about #\return instead of #\newline?
  (let ((pos (position delim line)))
    (cond (pos
           (cons (subseq line 0 pos)
                 (parse-with-delimiter (subseq line (1+ pos)) delim)))
          (t
           (list line)))))

(defun parse-with-delimiters (line &optional (delimiters '(#\newline)))
  "Breaks LINE into a list of strings, using DELIMITERS as a 
   breaking point."
  ;; what about #\return instead of #\newline?
  (let ((pos (position-if #'(lambda (character) (find character delimiters))
			    line)))
    (cond (pos
           (cons (subseq line 0 pos)
                 (parse-with-delimiters (subseq line (1+ pos)) delimiters)))
          (t
           (list line)))))


;;; subst:sublis::substitute:?  -- cl needs a parallel-substitute for
;;; performing many substitutions in a sequence in parallel.
(defun parallel-substitute (alist string)
  "Makes substitutions for characters in STRING according to the ALIST. 
   In effect, PARALLEL-SUBSTITUTE can perform several SUBSTITUTE
   operations simultaneously."
  (declare (simple-string string))
  ;; This function should be generalized to arbitrary sequences and
  ;; have an arglist (alist sequence &key from-end (test #'eql) test-not
  ;; (start 0) (count most-positive-fixnum) end key).
  (if alist
      (let* ((length (length string))
	     (result (make-string length)))
	(declare (simple-string result))
	(dotimes (i length)
	  (let ((old-char (schar string i)))
	    (setf (schar result i)
		  (or (second (assoc old-char alist :test #'char=))
		      old-char))))
	result)
      string))

(defun parse-with-string-delimiter (delim string &key (start 0) end)
  "Returns up to three values: the string up to the delimiter DELIM
   in STRING (or NIL if the field is empty), the position of the beginning
   of the rest of the string after the delimiter, and a value which, if
   non-NIL (:delim-not-found), specifies that the delimiter was not found."
  (declare (simple-string string))
  ;; Conceivably, if DELIM is a string consisting of a single character,
  ;; we could do this more efficiently using POSITION instead of SEARCH.
  ;; However, any good implementation of SEARCH should optimize for that
  ;; case, so nothing to worry about.
  (setq end (or end (length string)))
  (let ((delim-pos (search delim string :start2 start :end2 end))
	(dlength (length delim)))
    (cond ((null delim-pos)		
	   ;; No delimiter was found. Return the rest of the string,
	   ;; the end of the string, and :delim-not-found.
	   (values (subseq string start end) end :delim-not-found))
	  ((= delim-pos start)		
	   ;; The field was empty, so return nil and skip over the delimiter.
	   (values nil (+ start dlength)))
	  ;; The following clause is subsumed by the last cond clause,
	  ;; and hence should probably be eliminated.
	  (t				
	   ;; The delimiter is in the middle of the string. Return the
	   ;; field and skip over the delimiter. 
	   (values (subseq string start delim-pos)
		   (+ delim-pos dlength))))))

(defun parse-with-string-delimiter* (delim string &key (start 0) end
					   include-last)
  "Breaks STRING into a list of strings, each of which was separated
   from the previous by DELIM. If INCLUDE-LAST is nil (the default),
   will not include the last string if it wasn't followed by DELIM
   (i.e., \"foo,bar,\" vs \"foo,bar\"). Otherwise includes it even if
   not terminated by DELIM. Also returns the final position in the string."
  (declare (simple-string string))
  (setq end (or end (length string)))
  (let (result)
    (loop
     (if (< start end)
	 (multiple-value-bind (component new-start delim-not-found)
	     (parse-with-string-delimiter delim string :start start :end end)
	   (when delim-not-found 
	     (when include-last
	       (setq start new-start)
	       (push component result))
	     (return))
	   (setq start new-start)
	   (push component result))
	 (return)))
    (values (nreverse result) 
	    start)))

;; set in user-manual
(defun split-string (string &key (item #\space) (test #'char=))
  ;; Splits the string into substrings at spaces.
  (let ((len (length string))
	(index 0) result)
    (dotimes (i len
		(progn (unless (= index len)
			 (push (subseq string index) result))
		       (reverse result)))
      (when (funcall test (char string i) item)
	(unless (= index i);; two spaces in a row
	  (push (subseq string index i) result))
	(setf index (1+ i))))))

(defun format-justified-string (prompt contents &optional (width 80)
				       (stream *standard-output*))
  (let ((prompt-length (+ 2 (length prompt))))
    (cond ((< (+ prompt-length (length contents)) width)
	   (format stream "~%~A- ~A" prompt contents))
	  (t
	   (format stream "~%~A-" prompt)
	   (do* ((cursor prompt-length)
		 (contents (split-string contents) (cdr contents))
		 (content (car contents) (car contents))
		 (content-length (1+ (length content)) (1+ (length content))))
	       ((null contents))
	     (cond ((< (+ cursor content-length) width)
		    (incf cursor content-length)
		    (format stream " ~A" content))
		   (t
		    (setf cursor (+ prompt-length content-length))
		    (format stream "~%~A  ~A" prompt content)))))))
  (finish-output stream))

(defun number-to-string (number &optional (base 10))
  (cond ((zerop number) "0")
	((eql number 1) "1")
	(t
	 (do* ((len (1+ (truncate (log number base)))) 
	       (res (make-string len))
	       (i (1- len) (1- i))
	       (q number)		; quotient
	       (r))			; residue
	     ((zerop q)			; nothing left
	      res)
	   (declare (simple-string res)
		    (fixnum len i r))
	   (multiple-value-setq (q r) (truncate q base))
	   (setf (schar res i) 
		 (schar "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ" r))))))

(defun null-string (string &optional (start 0) end)
  "Returns T if STRING is the null string \"\" between START and END."
  (unless end (setf end (length string)))
  (string-equal string "" :start1 start :end1 end))

;;;; ********************************
;;;; Time ***************************
;;;; ********************************
(defun time-string (&optional universal-time)
  (unless universal-time (setf universal-time (get-universal-time)))
  (multiple-value-bind (secs min hour date month year dow)
      (decode-universal-time universal-time)
    (format nil "~@:(~A ~A-~A-~A ~2,'0d:~2,'0d:~2,'0d~)"
	    (svref '#("Mon" "Tue" "Wed" "Thu" "Fri" "Sat" "Sun") dow)
	    date 
	    (svref '#(0 "Jan" "Feb" "Mar" "Apr" "May"
			"Jun" "Jul" "Aug" "Sep" "Oct"
			"Nov" "Dec")
		   month)
	    (mod year 100)
	    hour min secs)))

#-EXCL
(defmacro if* (condition true &rest false)
  `(if ,condition ,true (progn ,@false)))

;;; ********************************
;;; misc ***************************
;;; ********************************
(defun eqmemb (item list &key (test #'equal))
  "Checks whether ITEM is either equal to or a member of LIST."
  (if (listp list)
      (member item list :test test)
      (funcall test item list)))

#-LISPM
(defun neq (x y)
  "not eq"
  (not (eq x y)))

(defun car-eq (x y)
  "Checks whether Y is eq to the car of X."
  (and (listp x) ; consp?
       (eq (car x) y)))

(defun dremove (item list)
  "Destructive remove which replaces the original list with the list
   that results when ITEM is deleted from LIST."
  ;; This is safe only because of the way delete works.
  (displace list (delete item list :test #'eq)))

(defun displace (list val)
  "Replaces LIST with VAL by destructively modifying the car and cdr of LIST.
   Warning: VAL must not share list structure with LIST or you'll be sorry."
  (when list
    ;; Can't alter NIL.
    (rplaca list (car val))
    (rplacd list (cdr val))))

(defun tailpush (item list)
  "Pushes ITEM onto the tail of LIST. Does not work if the list is null."
  (when list
    (rplacd (last list) (list item))))

(defun explode (symbol)
  (map 'list #'identity (symbol-name symbol)))

(defun implode (list &optional (package *package*))
  (intern (map 'string #'identity list) package))

(defun crush (a b &optional (package *package*))
  (implode (append (explode a) (explode b)) package))

(defun listify-string (string)
  "Turns a string into a list of symbols."
  (let ((eof (gensym))
	(result nil)
	(start 0)
	item)
    (loop
     (multiple-value-setq (item start)
	 (read-from-string string nil eof :start start))
     (when (eq item eof)
       (return result))
     (setq result (nconc result (list item))))))

(defun and-list (list)
  (dolist (item list t)
    (unless item
      (return nil))))

(defun or-list (list)
  (dolist (item list nil)
    (unless item
      (return t))))

(defun lookup (symbol environment)
  (dolist (frame environment)
    (let ((binding (assoc symbol frame)))
      (when binding
	(return (cdr binding))))))


;;; we define a variable to be a symbol of the form ?NAME, i.e., a
;;; symbol whose first character is #\?.
(defun make-variable (x)
  (make-symbol (format nil "?~a" x)))

(defun variablep (item)
  "Returns T if ITEM is a variable, namely a symbol of the form ?NAME,
   whose first character is a question-mark."
  (and (symbolp item)
       (char= (char (symbol-name item) 0)
              #\?)))

(defmacro dofile ((var filename &optional return-form) &body body)
  "Opens the specified file for input, reads successive lines 
   from the file, setting the specified variable <var> to 
   each line. When end of file is reached, the value of <return-form>
   is returned."
  (let ((eof (gensym "EOF"))
	(stream (gensym "STREAM")))
    `(with-open-file (,stream ,filename :direction :input)
       (do ((,var (read-line ,stream nil ,eof)
		  (read-line ,stream nil ,eof)))
	   ((eq ,var ,eof)
	    ,return-form)
	 ,@body))))

#+allegro-v4.1
(add-initialization "lep-init for dofile"
                    '(lep::eval-in-emacs "(put 'dofile 'fi:lisp-indent-hook '(like with-open-file))")
                    '(:lep))

(defun make-plist (keys data &optional (plist '()))
  "Constructs a property list from keys and data (addition to plist)."
  (cond ((and (null data) (null keys))
	 plist)
	((or  (null data) (null keys))
	 (error "The lists of keys and data are of unequal length."))
	(t
	 (list* (car keys)
		(car data)
		(make-plist (cdr keys) (cdr data) plist)))))

#-EXCL (defvar *keyword-package* (find-package 'keyword))
(defun make-keyword (symbol)
  (intern (IF (SYMBOLP SYMBOL)
	      (symbol-name symbol)
	      SYMBOL)
	  #-excl *keyword-package* #+excl excl:*keyword-package*))

;; more additions, 2/26/93 bwm from net

; This is a little collection of interesting examples, utilities, and
; one brain-bender. The file may be loaded into LISP (but only the last
; of the four definitions of transpose will remain).
; Tom Kramer
; kramer@cme.nist.gov
; 2/15/93

;; minor editing, removal of egregious usage of prog (but not some others
;; that still need to be removed :-) and deletions of example code -
;; 2/26/93 miller@cs.rochester.edu

; The symbol "==>" used below means "returns"

;**************************************************************************
; Here are four ways to transpose a matrix represented as a list of lists.
; The second one is perverted and awkward but fun. The last is the best.
; These illustrate interation with "prog" and "do" vs. recursion and the
; use of mapcar.

; (setq example '((A B C D) (E F G H) (I J K L)))

; Then (transpose example) ==> ((A E I) (B F J) (C G K) (D H L)).
; Also, (transpose nil) ==> nil.

;(defun transpose (matrix)
;  (mapcar #'(lambda (x) (prog1 (mapcar #'first matrix)
;			       (setq matrix (mapcar #'rest matrix))))
;	  (first matrix)))
;
;
;(defun transpose (matrix)
;  (prog (result temp)
;	(setq temp (first matrix))
;	loop1
;	(cond (temp
;	       (push nil result)
;	       (setq temp (rest temp))
;	       (go loop1)))
;	loop2
;	(cond ((null temp)
;	       (cond ((null (setq temp (pop matrix)))
;		      (return result)))))
;	(setq result (append (rest result)
;			     (list (append (first result)
;					   (list (pop temp))))))
;	(go loop2)))
;
;
;(defun transpose (matrix)
;  (do ((result (mapcar #'(lambda (x) nil) (first matrix))
;	       (mapcar #'(lambda (x) (append x (list (pop row)))) result))
;       (row (pop matrix) (pop matrix)))
;      ((null row) result)))


(defun transpose (matrix)
  (cond ((first matrix)
	 (cons (mapcar #'first matrix)
	       (transpose (mapcar #'rest matrix))))))

; **************************************************************************

; This is set of functions (fetch, place, putlink, pluck), for handling
; property lists which may be trees. A couple auxiliary functions are
; also defined.

;				 FETCH

; Fetch retrieves values from property lists which may be trees,
; "fetch" will go as far down the tree as you like. Fetch also works
; with the property lists of symbols.

; Example 1 - a disembodied property list
;	(setq jack '(jack toolbox (toolbox tool1 saw tool2 hammer) age 21))
; 	(fetch jack 'toolbox 'tool2) ==> hammer

; Example 2 - a property list of a symbol
;       (setf (symbol-plist 'jill)
;             '(toolbox (toolbox tool1 saw tool2 hammer) age 21))
;       (fetch 'jill 'toolbox 'tool2) ==> hammer

; Fetch takes any number of arguments. The arguments are all evaluated. 
; Each argument should be one level deeper in the tree than the previous one.
; The first argument should evaluate to a disembodied property list or
; a symbol.  All other arguments should evaluate to the names of properties.

; Fetch is perfectly happy with arguments naming properties that don't exist.
; It just returns nil.  For example,
;           (fetch jack 'car 'make) ==> nil
;           (fetch 'jill 'car 'make) ==> nil

; Fetch can deal with a combination of disembodied property lists and
; property lists of symbols.

; Example 3
; suppose we give "hammer" some properties
; (setf (symbol-plist 'hammer) '(color brown weight 2))
; now, although jack and the property list of jill have not changed,
; (fetch jack 'toolbox 'tool2 'weight) ==> 2
; (fetch 'jill 'toolbox 'tool2 'weight) ==> 2

(defun fetch (item &rest argl)
  (loop
    (cond ((null argl) (return item)))
    (setq item (cond ((listp item)
                      (list-get item (pop argl)))
                     (t
                      (get item (pop argl)))))))

(defun list-get (liz indicator)
  (cond ((not (typep liz 'list))
	 (error "Non-list first argument to list-get."))
	((not (or (symbolp indicator) (integerp indicator)))
	 (error "Non-symbol, non-integer indicator in list-get"))
	(t
	 (list-get-aux (cdr liz) indicator))))

(defun list-get-aux (liz indicator)
  (loop
    (cond ((null liz) (return nil))
          ((eq (car liz) indicator) (return (cadr liz))))
    (setq liz (cddr liz))))

;                             PLACE

; Place is used to put values into disembodied property lists or to
; put values in the property lists of symbols.
; Place goes along a path specified by the arguments through as many levels
; of the list as you like.

; Example; If jack is as set above
;    (place jack 'toolbox 'tool2 'rasp)
; results in jack being changed:
;    jack ==> (jack toolbox (toolbox tool1 saw tool2 rasp) age 21)

; Place takes three or more arguments. The arguments are all evaluated.
; Each argument should be one level deeper in the property list than
; the previous one.  Each argument but the first and last should evaluate to
; a symbol or integer which is a property.  The first argument may evaluate
; to a symbol or a list.

; Any list that place has to deal with must have an odd number of
; members.  New properties go after the first element.  The function does
; NOT check for an odd number of members, and it will do things you
; probably don't want done to a list with an even number of members.

; Place checks that there are no null arguments (except possibly
; the last one), and that there are at least three arguments.

; If any check fails, the function causes a break in the program requiring
; operator intervention and prints an appropriate error message.

; Place adds to the property list in an appropriate way,
; either replacing a branch or an end node, or constructing and adding a
; new branch.  The auxiliary function "listup" is called to build a new
; branch or select a new end node.

; Note that place adds a COPY of its last argument to the property
; list of its first argument.  This ensures that if the last argument is
; changed later, the property list will not be changed.  This also means
; that "place" cannot be used to link existing lists together.  The function
; "putlink" has been written to do that.  It is identical to "place"
; except that the copying feature has been removed.  The documentation for
; putlink gives some examples of how the action of putlink differs from that
; of place.

(defun place (liz &rest argl)
  (prog (item prop value)
	(cond ((member nil (butlast argl))
	       (error "Null middle argument to place."))
	      (( < (length argl) 2)
	       (error "Too few arguments to place.")))
	(setq item argl)

	checkloop
	(cond ((not (or (symbolp (car item)) (integerp (car item))))
	       (error "Non-symbol, non-integer middle argument to place."))
	      ((cddr item)
	       (setq item (cdr item))
	       (go checkloop)))

	(cond ((symbolp liz)   ; make liz an even list
	       (setq value (symbol-plist liz))
	       (cond ((null value) ; if no plist, make one
		      (setf (symbol-plist liz)
			    (list (car argl) (listup (copy-tree argl))))
		      (return (cadr (symbol-plist liz))))
		     (t
		      (setq liz (cons 'dummy value)))))
	      ((and liz (listp liz)))
	      (t
	       (error "First argument to place not a list or symbol.")))

	outer-loop               ; new level of tree
	(setq value (cdr liz)
	      item  value
	      prop (pop argl))
	(cond ((null value)
	       (rplacd liz (list prop
				 (listup (cons prop (copy-tree argl)))))
	       (return (caddr liz))))

	inner-loop               ; check this level of the tree
	(cond ((null item)
	       (insert-first value prop
			     (listup (cons prop (copy-tree argl))))
	       (return (cadr value)))
	      ((eq prop (car item))
	       (setq value (cadr item))
	       (cond ((or (atom value) (null (cdr argl)))
		      (rplaca (cdr item)
			      (listup (cons prop (copy-tree argl))))
		      (return (cadr item)))
		     (t
		      (setq liz value)
		      (go outer-loop))))
	      (t
	       (setq item (cddr item))
	       (go inner-loop)))))


(defun listup (liz)
  (cond ((null (cdr liz)) nil)
	((equal (length liz) 2) (cadr liz))
	((equal (length liz) 3) liz)
	(t (list (car liz) (cadr liz) (listup (cdr liz))))))

(defun insert-first (liz prop value)
  (rplacd liz (nconc (list value (car liz)) (cdr liz)))
  (rplaca liz prop))

; Suppose the property list of vw is set up before each example to be:
; (setf (symbol-plist 'vw) '(name vw type (type subtype sedan color green)))

; Example 1. This is a simple replacement of a leaf node.
;      (place 'vw 'type 'subtype 'convertible) ==> convertible
;      Then (symbol-plist 'vw) ==>
;                      (name vw type (type subtype convertible color green))

; Example 2. In this example we build an entirely new branch.
;      (place 'vw 'windows 'rear 'shading 'yes) ==>
;                      (windows rear (rear shading yes))
;      Then (symbol-plist 'vw) ==>(windows (windows rear (rear shading yes))
;			  name vw type (type subtype sedan color green))

; Example 3. The last argument to place may be a list.
;      (place 'vw 'type 'subtype '(subtype a b)) ==> (subtype a b)
;      Then (symbol-plist 'vw) ==>
;                    (name vw type (type subtype (subtype a b) color green)

; Example 4. The last argument to place may be nil.
;      (place 'vw 'type nil) ==> nil
;      Then (symbol-plist 'vw) ==> (name vw type nil)

; Example 5. The following call to place is bad because of the "nil" argument
;     which is not the last argument.
;     (place 'vw 'type nil 'oops) ==> Error: Null argument to place. 

; Example 6. The following call to place is bad because of too few arguments.
;     (place 'vw 'oops) ==> Error: Too few arguments to "place". 

; Example 7. The following call to place is bad because of the list argument
;      which is not the last argument.      
;      (place 'vw 'name '(name first joe) 'second 'schmidt) ==>
; 		Error: Non-symbol, non-integer middle argument to "place".


;                           PUTLINK

(defun putlink (liz &rest argl)
  (prog (item prop value)
	(cond ((member nil (butlast argl))
	       (error "Null middle argument to putlink."))
	      (( < (length argl) 2)
	       (error "Too few arguments to putlink.")))
	(setq item argl)

	checkloop
	(cond ((not (or (symbolp (car item)) (integerp (car item))))
	       (error "Non-symbol, non-integer middle argument to putlink."))
	      ((cddr item)
	       (setq item (cdr item))
	       (go checkloop)))

	(cond ((symbolp liz)   ; make liz an even list
	       (setq value (symbol-plist liz))
	       (cond ((null value) ; if no plist, make one
		      (setf (symbol-plist liz)
			    (list (car argl) (listup argl)))
		      (return (cadr (symbol-plist liz))))
		     (t
		      (setq liz (cons 'dummy value)))))
	      ((and liz (listp liz)))
	      (t
	       (error "First argument to putlink not a list or symbol.")))

	outer-loop               ; new level of tree
	(setq value (cdr liz)
	      item  value
	      prop (pop argl))
	(cond ((null value)
	       (rplacd liz (list prop (listup (cons prop argl))))
	       (return (caddr liz))))

	inner-loop               ; check this level of the tree
	(cond ((null item)
	       (insert-first value prop (listup (cons prop argl)))
	       (return (cadr value)))
	      ((eq prop (car item))
	       (setq value (cadr item))
	       (cond ((or (atom value) (null (cdr argl)))
		      (rplaca (cdr item) (listup (cons prop argl)))
		      (return (cadr item)))
		     (t
		      (setq liz value)
		      (go outer-loop))))
	      (t
	       (setq item (cddr item))
	       (go inner-loop)))))

; Example 1.  What happens if value of the last argument is changed later.

; a. If place is used
;     (setq liz '(a b c)) ==> (a b c)
;     (setq newend '(e f g)) ==> (e f g)
;     (place liz 'b newend) ==> (e f g)
;     liz ==> (a b (e f g))
;     (rplaca newend 'x) ==> (x f g)
;     newend ==> (x f g)
;     liz ==> (a b (e f g))

; b. If putlink is used
;      (setq liz '(a b c)) ==> (a b c) 
;      (setq newend '(e f g)) ==> (e f g) 
;      (putlink  liz 'b newend) ==> (e f g) 
;      liz ==> (a b (e f g)) 
;      (rplaca newend 'x) ==> (x f g)
;      newend ==> (x f g)
;      liz ==> (a b (x f g))

; Example 2.  Place will not crosslink a property list.  Putlink will.
; In this case, when putlink is used to crosslink the list, a change in
; a value in one of the linked branches changes the corresponding value
; in another branch.

; a. (setq liz '(liz prop1 (prop1 a (a b c)) prop2 nil)) ==>
;				(liz prop1 (prop1 a (a b c)) prop2 nil)
;    (place liz 'prop2 'a (fetch liz 'prop1 'a)) ==> (prop2 a (a b c))
;    liz ==> (liz prop1 (prop1 a (a b c)) prop2 (prop2 a (a b c)))
;    (place liz 'prop1 'a 'b 3) ==> 3
;    liz ==> (liz prop1 (prop1 a (a b 3)) prop2 (prop2 a (a b c)))

; b. (setq liz '(liz prop1 (prop1 a (a b c)) prop2 nil)) ==> 
;                                (liz prop1 (prop1 a (a b c)) prop2 nil)
;    (putlink liz 'prop2 'a (fetch liz 'prop1 'a)) ==> (prop2 a (a b c))
;    liz ==> (liz prop1 (prop1 a (a b c)) prop2 (prop2 a (a b c)))
;    (place liz 'prop1 'a 'b '3) ==> 3
;    liz ==> (liz prop1 (prop1 a (a b 3)) prop2 (prop2 a (a b 3)))

; Example 3. Place will not make a list which is a member of itself.
; Putlink will. If your LISP does not stop printing at certain depth of
; nesting, part b will print until you stop it.

; a. (setq liz '(a b c)) ==> (a b c)
;    (place liz 'b liz) ==> (a b c)
;    liz ==> (a b (a b c))

; b. (setq liz '(a b c)) ==> (a b c)
;    (putlink liz 'b liz) ==> (a b (a b (a b (a b (a b #)))))
;    liz ==> (a b (a b (a b (a b (a b (a b (a b #)))))))
;    In this example, printing stops at depth 5.

;                         PLUCK

; Pluck removes a property and its value from
; a property list (either the plist of an atom or a disembodied property
; list).  Pluck, however, goes down along a path specified in the arguments
; as far as you like into the list.  In this regard it is like "fetch" and
; "place".  Pluck takes any number of arguments, all of which are evaluated.
; The first argument must evaluate to  either an atom with a property list
; or to a disembodied property list.  The last argument must evaluate to the
; property to be removed.  The middle arguments must evaluate to atoms which
; are property names.

; The value returned by pluck is t if something was plucked and nil if not.

(defun pluck (&rest argl &aux liz)
  (setq liz (apply 'fetch (butlast argl)))
  (cond ((oddp (length liz))              ; be sure the list handed to remf
	 (setq liz (cons (gensym) liz))   ; has an even number of entries
	 (remf liz (car (last argl))))
	((eq (car liz) (car (last argl))) ; need to do this since remf will
	 (rplaca liz (caddr liz))         ; not alter structure if property
	 (rplacd liz (cdddr liz))         ; is at head of list
	 t)
	(t
	 (remf liz (car (last argl))))))  ; if even list and prop in middle

; Example 1. (setq liz '(a b (b f g h k) d e)) ==> (a b (b f g h k) d e)
;            liz ==> (a b (b f g h k) d e)
;            (pluck liz 'b 'h) ==> t
;            liz ==> (a b (b f g) d e)
;            (pluck liz 'b) ==> t
;            liz ==> (a d e)

;****************************************************************************
;                             MAPT

; If val is nil or missing, mapt returns a list of all those elements of
; the list "liz" for which (func element) is non-nil.

; If val is non-nil, mapt returns a list of all values of (func element)
; which are non-nil, in the same order as the elements.

; I don't know how the rest of the world gets along without this function.

(defun mapt (func liz &optional val)
  (cond (val
	 (mapcan #'(lambda (x)
		     (cond ((setq val (funcall func x)) (list val)))) liz))
	(t
	 (mapcan #'(lambda (x)
		     (cond ((funcall func x) (list x)))) liz))))


; Example 1 - (mapt #'numberp '(a 1 4 b c 2.3)) ==> (1 4 2.3)

; Example 2 - (mapt #'(lambda (item) (cond ((numberp item) (1+ item))))
;                    '(a 1 4 b c 2.3) t) ==> (2 5 3.3)

;****************************************************************************
;                             PUSHEND

; This function is like push, except it adds a new-item at the end of
; a list, not the beginning, and the second argument (list-end) is a
; pointer to the last item on the list, not the first.  list-end is
; reset so that it points at the new last item.  To make use of this
; function, some other variable is normally set to point at the front
; of the list.

; Using this function to add items at the end of a long list is much
; faster than nconc'ing the list with a list of the new item, because
; nconc'ing requires traversing the entire list.

(defmacro push-end (new-item list-end)
  `(nconc ,list-end (setq ,list-end (list ,new-item))))

; Example:
; (setq liz '(1 2 3)) => (1 2 3)
; (setq lass (last liz)) => (3)
; (push-end 4 lass) => (3 4)     -- the returned value is not usually used
; liz => (1 2 3 4)
; (push-end 5 lass) => (4 5)
; liz => (1 2 3 4 5)

;;; ********************************
;;; Noting Progress ****************
;;; ********************************
(defmacro noting-progress ((&optional (width 70)) &body body)
  "Every time NOTE-PROGRESS is called within the body of a NOTING-PROGRESS
   macro, it prints out a dot. Every width number of dots it also outputs
   a carriage return."
  (let ((dot-counter (gensym "DOT")))
    `(let ((,dot-counter 0))
       (declare (special ,dot-counter))
       (flet ((note-progress ()
		(incf ,dot-counter)
		(when (> ,dot-counter ,width)
		  (setf ,dot-counter 0)
		  (terpri))
		(princ #\.)))
	 ,@body))))

;;; *EOF*
