;;;
;;;   KNOWBEL knowledge representation system
;;;    
;;;    author: Bryan M. Kramer
;;;    
;;;    
;;; Copyright (c) 1990, 1991 University of Toronto, Toronto, ON
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all copies and
;;; supporting documentation.
;;;
;;; The University of Toronto provides this software "as is" without
;;; express or implied warranty.
;;;

;;;    
;;;    



;? functions for graphical display and manipulation of KNOWBEL knowledge bases


(eval-when (load compile eval)
  (use-package :cw))




#+cw-x
(defmacro with-cursor (window cursor &rest body)
  (let ((v (gensym "OLD-CURSOR"))
	(w (gensym "WINDOW")))
    `(let* ((,w ,window)
	    (,v (if (cw:window-stream-p ,w) (cw:window-stream-mouse-cursor ,w))))
       (unwind-protect
	   (progn
	     (setf (cw:window-stream-mouse-cursor ,w) ,cursor)
	     ,@body)
	 (if (and ,v (not (eq (cw:window-stream-status ,w) :flushed))) (setf (cw:window-stream-mouse-cursor ,w) ,v))
	 )
       )
    )
  )


(defun mygetkbed (&optional (name nil) (init t) (default-history '(1980 +)))
  (activate *output-window*)
  (clear *output-window*)
  (let ((*standard-output* *output-window*))
    (format *output-window* "File name: `~a'~%" name)
    (if (not (loadkb name init default-history))
      (format *output-window* "~&Error: file '~a' not found!~%" name)
      (format *output-window* "~&Loaded~%")
      )
    )
  (deactivate *input-window*)
  )




(def-cond-struct (graph-link (:conc-name gl-))
  dest style
  )


(def-cond-struct (graph-node (:conc-name gn-))
  x y w h object visited children parents visual-claimer y-distance-to-farthest-son style fn pfn
  )

(def-cond-struct (style (:conc-name style-))
  font icon color box
  )

(def-cond-struct (line-style)
  dashing color width
  )



(defmacro style-box-width (style)
  `(doloop (w (style-box ,style)) :vars ((sum 0 (+ sum w))) :result sum)
  )





(defvar *minimum-x-distance* 15)
(defvar *minimum-y-distance* 4)
(defvar *max-object-display* 400)




(defun gross-token-level (token belief)
  (let ((level (tok-inst-level token)))
    (if level
	level
      (dofind (parent-prop (tok-parents token))
	      (and (belief-satisfies belief (prop-belief parent-prop))
		   (gross-token-level (prop-dest parent-prop) belief)
		   )
	      )))
  )


(defvar *meta-class-style* (make-style :font *font1* :box (list 2 1 2 1) :color black))
(defvar *meta-isa-link-style* (make-line-style :dashing nil :color blue :width 2))
(defvar *class-style* (make-style :font *font1* :box (list 2 1) :color black))
(defvar *token-style* (make-style :font *font1* :box (list 3 1) :color turquoise))
(defvar *time-node-style* (make-style :font *font1* :box (list 3 1) :color turquoise))
(defvar *clause-node-style* (make-style :font *font1* :box (list 1 1) :color green))
(defvar *meta-explanation-link-style* (make-line-style :dashing nil :color red :width 2))
(defvar *copy-style* (make-style :font *font1* :box (list 1 1 1 1) :color blue))
(defvar *error-style* (make-style :font *font1* :box (list 4 1) :color red))
(defvar *isa-link-style* (make-line-style :dashing nil :color blue :width 2))
(defvar *inst-link-style* (make-line-style :dashing nil :color red :width 2))
(defvar *attr-link-style* (make-line-style :dashing nil :color green :width 2))
(defvar *val-link-style* (make-line-style :dashing '(5 5) :color green :width 2))
(defvar *time-link-int-style* (make-line-style :dashing nil :color green :width 2))
(defvar *time-link-style* (make-line-style :dashing '(5 5) :color green :width 2))
(defvar *time-same-style* (make-line-style :dashing '(5 5) :color blue :width 2))
(defvar *time-conventional-link-style* (make-line-style :dashing '(5 5) :color black :width 2))


(defun gn-layout-x (node style)
  (-- (gn-visited node))
  (when (>= 0 (gn-visited node))
    (let ((tree_x (round (* 0.1 (gn-y-distance-to-farthest-son node))))
	  (extra_x (round (* 0.17 (gn-h node) (length (gn-children node))))))
      (doloop (child-link (gn-children node))
	(let* ((child (gl-dest child-link))
	       (min_x (cond ((eq style :tree) tree_x)
			    ((eq node (gn-visual-claimer child)) 0)
			    (t tree_x))))
	  (setf (gn-x child)
	    (max (gn-x child)
		 (+ (gn-x node) (gn-w node) (max *minimum-x-distance* min_x extra_x))))
	
	  (gn-layout-x child style)
	  )
	))
    ))

 
  


(defun gn-layout-y (node level visited_mark)
  (if (not (eql visited_mark (gn-visited node)))
    (let ((new-level level))
      (doloop (child-link (gn-children node))
	(let ((child (gl-dest child-link)))
	  (when (null (gn-visual-claimer child))
	    (setf (gn-visual-claimer child) node))
	  ))
      (setf (gn-visited node) visited_mark)
      (doloop (child-link (gn-children node))
	(setf new-level (gn-layout-y (gl-dest child-link) new-level visited_mark))
	)
      (let ((count 0)
	    (sum 0)
	    (first-height (if (gn-children node) (gn-h (gl-dest (car (gn-children node))))))
	    (average new-level))
	(doloop (child-link (gn-children node))
	  (let ((child (gl-dest child-link)))
	    (when (eql node (gn-visual-claimer child))
	      (++ count)
	      (++ sum (gn-y child))
	      ))
	  )
	(when (> count 0)
	  (setf average (round (- (/ sum count) (/ (- first-height (gn-h node)) 2)))))
	(setf (gn-y node) average)
	(when (= count 0)
	  (++ new-level (gn-h node)))
	(setf new-level (max new-level (+ (gn-y node) (gn-h node) 5)))
	)    
      new-level)
    level)
  )


  
(defun gn-layout (roots nodes style)
  (let ((y 0)
	(key (gensym)))
    (doloop (node roots)
      (setf y (gn-layout-y node y key))
      )
    (doloop (node nodes)
      (setf (gn-visited node) (length (gn-parents node)))
      (setf (gn-x node) 0)
      (let ((farthest-y 0))
	(doloop (child-link (gn-children node))
	  (setf farthest-y (max farthest-y (abs (- (gn-y node) (gn-y (gl-dest child-link))))))
	  )
	(setf (gn-y-distance-to-farthest-son node) farthest-y))
      )
    (doloop (node roots)
      (gn-layout-x node style)
      )
    )
  )


(defun check-clos-object-p (object) nil)
(defun get-clos-object-name (object use-pfn) nil)


(defun gn-object-name (node &optional (use-pfn t))
  (cond ((check-clos-object-p (gn-object node)) (get-clos-object-name (gn-object node) use-pfn))
	((and use-pfn (gn-pfn node)) (funcall (gn-pfn node) node))
	((kb-attr-p (gn-object node))
	 (let* ((*print-pretty* nil)
		(val (format nil "~a:~a:~a"
			     (recons-attr-class (gn-object node))
			     (attr-label (gn-object node))
			     (if (not (kb-token-p (attr-value (gn-object node))))
			       (normal-clause (attr-value (gn-object node)))
			       ""))))
	   (if (> (length val) 100)
	     (subseq val 0 58)
	    val)))
	((kb-token-p (gn-object node)) (string (tok-name (gn-object node))))
	((time-point-p (gn-object node))
	 (let ((interval (point-interval (gn-object node))))
	   (if (kb-token-p (time-not-shared interval))
	     (normal-clause (time-not-shared interval))
	     (normal-clause (gn-object node)))
	   ))
	((numberp (gn-object node)) (pretty-time (gn-object node)))
	((active-clause-p (gn-object node))
	 (let* ((*print-pretty* nil)
		(val (format nil "~a" (normal-clause (gn-object node) 6 3))))
	   (if (> (length val) 40)
	     (subseq val 0 38)
	    val)))
	(t (let* ((*print-pretty* nil)
		  (val (format nil "~a" (normal-clause (gn-object node) 6 3))))
	     (if (> (length val) 40)
	       (subseq val 0 38)
	      val)))
	))



(defun gn-size (node)
  (let* ((name (gn-object-name node))
	 (style (gn-style node))
	 (factor (if (position #\newline name) 2 1))
	 (w (floor (/ (font-string-width (style-font style) name) factor)))
	 (lh (* factor (font-character-height (style-font style))))
	 (box (* 2 (doloop (w (style-box style)) :vars ((sum 0 (+ sum w))) :result sum)))
	 )
    (setf (gn-w node) (+ box w))
    (setf (gn-h node) (+ box lh))
    )
  )


(defmacro find-node (object node-list)
  `(mlet ((object ,object))
     (doloop (node ,node-list)
      :when (eql (gn-object node) object)
      :return node
       )
     ))


(defun node-style (object belief)
  (cond ((kb-token-p object)
	 (case (gross-token-level object belief)
	       ((0 nil) (values *token-style* *isa-link-style* *inst-link-style*))
	       (1 (values *class-style* *isa-link-style* *inst-link-style*))
	       (t (values *meta-class-style* *meta-isa-link-style* *inst-link-style*))
	       ))
	(t (values *error-style* *isa-link-style* *inst-link-style*)))
  )
	   



(defun gn-link-nodes (parent child link-style)
  (push (make-graph-link :dest child :style link-style) (gn-children parent))
  (push parent (gn-parents child))
  )


(defun build-kb-graph-aux (object nodes limit depth belief inst-depth attributes)
  (let ((found (find-node object (car nodes))))
    (if found
     found
      (multiple-value-bind (style link-style inst-link-style) (node-style object belief)
	(let ((new-node (make-graph-node :object object :style style)))
	  (tconc nodes new-node)
	  (setf (car limit) (- (car limit) 1))
	  (when (or (null depth) (> depth 1))
	    (doloop (child-link (tok-children object))
	      (when (and (> (car limit) 0) (belief-satisfies belief (prop-belief child-link)))
		(let ((child (build-kb-graph-aux (prop-src child-link) nodes limit
						 (if (null depth) nil (- depth 1)) belief inst-depth attributes)))
		  (gn-cycle-link new-node child link-style nodes)
		  )))
	    )
	  (when (or (null inst-depth) (> inst-depth 1))
	    (doloop (child-link (tok-instances object))
	      (when (and (> (car limit) 0) (belief-satisfies belief (prop-belief child-link)))
		(let ((child (build-kb-graph-aux (prop-src child-link) nodes limit depth  belief
						 (if (null inst-depth) nil (- inst-depth 1)) attributes)))
		  (gn-cycle-link new-node child inst-link-style nodes)
		  )))
	    )
	  (when (and (numberp attributes) (> attributes 0))
	    (doloop (attr-link (lookup-index-cdr (tok-attrs object) :all))
	      (when (and (> (car limit) 0) (belief-satisfies belief (attr-belief attr-link)))
		(let ((attr (build-kb-graph-aux attr-link nodes limit depth  belief inst-depth (- attributes 1))))
		  (gn-link-nodes new-node attr *attr-link-style*)
		  (when (kb-token-p (attr-value attr-link))
		    (let ((val (build-kb-graph-aux (attr-value attr-link) nodes limit depth  belief inst-depth (- attributes 1))))
		      (if (gn-reachable attr (list val)) ; a cycle!
			(let ((copy-node (make-graph-node :object (gn-object val) :style *copy-style*)))
			  (tconc nodes copy-node)
			  (gn-link-nodes attr copy-node *val-link-style*))
			(gn-link-nodes attr val *val-link-style*))
		      )
		    )))
	      )
	    )
	  new-node)
	)
      ))
  )


(defun build-kb-graph (roots depth belief inst-depth attributes limit &optional p-root-nodes p-nodes)
  (let ((root-nodes (or p-root-nodes (tconc)))
	(nodes (or p-nodes (tconc))))
    (doloop (root roots)
      (let ((new-root (build-kb-graph-aux root nodes limit depth belief inst-depth attributes)))
	(tconc root-nodes new-root)))
    (doloop (node (car nodes))
     :when (not (gn-reachable node (car root-nodes)))
      (tconc root-nodes node))
    (values nodes root-nodes)
    )
  )


(defun gn-linked (node1 node2)
  (or (member node1 (gn-parents node2))
      (member node2 (gn-parents node1)))
  )


(defun gn-reachable (node from-list)
  (cond ((member node from-list) t)
	(t (doloop (p (gn-parents node))
	     :when (gn-reachable p from-list)
	     :return t)))
  )
  
(defun build-relationships-graph (objects belief)
  (let ((nodes (tconc))
	(roots (tconc))
	(extra-nodes (tconc)))
    (doloop (object objects)
      (when (not (find-node object (car nodes)))
	(multiple-value-bind (style link-style inst-link-style) (node-style object belief)
	  (let ((new-node (make-graph-node :object object :style style)))
	    (tconc nodes new-node))))
      )
    (doloop (node (car nodes))
      (let ((object (gn-object node)))
	(multiple-value-bind (style link-style inst-link-style) (node-style object belief)
	  (doloop (prop (tok-parents object))
	    (when (belief-satisfies belief (prop-belief prop))
	      (let ((parent (find-node (prop-dest prop) (car nodes))))
		(when (and parent (not (eq parent node)))
		  (if (gn-reachable parent (list node)) ; a cycle!
		    (let ((new-node (make-graph-node :object (gn-object node) :style *copy-style*)))
		      (tconc extra-nodes new-node)
		      (gn-link-nodes parent new-node link-style))
		    (gn-link-nodes parent node link-style))
		  )
		)
	      )
	    )
	  (doloop (prop (tok-inst-of object))
	    (when (belief-satisfies belief (prop-belief prop))
	      (let ((parent (find-node (prop-dest prop) (car nodes))))
		(when (and parent (not (eq parent node)))
		  (if (gn-reachable parent (list node)) ; a cycle!
		    (let ((new-node (make-graph-node :object (gn-object node) :style *copy-style*)))
		      (tconc extra-nodes new-node)
		      (gn-link-nodes parent new-node inst-link-style))
		    (gn-link-nodes parent node inst-link-style))
		  )
		)
	      )
	    )
	  )))
    (lconc nodes (car extra-nodes))
    (doloop (node (car nodes))
      (when (null (gn-parents node))
	(tconc roots node))
      )
    (when (null (car roots))
      (doloop (node (car nodes))
       :when (gn-children node) :return (tconc roots (caar nodes))))
    (doloop (node (car nodes))
     :when (not (gn-reachable node (car roots)))
      (tconc roots node))
    (values nodes roots)
    ))


(defun gn-cycle-link (src dest style nodes)
  (if (gn-reachable src (list dest))
    (let ((new-node (make-graph-node :object (gn-object dest) :style *copy-style*)))
      (tconc nodes new-node)
      (gn-link-nodes src new-node style)
      )
    (gn-link-nodes src dest style)
    )
  )

(defmacro time-link-style (p1 p2)
  `(if (and (time-point-p ,p1) (time-int-p (point-interval ,p2)) (eq (point-interval ,p1) (point-interval ,p2)))
     *time-link-int-style*
     *time-link-style*
     )
  )


(defun build-time-graph-sub (point direction belief nodes init)
  (if (not (find-node point (car nodes)))
    (let ((new-node (make-graph-node :object point :style *time-node-style*)))
      (tconc nodes new-node)
      (when (time-point-p point)
	(doloop (link (if (or (null direction) init) (point-preceded-by point) nil))
	 :when (belief-satisfies belief (point-link-belief link))
	  (let ((sub-node (build-time-graph-sub (point-link-source link) nil belief nodes t)))
	    (when (not (gn-linked new-node sub-node))
	      (gn-cycle-link sub-node new-node (time-link-style (point-link-source link) point) nodes))
	    )
	  )
	(doloop (link (if (or init direction) (point-precedes point) nil))
	 :when (belief-satisfies belief (point-link-belief link))
	  (let ((sub-node (build-time-graph-sub (point-link-dest link) t belief nodes t)))
	    (when (not (gn-linked new-node sub-node))
	      (gn-cycle-link new-node sub-node (time-link-style (point-link-dest link) point) nodes))
	    )
	  )
	(doloop (link (car (point-same-as point)))
	 :when (belief-satisfies belief (point-link-belief link))
	  (let ((same-point (if (eq (point-link-dest link) point) (point-link-source link) (point-link-dest link))))
	    (let ((sub-node (build-time-graph-sub same-point direction belief nodes t)))
	      (when (not (gn-linked new-node sub-node))
		(gn-cycle-link new-node sub-node *time-same-style* nodes))
	      )
	    )
	  )
	)
      new-node)
    (find-node point (car nodes))
    )
  )

(defun time-graph-sort (a b)
  (if (eql (gn-object a) (gn-object b))
    (not (gn-reachable a (list b)))
    (not (time> (gn-object a) (gn-object b)))
    )
  )

(defun build-time-graph-conventional (nodes)
  (let ((conv-nodes (doloop (node (car nodes)) :when (not (time-point-p (gn-object node))) :collect node)))
    (doloop :iter (tail (sort conv-nodes #'time-graph-sort) (cdr tail))
     :while (cdr tail)
      (gn-cycle-link (car tail) (cadr tail) *time-conventional-link-style* nodes)
      )
    )
  )



(defun time-point-button-fn (node r mouse-state mouse-transition)
  (let ((obj (gn-object node)))
    (format t "~%~%~a~%" (normal-clause obj))
    (doloop (link (point-precedes obj))
      (format t "~&precedes ~a ~a" (normal-clause (point-link-dest link)) (normal-clause (point-link-belief link)))
      )
    (doloop (link (point-preceded-by obj))
      (format t "~&preceded by ~a ~a" (normal-clause (point-link-source link)) (normal-clause (point-link-belief link)))
      )
    (doloop (link (car (point-same-as obj)))
      (format t "~&same as ~a ~a"
	      (normal-clause (if (eq (point-link-dest link) obj) (point-link-source link) (point-link-dest link)))
	      (normal-clause (point-link-belief link)))
      )
    )
  )


(defun time-point-pfn (node)
  (cond ((time-point-p (gn-object node))
	 (let ((interval (point-interval (gn-object node))))
	   (if (kb-token-p (time-not-shared interval))
	     (normal-clause (time-not-shared interval))
	     (normal-clause (gn-object node)))
	   ))	
	((numberp (gn-object node)) (pretty-time (gn-object node)))
	(t "????")
	)
  )



(defun build-time-graph (point belief)
  (let ((nodes (tconc))
	(roots (tconc))
	(real-roots (tconc))
	minus-node
	plus-node)
    (if (listp point)
      (doloop (point point) (build-time-graph-sub point t belief nodes :init))
      (build-time-graph-sub point t belief nodes :init)
      )
    (doloop (node (car nodes))
      (setf (gn-fn node) 'time-point-button-fn)
      (setf (gn-pfn node) 'time-point-pfn)
      (when (null (gn-parents node))
	(tconc roots node))
      )
    (setf minus-node (build-time-graph-sub :- t belief nodes t))
    (setf plus-node (build-time-graph-sub :+ t belief nodes t))
    (build-time-graph-conventional nodes)
    (doloop (node (car nodes))
      (when (null (gn-parents node))
	(tconc roots node))
      )
    (when (not (member minus-node (car roots))) (tconc roots minus-node))
    (doloop (node (car nodes))
     :when (not (gn-reachable node (car roots)))
      (tconc roots node)
     :when (and (not (eq node plus-node)) (null (gn-children node)))
      (gn-cycle-link node plus-node *time-link-style* nodes))
    (doloop (node (car roots))
     :when (and (not (eq node minus-node)) (null (gn-parents node)))
      (gn-cycle-link minus-node node *time-link-style* nodes))
    (doloop (node (car nodes))
      (when (null (gn-parents node))
	(tconc real-roots node))
      )
    (values nodes real-roots)
    ))
   

(defun build-explanation-graph-sub (clause nodes depth)
  (if (> depth 0)
    (if (not (find-node clause (car nodes)))
      (let ((new-node (make-graph-node :object clause :style *clause-node-style*)))
	(tconc nodes new-node)
	(when (active-clause-p clause)
	  (doloop (just (ac-justifications clause))
	    (if (consp just)
	      (let ((style (if (eq (car just) :meta) *meta-explanation-link-style* *time-link-style*)))
		(doloop (sjust (if (eq (car just) :meta) (cdr just) just))
		 :vars ((sub-node nil (build-explanation-graph-sub sjust nodes (- depth 1))))
		  (when (eq sjust :meta) (setf style *meta-explanation-link-style*))
		  (when sub-node (gn-cycle-link new-node sub-node style nodes)))
		)
	      (let ((sub-node (build-explanation-graph-sub just nodes (- depth 1))))
		(when sub-node (gn-cycle-link new-node sub-node *time-link-style* nodes))
		)
	      )
	    )
	  )
	new-node)
      (find-node clause (car nodes))
      )
   nil)
  )

(defun build-explanation-graph (clause depth)
  (let ((nodes (tconc))
	(roots (tconc)))
    (build-explanation-graph-sub clause nodes depth)
    (doloop (node (car nodes))
      (when (null (gn-parents node))
	(tconc roots node))
      )
    (doloop (node (car nodes))
     :when (not (gn-reachable node (car roots)))
      (tconc roots node))
    (values nodes roots)
    ))



(defun build-theory-graph (theory)
  (let ((nodes (tconc))
	(roots (tconc)))
    (doloop (object (theory-clauses theory))
      (when (not (find-node object (car nodes)))
	(let ((new-node (make-graph-node :object object :style *clause-node-style*)))
	  (tconc nodes new-node)))
      )
    (doloop (node (car nodes))
      (let ((object (gn-object node)))
	(doloop (just (ac-justifications object))
	  (let ((just-node (find-node just (car nodes))))
	    (when (and just-node (not (eq just-node node)))
	      (gn-cycle-link node just-node *time-link-style* nodes)
	      )
	    )
	  )))
    (doloop (node (car nodes))
      (when (null (gn-parents node))
	(tconc roots node))
      )
    (when (null (car roots))
      (doloop (node (car nodes))
       :when (gn-children node) :return (tconc roots (caar nodes))))
    (doloop (node (car nodes))
     :when (not (gn-reachable node (car roots)))
      (tconc roots node))
    (values nodes roots)
    )
  )
 
	
	     
(defun display-collect-ancestors (object belief)
  (let* ((parents (lconc nil (tok-inst-of object)))
	 (new (car parents)))
    (while new
	   (let ((ancest (tok-inst-of (prop-dest (car new)))))
	     (while ancest
		    (if (not (member (car ancest) (car parents)))
			(tconc parents (car ancest)))
		    (setf ancest (cdr ancest))
		    )
	     )
	   (setq new (cdr new))
	   )
    (cons object (doloop (prop (car parents)) :collect (prop-dest prop)))
    )
  )



(defun display-collect-parents (object belief)
  (let* ((parents (lconc nil (tok-parents object)))
	 (new (car parents)))
    (while new
	   (let ((supers (tok-parents (prop-dest (car new)))))
	     (while supers
		    (if (not (member (car supers) (car parents)))
			(tconc parents (car supers)))
		    (setf supers (cdr supers))
		    )
	     )
	   (setq new (cdr new))
	   )
    (cons object (doloop (prop (car parents)) :collect (prop-dest prop)))
    )
  )


;;  function name: graph-button-down
;;  side effects: if the graph button is selected and this 
;;  function gets called then pop up the menu with choices
;;  for displaying graphs (and call the function that is chosen).


(comment  (let* ((isa-list (make-pop-up-menu
		    `(("Is-a Depth 4" (4 0))
		      ("No Depth Limit" (nil 0))))
	 (graph-list (make-pop-up-menu
		      `(("All links" gr-all)
			("Isa links -> " ,isa-list nil)
			("Instance links" gr-ins)))))
    (multiple-value-bind (choice item)
	(pop-up-menu-choose graph-list)
      (when item (gr-isa (car choice) (cadr choice)))
      (dbgn-print 'item item 'ch choice 'gl graph-list 'il isa-list)
      )
    )
  ))

(defvar *graph-menu* nil)


(defun gr-isa (depth inst-depth attr-depth &optional name)
  (clear *input-window*)
  (reset *input-window*)
  (write-title *input-window* "Isa links")
  (activate *input-window*)
  (format *input-window* 
	  "~% Name of object ?~% ")
  (let* ((ans (or name (multi-read *input-window*)))
	 (tok (lookup-type ans)))
    (if tok
      (display-kb tok depth inst-depth attr-depth)
      (format *output-window* "Bad token name '~a'~%" ans)
      )
    (deactivate *input-window*)
    )
  )

(defun graph-button-down ()
  ;; the menu list that is selected if the graph button is selected
  (when (null *graph-menu*)
    (setf *graph-menu* (make-pop-up-menu
			`(("IS-A Depth 3" (3 0 0))
			  ("IS-A No Depth Limit" (nil 0 0))
			  ("All Depth 3" (3 2 0))
			  ("All no limit" (nil 10 0))
			  ("Instance" (0 10 0))
			  ("Attributes (3)" (0 0 3)))))
    )
  (multiple-value-bind (choice item)
      (pop-up-menu-choose *graph-menu*)
    (when item (gr-isa (car choice) (cadr choice) (caddr choice)))
    )
  )



(defun gn-intersect-p (node region)
  (let ((node-right (+ (gn-x node) (gn-w node)))
	(node-top (+ (gn-y node) (gn-h node))))
    (or (and (>= (region-right region) (gn-x node))
	     (<= (region-left region) node-right)
	     (>= (region-top region) (gn-y node))
	     (<= (region-bottom region) node-top))
	(and (>= node-right (region-left region))
	     (<= (gn-x node) (region-right region))
	     (>= node-top (region-bottom region))
	     (<= (gn-y node) (region-top region))))
    )
  )


(defun box-intersect-p (region x y right top)
  (or (and (>= (region-right region) x)
	   (<= (region-left region) right)
	   (>= (region-top region) y)
	   (<= (region-bottom region) top))
      (and (>= right (region-left region))
	   (<= x (region-right region))
	   (>= top (region-bottom region))
	   (<= y (region-top region))))
  )

(defun gn-draw (win node)
  (let ((s (gn-object-name node))
	(border-w (style-box-width (gn-style node))) )
    (setf (window-stream-x-position win) (+ border-w (gn-x node)))
    (setf (window-stream-y-position win) (+ border-w (gn-y node) (font-baseline (style-font (gn-style node)))))
    (setf (window-stream-font win) (style-font (gn-style node)))
    (format win s)
    (doloop (lw (style-box (gn-style node)))
     :vars ((draw t)
	    (x (gn-x node))
	    (y (gn-y node))
	    (w (- (gn-w node) 1))
	    (h (- (gn-h node) 1)))
      (when (and draw (> lw 0))
	(draw-rectangle-xy win x y w h :brush-width lw :color (style-color (gn-style node)))
	)
      (setf x (+ x lw))
      (setf y (+ y lw))
      (setf h (- h lw lw))
      (setf w (- w lw lw))
      (setf draw (not draw))
      ))
  )


(defun gn-mouse-redraw (region &rest args)
  (declare (ignore args))
  (when (and (active-region-p region) (logtest (mouse-state-button-state (get-mouse-state)) *left-button-down*))
    (let ((w (active-region-parent region)))
      (with-graphics-batching
       (clear-area w (make-region :bottom (active-region-bottom region)
				  :left (active-region-left region)
				  :height (active-region-height region)
				  :width (active-region-width region)))
       (gn-draw w (active-region-get region 'name))
       )
      )
    )
  )


(defun gn-redraw (region &rest args)
  (declare (ignore args))
  (when (and (active-region-p region))
    (let ((w (active-region-parent region)))
      (with-graphics-batching
       (clear-area w (make-region :bottom (active-region-bottom region)
				  :left (active-region-left region)
				  :height (active-region-height region)
				  :width (active-region-width region)))
       (gn-draw w (active-region-get region 'name))
       )
      )
    )
  )


(defun resize-active-region (region node)
  (setf (active-region-bottom region) (gn-y node))
  (setf (active-region-left region) (gn-x node))
  (setf (active-region-height region) (gn-h node))
  (setf (active-region-width region) (gn-w node))
  )
     


(defun gn-mouse-hilight (region &rest args)
  (declare (ignore args))
  (when (and (active-region-p region) (logtest (mouse-state-button-state (get-mouse-state)) *left-button-down*))
    (let ((w (active-region-parent region)))
      (with-graphics-batching
       (draw-rectangle-xy w (+ (active-region-left region) 2)
			  (+ (active-region-bottom region) 2)
			  (- (active-region-width region) 4)
			  (- (active-region-height region) 5)
			  :brush-width 4)
       )
      )
    )
  )


(defun gn-draw-links (win node region)
  (let ((sx (+ (gn-x node) (gn-w node)))
	(sy (+ (gn-y node) (round (/ (gn-h node) 2)))))
    (doloop (child-link (gn-children node))
      (let ((child (gl-dest child-link)))
	(when (or (null region)
		  (box-intersect-p region
				   (min (gn-x node) (gn-x child)) (min (gn-y node) (gn-y child))
				   (max (gn-x node) (gn-x child)) (max (gn-y node) (gn-y child))))
	  (let ((style (gl-style child-link))
		(dx (gn-x child))
		(dy (+ (gn-y child) (round (/ (gn-h child) 2)))))
	    (draw-line-xy win sx sy dx dy
			  :color (line-style-color style)
			  :dashing (line-style-dashing style)
			  :brush-width (line-style-width style))
	    ))
	;(gn-draw-links win child region)
	))
    ))



(defun gd-redraw (win region)
  (let ((old-cursor (window-stream-mouse-cursor win)))
    (unwind-protect
	(progn
	  (setf (window-stream-mouse-cursor win) *mouse-cursor-timer*)
	  (if region
	    (clear-area win region)
	    (clear-rectangle-xy win 0 0 (window-stream-extent-width win) (window-stream-extent-height win))
	    )
	  (when region
	    (setf (region-width region) (region-width region))
	    (setf (region-height region) (region-height region))
	    )
	  (let ((graph (window-stream-get win 'graph)))
	    (let ((ptr (cdr graph)))
	      (doloop :while ptr
		(with-graphics-batching
		 (doloop :for i :from 1 :to 15
		  :when (null ptr) :return t
		  :when (or (null region) (gn-intersect-p (car ptr) region))
		   (gn-draw win (car ptr))
		  :when (null (pop ptr)) :return t
		   )
		 )
		)
	      )
	    (doloop (node (cdr graph))
	      (with-graphics-batching
	       (gn-draw-links win node region)
	       )
	      )
	    )
	  )
      (setf (window-stream-mouse-cursor win) old-cursor)
      )
    )
  )

(defun graph-extent (graph)
  (let ((w 0)
	(h 0))
    (doloop (node (cdr graph))
      (let ((tw (+ (gn-x node) (gn-w node)))
	    (th (+ (gn-y node) (gn-h node))))
	(if (> tw w) (setf w tw))
	(if (> th h) (setf h th))
	))
    (values (+ w 1) (+ 1 h)))
  )

(defun graph-expand-x (graph sx)
  (doloop (node (cdr graph))
    (setf (gn-x node) (floor (* sx (gn-x node))))
    )
  )

(defun graph-expand-y (graph sy)
  (doloop (node (cdr graph))
    (setf (gn-y node) (floor (* sy (gn-y node))))
    )
  )

(defun graph-expand-xy (graph sx sy)
  (doloop (node (cdr graph))
    (setf (gn-x node) (floor (* sx (gn-x node))))
    (setf (gn-y node) (floor (* sy (gn-y node))))
    )
  )




(defun graph-window-activate (win graph)
  (doloop (node (cdr graph))
    (let ((r (make-active-region
	      :left (gn-x node)
	      :bottom (gn-y node)
	      :height (gn-h node)
	      :width (gn-w node)
	      :parent win
	      :mouse-cursor *mouse-cursor-+*
	      :activate-p (not (eq (gn-fn node) :none)))))
      (setf (active-region-get r 'name) node)
      (setf (active-region-mouse-cursor-in r) (list #'gn-mouse-hilight))
      (modify-active-region-method r :mouse-cursor-out :after 'gn-mouse-redraw)
      (modify-active-region-method r :left-button-down :after 'gn-mouse-hilight)
      (modify-active-region-method r :left-button-up :after 'gd-button-down)))
  )


(defun create-graph-window (title graph width height)
  (let ((region (get-box-region (min width 1050) (min height 700))))
    (let ((graph-win (make-window-stream :left (region-left region)
					 :bottom (region-bottom region)
					 :height (region-height region)
					 :width (region-width region)
					 ;; :parent *gd*
					 :title title
					 :activate-p t)))
      (with-cursor graph-win *mouse-cursor-timer*
	(setf (window-stream-truncate-lines-p graph-win) t)
	(setf (window-stream-get graph-win 'graph) graph)
	(setf (window-stream-extent-width graph-win) width)
	(setf (window-stream-extent-height graph-win) height)
	(enable-window-stream-extent-scrolling graph-win)
	(enable-window-stream-event-handling graph-win)
	(modify-window-stream-method graph-win :flush :after #'(lambda (&rest xxx) nil))
	(setf (getf (mp:process-property-list (window-stream-event-handler graph-win)) :browser) graph-win)
	(modify-window-stream-method graph-win :repaint :after 'gd-redraw)
	(graph-window-activate graph-win graph)
	(repaint graph-win)
	)
      graph-win)
    )
  )

    


(defun display-kb (root depth inst-depth attributes &optional command (belief (std-belief)))
  (let ((limit (list *max-object-display*))
	title)
    (multiple-value-bind (nodes roots)
	(case command
	  (:parents (setf title (format nil "Parent graph for ~a" (tok-name root)))
		    (build-relationships-graph (display-collect-parents root belief) belief))
	  (:ancestors (setf title (format nil "Ancestor graph for ~a" (tok-name root)))
		      (build-relationships-graph (display-collect-ancestors root belief) belief))
	  (t (setf title (format nil "Object graph rooted at ~a (~a, ~a, ~a)"
				 (if (listp root)
				   (format nil "~a" (doloop (item root) :collect (tok-name item)))
				   (tok-name root))
				 depth inst-depth attributes))
	     (build-kb-graph (if (listp root) root (list root)) depth belief inst-depth attributes limit))
	  )
      (when (< (car limit) 2)
	(format *output-window* "~%Warning: only ~d objects displayed~%" *max-object-display*))
      (let ((graph (cons (car roots) (car nodes))))
	(doloop (node (cdr graph))
	  (gn-size node)
	  )
	(gn-layout (car graph) (cdr graph) *class-style*)
	(multiple-value-bind (width height) (graph-extent graph)
	  (let ((win (create-graph-window title graph width height)))
	    (setf (window-stream-get win :depth) depth)
	    (setf (window-stream-get win :inst-depth) inst-depth)
	    )
	  ))
      )
    )
  )


(defmacro browse (object &key (subclasses 3) (instances 2) (attributes 0) (belief '(std-belief)))
  `(let ((obj (lookup-type ,object)))
     (if obj
       (display-kb obj ,subclasses ,instances ,attributes nil ,belief)
       (list 'no-such-object object)
       )
     )
  )

(defmacro browse-isa (object &key (subclasses 10) (instances 0) (attributes 0) (belief '(std-belief)))
  `(let* ((obj-spec ,object)
	  (obj (if (listp obj-spec)
		 (doloop (item obj-spec) :collect (lookup-type item))
		 (lookup-type obj-spec))))
     (if obj
       (display-kb obj ,subclasses ,instances ,attributes nil ,belief)
       (list 'no-such-object object)
       )
     )
  )

(defmacro browse-instances (object &key (subclasses 0) (instances 2) (attributes 0) (belief '(std-belief)))
  `(let ((obj (lookup-type ,object)))
     (if obj
       (display-kb obj ,subclasses ,instances ,attributes nil ,belief)
       (list 'no-such-object object)
       )
     )
  )

(defmacro browse-parents (object &key (belief '(std-belief)))
  `(let ((obj (lookup-type ,object)))
     (if obj
       (display-kb obj 0 0 0 :parents ,belief)
       (list 'no-such-object object)
       )
     )
  )

(defmacro browse-ancestors (object &key (belief '(std-belief)))
  `(let ((obj (lookup-type ,object)))
     (if obj
       (display-kb obj 0 0 0 :ancestors ,belief)
       (list 'no-such-object object)
       )
     )
  )





(defun display-time-point (root &optional belief)
  (let ((belief (or belief (std-belief)))
	(title "time graph")
	(classes (list (list :c))))
    (multiple-value-bind (nodes roots)
	(build-time-graph root belief)
      (let ((graph (cons (car roots) (car nodes))))
	(doloop (node (cdr graph))
	  (gn-size node)
	  (setf (gn-y node) 0)
	  )
	(gn-layout (car graph) (cdr graph) *class-style*)
	(graph-expand-xy graph 1.3 1.8)
	(doloop (node (cdr graph))
	 (cond ((not (time-point-p (gn-object node))) (add-bucket classes :c node))
	       (t (add-bucket classes (point-interval (gn-object node)) node)))
	  )
	(doloop (bucket classes)
	  :iter (y 0 (+ y h *minimum-y-distance*))
	  :iter (h 0 0)
	  (doloop (node (cdr bucket))
	    (setf (gn-y node) y)
	    :when (> (gn-h node) h)
	    (setq h (gn-h node))
	    )
	  )
	(multiple-value-bind (width height) (graph-extent graph)
	  (let ((win (create-graph-window title graph width height)))
	    (setf (window-stream-get win :graph) 'time)
	    )
	  ))
      )
    )
  )


(defun display-active-theory (theory)
  (let ((title "active theory"))
    (multiple-value-bind (nodes roots)
	(build-theory-graph theory)
      (let ((graph (cons (car roots) (car nodes))))
	(doloop (node (cdr graph))
	  (gn-size node)
	  )
	(gn-layout (car graph) (cdr graph) *class-style*)
	(multiple-value-bind (width height) (graph-extent graph)
	  (let ((win (create-graph-window title graph width height)))
	    (setf (window-stream-get win :graph) 'time)
	    )
	  ))
      )
    )
  )



(defun display-explanation (clause &optional (depth 3))
  (let ((title "explanation"))
    (multiple-value-bind (nodes roots)
	(build-explanation-graph clause depth)
      (let ((graph (cons (car roots) (car nodes))))
	(doloop (node (cdr graph))
	  (gn-size node)
	  )
	(gn-layout (car graph) (cdr graph) *class-style*)
	(multiple-value-bind (width height) (graph-extent graph)
	  (let ((win (create-graph-window title graph width height)))
	    (setf (window-stream-get win :graph) 'explanation)
	    (setf (window-stream-get win :depth) depth)
	    )
	  ))
      )
    )
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; function name: gd-button-down
;;; pop-up a menu when right button is down and the mouse is within a box
;;;
;;; input: a window stream
;;;        mouse-state and mouse-event (not used)
;;; return: pop up menus when mouse right button is down
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar *gd-button-down-menu* nil)


(defun beep ()
  (xlib::bell xcw::x-display)
  (xcw::x-flush))


(defun object-button-down (r mouse-state mouse-transition)
  (declare (ignore mouse-state mouse-transition))
  (when (null *gd-button-down-menu*)
    (setf *gd-button-down-menu* (make-pop-up-menu '(
						    ("Edit" :edit)
						    ("Subgraph" :subgraph)
						    ("Instances" :instances)
						    ("Parents" :parents)
						    ("Ancestors" :ancestors)
						    ("Attributes" :attributes)
						    )
						  :title "Node operations")))
  (multiple-value-bind (choice item)
      (pop-up-menu-choose *gd-button-down-menu*)
    (when item
      (case choice
	(:edit (gd-edit r))
	(:instances (let ((win (active-region-parent r)))
		     (display-kb (gn-object (active-region-get r 'name))
				 5 5 0)))
	(:subgraph (let ((win (active-region-parent r)))
		     (display-kb (gn-object (active-region-get r 'name))
				 (window-stream-get win :depth)
				 (window-stream-get win :inst-depth)
				 (window-stream-get win :attr-depth))))
	(:parents (display-kb (gn-object (active-region-get r 'name)) nil nil nil :parents))
	(:ancestors (display-kb (gn-object (active-region-get r 'name)) nil nil nil :ancestors))
	(:attributes (display-kb (gn-object (active-region-get r 'name)) 1 0 2 nil))
	))
    ))

(defvar *clause-button-down-menu* nil)

(defun clause-button-down (r mouse-state mouse-transition)
  (declare (ignore mouse-state mouse-transition))
  (when (null *gd-button-down-menu*)
    (setf *clause-button-down-menu* (make-pop-up-menu '(
						    ("Subgraph" :subgraph)
						    ("Detail" :detail)
						    )
						  :title "Node operations")))
  (multiple-value-bind (choice item)
      (pop-up-menu-choose *clause-button-down-menu*)
    (when item
      (case choice
	(:detail (format *output-window* "~a" (normal-clause (gn-object (active-region-get r 'name)))))
	(:subgraph (let ((win (active-region-parent r)))
		     (display-explanation (gn-object (active-region-get r 'name))
					  (window-stream-get win :depth))))
	))
    ))


(defun gd-button-down (r mouse-state mouse-transition)
  (mp:process-run-function "gd-button-down"
			   #'(lambda nil
			       (let* ((node (active-region-get r 'name))
				      (fn (gn-fn node)))
				 (gn-redraw r)
				 (cond (fn
					(cond ((eq fn :none) nil)
					      ((listp fn) (apply (car fn) node r mouse-state mouse-transition (cdr fn)))
					      (t (apply fn node r mouse-state mouse-transition nil))))
				       ((kb-token-p (gn-object node)) (object-button-down r mouse-state mouse-transition))
				       ((active-clause-p (gn-object node)) (clause-button-down r mouse-state mouse-transition))
				       (t (beep))
				       ))
			       )
			   )
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; function name : gd-edit
;;; call the editor (pop up the editor for this object)
;;;
;;; input : region and useless arguments
;;; output : don't care
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun gd-edit (reg &rest args)
  (declare (ignore args))
  (let* ((node (active-region-get reg 'name))
	 (obj (gn-object node)))
    (cond ((kb-token-p obj) (display-object-editor obj))
	  ((prop-object-p obj) (display-object-editor (prop-dest obj)))
	  (t (format *output-window* "~&Can't identify ~a~%" (normal-clause obj))))
    )
  )


(defun recons-attr-class-parents (props attr)
  (doloop (prop props)
   :splice
    (append
     (when (kb-attr-p (prop-dest prop)) (list (attr-label (prop-dest prop))))
     (recons-attr-class-parents (tok-parents (prop-dest prop)) attr)
     ) 
    )
  )

(defun recons-attr-class (attr)
  (doloop (prop (tok-inst-of attr))
   :splice
    (append
     (when (kb-attr-p (prop-dest prop)) (list (attr-label (prop-dest prop))))
     (recons-attr-class-parents (tok-parents (prop-dest prop)) attr)
     )
    )
  )


(defun recons-obj (obj)
  (normal-clause
   (list 'instance
	(doloop (prop (tok-inst-of obj))
	  :collect (list (tok-name (prop-dest prop)) (prop-history prop) (normal-belief (prop-belief prop))))
	'isa
	(doloop (prop (tok-parents obj))
	  :collect (list (tok-name (prop-dest prop)) (prop-history prop) (normal-belief (prop-belief prop))))
	'with
	(doloop (attr (lookup-index-cdr (tok-attrs obj) :all))
	  :collect
	  (list (recons-attr-class attr)
		(list (attr-label attr)
		      (if (kb-token-p (attr-value attr)) (tok-name (attr-value attr)) (attr-value attr))
		      (attr-history attr)
		      (normal-belief (attr-belief attr))
		      )))
	))
  )



(defun w-query (query &optional (belief (std-belief)) (theory *theory*))
  (let ((variables :all)
	(horn t)
	(active-theory (make-theory :kb (theory-kb theory)))
	(answers (tconc)))
    (multiple-value-bind (result state) (backward-inference query theory variables horn active-theory belief)
      (do nil
	  ((null result) t)
	(when (null (clause-expression (ac-clause result)))
	  (tconc answers
		 (doloop (item (ac-collect result))
		   (format nil "~a" (normal-clause item))))
	  )
	(setq result (backward-inference-loop state))
	)
      )
    )
  )




(defun query-button-down ()
  (clear *input-window*)
  (reset *input-window*)
  (write-title *input-window* "Query Input")
  (activate *input-window*)
  (format *input-window* "~%Enter query: ")
  (select *input-window*)
  (let* ((in (multi-read *input-window*))
	 (*standard-output* *output-window*)
	 (query (list 'not (read-from-string in nil :error))))
    (format *output-window* "~&q: ~a~%" query)
    (query1 query)
    )
  (deactivate *input-window*)
  )

(defun build-object-layout (x y node)
  (gn-size node)
  (setf (gn-x node) x)
  (setf (gn-y node) (car y))
  (-- (car y) (gn-h node))
  node
  )

(defun object-of-interest (node)
  (let ((obj (gn-object node)))
    (cond ((kb-token-p obj) obj)
	  ((prop-object-p obj) (prop-dest obj))
	  (t nil))
    )
  )


(defun object-edit-untell (node region)
  (cond ((not (eq (untell (gn-object node)) :error))
	 (gn-size node)
	 (resize-active-region region node)
	 (gn-redraw region))
	(t (beep)))
  )


(defun object-edit-undo (node region)
  (cond ((not (eq (undo-untell (gn-object node)) :error))
	 (gn-size node)
	 (resize-active-region region node)
	 (gn-redraw region))
	(t (beep)))
  )


(defvar *object-edit-button-menu* nil)


(defun object-edit-button-fn (node r mouse-state mouse-transition &rest args)
  (declare (ignore mouse-state mouse-transition))
  (when (null *object-edit-button-menu*)
    (setf *object-edit-button-menu* (make-pop-up-menu '(
							("Untell" :untell)
							("Undo" :undo)
							("Edit" :edit)
							("Subgraph" :subgraph)
							("Parents" :parents)
							("Ancestors" :ancestors)
							("Attributes" :attributes)
							)
						      :title "Node operations")))
  (multiple-value-bind (choice item)
      (pop-up-menu-choose *object-edit-button-menu*)
    (when item
      (case choice
	(:untell (object-edit-untell node r))
	(:undo (object-edit-undo node r))
	(:edit (gd-edit r))
	(:subgraph (let ((win (active-region-parent r)))
		     (display-kb (object-of-interest node) 3 2 0)))
	(:parents (display-kb (object-of-interest node) nil nil nil :parents))
	(:ancestors (display-kb (object-of-interest node) nil nil nil :ancestors))
	(:attributes (display-kb (object-of-interest node) 1 0 2 nil))
	))
    )
  )

(defun obj-ed-name-fn (node)
  (let ((obj (gn-object node)))
    (cond ((kb-attr-p obj)
	   (let* ((*print-pretty* nil)
		  (val (format nil "[~a][~a] ~a: ~a"
			       (normal-clause (attr-history obj))
			       (normal-belief (attr-belief obj))
			       (attr-label obj)
			       (normal-clause (attr-value obj)))))
	     (if (> (length val) 100)
	       (subseq val 0 99)
	      val)))
	  ((prop-object-p obj)
	   (let* ((*print-pretty* nil)
		  (val (format nil "[~a][~a] ~a"
			       (normal-clause (prop-history obj))
			       (normal-belief (prop-belief obj))
			       (tok-name (prop-dest obj)))))
	     (if (> (length val) 100)
	       (subseq val 0 99)
	      val)))
	  (t (gn-object-name node nil))
	  ))
  )


(defvar *obj-keyword-style* (make-style :font *font1* :box (list 0 3) :color turquoise))
(defvar *obj-prop-style* (make-style :font *font1* :box (list 0 3) :color turquoise))


(defun build-object-graph (obj)
  (let ((y (list 10000))
	(nodes (tconc)))
    (tconc nodes (build-object-layout 0 y (make-graph-node :pfn #'obj-ed-name-fn
							   :fn :none
							   :object "instance"
							   :style *obj-keyword-style*)))
    (doloop (prop (tok-inst-of obj))
      (tconc nodes (build-object-layout 15 y (make-graph-node :pfn #'obj-ed-name-fn
							      :fn #'object-edit-button-fn
							      :object prop
							      :style *obj-prop-style*)))
      )
    (tconc nodes (build-object-layout 0 y (make-graph-node :pfn #'obj-ed-name-fn
							   :fn :none
							   :object "is a"
							   :style *obj-keyword-style*)))
    (doloop (prop (tok-parents obj))
      (tconc nodes (build-object-layout 15 y (make-graph-node :pfn #'obj-ed-name-fn
							      :fn #'object-edit-button-fn
							      :object prop
							      :style *obj-prop-style*)))
      )
    (tconc nodes (build-object-layout 0 y (make-graph-node :pfn #'obj-ed-name-fn
							   :fn :none
							   :object "with"
							   :style *obj-keyword-style*)))
    (doloop (attr (lookup-index-cdr (tok-attrs obj) :all))
      (doloop (prop (tok-inst-of attr))
	(tconc nodes (build-object-layout 15 y (make-graph-node :pfn #'obj-ed-name-fn
								:fn (list #'object-edit-button-fn attr)
								:object prop
								:style *obj-prop-style*)))
	)
      (tconc nodes (build-object-layout 45 y (make-graph-node :pfn #'obj-ed-name-fn
							      :fn #'object-edit-button-fn
							      :object attr
							      :style *obj-prop-style*)))
      )
    (-- (car y) 20)
    (doloop (node (car nodes))
      (-- (gn-y node) (car y))
      )
    (cons nil (car nodes))
    )
  )


(defun display-object-editor (obj)
  (let ((graph (build-object-graph obj))
	(title (format nil "~a" (normal-clause obj))))
    (multiple-value-bind (width height) (graph-extent graph)
      (let ((win (create-graph-window title graph width height)))
	(setf (window-stream-get win :object) obj)
	)
      )
    )
  )

(defmacro edit-object (obj)
  `(let ((object (lookup-type ,obj)))
     (if object
       (display-object-editor object)
       (list 'no-such-object obj)
       ))
  )
