;;;
;;;   KNOWBEL knowledge representation system
;;;    
;;;    author: Bryan M. Kramer
;;;    
;;;    
;;; Copyright (c) 1990, 1991 University of Toronto, Toronto, ON
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all copies and
;;; supporting documentation.
;;;
;;; The University of Toronto provides this software "as is" without
;;; express or implied warranty.
;;;

;;;    
;;;    


;? forward chaining


(defmacro clause-make-sigma (clause)
  (let ((cvar (gensym "CLAUSE")))
    `(let ((,cvar ,clause))
       (make-array (list (clause-n-vars ,cvar) 2))
       )
    )
  )


(defun copy-sigma (sigma &optional (extend 0) clause)
  (let* ((dim (array-dimensions sigma))
	 (copy (make-array (list (+ (car dim) extend) (cadr dim)))))
    (doloop :for i :from 0 :to (- (car dim) 1)
      (doloop :for j :from 0 :to (- (cadr dim) 1)
	(setf (aref copy i j) (aref sigma i j))
	)
      )
    (if (> extend 0)
      (doloop :for i :from (car dim)
	:for j :from 0 :to (- extend 1)
	(setf (aref copy i 0) '*none*)
	(setf (aref copy i 1) (var-type (svref (clause-vars clause) j)))
	))
    copy)
  )





(defun rebuild-sigma (sigma keep-clause discard-clause &optional (set-n t))
  (let* ((n2 (if discard-clause (clause-n-vars discard-clause) 0))
	 (n1 (clause-n-vars keep-clause))
	 (n (+ n1 n2))
	 (copy (make-array (list n 2))))
    (doloop :for i :from 0 :to (- n1 1)
     :if (eq (aref sigma i 0) '*none*)
      (setf (aref copy i 0) (aref (clause-vars keep-clause) i))
      (setf (aref copy i 1) (aref sigma i 1))
     :else
      (setf (aref copy i 0) (aref sigma i 0))
      (setf (aref copy i 1) nil)
      )
    (doloop :for i :from n1 :to (- n 1)
     :if (eq (aref sigma i 0) '*none*)
      (setf (aref copy i 0) (make-clause-var :number i :name (list n1 n2 n) :type (aref sigma i 1) :owner keep-clause))
      (setf (aref copy i 1) (aref sigma i 1))
     :else
      (setf (aref copy i 0) (aref sigma i 0))
      (setf (aref copy i 1) nil)
      )
    (doloop :for i :from 0 :to (- n 1)
     :if (not (aref copy i 1))
      (setf (aref copy i 1) (or (aref sigma i 1) t))
      (setf (aref copy i 0) (apply-sigma-copy-sub (aref copy i 0) copy keep-clause keep-clause discard-clause))
      )
    (when set-n (setf (clause-n-vars keep-clause) n))
    copy)
  )


;;; the state could contain a forward agenda, some atoms could cause termination of this path
;;; and placing of the rest of the task onto the agenda


(defun chain-forward-prim (atoms fn belief theory sigma active-clause state temp-theory props)
  (doloop :for gen :set :init
   :for atom :set (apply-sigma (car atoms) sigma nil (ac-clause active-clause) nil)
   :while gen
   :for new-sigma := (copy-sigma sigma)
    (multiple-value-bind (success continuation justifications)
	(if (consp fn)
	  (funcall (car fn)
		   atom new-sigma gen belief active-clause theory state)
	  (funcall fn
		   atom new-sigma gen belief active-clause theory))
      (setf gen continuation)
      (if! (eq success t)
	(chain-forward-atom (cdr atoms) belief theory
			    (rebuild-sigma new-sigma (ac-clause active-clause) nil)
			    active-clause state temp-theory props)
	)
      ))
  )


(defmacro deftell (name args &body body)
  (let ((extra-args (and (listp name) (cdr (member :args name)))))
    `(let ((tell-fn ,(if extra-args `(cons #'(lambda ,args ,@body) args) `#'(lambda ,args ,@body))))
       ,@(if (consp name)
	   (doloop (n name)
	    :for result :set (tconc)
	    :when (eq n :args)
	    :return (car result)
	     (tconc result `(setf (get ',n :tell-fn) tell-fn))
	    :return (car result)
	     )
	   `((setf (get ',name :tell-fn) tell-fn))
	   )
       )
    )
  )


(deftell attr (clause belief theory sigma active-clause state)
  (dlet* (((* obj label class-label val * time) (car (clause-expression clause))))
    (if (and
	 (not (clause-var-p obj))
	 (not (clause-var-p class-label))
	 (not (clause-var-p val))
	 (clause-var-p time)
	 (time-int-p (var-type time)))
      (let ((r (tell `(,obj with ((,class-label (,(if (or (clause-wild-card-p label)
						       (clause-var-p label)) '_ label)
						 ,val ,(var-type time)))))
		     theory nil belief t)))
	r)
      )
    )
  )


(deftell (instance instance-of) (clause belief theory sigma active-clause state)
  (dlet* (((* obj class time) (car (clause-expression clause))))
    (if (and
	 (not (clause-var-p obj))
	 (not (clause-var-p class))
	 (clause-var-p time)
	 (time-int-p (var-type time)))
      (tell `(,obj instance ((,class ,(var-type time))))
	    theory nil belief t)
      )
    )
  )


(deftell (is-a isa) (clause belief theory sigma active-clause state)
  (dlet* (((* obj class time) (car (clause-expression clause))))
    (if (and
	 (not (clause-var-p obj))
	 (not (clause-var-p class))
	 (clause-var-p time)
	 (time-int-p (var-type time)))
      (tell `(,obj isa ((,class ,(var-type time))))
	    theory nil belief t)
      )
    )
  )


(defun forward-fix-sigma (sigma clause)
  (doloop :for i :from 0 :to (- (car (array-dimensions sigma)) 1)
    :for val := (aref sigma i 0)
    :if (and (unify-var-p val clause nil)
	     (eq (var-number val) i))
    (setf (aref sigma i 0) '*none*)
    )
  sigma
  )


(defun forward-build-resolvent (clause sigma time)
  (let* ((new-clause (make-clause))
	 (vars (subst-sigma (forward-fix-sigma sigma clause) new-clause clause nil))
	 (resolvent (tconc)))
    (doloop (atom (clause-expression clause))
      :when (not (eq (car atom) 'not))
      (tconc-unique resolvent (apply-sigma atom sigma new-clause clause nil))
      )
    (setf (clause-expression new-clause) (car resolvent))
    (setf (clause-n-vars new-clause) (array-dimension vars 0))
    (setf (clause-vars new-clause) vars)
    (setf (clause-belief-time new-clause) time)
    new-clause)
  )


(defun tell-prop (atom belief theory sigma active-clause state temp-theory)
  (let* ((new-clause (forward-build-resolvent (ac-clause active-clause) sigma belief))
	 (fn (and (symbolp (car atom)) (get (car atom) :tell-fn)))
	 (r (if fn
	      (apply (if (consp fn) (car fn) fn)
		     new-clause belief theory sigma active-clause state (if (consp fn) (cdr fn))))))
    (if r
      r
      (list (stash new-clause temp-theory nil belief))
      ))
  )



(defun chain-forward-do (belief theory sigma active-clause state temp-theory props)
  (doloop (atom (clause-expression (ac-clause active-clause)))
    :when (not (eq (car atom) 'not))
    (lconc props (tell-prop atom belief theory sigma active-clause state temp-theory))
    )
  )


(defun chain-forward-theory (atoms belief theory sigma active-clause state temp-theory props)
  (let ((atom (apply-sigma (car atoms) sigma nil (ac-clause active-clause) nil)))
    (doloop (pair (lookup-theory-index temp-theory t (cadr atom)))
     :for pred :set (car (cadr atom))
     :for clause := (car pair)
     :for expr := (clause-expression clause)
     :when (and (null (cdr expr)) (eql (caar expr) pred))    
      (let* ((new-sigma (copy-sigma sigma (clause-n-vars clause) clause))
	     (match (unify-work (cadr atom) (car expr) new-sigma (ac-clause active-clause) clause belief))
	     )
	(if match
	  (chain-forward-atom (cdr atoms) belief theory
			      (rebuild-sigma new-sigma (ac-clause active-clause) clause)
			      active-clause state temp-theory props)
	  ))
      ))
  )


(defun chain-forward-atom (atoms belief theory sigma active-clause state temp-theory props)
  (if! (null atoms)
    (chain-forward-do belief theory sigma active-clause state temp-theory props)
   elseif (eq (caar atoms) 'not)
    (let* ((pred (caadar atoms))
	   (fn (if (symbolp pred) (get pred 'lisp-fn))))
      (if! fn
	(chain-forward-prim atoms fn belief theory sigma active-clause state temp-theory props)
       else
	(chain-forward-theory atoms belief theory sigma active-clause state temp-theory props)
	)
      )
   else
    (chain-forward-atom (cdr atoms) belief theory sigma active-clause state temp-theory props)
    )
  )



(defun forward-copy-clause-vars (clause vars &optional sigma)
  (let* ((d (if sigma (array-dimensions sigma)))
	 (n (if d (car d) (length vars)))
	 (v (make-array n)))
    (doloop :for i :from 0 :to (- n 1)
      :for var := (if (and sigma (clause-var-p (aref sigma i 0))) (aref sigma i 0) (svref vars i))
      (setf (svref v i) (make-clause-var :number (var-number var)
					 :name (list :*copy* i)
					 :type (var-type var)
					 :owner clause))
      )
    v)
  )


(defun forward-copy-clause-expression (vars expression)
  (cond ((null expression) nil)
	((clause-var-p expression)
	 (svref vars (var-number expression)))
	((consp expression)
	 (doloop (x expression)
	   :collect (forward-copy-clause-expression vars x) :atom-tail y (forward-copy-clause-expression vars y)))
	(t expression))
  )


(defun forward-copy-clause (clause &optional sigma)
  (let* ((new (make-clause :number (list :*copy* (clause-number clause))))
	 (vars (forward-copy-clause-vars new (clause-vars clause) sigma)))
    (setf (clause-vars new) vars)
    (setf (clause-n-vars new) (length vars))
    (setf (clause-expression new) (forward-copy-clause-expression vars (clause-expression clause)))
    new)
  )


(defun forward-copy-fix-sigma (sigma vars)
  (doloop :for i :from 0 :to (- (car (array-dimensions sigma)) 1)
    (setf (aref sigma i 0) (forward-copy-clause-expression vars (aref sigma i 0)))
    )
  )



(defun chain-forward-clause (clause belief theory state temp-theory)
  (let* ((init-sigma (cdr clause))
	 (active-clause (make-active-clause :clause (forward-copy-clause (car clause) init-sigma)))
	 (props (tconc)))
    (forward-copy-fix-sigma init-sigma (clause-vars (ac-clause active-clause)))
    (chain-forward-atom (clause-expression (ac-clause active-clause))
			belief theory init-sigma active-clause state temp-theory props)
    (car props))
  )


(defun lookup-attr-index (token prop theory belief)
  (let ((tok-index (lookup-index (theory-neg-indices theory) 'attr :_attr_arg_1_token token))
	(type-indices (lookup-index (theory-neg-indices theory) 'attr :_attr_arg_1_var)))
    (doloop (index (cdr type-indices))
     :when (weak-type-instance token (car index) belief)
     :splice (doloop (item (cdr index)) :collect item) :init (doloop (item tok-index) :collect item)
      ))
  )


(defun lookup-inst-index (token prop theory belief)
  (let ((tok-index (lookup-index (theory-neg-indices theory) 'instance-of :_inst_arg_1_token token))
	(var-indices (lookup-index (theory-neg-indices theory) 'instance-of :_inst_arg_1_var)))
    (doloop (index (cdr var-indices))
     :when (weak-type-instance token (car index) belief)
     :splice (doloop (item (cdr index)) :collect item) :init (doloop (item tok-index) :collect item))
    )
  )
  

(defun chain-forward-match-prim (prop atom clause state belief)
  (and (consp atom)
       (eq (car atom) 'not)
       (let (r
	     (atom (cadr atom))
	     (sigma (istate-get-sigma state (clause-n-vars clause) 2)))
	 (setq r (cond ((kb-attr-p prop)
		(doloop (prop-type-prop (tok-inst-of (attr-token prop)))
		  (reset-sigma sigma clause)
		 :some
		  (dlet* (((pred obj label class-label value propx time) atom))
		    (and (eq pred 'attr)
			 (or (clause-wild-card-p obj) (unify-work obj (attr-from prop) sigma clause nil belief))
			 (or (clause-wild-card-p label) (unify-work label (attr-label prop) sigma clause nil belief))
			 (or (clause-wild-card-p class-label)
			     (unify-work class-label (attr-token-class-label (prop-dest prop-type-prop))
					 sigma clause nil belief))
			 (or (clause-wild-card-p value) (unify-work value (attr-value prop) sigma clause nil belief))
			 (or (clause-wild-card-p propx) (unify-work propx propx sigma clause nil belief))
			 (unify-time (attr-history prop) time sigma clause belief)
			 ))))
	       ((prop-object-p prop :inst)
		(dlet* (((pred obj class time) atom))
		  (and (member  pred '(instance inst instance-of))
		       (unify-work obj (prop-src prop) sigma clause nil belief)
		       (unify-work class (prop-dest prop) sigma clause nil belief)
		       (unify-time (prop-history prop) time sigma clause belief))))
	       (t nil)))
	 (if r (rebuild-sigma sigma clause nil nil))
	 ))		  
  )


(defun chain-forward-match-theory (atom1 atom2 clause1 clause2 state belief) ;; clause1 is the one that will be used in chaining
  (let (r
	(sigma (istate-get-sigma state (+ (clause-n-vars clause1) (clause-n-vars clause2)) 2)))
    (reset-sigma sigma clause1 clause2)
    (setq r (if! (and (eq (car atom1) 'not)
		      (not (eq (car atom2) 'not)))
	      (unify-work (cadr atom1) atom2 sigma clause1 clause2 belief)
	     elseif (and (not (eq (car atom1) 'not))
			 (eq (car atom2) 'not))
	      (unify-work atom1 (cadr atom2) sigma clause1 clause2 belief)
	      )
	  )
    (if r (rebuild-sigma sigma clause1 clause2 nil))
    )
  )


(defun find-potential-matches (prop theory belief state)
  (cond ((kb-attr-p prop)
	 (doloop (item (lookup-attr-index (attr-from prop) prop theory belief))
	   :for sigma := (chain-forward-match-prim prop (cadr item) (car item) state belief)
	   :when sigma 
	   :collect (cons (car item) sigma)))
	((prop-object-p prop :isa)
	 nil)
	((prop-object-p prop :inst)
	 (doloop (item (lookup-inst-index (prop-src prop) prop theory belief))
	   :for sigma := (chain-forward-match-prim prop (cadr item) (car item) state belief)
	   :when sigma
	   :collect (cons (car item) sigma))
	 )
	((clause-p prop)
	 (let* ((atom (car (clause-expression prop)))
		(literal (if (eq (car atom) 'not) (cadr atom) atom)))
	   (doloop (item (lookup-theory-index theory (eq (car atom) 'not) literal))
	     :for sigma := (chain-forward-match-theory (cadr item) atom (car item) prop state belief)
	     :when sigma
	     :collect (cons (car item) sigma))))
	(t nil))
  )





(defun tell-forward-chain (props forward-theory theory belief context discard-temp)
  (doloop :while props
   :for temp-theory :set (if discard-temp (new-theory (theory-kb theory) nil) theory)
   :for state :set (make-inference-state :root-theory theory :belief belief :context context)
   :for forward-clauses := nil
   :for new-props := (tconc)
    (doloop (prop props)
      (doloop (match (find-potential-matches prop forward-theory belief state))
	(if (not (assoc (car match) forward-clauses))
	  (push match forward-clauses)
	  )))
    (if forward-clauses
      (let ()
	(doloop (clause forward-clauses)
	  (lconc new-props (chain-forward-clause clause belief theory state temp-theory))
	  ))
      )
    (setq props (car new-props))
    )
  )



(defun tell-forward (expr &key (theory *theory*) default-history (belief (std-belief)) context forward-theory (discard-temp t))
  (multiple-value-bind (token errors props)
      (real-tell expr :theory theory :default-history default-history :belief belief :context context)
    (let ((forward (and forward-theory
			(if (theory-p forward-theory)
			  forward-theory
			  (theory-find-theory theory forward-theory)))))
      (if forward
	(tell-forward-chain props forward theory belief context discard-temp)
	)
      )
    token)
  )

(defvar *tftc* 1)


(defun test-forward nil
  (++ *tftc*)
  (let* ((name (intern (format nil "FORWARD~d" *tftc*)))
	 (name2 (intern (format nil "FORINSTA~d" *tftc*)))
	 (name3 (intern (format nil "FORINSTB~d" *tftc*)))
	 (xvar (intern (format nil "$X/~a" name))))
    (tell `(,name instance (s-class)
		  with (
			(attribute
			 (a1 number)
			 (a2 number)
			 (a3 number)
			 (a4 number)
			 (a5 number))
			(deductive-rule
			 (f1 (clause (=> (attr ,xvar * "a1" $v * $t@)
					 (+ $v 99 $r)
					 (attr $x * "a2" $r * $t@))
				     forward))
			 (f2 (clause (=> (attr ,xvar * "a1" $v * $t@)
					 (+ $v 199 $r)
					 (xxx $x $r $t@))
				     forward))
			 (f3 (clause (=> (xxx ,xvar $r $t@)
					 (> $r 300)
					 (attr $x * "a3" $r * $t@))
				     forward))
			 (f4 (clause (=> (xxx ,xvar $r $t@)
					 (< $r 300)
					 (attr $x * "a3" $r * $t@))
				     forward))			 
			 (f5 (clause (=> (attr ,xvar * "a3" $v * $t@)
					 (+ $v 99 $r)
					 (and
					  (attr $x * "a5" $r * $t@)
					  (attr $x * "a4" $r * $t@)))
				     forward))
			 )
			)
		  ))
    (time (tell-forward `(,name2 instance (,name) with ((a1 (_ 3)))) :forward-theory (theory-find-theory *theory* 'forward)))
    (time (tell-forward `(,name3 instance (,name) with ((a1 (_ 9)))) :forward-theory (theory-find-theory *theory* 'forward)))
    (kb-print-token (lookup-type name2))
    (kb-print-token (lookup-type name3))
    )
  )