;;;
;;;   KNOWBEL knowledge representation system
;;;    
;;;    author: Bryan M. Kramer
;;;    
;;;    
;;; Copyright (c) 1990, 1991 University of Toronto, Toronto, ON
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all copies and
;;; supporting documentation.
;;;
;;; The University of Toronto provides this software "as is" without
;;; express or implied warranty.
;;;

;;;    
;;;    


;? functions to dump and restore a KNOWBEL knowledge base

(eval-when (load compile eval)
  (provide 'dump)
  (require 'tools)
  )


;;; (in-package :knowbel-dump :use '(:lisp))
;;; (export 'dump-kb-file)

(eval-when (load compile eval)

(defmacro dump-hash (hash hash-num object)
  `(let ((nnx (car ,hash-num)))
     (++ (car ,hash-num))
     (setf (gethash ,object ,hash) nnx)
     nnx)
  )
)

(defconstant *theory-define* :o)
(defconstant *theory-end* :p)
(defconstant *theory-ref* :q)
(defconstant *clause-var-short* :r)
(defconstant *clause-var* :s)
(defconstant *description* :t)
(defconstant *clause* :u)
(defconstant *clause-def* :v)
(defconstant *token* :w)
(defconstant *anonymous-token* :a)
(defconstant *meta-type* :b)
(defconstant *belief* :c)
(defconstant *belief-define* :d)
(defconstant *time-link* :e)
(defconstant *time-link-define* :f)
(defconstant *time-point-same* :g)
(defconstant *time-point-same-define* :x)
(defconstant *time-point* :h)
(defconstant *time-point-define* :i)
(defconstant *conventional-time-point* :j)
(defconstant *interval* :k)
(defconstant *interval-define* :l)
(defconstant *token-define* :m)
(defconstant *anonymous-token-define* :n)
(defconstant *belief-point* :y)



(defun dump-theory (th theory hash hash-num result)
  (when (not (gethash th hash))
    (tconc result
	   (list *theory-define* (dump-hash hash hash-num th)
		 (dump-expression (theory-name th) theory hash hash-num result)
		 (doloop (thi (theory-includes th)) :collect (dump-theory thi theory hash hash-num result))
		 (theory-horn th)))
    (doloop (clause (theory-clauses th))
      (tconc result (dump-clause clause theory hash hash-num result)))
    (tconc result (list *theory-end* (gethash th hash)))
    )
  (list *theory-ref* (gethash th hash))
  )



(defun dump-clause-expression (exp theory hash hash-num result vars)
  (cond ((null exp) nil)
	((consp exp) (cons (dump-clause-expression (car exp) theory hash hash-num result vars)
			   (dump-clause-expression (cdr exp) theory hash hash-num result vars)))
	((clause-var-p exp)
	 (if (member exp (car vars))
	   (list *clause-var-short* (dump-expression (var-number exp) theory hash hash-num result)
		 (dump-expression (var-owner exp) theory hash hash-num result))
	   (progn
	     (tconc vars exp)
	     (list *clause-var* (dump-expression (var-number exp) theory hash hash-num result)
		   (dump-expression (var-name exp) theory hash hash-num result)
		   (dump-expression (var-owner exp) theory hash hash-num result)
		   (dump-expression (var-type exp) theory hash hash-num result)
		   (dump-expression (var-meta-type exp) theory hash hash-num result)
		   ))))
	((token-description-p exp)
	 (list *description*
	       (dump-clause-expression (tok-desc-instance-of exp) theory hash hash-num result vars)
	       (dump-clause-expression (tok-desc-attrs exp) theory hash hash-num result vars)))
	(t (dump-expression exp theory hash hash-num result)))
  )


(defun dump-clause (clause theory hash hash-num result)
  (if (gethash clause hash)
    (list *clause* (gethash clause hash))
    (let ((vars (tconc)))
      (list *clause-def* (dump-hash hash hash-num clause)
	    (dump-expression (clause-number clause) theory hash hash-num result)
	    (dump-clause-expression (clause-expression clause) theory hash hash-num result vars)
	    (dump-belief (clause-belief-time clause) theory hash hash-num result t)
	    (dump-expression (clause-props clause) theory hash hash-num result)
	    )))
  )


(defun dump-token-identify (token theory hash hash-num result knownp)
  (cond ((eq (lookup-type (tok-name token) theory) token)
	 (if knownp (tok-name token) (list *token* (tok-name token))))
	((gethash token hash)
	 (if knownp (gethash token hash) (list *anonymous-token* (gethash token hash))))
	(t (dump-token token theory hash hash-num result)
	   (if knownp (gethash token hash) (list *anonymous-token* (gethash token hash)))))
  )


(defun dump-expression (exp theory hash hash-num result)
  (cond ((consp exp) (cons (dump-expression (car exp) theory hash hash-num result)
			   (dump-expression (cdr exp) theory hash hash-num result)))
	((time-int-p exp) (dump-interval exp theory hash hash-num result nil))
	((kb-token-p exp) (dump-token-identify exp theory hash hash-num result nil))
	((clause-p exp) (dump-clause exp theory hash hash-num result))
	((theory-p exp) (dump-theory exp theory hash hash-num result))
	((stringp exp) exp)
	((meta-type-p exp) (list *meta-type* (dump-expression (meta-type-type exp) theory hash hash-num result)))
	((vectorp exp) (break))
	(t exp))
  )

(defun dump-belief (belief-int theory hash hash-num result knownp)
  (if (not (listp belief-int))
    (if knownp belief-int (list *belief-point* belief-int))
    (let ((n (gethash belief-int hash)))
      (if n
	(list *belief* n)
	(list* *belief-define* (dump-hash hash hash-num belief-int) belief-int)
	)
      )
    ) 
  )

(defun dump-time-link (link theory hash hash-num result)
  (if (gethash link hash)
    (list *time-link* (gethash link hash))
    (list *time-link-define* (dump-hash hash hash-num link)
	  (dump-time-point (point-link-source link) theory hash hash-num result)
	  (dump-time-point (point-link-dest link) theory hash hash-num result)
	  (dump-belief (point-link-belief link) theory hash hash-num result t))
    )
  )



(defun dump-time-same (same theory hash hash-num result)
  (if (gethash same hash)
    (list *time-point-same* (gethash same hash))
    (list* 'tpsd*time-point-same-define*
	   (dump-hash hash hash-num same)
	   (doloop (link (car same)) (dump-time-link link theory hash hash-num result)))
    )
  )
      



(defun dump-time-point (point theory hash hash-num result)
  (if (time-point-p point)
    (let ((n (gethash point hash)))
      (if n
	(list *time-point* n)
	(progn
	  (list *time-point-define*
		(dump-hash hash hash-num point)
		(dump-expression (point-interval point) theory hash hash-num result)
		(doloop (link (point-precedes point))
		  (dump-time-link link theory hash hash-num result))
		(doloop (link (point-preceded-by point))
		  (dump-time-link link theory hash hash-num result))
		(dump-time-same (point-same-as point) theory hash hash-num result)
		)
	  )
	)
      )
    (list *conventional-time-point* point)
    )
  )

(defun dump-interval (int theory hash hash-num result knownp)
  (let ((n (gethash int hash)))
    (if n
      (if knownp n (list *interval* n))
      (let ((n (dump-hash hash hash-num int)))
	(if (null (time-not-shared int))
	  (list *interval-define* n
		(dump-time-point (time-start int) theory hash hash-num result)
		(dump-time-point (time-end int) theory hash hash-num result))
	  (list *interval-define* n
		(dump-time-point (time-start int) theory hash hash-num result)
		(dump-time-point (time-end int) theory hash hash-num result)
		(dump-expression (time-not-shared int) theory hash hash-num result))
	  ))
      )
    )
  )
  


(defun dump-token (token theory hash hash-num result)
  (let ((anonymous (not (eq (lookup-type (tok-name token) theory) token)))
	(desc (tconc)))
    (if anonymous
      (progn (tconc desc *anonymous-token-define*)
	     (tconc desc (or (gethash token hash) (dump-hash hash hash-num token))))
      (progn (tconc desc *token-define*)
	     (tconc desc (tok-name token)))
      )
    (tconc desc
	   (doloop (prop (tok-inst-of token))
	    :splice (list (dump-token-identify (prop-dest prop) theory hash hash-num result t)
			   (dump-interval (prop-history prop) theory hash hash-num result t)
			   (dump-belief (prop-belief prop) theory hash hash-num result t)
			   )))
    (tconc desc
	   (doloop (prop (tok-parents token))
	    :splice (list (dump-token-identify (prop-dest prop) theory hash hash-num result t)
			   (dump-interval (prop-history prop) theory hash hash-num result t)
			   (dump-belief (prop-belief prop) theory hash hash-num result t)
			   )))
    (tconc desc
	   (doloop (attr-group (tok-attrs token))
	    :collect
	     (cons (car attr-group)
		   (doloop (attr (cdr attr-group))
		    :collect (dump-token-identify attr theory hash hash-num result t)))
	     ))
    (if (kb-attr-p token)
      (lconc desc (list (dump-expression (attr-value token) theory hash hash-num result)
			(dump-interval (attr-history token) theory hash hash-num result t)
			(dump-belief (attr-belief token) theory hash hash-num result t)
			)))
    (tconc result (car desc)))
  )

(defun dump-token-driver (token theory hash hash-num stream)
  (let ((result (tconc)))
    (dump-token token theory hash hash-num result)
    (doloop (item (car result)) (print item stream))
    (doloop (prop (tok-instances token))
      (let ((tok (prop-src prop)))
	(when (and (or (null (tok-name tok))
		       (not (eq (lookup-type (tok-name tok) theory) tok)))
		   (not (gethash tok hash)))
	  (dump-token-driver tok theory hash hash-num stream)
	  ))
      )
    )
  )
  
      

(defun dump-kb (theory stream)
  (let ((hash (make-hash-table))
	(hash-num (list 0))
	(result (tconc))
	(*print-pretty* nil)
	(*print-level* nil)
	(*print-length* nil))
    (dump-theory theory theory hash hash-num result)
    (doloop (item (car result)) (print item stream))
    (maphash #'(lambda (key val)
		 (dump-token-driver val theory hash hash-num stream)		    
		 )
	     (theory-tokens theory)
	     )
    (print (list :size (car hash-num)) stream)
    )
  )


(defun dump-kb-file (theory file)
  (with-open-file (s file :direction :output)
    (dump-kb theory s)
    )
  )


(defun restore-find-size (stream)
  (let ((point (file-length stream)))
    (when point
      (setf point (- point 3))      
      (while (and (file-position stream point) (not (eql (read-char stream) #\()))
	(-- point)
	)
      (file-position stream point)
      (let ((x (read stream)))
	(if (listp x) (cadr x))
	)))
  )


(eval-when (load compile)
  
(defmacro restore-reference (item maker-fn &rest maker-args)
  `(let ((obj (svref objects (cadr ,item))))
     (if obj
       obj
       (progn (setf obj (,maker-fn ,@maker-args))
	      (setf (svref objects (cadr ,item)) obj)
	      obj))
     )
  )

)

(defun restore-theory-ref (item objects theory)
  (restore-reference item new-theory (theory-kb theory) nil)
  )

(defun restore-kb-theory (item objects theory)
  (dlet* (((n name include horn) (cdr item)))
    (let ((th (theory-find-theory theory name)))
      (if (not th) (setq th (svref objects n)))
      (if (not th)
	(setq th (new-theory (theory-kb theory) name))
	(name-theory th name))
      (setf (svref objects n) th)
      (setf (theory-includes th)
	(doloop (include-th include)
	 :collect (restore-theory-ref include-th objects theory)))
      (setf (theory-horn th) horn)
      th))
  )


(defun restore-kb-input (item objects theory)
  (when (listp item)
    (case (car item)
      (*theory-define* (restore-kb-theory item objects theory))
      (*theory-end* nil)
      (*theory-ref* (restore-theory-ref item objects theory))
      (*clause-var-short* )
      (*clause-var* )
      (*description* )
      (*clause* (restore-reference item make-clause))
      (*clause-def* )
      (*token* (restore-reference item make-kb-token))
      (*anonymous-token* (restore-reference item make-kb-token))
      (*meta-type* (make-meta-type :type (restore-kb-input (cadr item))))
      (*belief-point* (cadr item))
      (*belief* (restore-reference item list nil))
      (*belief-define* )
      (*time-link* (restore-reference item list nil))
      (*time-link-define* )
      (*time-point-same* (restore-reference item tconc nil))
      (*time-point-same-define* )
      (*time-point* (restore-reference item make-time-point))
      (*time-point-define* )
      (*conventional-time-point* )
      (*interval* (restore-reference item make-time-int))
      (*interval-define* )
      (*token-define* )
      (*anonymous-token-define* )
      (t (error "bogus dump file"))
      ))
  )


(defun restore-kb-file (file)
  (with-open-file (stream file :direction :input)
    (let ((nobjs (restore-find-size stream)))
      (if nobjs
	(let* ((kb (new-kb :clause-number 0))
	       (theory (new-theory kb 'root))
	       (objects (make-array (+ nobjs 10)))
	       (theory-stack nil))

	  (setq *theory* theory)
	  (setq *all-time* (make-time-int :start :- :end :+ ))
	  (setf (theory-root-object theory) *root-type*)
	  (setf (kb-all-time kb) *all-time*)
	  (setf (kb-meta-attr kb) (find-string "meta-rule" theory))
	  (doloop (x *built-in-type-symbols*)
	    (push (type-name x theory) *built-in-type-names*)
	    (push (cons (type-name x theory) x) *built-in-type-alist*)
	    )
      
	  (file-position stream :start)
	  (push theory theory-stack)
	  (doloop :iter (item (read stream nil :done) (read stream nil :done))
	   :while (not (eq item :done))
	    (if (listp item)
	      (case (car item)
		(*theory-define* (push (restore-kb-input item objects theory) theory-stack)
				 (setq theory (car theory-stack)))
		(*theory-end* (pop theory-stack)
			      (setq theory (car theory-stack)))
		(t (restore-kb-input item objects theory))
		))
	    )
	  (setq *root-type* (lookup-type 'proposition theory))
	  theory
	  ))
      )
    )
  )
