function y=frule(a,c,z,x,f2,f1,f3)
% FRULE Evaluates sets of fuzzy rules.
%
% FRULE(A,C,Z,X)
%   A - Antecedant table (R rules x N inputs).
%   C - Consequence table (R rules x M inputs).
%   Z - Grouped support and grades for inputs and outputs in order.
%   X - n element column vector(s) of input variables.
% Returns output (column) vectors. Uses Centroid of Max-of-Min.
%
% FRULE(A,C,Z,X,'F1','F2','F3')
%   F1 - name of union function (default = 'or').
%   F2 - name of intersection function (default = 'and').
%   F3 - name of defuzzification function (default = 'centroid').
% Returns output vectors. Uses F3 of F1-of-F2.
%
% EXAMPLE: xS = 0:.1:10; xG = [small(xS); medium(xS); large(xS)];
%          yS = 0:.1:10; yG = [small(yS); medium(yS); large(yS)];
%          % "IF X IS LARGE, THEN Y IS MEDIUM"
%          % "IF X IS SMALL, THEN Y IS LARGE"
%          A = [3; 1];
%          C = [2; 3];
%          Z = group(xS, xG, yS, yG);
%          % "IF X IS 3, THEN WHAT IS Y?"
%          x = 3;
%          y = frule(A,C,Z,x)
%
% SEE ALSO: ifthen

% Mark Beale 6-24-93
% Copyright (c) 1993 by PWS Publishing Company

if nargin < 7, f3 = 'centroid';
  if nargin < 6, f1 = 'and';
    if nargin < 5, f2 = 'or';
    end
  end
end

[ar,ac] = size(a);
[cr,cc] = size(c);
[xr,xc] = size(x);

y = zeros(cc,xc);

for p=1:xc

  % FUZZIFY INPUTS

  memb = zeros(ar,ac);
  for i=1:ac
    si = ungroup(z,(i-1)*2+1);
    gi = ungroup(z,(i-1)*2+2);
    fuzzyx = fuzzyval(si,gi,x(i,p));
    memb(:,i) = fuzzyx(a(:,i));
  end

  % ANTECEDANT GRADES
  ant = feval(f1,memb')';

  % OUTPUTS
  for i=1:cc
    so = ungroup(z,(ac+i-1)*2+1);
    go = ungroup(z,(ac+i-1)*2+2);
    [gr,gc] = size(go);

    % FIND UNION OF ANTECEDANT GRADES FOR EACH OUTPUT FUZZY SET

    clip = zeros(gr,1);
    for j = 1:gr
      ind = find(c(:,i) == j);
      clip(j) = feval(f2,ant(ind));
    end

    % TAKE UNION OF CLIPPED OUTPUT FUZZY SETS

    j = find(clip ~= 0);
    g = feval(f2,feval(f1,go(j,:),clip(j)*ones(1,gc)));

    %  FIND CENTROID

    y(i,p) = feval(f3,so,g);
  end
end
