function y = eqshape(t,s,n,f)
% EQSHAPE Creates a group of equally spaced fuzzy set shapes.
%
% EQSHAPE(T,S,N)
%   T - Fuzzy set shape type (see below).
%   S - Vector containing min and max support values.
%   N - The number of fuzzy sets to define.
% Returns group containing the limits of N fuzzy sets
%   spaced equally between min and max support values.
%
% EQSHAPE(T,S,N,'extend')
% Returns group where the far left and right fuzzy sets are
%   extended to -1e100 and +1e100.
%
% There are 6 types of fuzzy set shapes:
%   Singleton: T = 1,  Rectangle: T = 2,  Triangle:  T = 3
%   Trapezoid: T = 4,  Bump:      T = 5,  Flat Bump: T = 6
%
% EXAMPLE: s = 0:.1:10;
%          z = eqshape(5,s,7);
%          g = shapeval(z,s);
%          plotfs(s,'Seven Equally Spaced Bumps');
%          plot(s,g);
%          hold off
%
% SEE ALSO: shape, shapeval, group, ungroup

% Mark Beale 6-24-93
% Copyright (c) 1993 by PWS Publishing Company

if nargin < 4, f = 0; end

mins = min(s);
maxs = max(s);

if n == 1
  if t == 1
    v = (maxs+mins)*0.5;
  elseif t == 2
    v = [.67 .33]*mins + [.33 .67]*maxs;
  elseif t == 3
    v = [1 .5 0]*mins + [0 .5 1]*maxs;
  elseif t == 4
    v = [1 .67 .33 0]*mins + [0 .33 .67 1]*maxs;
  elseif t == 5
    v = [1 .75 .5 .25 0]*mins + [0 .25 .5 .75 1]*maxs;
  elseif t == 6
    v = [1 .8 .6 .4 .2 0]*mins + [0 .2 .4 .6 .8 1]*maxs;
  end

else
  base = (maxs-mins)/(n-1);
  cent = (mins+[0:(n-1)]'*base);

  z1 = ones(n,1);
  z2 = cent*ones(1,t);

  if t == 1
    v = cent;
  elseif t == 2
    v = [z1*[-.5 .5]*base+z2];
    if f
      v(1,1) = -1e100;
      v(n,2) = 1e100;
    end
  elseif t == 3
    if f
      t = 4;
      v = [z1*[-1 0 0 1]*base+cent*ones(1,t)];
      v(1,1) = -1e101;
      v(1,2) = -1e100;
      v(n,3) = 1e100;
      v(n,4) = 1e101;
    else
      v = [z1*[-1 0 1]*base+z2];
    end
  elseif t == 4
    v = [z1*[-.75 -.25 .25 .75]*base+z2];
    if f
      v(1,1) = -1e101;
      v(1,2) = -1e100;
      v(n,3) = 1e100;
      v(n,4) = 1e101;
    end
  elseif t == 5
    v = [(z1*[-1 -.5 0 .5 1]*base+z2)];
    if f
      v(1,1) = -1e101;
      v(1,2) = -1e100;
      v(n,4) = 1e100;
      v(n,5) = 1e101;
    end
  elseif t == 6
    v = [(z1*[-.8333 -.5 -.1667 .1667 .5 .8333]*base+z2)];
    if f
      v(1,1) = -1e102;
      v(1,2) = -1e101;
      v(1,3) = -1e100;
      v(n,4) = 1e100;
      v(n,5) = 1e101;
      v(n,6) = 1e102;
    end
  end
end

y = zeros(1,2+n*(5+t));
y(1) = 12345.12345;
y(2) = n;
pos = (0:(n-1))*t+5*n+3;
col = ones(1,n)*t;
y(3:(5*n+2)) = reshape([pos; col; ones(1,n); col; zeros(1,n)],1,5*n);

y((5*n+3):(2+n*(5+t))) = reshape(v',1,n*t);
