/*
 * Copyright (c) 1992 The University of Utah and
 * the Center for Software Science (CSS).  All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation, and (3) all advertising
 * materials mentioning features or use of this software display the following
 * acknowledgement: ``This product includes software developed by the Center
 * for Software Science at the University of Utah.''
 *
 * THE UNIVERSITY OF UTAH AND CSS ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSS DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSS requests users of this software to return to css-dist@cs.utah.edu any
 * improvements that they make and grant CSS redistribution rights.
 *
 *      Utah $Hdr: tri_member.h 1.6 93/09/07$
 *      Author: Doug Orr, University of Utah CSS
 */

#ifndef	_tri_member_h_
#define	_tri_member_h_	 1

/*  tri_member:  "triangular" membership functions.  */

#include "geom_member.h"

template<class C> class tri_member : public geom_member<C> {
	C lx;
	C mx;
	C rx;
	float h;

	C b1;
	C b2;
	float s1;
	float s2;

	/* cached copies of computed values */
	C a;							// area
	C c;       						// centroid
	C m;							// moment
	
	float			calc_area () const;
	float			calc_centroid () const;

    public:
	tri_member (C lx_, C mx_, C rx, float h_, char * n_ = NULL);
	~tri_member ();

	degree 			is (const C value) const;
	C	 		output (const degree d) const;

	C			area () const			{return a;}
	C			centroid () const		{return c;}
	C			moment () const			{return m;}

	geom_member<C> * 	scale (const degree d) const;
	geom_member<C> * 	clip (const degree d) const;

};



template<class C>
tri_member<C>::	tri_member (C lx_, C mx_, C rx_, float h_, char * n_) :
	lx (lx_),
	mx (mx_),
	rx (rx_),
	h (h_),
	geom_member<C> (NewStr (n_))
{
	Assert ((lx < mx) && (mx < rx));

	/* calculate slopes and y-intercepts */
	s1 = h / (mx - lx);
	s2 = h / (mx - rx);
	
	b1 = (C)(-s1 * lx);
	b2 = (C)(-s2 * rx);

	/* cache a copy of the area, centroid, and moment */
	a = (C)calc_area ();
	float cent = calc_centroid ();
	c = (C)cent;
	m = (C)(a * cent);
}

template<class C>
tri_member<C>::~tri_member ()
{
}

#endif	/* _tri_member_h_ */
