/*
 * Copyright (c) 1992 The University of Utah and
 * the Center for Software Science (CSS).  All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation, and (3) all advertising
 * materials mentioning features or use of this software display the following
 * acknowledgement: ``This product includes software developed by the Center
 * for Software Science at the University of Utah.''
 *
 * THE UNIVERSITY OF UTAH AND CSS ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSS DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSS requests users of this software to return to css-dist@cs.utah.edu any
 * improvements that they make and grant CSS redistribution rights.
 *
 *      Utah $Hdr: fam_surf.C 1.2 93/09/08$
 *      Author: Doug Orr, University of Utah CSS
 */

#include <stdio.h>
#include <stdarg.h>
#include "gen.h"
#include "fam_surf.h"

/* FAM: implementation of 3D Bart Kosko fuzzy associative memories.

   FAM matrices define the action to be taken on an output variable,
   based on a series of fuzzy AND's of three input variables.   */


fam_surf<int>::fam_surf (int dim_, ...) :
	dim (dim_)
{
	/*  define the NxNxN FAM surface */
	/*  note: by convention, the first row is a complete set
	    of pointers to the ordered member functions.
	    the FAM matrices follow.  */

	surf = new fam<int> * [dim];
	fns = new geom_member<int> * [dim];

	if (! (surf && fns)) {
		cerr << "memory allocation problem\n";
		exit (1);
	}
	
	va_list args;
	va_start (args, format);

	for (int i=0; i<dim; i++)
		fns [i] = va_arg (args, geom_member<int>*);
		
	for (i=0; i<dim; i++)
		surf [i] = va_arg (args, fam<int> *);

	va_end (args);
}


void fam_surf<int>::apply (const fuzzy<int> & v1,
			   const fuzzy<int> & v2,
			   const fuzzy<int> & v3,
			   fuzzy<int> & result)
{

	/* for all defined rules... fire each rule. 

	   keep track of an output result by accumulating
	   the scaled moment and area represented by each firing rule.

	   finally, compute the output result by dividing the
	   moment by the area.  */

	int moment = 0;
	int area = 0;


	for (int i=0; i<dim; i++) {
		/* precompute the first element */
		degree d_v1 = v1.is(*fns [i]);
		if (surf [i] && (int)d_v1) {

			/* calculate the degree to which 
			   (v1 IS m1)  AND  (v2 IS m2) AND (v3 IS m3) THEN (r IS m4) */
			fam<int> * f = surf[i];
			
			for (fam_member<int> * m = f->rules (); m; m = m->next) {
				degree d = d_v1 && v2.is(*m->ir) && v3.is(*m->ic);

				moment += m->r->moment () * d;
				area += m->r->area () * d;
			}
		}
	}

	result =  (area) ? (moment / area) : 0;
}
