/*
 * Copyright (c) 1992 The University of Utah and
 * the Center for Software Science (CSS).  All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation, and (3) all advertising
 * materials mentioning features or use of this software display the following
 * acknowledgement: ``This product includes software developed by the Center
 * for Software Science at the University of Utah.''
 *
 * THE UNIVERSITY OF UTAH AND CSS ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSS DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSS requests users of this software to return to css-dist@cs.utah.edu any
 * improvements that they make and grant CSS redistribution rights.
 *
 *      Utah $Hdr: fam.C 1.4 93/09/05$
 *      Author: Doug Orr, University of Utah CSS
 */

#define	VERBOSE

#include <stdio.h>
#include <stdarg.h>
#include "gen.h"
#include "fam.h"

/* FAM: implementation of 2D Bart Kosko fuzzy associative memories.

   FAM matrices define the action to be taken on an output variable,
   based on a series of fuzzy AND's of two input variables.   */


fam<int>::fam (int dim_, ...) :
	dim (dim_)
{
	/*  define the NxN FAM matrix */
	/*  note: by convention, the first row is a complete set
	    of pointers to the ordered member functions.
	    the FAM matrix itself follows.  */

	fns = new geom_member<int> * [dim];
	matrix = new fam_member<int> * [dim*dim];

	if (! (fns && matrix)) {
		fprintf (stderr, "memory allocation problem\n");
		exit (1);
	}
	
	fam_member<int> * cur = NULL;
	first_rule = NULL;
	
	va_list args;
	va_start (args, format);

	for (int i=0; i<dim; i++)
		fns [i] = va_arg (args, geom_member<int>*);
		
	for (i=0; i<dim; i++)
		for (int j = 0; j < dim; j++) {
			geom_member<int> * m = va_arg (args, geom_member<int>*);
			if (m) {
				fam_member<int> * f =  new fam_member<int> (m, i, j, fns);
				if (! f) {
					fprintf (stderr, "memory allocation problem\n");
					exit (1);
				}
				
				/* Keep the active fam rules on a list so we don't have
				   to scan the whole array to find them.  Leave them in a matrix
				   in case we want to fire a given rule.  */

				if (! first_rule)
					first_rule = f;
				if (cur)
					cur->next = f;
				cur = f;
			
				matrix [(i*dim) + j] = cur;
			} else
				matrix [(i*dim) + j] = NULL;
		}

	va_end (args);
}


void fam<int>::apply (const fuzzy<int> & v1,
		      const fuzzy<int> & v2,
		      fuzzy<int> & result)
{

	/* for all defined rules... fire each rule. 

	   keep track of an output result by accumulating
	   the scaled moment and area represented by each firing rule.

	   finally, compute the output result by dividing the
	   moment by the area.  */

	int moment = 0;
	int area = 0;

	for (fam_member<int> * f = first_rule; f; f = f->next) {

		/* calculate the degree to which 
		       (v1 IS m1)  AND  (v2 IS m2) THEN (r IS m3) */
		
		degree d = v1.is (* f->ir) && v2.is (* f->ic);

		moment += f->r->moment () * d;
		area += f->r->area () * d;
	}

	result =  (area) ? (moment / area) : 0;
}
