/* NEFCON-I: an interactive system for realization of a neural fuzzy controller

   Copyright (C) 1994 

   Institut fuer Betriebssysteme und Rechnerverbund, Technische Universitaet
   Braunschweig, Bueltenweg 74/75, 38106 Braunschweig, Germany, 
   hereby disclaims all copyright interests in the program NEFCON-I 
   written by Hermann-Josef Diekgerdes and Roland Stellmach.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/




#include <string.h>
#include <IV-look/kit.h>
#include <IV-look/button.h>
#include <IV-look/choice.h>
#include <IV-look/telltale.h>
#include <IV-look/menu.h>
#include <InterViews/layout.h>
#include <InterViews/session.h>
#include <InterViews/rule.h>
#include <InterViews/hit.h>
#include <InterViews/observe.h>
#include <InterViews/border.h>
#include <InterViews/patch.h>
#include <InterViews/deck.h>
#include <InterViews/glyph.h>
#include <InterViews/font.h>
#include <OS/string.h>

#include "texte.h"
#include "global.h"
#include "regel_tafel.h"

/*
 *------------------------------------------------------------------------------
 *-------------------- Definition der Klasse : RegelTafel ----------------------
 *------------------------------------------------------------------------------
 */

declareActionCallback(RegelTafel);
implementActionCallback(RegelTafel);

RegelTafel::RegelTafel(const RegelBasis*, const LingVarSatz* lingvars)
{
  int spalten  = lingvars->anz_lingvars() * MAXNAME +
                 strlen(Txt(41)) + strlen(Txt(42)) + strlen(Txt(43));
  int zeilen   = 10;
  Action* cb   = new ActionCallback(RegelTafel)(this,&RegelTafel::_callback);
  _regel_tafel = new NfcStrBrowser(spalten, zeilen, cb);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RegelTafel::glyph()
 * Rueckgabewert : der RegelTafel entsprechendes Grafikelement
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
Glyph* RegelTafel::glyph()
{
  return NfcKit::instance()->text_drauf(Txt(44), _regel_tafel, true);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RegelTafel::zeig_regeln(RegelBasis*)
 * Parameter : RegelBasis* - Regel-Basis
 * Zweck : stellt angegebene Regel-Basis in dem NfcStrBrowser dar und merkt sich
 *         den Zeiger auf die Regel-Basis.
 *------------------------------------------------------------------------------
 */
void RegelTafel::zeig_regeln(const RegelBasis* regeln)
{
  char text[MAXSTRING];
  while(_regel_tafel->anzahl() > 0)
    _regel_tafel->loeschen(0);

  if(regeln != nil) {
    for(int i = 0; i < regeln->anzahl_regeln(); i++) {
      Regel* regel = regeln->hol_regel(i);
      for(RegelAusdruck ausdr = Praemisse; ausdr <= Konklusion; ausdr++) {
        if(ausdr == Praemisse)
          strcpy(text, Txt(41));               // "IF "
        else
          strcat(text, Txt(43));               // " THEN "
        for(int j = 0; j < regel->groesse(ausdr); j++) {
          NullTerminatedString lingvar(regel->bedingung(ausdr, j)->lingvar());
          strcat(text, lingvar.string());      // Name der ling. Variablen
          strcat(text, " = ");                 // " = "
          NullTerminatedString fuzzyset(regel->bedingung(ausdr, j)->fuzzyset());
          strcat(text, fuzzyset.string());     // Name des Fuzzy-Sets
          if(j != regel->groesse(ausdr) - 1)
            strcat(text, Txt(42));             // " AND "
        }
      }
    _regel_tafel->zur_auswahl(text);
    }
  }
  _regel_tafel->aktualisiere();
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RegelTafel::markiere(int)
 * Parameter : int - Nummer einer Regel
 * Zweck : markiert die Zeile des NfcStrBrowser's, in dem sich die Regel mit
 *         der angegebenen Nummer befindet.
 *------------------------------------------------------------------------------
 */
void RegelTafel::markiere(int nr)
{
  if(nr != _regel_tafel->auswahl_nr())
    _regel_tafel->markieren(nr);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RegelTafel::markiert()
 * Rueckgabewert : Nummer der Regel, die monemtan markiert ist. Falls keine
 *                 markiert ist : -1.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
int RegelTafel::markiert() { return _regel_tafel->auswahl_nr(); }

void RegelTafel::_callback()
{
  markiere(_regel_tafel->auswahl_nr());
  if(angebunden() != nil)
    angebunden()->markiere(_regel_tafel->auswahl_nr());
}


/*
 *------------------------------------------------------------------------------
 * Klasse : RegelMaske
 * Zweck  : Fuer jede linguistische Variable wird ein Eingabefeld erzeugt,
 *          mit dem sich ein Fuzzy-Set eingaben laesst. Diese Fuzzy-Sets werden
 *          zusammen mit den zugehoerigen Variablen als Regel interpretiert.
 *          Die Zuordnung der Eingabefelfer zu einzelnen Variablen erfolgt durch
 *          die Klasse SyntaxMaske.
 *------------------------------------------------------------------------------
 */
class RegelMaske : public InputHandler
{
  public : RegelMaske(int max_praemisse, int max_konklusion, Style*, Action*);
           ~RegelMaske() { delete _ed_feld; };
           void setz_fuzzyset(RegelAusdruck, int nr, String fuzzyset);
           String hol_fuzzyset(RegelAusdruck, int nr);
  private: void _init_maske(int gr_pra, int gr_kon);
           int _anz_praemisse,
               _anz_konklusion;
           EingabeFeld** _ed_feld;
};

RegelMaske::RegelMaske(int gr_pra, int gr_kon, Style* style, Action* setz_cb)
  : InputHandler(nil, style)
{
  _anz_praemisse  = gr_pra;
  _anz_konklusion = gr_kon;

  // Initialisieren der Eingabefelder fuer Praemisse und Konklusion :
  // Das Callback fuer das letzte Eingabefeld aktiviert die Setz-Funktion
  int anzahl = _anz_praemisse + _anz_konklusion;
  _ed_feld = new EingabeFeld*[anzahl];
  for(int i = 0; i < anzahl; i++) {
    if(i == anzahl - 1)
      _ed_feld[i] = new EingabeFeld(MAXNAME, LeerStr, setz_cb);
    else
      _ed_feld[i] = new EingabeFeld(MAXNAME, LeerStr, nil);
    append_input_handler(_ed_feld[i]->input_handler());
  }
  focus(_ed_feld[0]->input_handler());

  WidgetKit* wkit = WidgetKit::instance();
  LayoutKit* lkit = LayoutKit::instance();
  NfcKit* nfc_kit = NfcKit::instance();

 Glyph* konklusion = lkit->hbox();
  for(i = 0; i < _anz_konklusion; i++) {
    if(i != gr_kon - 1)
      konklusion->prepend(
        nfc_kit->text_davor(Txt(42), _ed_feld[--anzahl], false)
      );
    else
      konklusion->prepend(
        nfc_kit->text_davor(LeerStr, _ed_feld[--anzahl], false)
      );
    konklusion->prepend(lkit->hglue());
  }

  Glyph* praemisse = lkit->hbox();
  for(i = 0; i < _anz_praemisse; i++) {
    if(i != gr_pra - 1)
      praemisse->prepend(
        nfc_kit->text_davor(
          Txt(42),
          lkit->hbox(lkit->hglue(),  _ed_feld[--anzahl]),
          false
        )
      );
    else
      praemisse->prepend(lkit->vcenter(_ed_feld[--anzahl]));
    praemisse->prepend(lkit->hglue());
  }

  body(lkit->hbox(
         lkit->hglue(),
         nfc_kit->text_davor(Txt(41), praemisse, false),
         lkit->hglue(),
         nfc_kit->text_davor(Txt(43), konklusion, false),
         lkit->hglue()
       )
  );
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RegelMaske::setz_fuzzyset(RegelAusdruck, int, String)
 * Parameter : RegelAusdruck - Praemisse oder Konklusion
 *             int           - Nummer der Bedingung
 *             String        - Name des Fuzzy-Sets
 * Zweck : belegt das angegebene Eingabefeld mit gegebenem Namen.
 *------------------------------------------------------------------------------
 */
void RegelMaske::setz_fuzzyset(RegelAusdruck ausdruck, int nr, String name)
{
  if(ausdruck == Praemisse) {
    if(nr >= 0 && nr < _anz_praemisse)
      _ed_feld[nr]->text(name);
  } else
    if(nr >= 0 && nr < _anz_konklusion)
      _ed_feld[_anz_praemisse + nr]->text(name);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RegelMaske::hol_fuzzyset(RegelAusdruck, int)
 * Parameter : RegelAusdruck - Praemisse oder Konklusion
 *             int           - Nummer der Bedingung
 * Rueckgabewert : Name eines Fuzzy-Sets
 * Zweck : liefert den Eintrag des spezifizierten Eingabefeldes.
 *------------------------------------------------------------------------------
 */
String RegelMaske::hol_fuzzyset(RegelAusdruck ausdruck, int nr)
{
  if(ausdruck == Praemisse) {
    if(nr >= 0 && nr < _anz_praemisse)
      return _ed_feld[nr]->text();
  } else
    if(nr >= 0 && nr < _anz_konklusion)
      return _ed_feld[_anz_praemisse + nr]->text();
  return LeerStr;
}


/*
 *------------------------------------------------------------------------------
 * Klasse : SyntaxMaske
 * Zweck  : Fuer jede linguistische Variable wird ein NfcAuswahlButton erzeugt,
 *          mit dem sich eine der ling. Variablen auswaehlen laesst. Diese
 *          dienen zur Interpretation der Eingabefelder aus der RegelMaske, d.h.
 *          jedem AuswahlButton ist ein Eingabefeld zugeordnet, mit dem ein
 *          Fuzzy-Set der ausgewaehlten Variable eingegeben werden kann.
 *------------------------------------------------------------------------------
 */
class SyntaxMaske : public MonoGlyph
{
  public : SyntaxMaske(int anz_eingabe, int anz_ausgabe);
           ~SyntaxMaske();
           void zur_auswahl_lingvar(RegelAusdruck, String lingvar);
           String hol_lingvar(RegelAusdruck, int nr);
           void setz_lingvar(RegelAusdruck, int nr, String lingvar);
  private: int _anz_eingabevar,
               _anz_ausgabevar;
           String _leer;
           AuswahlButton** _lingvar_pra;
           AuswahlButton** _lingvar_kon;
};

SyntaxMaske::SyntaxMaske(int anz_eingabe, int anz_ausgabe)
{
  _anz_eingabevar = anz_eingabe;
  _anz_ausgabevar = anz_ausgabe;
  _leer           = Txt(45);
  WidgetKit* wkit = WidgetKit::instance();
  LayoutKit* lkit = LayoutKit::instance();
  NfcKit* nfc_kit = NfcKit::instance();

  _lingvar_kon = new AuswahlButton*[_anz_ausgabevar];
  for(int i = 0; i < _anz_ausgabevar; i++)
    _lingvar_kon[i] = new AuswahlButton(wkit, nil);
  _lingvar_pra  = new AuswahlButton*[_anz_eingabevar];
  for(i = 0; i < _anz_eingabevar; i++)
    _lingvar_pra[i] = new AuswahlButton(wkit,  nil);

  if(_anz_eingabevar > 1)
    zur_auswahl_lingvar(Praemisse, _leer);
  if(_anz_ausgabevar > 1)
    zur_auswahl_lingvar(Konklusion, _leer);

  Glyph* konklusion = lkit->hbox();
  for(i = 0; i < _anz_ausgabevar; i++)
    if(i != 0)
       konklusion->append(nfc_kit->text_davor(Txt(42), _lingvar_kon[i], false));
    else
       konklusion->append(nfc_kit->text_davor(LeerStr, _lingvar_kon[i], false));

  Glyph* praemisse = lkit->hbox();
  for(i = 0; i < _anz_eingabevar; i++)
    if(i != 0)
       praemisse->append(nfc_kit->text_davor(Txt(42), _lingvar_pra[i], false));
    else
       praemisse->append(nfc_kit->text_davor(LeerStr, _lingvar_pra[i], false));

  body(lkit->hbox(
         lkit->hglue(),
         nfc_kit->text_davor(Txt(41), praemisse, false),
         nfc_kit->text_davor(Txt(43), konklusion, false),
         lkit->hglue()
       )
  );
}

SyntaxMaske::~SyntaxMaske()
{
  delete _lingvar_pra;
  delete _lingvar_kon;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : SyntaxMaske::zur_auswahl_lingvar(RegelAusdruck, String)
 * Parameter : RegelAusdruck - Praemisse oder Konklusion
 *             String        - Name der ling. Variablen
 * Zweck : Stellt gegebene ling. Variable in allen Buttons, entweder der
 *         Praemisse oder der Konklusion, zur Auswahl.
 *------------------------------------------------------------------------------
 */
void SyntaxMaske::zur_auswahl_lingvar(RegelAusdruck ausdruck, String lingvar)
{
  int i;
  if(ausdruck == Praemisse)
    for(i = 0; i < _anz_eingabevar; i++)
      _lingvar_pra[i]->zur_auswahl(lingvar);
  else
    for(i = 0; i < _anz_ausgabevar; i++)
      _lingvar_kon[i]->zur_auswahl(lingvar);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : SyntaxMaske::hol_lingvar(RegelAusdruck, int)
 * Parameter : RegelAusdruck - Praemisse oder Konklusion
 *             int           - Nummer ling. Variablen
 * Rueckgabewert : Name der ling. Variablen
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
String SyntaxMaske::hol_lingvar(RegelAusdruck ausdruck, int nr)
{
  if(ausdruck == Praemisse) {
    if(nr >= 0 && nr < _anz_eingabevar)
      if(_lingvar_pra[nr]->auswahl() != _leer)
        return _lingvar_pra[nr]->auswahl();
  } else
    if(nr >= 0 && nr < _anz_ausgabevar)
      if(_lingvar_kon[nr]->auswahl() != _leer)
        return _lingvar_kon[nr]->auswahl();
  return LeerStr;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : SyntaxMaske::setz_lingvar(RegelAusdruck, int, String)
 * Parameter : RegelAusdruck - Praemisse oder Konklusion
 *             int           - Nummer ling. Variablen
 *             String        - Name einer ling. Variablen
 * Zweck : Setzt den gewaehlten AuswahlButton auf den gegebenen String.
 *------------------------------------------------------------------------------
 */
void SyntaxMaske::setz_lingvar(RegelAusdruck ausdruck, int nr, String name)
{
  if(name == LeerStr)
    name = _leer;
  if(ausdruck == Praemisse) {
    if(nr >= 0 && nr < _anz_eingabevar)
      _lingvar_pra[nr]->auswahl(name);
  } else
    if(nr >= 0 && nr < _anz_ausgabevar)
      _lingvar_kon[nr]->auswahl(name);
}


/*
 *------------------------------------------------------------------------------
 *--------------------- Definition der Klasse : NumRegelEd ---------------------
 *------------------------------------------------------------------------------
 */

declareActionCallback(NumRegelEd);
implementActionCallback(NumRegelEd);

NumRegelEd::NumRegelEd(
              NfcAusgabe* ausgabe,
              RegelBasis* regelbasis,
              LingVarSatz* lingvarsatz
            )
  : RegelTafel(regelbasis, lingvarsatz)
{
  ausgabe_feld = ausgabe;

  WidgetKit* wkit = WidgetKit::instance();
  LayoutKit* lkit = LayoutKit::instance();
  Action* cb;

  _nfc_kit     = NfcKit::instance();
  _regelbasis  = regelbasis;
  _lingvarsatz = lingvarsatz;
  _regel_nr    = -1;

  // Masken fuer Eingabe der Regeln erstellen
  cb = new ActionCallback(NumRegelEd)(this,&NumRegelEd::_setz_regel_cb);
  _regel_maske  = new RegelMaske(
                    _lingvarsatz->anz_lingvars(EingabeVar),
                    _lingvarsatz->anz_lingvars(AusgabeVar),
                    wkit->style(), cb
                  );
  _syntax_maske = new SyntaxMaske(
                    _lingvarsatz->anz_lingvars(EingabeVar),
                    _lingvarsatz->anz_lingvars(AusgabeVar)
                  );

  // Namen der ling. Variablen in der Syntax-Maske eintragen
  for(int i = 0; i < _lingvarsatz->anz_lingvars(); i++) {
    String name = _lingvarsatz->hol_lingvar(i)->name();
    if(_lingvarsatz->hol_lingvar(i)->typ() == EingabeVar) {
      _syntax_maske->zur_auswahl_lingvar(Praemisse, name);
      _syntax_maske->setz_lingvar(Praemisse, i, name);
    } else {
      _syntax_maske->zur_auswahl_lingvar(Konklusion, name);
      _syntax_maske->setz_lingvar(Konklusion, i, name);
    }
  }

  // Buttons fuer 'loeschen', 'setzen' und 'neu' erzeugen
  cb = new ActionCallback(NumRegelEd)(this,&NumRegelEd::_loesch_regel_cb);
  Glyph* loesch_bt = wkit->push_button(Txt(1), cb);

  cb = new ActionCallback(NumRegelEd)(this,&NumRegelEd::_setz_regel_cb);
  Glyph* setz_bt = wkit->push_button(Txt(2), cb);

  cb = new ActionCallback(NumRegelEd)(this,&NumRegelEd::_reset_regel_cb);
  Glyph* neu_bt = wkit->push_button(Txt(3), cb);

  Coord rand = 10;
  Glyph* regel_glyph = lkit->vbox(
                         lkit->vcenter(lkit->vglue(rand)),
                         _nfc_kit->text_davor(Txt(61), _syntax_maske, false),
                         lkit->vglue(rand),
                         _nfc_kit->text_davor(Txt(62), _regel_maske, false),
                         lkit->vglue(rand),
                         new HRule(wkit->foreground(), 1),
                         lkit->vglue(rand),
                         lkit->hbox(
                           lkit->hglue(),
                           loesch_bt, lkit->hglue(),
                           setz_bt, lkit->hglue(),
                           neu_bt, lkit->hglue()
                         ), lkit->vglue(rand)
                       );

  _glyph = lkit->vbox(
             lkit->vcenter(RegelTafel::glyph()),
             lkit->vglue(rand),
             _nfc_kit->text_drauf(Txt(48), regel_glyph, true),
             lkit->vglue(rand)
           );
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NumRegelEd::markiere(int)
 * Parameter : int - Nummer einer Regel
 * Zweck : markiert Regel mit gegebener Nummer.
 *------------------------------------------------------------------------------
 */
void NumRegelEd::markiere(int nr)
{
  RegelTafel::markiere(nr);

  _regel_nr = nr;
  String lingvar_name, fuzzyset_name;
  Regel* regel = _regelbasis->hol_regel(nr);
  LingVarTyp typ;
  int anzahl;
  RegelAusdruck ausdr;

  if(regel == nil)        // keine Regel ausgewaehlt, also Default-Einstellung
    for(ausdr = Praemisse; ausdr <= Konklusion; ausdr++) {
      typ    = (ausdr == Praemisse) ? EingabeVar : AusgabeVar;
      anzahl = _lingvarsatz->anz_lingvars(typ);
      for(int i = 0; i < anzahl; i++) {
        if(ausdr == Praemisse)
           lingvar_name = _lingvarsatz->hol_lingvar(EingabeVar, i)->name();
        else
           lingvar_name = _lingvarsatz->hol_lingvar(AusgabeVar, i)->name();
        _syntax_maske->setz_lingvar(ausdr, i, lingvar_name);
        _regel_maske->setz_fuzzyset(ausdr, i, LeerStr);
      }
    }
  else                    // ling. Variablen und Fuzzy-Sets der Regel anzeigen
    for(ausdr = Praemisse; ausdr <= Konklusion; ausdr++) {
      anzahl = regel->groesse(ausdr);
      for(int i = 0; i < anzahl; i++) {
        lingvar_name  = regel->bedingung(ausdr, i)->lingvar();
        fuzzyset_name = regel->bedingung(ausdr, i)->fuzzyset();
        _syntax_maske->setz_lingvar(ausdr, i, lingvar_name);
        _regel_maske->setz_fuzzyset(ausdr, i, fuzzyset_name);
      }
      typ    = (ausdr == Praemisse) ? EingabeVar : AusgabeVar;
      anzahl = _lingvarsatz->anz_lingvars(typ);
      for(i = regel->groesse(ausdr); i < anzahl; i++) {
        _syntax_maske->setz_lingvar(ausdr, i, LeerStr);
        _regel_maske->setz_fuzzyset(ausdr, i, LeerStr);
      }
    }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NumRegelEd::markiere(Regel*)
 * Parameter : Regel* - irgendeine Regel
 * Zweck : markiert angegebene Regel.
 *------------------------------------------------------------------------------
 */
void NumRegelEd::markiere(Regel* regel)
{
  boolean gefunden = false;
  int anzahl = _regelbasis->anzahl_regeln();
  for(int i = 0; !gefunden && i < anzahl; i++)
    if(*_regelbasis->hol_regel(i) == *regel) {
      gefunden = true;
      markiere(i);
      if(angebunden() != nil)
        angebunden()->markiere(i);
    }
}

/*
 *----------------------------- lokale Funktionen ------------------------------
 */

void NumRegelEd::_loesch_regel_cb()
{
  Regel regel;
  if(_lies_regel(regel)) {
    if(_regelbasis->del_regel(&regel))
      ausgabe_feld->ausgeben(Meldung, Txt(57)); // Regel geloescht
    else
      ausgabe_feld->ausgeben(Fehler, Txt(56));  // Regel nicht gefunden
    zeig_regeln(_regelbasis);
    if(angebunden() != nil)
      angebunden()->zeig_regeln(_regelbasis);
    _regel_nr = -1;
  }
}

void NumRegelEd::_setz_regel_cb()
{
  Regel regel;
  if(_lies_regel(regel)) {
    _regelbasis->add_regel(&regel);
    ausgabe_feld->ausgeben(Meldung, Txt(58));   // Regel gesetzt
    zeig_regeln(_regelbasis);
    markiere(&regel);
    if(angebunden() != nil)
      angebunden()->zeig_regeln(_regelbasis);
  }
}

void NumRegelEd::_reset_regel_cb() { markiere(_regel_nr); }

boolean NumRegelEd::_lies_regel(Regel& neue_regel)
{
  int anzahl;
  LingVarTyp typ;

  // zuerst die Groesse von Praemisse und Konklusion feststellen ...
  int groesse_pra = 0;
  int groesse_kon = 0;

  for(RegelAusdruck ausdruck = Praemisse; ausdruck <= Konklusion; ausdruck++) {
    typ = (ausdruck == Praemisse) ? EingabeVar : AusgabeVar;
    anzahl = _lingvarsatz->anz_lingvars(typ);
    for(int i = 0; i < anzahl; i++)
      if(_syntax_maske->hol_lingvar(ausdruck, i) != LeerStr) {
        if(ausdruck == Praemisse) 
          groesse_pra++;
        else
          groesse_kon++;
      }
  }

  if(groesse_pra == 0 || groesse_kon == 0) {
    ausgabe_feld->ausgeben(Fehler, Txt(55));
    return false;
  }

  // ... danach eine entsprechende Regel erzeugen ...
  Regel regel(groesse_pra, groesse_kon);

  // ... und initialisieren
  boolean ok = true;
  for(ausdruck = Praemisse; ausdruck <= Konklusion; ausdruck++) {
    int zaehler = 0;
    LingVar* lingvar;
    FuzzySet* fuzzyset;
    typ     = (ausdruck == Praemisse) ? EingabeVar : AusgabeVar;
    anzahl  = _lingvarsatz->anz_lingvars(typ);
    for(int i = 0; i < anzahl && ok; i++) {
      ok = false;
      String lingvar_name  = _syntax_maske->hol_lingvar(ausdruck, i);
      String fuzzyset_name = _regel_maske->hol_fuzzyset(ausdruck, i);
      if(lingvar_name != LeerStr) {
        if(fuzzyset_name != LeerStr) {
          lingvar = _lingvarsatz->hol_lingvar(lingvar_name);
          if(lingvar != nil) {
            fuzzyset = lingvar->hol_fuzzyset(fuzzyset_name);
            if(fuzzyset != nil) {
              regel.bedingung(ausdruck, zaehler++, lingvar_name, fuzzyset_name);
              ok = true;
            } else // Fuzzy-Set nicht gefunden
              ausgabe_feld->ausgeben(Fehler, Txt(50), fuzzyset_name, Txt(51));
          } else   // Variable nicht gefunden; sollte eigentlich nicht vorkommen
            ausgabe_feld->ausgeben(Fehler, Txt(54), lingvar_name, Txt(51));
        } else     // kein Fuzzy-Set angegeben
          ausgabe_feld->ausgeben(Fehler, Txt(52), lingvar_name, Txt(53));
      } else       // ling. Var ist 'leer'
        if(fuzzyset_name != LeerStr)
          ausgabe_feld->ausgeben(Fehler, Txt(49));
        else
          ok = true;
    }
  }
  neue_regel = regel;
  return ok;
}
