/* NEFCON-I: an interactive system for realization of a neural fuzzy controller

   Copyright (C) 1994 

   Institut fuer Betriebssysteme und Rechnerverbund, Technische Universitaet
   Braunschweig, Bueltenweg 74/75, 38106 Braunschweig, Germany, 
   hereby disclaims all copyright interests in the program NEFCON-I 
   written by Hermann-Josef Diekgerdes and Roland Stellmach.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/




#include <string.h>
#include <stdio.h>
#include <IV-look/dialogs.h>
#include <IV-look/choice.h>
#include <IV-look/telltale.h>
#include <InterViews/display.h>
#include <InterViews/background.h>
#include <InterViews/target.h>
#include <InterViews/font.h>
#include <InterViews/patch.h>
#include <InterViews/canvas.h>
#include <InterViews/border.h>

#include "texte.h"
#include "nfclook.h"

extern "C" {
  extern double strtod(const char*, char**);
}

implementActionCallback(EingabeFeld);

declareFieldEditorCallback(EingabeFeld)
implementFieldEditorCallback(EingabeFeld)


/*
 *------------------------------------------------------------------------------
 *-------------------- Definition der Klasse : AusgabeFeld ---------------------
 *------------------------------------------------------------------------------
 */

AusgabeFeld::AusgabeFeld(int spalten, int zeilen) : Patch(nil)
{
  WidgetKit* kit    = WidgetKit::instance();
  LayoutKit* layout = LayoutKit::instance();

  FontBoundingBox font_box;
  kit->font()->font_bbox(font_box);
  Coord breite = spalten * (font_box.left_bearing() + font_box.right_bearing());
  Coord hoehe  = zeilen * (font_box.ascent() + font_box.descent());
  _max_zeilen  = 40;
  _feld        = new TBScrollBox();
  Resource::ref(_feld);

  body(
    layout->hbox(
       layout->natural_span(layout->vcenter(_feld, 1), breite, hoehe),
       kit->vscroll_bar(_feld)
    )
  );
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : AusgabeFeld::ausgeben(String)
 * Parameter : String - auszugebender Text
 * Zweck : schreibt Text ins Ausgabefeld.
 *------------------------------------------------------------------------------
 */
void AusgabeFeld::ausgeben(String ausgabe)
{
  WidgetKit* kit = WidgetKit::instance();
  _feld->append(kit->label(ausgabe));
  while(_feld->count() > _max_zeilen)
    _feld->remove(0);

  //falls das Feld initialisiert wird bevor es sichtbar ist, nicht scrollen
  if(canvas() != nil)
    _feld->scroll_to(Dimension_Y, 0);
}


/*
 *------------------------------------------------------------------------------
 *----------------- Definition der Klasse : NfcStrBrowser ----------------------
 *------------------------------------------------------------------------------
 */
NfcStrBrowser::NfcStrBrowser(int spalten, int zeilen,
                             Action* cb_ok, Action* cb_abbruch)
{
  Coord rand = 3;

  _kit     = WidgetKit::instance();
  _auswahl = new FileBrowser(_kit, cb_ok, cb_abbruch);
  _strings = new StringList;

  FontBoundingBox font_box;
  _kit->font()->font_bbox(font_box);
  Coord breite = spalten * (font_box.left_bearing() + font_box.right_bearing());
  Coord hoehe  = zeilen * (font_box.ascent() + font_box.descent());

  _kit->begin_style("NfcEditorStyle","NfcEditorFont");
  LayoutKit& layout = *LayoutKit::instance();
  body(
    layout.hbox(
      layout.hglue(rand, 0, 0),
      layout.natural_span(layout.vcenter(_auswahl, 1), breite, hoehe),
      _kit->vscroll_bar(_auswahl->adjustable())
    )
  );
  _kit->end_style();
}

NfcStrBrowser::~NfcStrBrowser()
{
  if(_strings->count() > 0)
    delete _strings;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcStrBrowser::zur_auswahl(String)
 * Parameter : String - zur Auswahl stehender Text
 * Zweck : Stellt angegebenen Text als letzten Text im StrBrowser dar.
 *------------------------------------------------------------------------------
 */
void NfcStrBrowser::zur_auswahl(String neuer_str)
{
  _strings->append(neuer_str);
  Glyph* label = new Target(_kit->label(neuer_str), TargetPrimitiveHit);
  TelltaleState* status = new TelltaleState(TelltaleState::is_enabled);
  _auswahl->append_selectable(status);
  _auswahl->append(
     new ChoiceItem(status, label, _kit->bright_inset_frame(label))
  );
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcStrBrowser::ausgewaehlt()
 * Rueckgabewert : liefert ausgewaehlten Text oder einen Leer-String.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
String NfcStrBrowser::ausgewaehlt()
{
  if(_auswahl->selected() >= 0 && _auswahl->selected() < anzahl())
    return _strings->item_ref(_auswahl->selected());
  else
    return LeerStr;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcStrBrowser::loeschen(GlyphIndex)
 * Parameter : GlyphIndex - Nummer des Textes
 * Zweck : loescht Text mit angegebener Nummer aus dem StrBrowser.
 *------------------------------------------------------------------------------
 */
void NfcStrBrowser::loeschen(GlyphIndex nr)
{
  if(nr >= 0 && nr < anzahl()) {
    _auswahl->select(-1);
    _auswahl->remove_selectable(nr);
    _auswahl->remove(nr);
    _strings->remove(nr);
  }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcStrBrowser::markieren(int)
 * Parameter : int - Nummer des Textes
 * Zweck : markiert Text mit angegebener Nummer im StrBrowser.
 *------------------------------------------------------------------------------
 */
void NfcStrBrowser::markieren(int nr)
{
  if(nr >= -1 && nr < anzahl())
    _auswahl->select(nr);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcStrBrowser::markieren(String)
 * Parameter : String - Text
 * Zweck : markiert angegebenen Text im StrBrowser.
 *------------------------------------------------------------------------------
 */
void NfcStrBrowser::markieren(String str)
{
  boolean gefunden = false;
  for(GlyphIndex i = 0;  !gefunden && i < anzahl(); i++)
    if(_strings->item_ref(i) == str) {
      gefunden = true;
      _auswahl->select(i);
     }
  if(!gefunden)
    _auswahl->select(-1);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcStrBrowser::anzahl()
 * Rueckgabewert : Anzahl der Texte im StrBrowser
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
GlyphIndex NfcStrBrowser::anzahl() { return _strings->count(); }

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcStrBrowser::aktualisiere()
 * Zweck : zeichnet den StrBrowser neu.
 *------------------------------------------------------------------------------
 */
void NfcStrBrowser::aktualisiere() { _auswahl->refresh(); }


/*
 *------------------------------------------------------------------------------
 *--------------------- Definition der Klasse : NfcKit -------------------------
 *------------------------------------------------------------------------------
 */

NfcKit::NfcKit()
{
  _layout_kit = LayoutKit::instance();
  _widget_kit = WidgetKit::instance();
}

NfcKit* NfcKit::_instance;

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcKit::instance()
 * Rueckgabewert : Zeiger auf statischen NfcKit.
 * Zweck : liefert Zeiger auf statischen NfcKit; falls noch keiner existiert,
 *         wird einer mittles 'new' erzeugt.
 *------------------------------------------------------------------------------
 */
NfcKit* NfcKit::instance()
{
  if(NfcKit::_instance == nil)
    NfcKit::_instance = new NfcKit;
  return NfcKit::_instance;
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcKit::text_drauf(char* , Glyph*, boolean)
 * Parameter : char*   - Text
 *             Glyph*  - Grafikelement
 *             boolean - mit oder ohne Rand ums Grafikelement
 * Rueckgabewert : neues Grafikelement
 * Zweck : Versieht ein gegebenes Grafikelement mit einem Text. Der Text
 *         erscheint auf dem Grafikelement.
 *------------------------------------------------------------------------------
 */
Glyph* NfcKit::text_drauf(const char* text, Glyph* obj, boolean mit_rand)
{
  Glyph *elem, *erg;

  _widget_kit->begin_style("NfcInfoStyle","NfcInfoFont");
  Style* style = _widget_kit->style();
  if(mit_rand)
    elem = new Border(
             _layout_kit->margin(obj, 2), _widget_kit->foreground(), 1
           );
  else
    elem = obj;
  erg = _layout_kit->vbox(
          _layout_kit->vcenter(
            _layout_kit->hbox(_widget_kit->label(text), _layout_kit->hglue())
          ), _layout_kit->vcenter(elem)
        );
  _widget_kit->end_style();
  return erg;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcKit::text_davor(char* , Glyph*, boolean)
 * Parameter : char*   - Text
 *             Glyph*  - Grafikelement
 *             boolean - mit oder ohne Rand ums Grafikelement
 * Rueckgabewert : neues Grafikelement
 * Zweck : genau wie oben, nur dass der Text vor dem Grafikelement erscheint.
 *------------------------------------------------------------------------------
 */
Glyph* NfcKit::text_davor(const char* text, Glyph* obj, boolean mit_rand)
{
  Glyph *elem, *erg;

  _widget_kit->begin_style("NfcInfoStyle","NfcInfoFont");
  if(mit_rand)
    elem = new Border(
             _layout_kit->margin(obj, 2), _widget_kit->foreground(), 1
           );
  else
    elem = obj;
  erg = _layout_kit->hbox(
          _layout_kit->center(_widget_kit->label(text)),
          _layout_kit->center(elem)
        );
  _widget_kit->end_style();
  return erg;
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcKit::menue(MenueStruktur*, WidgetKit*, LayoutKit*)
 * Parameter : MenueStruktur* - Liste der einzelnen Menuepunkte
 *             WidgetKit*     - zu benutzendes WidgetKit
 *             LayoutKit*     - zu benutzendes LayoutKit
 * Rueckgabewert : erstelltes Menue.
 * Zweck : Macht aus einer Liste von Menuepunkt-Definitionen ein Menue.
 *         Es gibt 3 Arten von Menuepunkten :
 *            1       - check_menu_item
 *            2       - radio_menu_item
 *            default - menu_item
 *         (siehe ..../iv-3.1/src/examples/menu)
 *------------------------------------------------------------------------------
 */
Menu* NfcKit::menue(MenueStruktur* strukt, WidgetKit* wkit, LayoutKit* lkit)
{
  wkit->begin_style("NfcInfoStyle","NfcInfoFont");
  Menu* menue_balken = wkit->menubar();
  MenuItem* elem;
  for(MenueStruktur* ms = strukt; ms->eintrag != -1; ms++) {
    if(ms->eintrag == -2)
      elem = wkit->menu_item_separator();
    else {
      elem = wkit->menubar_item(wkit->fancy_label(Txt(ms->eintrag)));
      elem->menu(_unter_menue(ms->unter_menue, ms->typ, wkit, lkit));
      if(ms->cb != nil)
        elem->action(ms->cb);
    }
    menue_balken->append_item(elem);
  }
  wkit->end_style();
  return menue_balken;
}

Menu* NfcKit::_unter_menue(
        MenueStruktur* strukt, int typ, WidgetKit* wkit, LayoutKit* lkit
      )
{
  Menu* menue = wkit->pulldown();
  TelltaleGroup* group = nil;
  for(MenueStruktur* ms = strukt; ms->eintrag != -1; ms++)
    if(ms->eintrag == -2)
      menue->append_item(wkit->menu_item_separator());
    else {
      Glyph* glyph = wkit->fancy_label(Txt(ms->eintrag));
      MenuItem* elem;
      switch(typ) {
        case 1:
          elem = wkit->check_menu_item(glyph);
          break;
        case 2:
          if(group == nil)
            group = new TelltaleGroup;
          elem = wkit->radio_menu_item(group, glyph);
          break;
        default:
          elem = wkit->menu_item(glyph);
          break;
      }
      if(ms->cb == nil && ms->unter_menue != nil)
        elem->menu(_unter_menue(ms->unter_menue, ms->typ, wkit, lkit));
      else
        elem->action(ms->cb);
      menue->append_item(elem);
    }
  return menue;
}


/*
 *------------------------------------------------------------------------------
 *--------------------- Definition der Klasse : EingabeFeld --------------------
 *------------------------------------------------------------------------------
 */

EingabeFeld::EingabeFeld(int spalten, float def, Action* ok, Action* abbruch)
{
  // Default-Ausgabe ist eine Zahl
  char str[MAXSTRING];
  _float2str(def, str);
  _init(spalten, str, ok, abbruch);
}

EingabeFeld::EingabeFeld(int spalten, String def, Action* ok, Action* abbruch)
{
  // Default-Ausgabe ist ein Text
  _init(spalten, def, ok, abbruch);
}

EingabeFeld::~EingabeFeld()
{
  if(_ok_cb != nil)
    Resource::unref(_ok_cb);
  if(_abbruch_cb != nil)
    Resource::unref(_abbruch_cb);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : EingabeFeld::edit()
 * Zweck : Initiiert das Editieren.
 *------------------------------------------------------------------------------
 */
void EingabeFeld::edit() { _editor->edit(); }

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : EingabeFeld::hol_wert(FuzzyTyp&)
 *                   EingabeFeld::hol_wert(int&)
 * Parameter : FuzzyTyp&, int& - eingegebene Zahl
 * Rueckgabewert : Erfolg der Aktion.
 * Zweck : Belegt uebergebene Variable mit der im Editor eingegebenen Zahl.
 *------------------------------------------------------------------------------
 */
boolean EingabeFeld::hol_wert(FuzzyTyp& zahl)
{
  const String str = _lies_text();
  double wert = 0;
  boolean ok = _str2fuzzy(str.string(), wert);
  zahl = (FuzzyTyp) wert;
  return ok;
}

boolean EingabeFeld::hol_wert(int& zahl)
{
  const String str = _lies_text();
  double wert = 0;
  boolean ok = _str2fuzzy(str.string(), wert);
  zahl = (int) wert;
  return ok;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : EingabeFeld::setz_wert(float)
 * Parameter : float - Zahl
 * Zweck : schreibt gegebene Zahl in den Editor.
 *------------------------------------------------------------------------------
 */
void EingabeFeld::setz_wert(float zahl)
{
  char str[MAXSTRING];
  _float2str(zahl, str);
  _editor->field(str);
  _editor->select(0);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : EingabeFeld::text()
 * Rueckgabewert : eingegebener Text.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
String EingabeFeld::text() { return _lies_text(); }

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : EingabeFeld::text(String)
 * Parameter : String - Text
 * Zweck : schreibt Text in den Editor.
 *------------------------------------------------------------------------------
 */
void EingabeFeld::text(String str) { _editor->field(str); }

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : EingabeFeld::input_handler()
 * Rueckgabewert : liefert einen, dem Editor entsprenden, Input-Handler.
 * Zweck : wird benoetigt, um mehrere Eingabefelder zu verbinden (d.h. nur ein
 *         Cursor fuer alle).
 *------------------------------------------------------------------------------
 */
InputHandler* EingabeFeld::input_handler() { return _editor; }

/*
 *--------------------------- lokale Funktionen --------------------------------
 */

void EingabeFeld::_init(int spalten, String def, Action* ok, Action* abbruch)
{
  _ok_cb      = ok;
  _abbruch_cb = abbruch;
  if(ok != nil)
    Resource::ref(_ok_cb);
  if(abbruch != nil)
    Resource::ref(_abbruch_cb);

  FieldEditorAction* ed_action = new FieldEditorCallback(EingabeFeld)(
                                   this,
                                   &EingabeFeld::_ok_callback,
                                   &EingabeFeld::_nook_callback
                                 );
  char sample[MAXSTRING];
  NullTerminatedString ndef(def);
  strcpy(sample, ndef.string());
  if(spalten > strlen(sample))
    for(int i = 0; i < spalten - def.length(); i++)
      strcat(sample, " ");
  String eintrag(sample);

  WidgetKit* kit =  WidgetKit::instance();
  LayoutKit* lkit =  LayoutKit::instance();
  kit->begin_style("NfcEditorStyle","NfcEditorFont");
  _editor = DialogKit::instance()->field_editor(
              eintrag.substr(0, spalten), kit->style(), ed_action
            );
  body(lkit->variable_span(_editor, fil, 0));
  kit->end_style();
}

void EingabeFeld::_ok_callback(FieldEditor*)
{
  if(_editor->parent() != nil)
    _editor->next_focus();
  if(_ok_cb != nil)
    _ok_cb->execute();
}

void EingabeFeld::_nook_callback(FieldEditor*)
{
  if(_editor->parent() != nil)
    _editor->next_focus();
  if(_abbruch_cb != nil)
    _abbruch_cb->execute();
}

String EingabeFeld::_lies_text()
{
  int anfang = 0;
  int ende   = 0;
  const String* text = _editor->text();

  // Leerzeichen am Anfang und am Ende des Textes loeschen
  for(int i = 0; i < text->length(); i++) {
    String ch = text->substr(i, 1);
    if(ch == " ") {
      if(i == anfang)
        anfang++;
      ende++;
    } else
      ende = 0;
  }
  if(anfang == text->length())
    return LeerStr;
  else
    return text->substr(anfang, text->length() - ende - anfang);
}

boolean EingabeFeld::_str2fuzzy(const char* str, double& zahl)
{
  NullTerminatedString nstr(str);
  char** rest;
  rest = 0;
  zahl = strtod(nstr.string(), rest);
  if (rest == 0)
    return true;
  return false;
}

void EingabeFeld::_float2str(float zahl, char* str)
{
  sprintf(str, "%.3f", zahl);
  int laenge = strlen(str);
  for(int i = laenge - 1; i > 0 && str[i] == '0' ; i--)
    str[i] = '\0';
  if(str[i] == '.')
    str[i] = '\0';
}


/*
 *------------------------------------------------------------------------------
 *--------------------- Definition der Klasse : NfcLabel -----------------------
 *------------------------------------------------------------------------------
 */

NfcLabel::NfcLabel(String str, WidgetKit* kit)
{
  _kit = kit;
  FontBoundingBox font_box;
  _kit->font()->font_bbox(font_box);
  _chr_breite = font_box.left_bearing() + font_box.right_bearing();
  _chr_hoehe  = font_box.ascent() + font_box.descent();
  _breite     = 0;
  _canvas     = nil;

  text(str);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcLabel::request(Requisition&)
 * Parameter : Requisition& - Groesse des Bildschirmplatzes
 * Zweck : Initialisiert Requisition mit dem benoetigtem Bildschirmplatz.
 *------------------------------------------------------------------------------
 */
void NfcLabel::request(Requisition& req) const
{
  Requirement ry(_chr_hoehe);
  req.require(Dimension_Y, ry);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcLabel::allocate(Canvas*, Allocation&, Extension&)
 * Parameter : Canvas*     - gesammter Bildschirmplatz
 *             Allocation& - zugewiesener Anteil
 *             Extension&  - was wird davon wirklich benoetigt ?
 * Zweck : Initialisiert Variablen aufgrund des zugewiesenen Bildschirmplatzes.
 *------------------------------------------------------------------------------
 */
void NfcLabel::allocate(Canvas* c, const Allocation& a, Extension& e)
{
  _breite = a.x_allotment().span();
  _canvas = c;
  _links  = a.left();
  _rechts = a.right();
  _oben   = a.top();
  _unten  = a.bottom();

  e.set(c, a);
  _init();
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcLabel::draw(Canvas*, Allocation&)
 * Parameter : Canvas*     - gesammter Bildschirmplatz
 *             Allocation& - zugewiesener Anteil
 * Zweck : Zeichnet den Text im zugewiesenen Bereich.
 *------------------------------------------------------------------------------
 */
void NfcLabel::draw(Canvas* c, const Allocation& a) const
{
  for(int i = 0; _text[i] != '\0' && i < _darstellbar; i++)
    c->character(
         _kit->font(),
         (long) _text[i],
         _chr_breite,
         _kit->foreground(),
         a.left() + _offset + i * _chr_breite,
         a.bottom()
       );
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcLabel::text(String)
 * Parameter : String - Text
 * Zweck : Setzt den darzustellenden Text.
 *------------------------------------------------------------------------------
 */
void NfcLabel::text(String text)
{
  _text = text;
  _init();
}

void NfcLabel::_init()
{
  // Feststellen, wieviel Zeichen darstellbar sind
  FontBoundingBox font_box;
  int i = _text.length();
  do {
    _kit->font()->string_bbox(_text.string(), i, font_box);
  } while(font_box.width() > _breite && --i > 0);
  _darstellbar = i;

  // evtl. mittig zentrieren
  _offset = (_breite - font_box.width()) / 2;
}


/*
 *------------------------------------------------------------------------------
 *------------------ Definition der Klasse : AuswahlButton ---------------------
 *------------------------------------------------------------------------------
 */

AuswahlButton::AuswahlButton(WidgetKit* kit, Action* cb, DimensionName dim)
  : InputHandler(nil, kit->style())
{
  _kit         = kit;
  _callback    = cb;
  _dimension   = dim;
  _ausgewaehlt = -1;
  _deck        = new Deck;
  _patch       = new Patch(_deck);
  body(new InputHandler(_patch, _kit->style()));
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : AuswahlButton::press(Event&)
 * Parameter : Event& - Ereignis
 * Zweck : registriert das Anklicken des Buttons und wechselt den dargestellten
 *         Text.
 *------------------------------------------------------------------------------
 */
void AuswahlButton::press(const Event&)
{
  _ausgewaehlt += 1;
  if(_ausgewaehlt >= _deck->count())
    _ausgewaehlt -= (int) _deck->count();
  _deck->flip_to(_ausgewaehlt);
  _patch->redraw();
  if(_callback != nil)
    _callback->execute();
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : AuswahlButton::zur_auswahl(String)
 * Parameter : String - zur Auswahl stehender Text
 * Zweck : nimmt einen neuen Text in die Liste der zur Auswahl stehenden Texte
 *         auf.
 *------------------------------------------------------------------------------
 */
void AuswahlButton::zur_auswahl(String str)
{
  _merk_strings.append(str);

  char text[MAXSTRING];
  NullTerminatedString nstr(str);
  sprintf(text, "%s%s%s", Txt(46), nstr.string(), Txt(47));

  _kit->begin_style("NfcEditorStyle", "NfcEditorFont");
  LayoutKit& lkit = *LayoutKit::instance();
  Glyph* eintrag;
  if(_dimension == Dimension_X)
    eintrag = lkit.hbox(lkit.hglue(), _kit->label(text), lkit.hglue());
  else {
    String hilf_str(text);
    eintrag = lkit.vbox(lkit.vcenter(lkit.vglue()));
    int anzahl = hilf_str.length();
    for(int i = 0; i < anzahl; i++)
      eintrag->append(lkit.hcenter(_kit->label(hilf_str.substr(i, 1))));
    eintrag->append(lkit.vglue());
  }
  _deck->append(eintrag);
  _kit->end_style();

  if(_ausgewaehlt < 0) {
    _ausgewaehlt = 0;
    _deck->flip_to(_ausgewaehlt);
    _patch->redraw();
  }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : AuswahlButton::auswahl(String)
 * Parameter : String - Text
 * Zweck : stellt gegebenen Text dar, falls dieser zur Auswahl steht.
 *------------------------------------------------------------------------------
 */
void AuswahlButton::auswahl(String str)
{
  boolean gefunden = false;
  for(int i = 0; i < _merk_strings.count() && !gefunden; i++)
    if(_merk_strings.item(i) == str) {
      gefunden = true;
      _ausgewaehlt = i;
      _deck->flip_to(i);
      _patch->redraw();
    }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : AuswahlButton::auswahl()
 * Rueckgabewert : momentan dargestellter Text.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
String AuswahlButton::auswahl()
{
  if(_ausgewaehlt != -1)
    return _merk_strings.item_ref(_ausgewaehlt);
  else
    return LeerStr;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : AuswahlButton::auswahl()
 * Rueckgabewert : Nummer des momentan dargestellten Textes.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
void AuswahlButton::auswahl_nr(int nr)
{
  if(nr >= 0 && nr != _ausgewaehlt) {
    _ausgewaehlt = nr % (int) _deck->count();
    _deck->flip_to(_ausgewaehlt);
    _patch->redraw();
  }
}


/*
 *------------------------------------------------------------------------------
 *--------------------- Definition der Klasse : NfcAction ----------------------
 *------------------------------------------------------------------------------
 */

NfcAction::NfcAction(int* var, int wert, Action* cb)
{
  _var      = var;
  _wert     = wert;
  _callback = cb;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcAction::execute()
 * Zweck : Initialisiert gegebene Variable und fuehrt Callback aus.
 *------------------------------------------------------------------------------
 */
void NfcAction::execute()
{
  *_var = _wert;
  if(_callback != nil)
    _callback->execute();
}


/*
 *------------------------------------------------------------------------------
 *------------------ Definition der Klasse : NfcDialogFenster ------------------
 *------------------------------------------------------------------------------
 */

NfcDialogFenster::NfcDialogFenster(Glyph* inhalt)
{
  _fenster = nil;
  _inhalt  = inhalt;
  _x_pos   = _y_pos = 0;
  if(_inhalt != nil)
    Resource::ref(_inhalt);
}

NfcDialogFenster::~NfcDialogFenster()
{
  if(_fenster != nil)
    delete _fenster;
  if(_inhalt != nil)
    Resource::unref(_inhalt);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcDialogFenster::position(Coord, Coord)
 * Parameter : Coord x - x-Position des Fensters
 *             Coord y - y-Position des Fensters
 * Zweck : Belegt Parameter mit Fensterposition.
 *------------------------------------------------------------------------------
 */
void NfcDialogFenster::position(Coord& x, Coord& y)
{
  x = _x_pos;
  y = _y_pos;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcDialogFenster::sichtbar()
 * Rueckgabewert : ist Fenster gerade sichtbar ?
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
boolean NfcDialogFenster::sichtbar()
{
  if(_fenster != nil)
    return _fenster->is_mapped();
  else
    return false;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcDialogFenster::einblenden(
 *                     Window*, String, String, Coord, Coord
 *                   )
 * Parameter : Window*    - Elternfenster
 *             String     - Name des Fensters
 *             String     - Name des Icons
 *             Coord x, y - Position des Fensters
 * Zweck : blendet Fenster ein. Falls Postion == -1 gilt, wird keine Position
 *         vorgegeben und das Fenster wird vom Benutzer per Maus plaziert.
 *------------------------------------------------------------------------------
 */
void NfcDialogFenster::einblenden(
       Window* w, String name, String icon_name, Coord x_pos, Coord y_pos
     )
{
  if(w != nil && _inhalt != nil) {
    WidgetKit* wkit = WidgetKit::instance();
    if(name != LeerStr)
      wkit->style()->attribute("name", name);
    if(icon_name != LeerStr)
      wkit->style()->attribute("iconName", icon_name);
    if(_fenster == nil)
      _fenster = new TopLevelWindow(
                   new Background(_inhalt, wkit->background())
                 );
    if(!_fenster->is_mapped()) {
      _fenster->display(w->display());
      //_fenster->align(0, 1);
      if(x_pos != -1)
        _fenster->place(x_pos, y_pos);
      _fenster->map();
      _fenster->display()->sync();
    }
  }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcDialogFenster::ausblenden()
 * Zweck : blendet Fenster aus.
 *------------------------------------------------------------------------------
 */
void NfcDialogFenster::ausblenden()
{
  if(_fenster != nil && _fenster->is_mapped()) {
    _x_pos = _fenster->left();
    _y_pos = _fenster->bottom();
    _fenster->unmap();
  }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcDialogFenster::body(Glyph*)
 * Parameter : Glyph* - Grafikelement
 * Zweck : Setzt den Fensterinhalt.
 *------------------------------------------------------------------------------
 */
void NfcDialogFenster::body(Glyph* inhalt)
{ 
  if(_inhalt != nil)
    Resource::unref(_inhalt);
  _inhalt = inhalt;
  if(_inhalt != nil)
    Resource::ref(_inhalt);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcDialogFenster::window()
 * Rueckgabewert : Zeiger auf TopLevelWindow.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
Window* NfcDialogFenster::window() { return _fenster; }


/*
 *------------------------------------------------------------------------------
 *----------------------- Definition der Klasse : NfcAusgabe -------------------
 *------------------------------------------------------------------------------
 */

NfcAusgabe::NfcAusgabe(AusgabeFeld* ausgabefeld)
  : MonoGlyph(ausgabefeld)
{
  _feld   = ausgabefeld;
  _toggle = false;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcAusgabe::ausgeben(AusgabeTyp, String)
 *                   NfcAusgabe::ausgeben(AusgabeTyp, String, String)
 *                   NfcAusgabe::ausgeben(AusgabeTyp, String, String, String)
 * Parameter : AusgabeTyp - Fehler, Warnung, Meldung oder Info
 *             String ..  - Texte
 * Zweck : Schreibt angegebene Texte ins Ausgabefeld, pro Text eine Zeile.
 *------------------------------------------------------------------------------
 */
void NfcAusgabe::ausgeben(AusgabeTyp typ, String str1)
{
  ausgeben(typ, str1, String(LeerStr), String(LeerStr));
}

void NfcAusgabe::ausgeben(AusgabeTyp typ, String str1, String str2)
{
  ausgeben(typ, str1, str2, String(LeerStr));
}

void NfcAusgabe::ausgeben(AusgabeTyp typ, String str1, String str2, String str3)
{
  if(_feld != nil) {
    NullTerminatedString nstr1(str1);
    NullTerminatedString nstr2(str2);
    NullTerminatedString nstr3(str3);
    char text[MAXSTRING];

    sprintf(text, "%s%s%s", nstr1.string(), nstr2.string(), nstr3.string());

    if(typ != Info && !_toggle)
      _feld->ausgeben(LeerStr);

    if(typ == Fehler)
      _feld->ausgeben(Txt(190));
    else if(typ == Warnung)
      _feld->ausgeben(Txt(189));
    _feld->ausgeben(text);

    // Bei Infos wird keine Leerzeile angehaengt (oft gehoeren hier mehrere
   // aufeinanderfolgende Ausgaben zusammen); Eine Trennung zu nachfolgenden
  // Meldungen geschieht durch explizietes Ausgeben eines LeerStrings.

    if(typ != Info && (_toggle = !_toggle))
      _feld->ausgeben(LeerStr);
    _feld->aktualisiere();
  }
}

