/* NEFCON-I: an interactive system for realization of a neural fuzzy controller

   Copyright (C) 1994 

   Institut fuer Betriebssysteme und Rechnerverbund, Technische Universitaet
   Braunschweig, Bueltenweg 74/75, 38106 Braunschweig, Germany, 
   hereby disclaims all copyright interests in the program NEFCON-I 
   written by Hermann-Josef Diekgerdes and Roland Stellmach.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/




#ifndef koosystem_h
#define koosystem_h

#include <IV-look/kit.h>
#include <InterViews/glyph.h>
#include <InterViews/color.h>

#include "lingvar.h"

/*
 * -----------------------------------------------------------------------------
 * Klasse : KooKreuz
 * Zweck  : Definiert ein Koordinatenkreuz mit 2 Achsen und einer gegebenen
 *          Anzahl von Einheiten. Innerhalb dieses Koordinatenkreuzes koennen
 *          dann Linien gezeichnet werden, ausserdem besteht die Moeglichkeit
 *          diese Linien zu markieren. Die verwendeten Einheiten sind die, mit
 *          denen die Achsen skaliert wurden, d.h. 0 ist immer ganz links bzw.
 *          ganz unten auf der Achse und 5 ganz rechts bzw. oben, wenn die
 *          Anzahl der Einheiten 5 betraegt.
 * -----------------------------------------------------------------------------
 */
class KooKreuz : public Glyph
{
  public : KooKreuz();
           virtual void request(Requisition&) const;
           virtual void allocate(Canvas*, const Allocation&, Extension&);
           virtual void draw(Canvas*, const Allocation&) const;
           float x_einh(Coord x);
           float y_einh(Coord y);
           void skaliere(DimensionName, int anzahl_einheiten, int null_pos);
           void beschriften(DimensionName, float einh, char*, int max) const;
           void linie(float x1, float y1, float x2, float y2) const;
           void zeig_linie(float x1, float y1, float x2, float y2) const;
           void aktualisiere();
  private: void _init();
           void _markierung(Coord x, Coord y) const;
           Coord _x_koo(float einheit) const;
           Coord _y_koo(float einheit) const;
           Coord _anz_einh_abz,           // darstellungsabhaengige Koordianten
                 _anz_einh_ord,
                 _laenge_abz,
                 _laenge_ord,
                 _x0, _y0,
                 _laenge_pfeil,           // konstante Koordinaten
                 _links, _rechts,
                 _oben,  _unten,
                 _rand_x, _rand_y,
                 _strich,
                 _chr_b, _chr_h;
           Canvas* _canvas;               // zum aktualisieren der Grafik
           int _max_str;                  // Maximale String-Laenge
           WidgetKit* _kit;
};

/*
 * -----------------------------------------------------------------------------
 * Klasse : KooSystem
 * Zweck  : Definiert ein Koordinatensystem fuer einen bestimmten Werte-
 *          Bereich. Innerhalb dieses Koordinatensystems koennen dann Linien
 *          gezeichnet und markiert werden.
 * -----------------------------------------------------------------------------
 */
class KooSystem : public KooKreuz
{
  public : KooSystem(
             FuzzyTyp von_x, FuzzyTyp bis_x, FuzzyTyp von_y, FuzzyTyp bis_y
           );
           void neue_skala(DimensionName, FuzzyTyp von, FuzzyTyp bis);
           virtual void draw(Canvas*, const Allocation&) const;
           void linie(FuzzyTyp x1, FuzzyTyp y1, FuzzyTyp x2, FuzzyTyp y2) const;
           void zeig_linie(
                  FuzzyTyp x1, FuzzyTyp y1, FuzzyTyp x2, FuzzyTyp y2
                ) const;
           boolean drin(FuzzyTyp x, FuzzyTyp y) const;
           FuzzyTyp xkoo2wert(Coord x);
           FuzzyTyp ykoo2wert(Coord y);
  private: void _skaliere(
                  DimensionName, FuzzyTyp von, FuzzyTyp bis, int& null_pos
                );
           float _such_einheit(FuzzyTyp intervall);
           float _einheit(DimensionName, FuzzyTyp) const;
           boolean _runden(DimensionName, float wert, int& einheit) const;
           boolean _darstellbar(float zahl, float einheit, float& rund) const;
           void _beschriften(DimensionName) const;
           float _einheit_x, _einheit_y;
           float _min_wert_x, _max_wert_x,
                 _min_wert_y, _max_wert_y;
           int _min_anz_einh;
};

/*
 * --------------------- Inlinie - Funktionen fuer KooKreuz --------------------
 */

inline Coord KooKreuz::_x_koo(float einheit) const
{
  if(_anz_einh_abz > 0)
    return _links + _rand_x + (_laenge_abz / _anz_einh_abz) * einheit;
  else
    return _links + _rand_x;
}

inline Coord KooKreuz::_y_koo(float einheit) const
{
  if(_anz_einh_ord > 0)
    return _unten + _rand_y + (_laenge_ord / _anz_einh_ord) * einheit;
  else
    return _unten + _rand_y;
}

#endif
