/* NEFCON-I: an interactive system for realization of a neural fuzzy controller

   Copyright (C) 1994 

   Institut fuer Betriebssysteme und Rechnerverbund, Technische Universitaet
   Braunschweig, Bueltenweg 74/75, 38106 Braunschweig, Germany, 
   hereby disclaims all copyright interests in the program NEFCON-I 
   written by Hermann-Josef Diekgerdes and Roland Stellmach.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/




#include <fstream.h>
#ifndef anwendung_h
#define anwendung_h

#include <OS/list.h>
#include <InterViews/action.h>

#include "global.h"
#include "lingvar.h"

/*
 *------------------------------------------------------------------------------
 * Klasse : Anwendung
 * Zweck  : Durch diese Klasse wird ein zu steuerndes Anwendungs-Programm
 *          definiert. Die Definition umfasst den Namen des Programms (evtl. mit
 *          Pfadangabe) sowie die Reihenfolge der Ein/Ausgabevariablen.
 *          Die Klasse 'Anwendung' stellt ausserdem Funktionen zur Kommunikation
 *          zum Anwendungs-Prg. zur Verfuegung.
 *          Die Verbindung zum Anwendungs-Prg verlaeuft ueber dessen Standart-
 *          Ein/Ausgabe.
 *          Voraussetzung fuer die Definition der Anwendung ist das vorliegen
 *          eines Satzes ling. Var.; dieser legt Anzahl und Namen der Ein/Aus-
 *          gabevariablen fest.
 *------------------------------------------------------------------------------
 */
class Anwendung
{
  public : Anwendung(const LingVarSatz*);
           ~Anwendung();
           Anwendung(const Anwendung&);

           void starten(Ausgabe*);
           boolean aktiv();
           void link(Action* ok, Action* nook);
           void unlink();
           FuzzyTyp* messwerte();
           void steuerwerte(FuzzyTyp*);

           void name(String);
           String name() const;

           void var(LingVarTyp, String name);
           String var(LingVarTyp, int nr);
           void loesch_var(LingVarTyp);
           int anzahl(LingVarTyp) const;

           Anwendung& operator=(const Anwendung&);
  private: void _init();
           int _lies_eingaben(int fd);       // Callback fuer Dispatcher
           const LingVarSatz* _lingvarsatz;
           Action* _ok_callback;             // Callback fuer '_lies_eingaben'
           Action* _nook_callback;           // Callback fuer '_lies_eingaben'

           CopyString _name;                 // Name des Anwendungs-Prg.'s
           StringList *_eingabe_var,         // zur Definition der Reihenfolge
                      *_ausgabe_var;         // der Ein/Ausgabevariablen

           FuzzyTyp* _mess_werte;            // Zeiger auf Messwerte
           FuzzyTyp* _eingabe_werte;         // Puffer fuer eingelesene Werte
           int *_eingabe_folge,              // Reihenfolge der Ein-
               *_ausgabe_folge;              // Ausgabe-Variablen

           int _merk_pid;                    // PID des Anwendungsprg.
           int _kanal[2];                    // File-Deskriptoren fuer Pipe
           ifstream* _quelle;                // Kanaele zur Kommunikation
           ofstream* _ziel;                  // mit Anwendungsprogramm
};

// Ruckgabewert 'nil' <=> Regelung ist fehlgeschlagen; z.B. Pendel ist umgekippt
inline FuzzyTyp* Anwendung::messwerte() { return _mess_werte; }

inline void Anwendung::name(String name) { _name = name; }

inline String Anwendung::name() const { return _name; }

inline int Anwendung::anzahl(LingVarTyp typ) const
{
  return _lingvarsatz->anz_lingvars(typ);
}

inline void Anwendung::unlink() { link(nil, nil); }

#endif
