/*
 * (c) foerster
 * geaendert : diekgers       (Kommentare stimmen nicht immer)
 */

/* ---------------------------------------------------------------------
   Implementationsdatei fuer das zu steuernde System des

               Neural-Fuzzy-Controller ARIC         (Berenji)

   Weitere Informationen bitte in

   Hamid R. Berenji
   - A Reinforcement Learning - Based Architecture for
     Fuzzy Logic Control
   aus : International Journal of Approximate Reasoning,
	 1992, Nummer 6, Seiten 267-292

   nachschlagen.
   --------------------------------------------------------------------- */

#include <math.h>
#include "aric.h"

#define g    9.81  /* Gravitationskonstante */
#define m_p  1.0   /* Masse des Pendels */
#define m_c  1.0   /* Masse des Wagens */
#define l    0.5   /* Laenge des Pendels */
#define mu_p 0.33  /* Reibungskoeffizient des Pendels am Wagen */
#define mu_c 0.33  /* Reibungskoeffizient des Wagens auf der Schiene */
#define h    0.1   /* Schrittweite beim Runge-Kutta-Verfahren */

/* Grenzwert des Pendels in Rad */
#define theta_grenze 0.6   /* entspricht 34.38 */

/* Grenzwert des Wagens in m */
#define x_grenze     10.0               

/*
 *----------------------------- (c) diekgers -----------------------------------
 */
float pi() { return acos(-1.0); }

int sgn(float x) { return (x >= 0) ? 1 : -1; }

float grad2bog(float grad) { return (grad / 180.0) * pi(); };

float bog2grad(float bogen) { return (bogen / pi()) * 180.0; };

PendelSim::PendelSim()
{
  _werte[0] = 0;
  _werte[1] = 0;
  _werte[2] = 0;
  _werte[3] = 0;
  _pi       = pi();
}

/*
 *------------------------------------------------------------------------------
 */


/* ====================================================================
   Als Funktionswert wird die 2te Ableitung der Differentielgleichung
   zurueckgegeben, die den Winkel des Pendels beschreibt.
   In :  t   (Winkel theta)
	 t_p (Winkelgeschwindigkeit)
	 x_p (Geschindigkeit des Wagens)
	 f   (die Kraft die am Fusspunkt auf das Pendel wirkt)
   Out:  theta''
   ==================================================================== */
float PendelSim::_winkel(float t, float t_p, float x_p, float f)
{
 float zaehler;
 float nenner;

 zaehler = g * sin(t) + cos(t) * (-f - m_p * l * pow(t_p, 2.0) * sin(t) +
           mu_c * ((float) sgn(x_p))) / (m_c + m_p) - (mu_p * t_p) / (m_p * l);
 nenner  = l * (4.0 / 3.0 - (m_p * pow(cos(t), 2.0)) / (m_c + m_p));

 return (zaehler / nenner);
}


/* ====================================================================
   Als Funktionswert wird die 2te Ableitung der Differentialgleichung
   zurueckgegeben, die die Position des Wagens beschreibt.
   In :  t    (Winkel theta)
	 t_p  (Winkelgeschwindigkeit)
	 t_2p (Winkelbeschleunigung)
	 x_p  (Geschindigkeit des Wagens)
	 f    (die Kraft die am Fusspunkt auf das Pendel wirkt)
   Out:  x''
   ==================================================================== */
float PendelSim::_position(float t, float t_p, float t_2p, float x_p, float f)
{
 float zaehler;
 float nenner;

 zaehler = f + m_p * l * (pow(t_p, 2.0) * sin(t) - t_2p * cos(t)) - mu_c *
           ((float) sgn(x_p));
 nenner  = m_c + m_p;

 return (zaehler / nenner);
}


/* ====================================================================
   Der neue Zustand des Systems nach Anwendung der Kraft 'force' wird
   errechnet. Das System ist durch zwei Differentialgleichungen 2ter
   Ordnung beschrieben und wird numerisch mit einem Runge-Kutta-Verfahren
   4ter Ordnung geloest. Dazu wurde das System 2ter Ordnung in ein System
   1ter verwandelt fuer das die notwendigen Parameter einfach zu bestimmen
   sind. Die Dgln. muessen nach y'' aufgeloest werden und werden in
   den Funktionen
       _winkel(theta, theta_punkt, x_punkt, force) bzw.
       position(theta, theta_punkt, theta_2punkt, x_punkt, force)
   berechnet. Die Schrittweite h ist als Konstante definiert.
   In : force       (Kraft die angewendet wird)
 	x           (Position des Wagens)
 	x_p         (Geschwindigkeit des Wagens)
 	theta       (Winkel des Pendels)
 	theta_punkt (Winkelgeschwindigkeit des Pendels)
   Out : Reinforcement des Zustandes
   ==================================================================== */
void PendelSim::simulation(float force)
{
 float y_k[4];    /* y_k sind die Statusvariablen */
 float k[4][4];   /* Parameter fuer Runge-Kutta   */

 /* printf("(PLANT) calculating new system status ...\n"); */

 y_k[0] = _werte[2];
 y_k[1] = _werte[3];
 y_k[2] = _werte[0];
 y_k[3] = _werte[1];

 /* Parameter K1 fuer Runge-Kutta */
 k[0][0] = y_k[1];
 k[0][1] = _winkel(y_k[0], y_k[1], y_k[3], force);
 k[0][2] = y_k[3];
 k[0][3] = _position(y_k[0], y_k[1], k[0][1], y_k[3], force);

 /* Parameter K2 fuer Runge-Kutta */
 k[1][0] = y_k[1] + 0.5 * h * k[0][1];
 k[1][1] = _winkel(
             y_k[0] + 0.5 * h * k[0][0],
             y_k[1] + 0.5 * h * k[0][1],
             y_k[3] + 0.5 * h * k[0][3],
             force
           );
 k[1][2] = y_k[3] + 0.5 * h * k[0][3];
 k[1][3] = _position(
             y_k[0] + 0.5 * h * k[0][0],
             y_k[1] + 0.5 * h * k[0][1],
             k[1][1],
             y_k[3] + 0.5 * h * k[0][3],
             force
           );

 /* Parameter K3 fuer Runge-Kutta */
 k[2][0] = y_k[1] + 0.5 * h * k[1][1];
 k[2][1] = _winkel(
             y_k[0] + 0.5 * h * k[1][0],
             y_k[1] + 0.5 * h * k[1][1],
             y_k[3] + 0.5 * h * k[1][3],
             force
           );
 k[2][2] = y_k[3] + 0.5 * h * k[1][3];
 k[2][3] = _position(
             y_k[0] + 0.5 * h * k[1][0],
             y_k[1] + 0.5 * h * k[1][1],
             k[2][1],
             y_k[3] + 0.5 * h * k[1][3],
             force
           );

 /* Parameter K4 fuer Runge-Kutta */
 k[3][0] = y_k[1] + h * k[2][1];
 k[3][1] = _winkel(
             y_k[0] + h * k[2][0],
             y_k[1] + h * k[2][1],
             y_k[3] + h * k[2][3],
             force
           );
 k[3][2] = y_k[3] + h * k[2][3];
 k[3][3] = _position(
             y_k[0] + h * k[2][0],
             y_k[1] + h * k[2][1],
             k[3][1],
             y_k[3] + h * k[2][3],
             force
           );

 /* Berechnung des neuen Zustandes mit Runge-Kutta */

 /* Winkel */
 _werte[2] = y_k[0] + (1.0 / 6.0) * h *
             (k[0][0] + 2.0 * k[1][0] + 2.0 * k[2][0] + k[3][0]);

 /* Winkelgeschwindigkeit */
 _werte[3] = y_k[1] +(1.0 / 6.0) * h *
             (k[0][1] + 2.0 * k[1][1] + 2.0 * k[2][1] + k[3][1]);

 /* Position */
 _werte[0] = y_k[2] + (1.0 / 6.0) * h *
             (k[0][2] + 2.0 * k[1][2] + 2.0 * k[2][2] + k[3][2]);

 /* Geschwindigkeit */
 _werte[1] = y_k[3] + (1.0 / 6.0) * h *
             (k[0][3] + 2.0 * k[1][3] + 2.0 * k[2][3] + k[3][3]);

 while (_werte[2] > _pi)
  {
   _werte[2] -= _pi;
   _werte[2] = -_pi + (_werte[2]);
  }
 while (_werte[2] < -_pi)
  {
   _werte[2] += _pi;
   _werte[2] = _pi - (_werte[2]);
  }
}
