//---------------------------------------------------------------------------
//
// FZNUM.H
//
//    Fuzzy number type declarations.
//
//    Based on A. Kaufmann, M. Gupta, "Introduction to Fuzzy Arithmetic",
//    Van Nostrand Reinhold, New York, 1991.
//
// Contents:
//
//    define NPLEV
//
//    class FzNum
//
//    inline ConfInt FzNum::GetBase()
//    inline double FzNum::GetBaseSize()
//
//    inline FzNum FzNum::operator+()
//
//    inline FzNum operator+( double opnd1, FzNum opnd2)
//    inline FzNum operator-( double opnd1, FzNum opnd2)
//    inline FzNum operator*( double opnd1, FzNum opnd2)
//
// Author: S. Deodato ( 03.03.94)
//
//---------------------------------------------------------------------------

#ifndef ___FZNUM_H
#define ___FZNUM_H

#include "confint.h"

#define NPLEV 10 // number of used presumption levels


//---------------------------------------------------------------------------
//
// class FzNum
//
//    Fuzzy number representation.
//
// Members:
//
//    ConfInt ciPLev[ NPLEV + 1]
//       presumption levels used as internal representation of the fuzzy
//       number
//
// Author: S. Deodato ( 03.03.94)
//
//---------------------------------------------------------------------------


class FzNum

{

   protected:

      ConfInt ciPLev[ NPLEV + 1];

   public:

      FzNum() {}

      FzNum( double value);
      FzNum( double left, double middle, double right);
      FzNum( double lBase, double lTop, double rTop, double rBase);

      ConfInt GetBase();
      double GetBaseSize();

      double operator[]( double value);

      FzNum operator+();
      FzNum operator-();

      FzNum operator+( double opnd2);
      FzNum operator+( FzNum opnd2);
      FzNum operator-( double opnd2);
      FzNum operator-( FzNum opnd2);
      FzNum operator*( double opnd2);
      FzNum operator*( FzNum opnd2);
      FzNum operator/( double opnd2);
      FzNum operator/( FzNum opnd2);

      FzNum operator+=( double opnd2);
      FzNum operator+=( FzNum opnd2);
      FzNum operator-=( double opnd2);
      FzNum operator-=( FzNum opnd2);
      FzNum operator*=( double opnd2);
      FzNum operator*=( FzNum opnd2);
      FzNum operator/=( double opnd2);
      FzNum operator/=( FzNum opnd2);

      friend FzNum operator+( double opnd1, FzNum opnd2);
      friend FzNum operator-( double opnd1, FzNum opnd2);
      friend FzNum operator*( double opnd1, FzNum opnd2);
      friend FzNum operator/( double opnd1, FzNum opnd2);

      friend ostream& operator<<( ostream& smStream, FzNum item);

      void print();

      friend FzNum min( FzNum opnd1, FzNum opnd2);
      friend FzNum max( FzNum opnd1, FzNum opnd2);

      friend FzNum sin( FzNum opnd);
      friend FzNum cos( FzNum opnd);

      friend double fzSM( FzNum opnd1, FzNum opnd2, double lambda = 0.5);
};


//---------------------------------------------------------------------------
//
// inline ConfInt FzNum::GetBase()
//
//    Return the confidence interval at presumption level 0.
//
// Argumets:
//
//    none
//
// Return value:
//
//    the requested confidence interval
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 26.08.94)
//
//---------------------------------------------------------------------------

inline ConfInt FzNum::GetBase()

{
  return ciPLev[ 0];
}


//---------------------------------------------------------------------------
//
// inline double FzNum::GetBaseSize()
//
//    Return the size of the confidence interval at presumption level 0.
//
// Argumets:
//
//    none
//
// Return value:
//
//    the requested size
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 26.08.94)
//
//---------------------------------------------------------------------------

inline double FzNum::GetBaseSize()

{
  return ciPLev[ 0].GetSize();
}


//---------------------------------------------------------------------------
//
// inline FzNum FzNum::operator+()
//
//    Apply the unary plus operator to a fuzzy number.
//
// Arguments:
//
//    none
//
// Return value:
//
//    the resulting fuzzy number
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 03.03.94)
//
//---------------------------------------------------------------------------

inline FzNum FzNum::operator+()

{
   return *this;
}


//---------------------------------------------------------------------------
//
// inline FzNum operator+( double opnd1, FzNum opnd2)
//
//    Addiction between a singleton and a fuzzy number.
//
// Arguments:
//
//    double opnd1
//    FzNum opnd2
//       addiction operands
//
// Return value:
//
//    the resulting fuzzy number
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 03.03.94)
//
//---------------------------------------------------------------------------

inline FzNum operator+( double opnd1, FzNum opnd2)

{
   return opnd2 + opnd1;
}


//---------------------------------------------------------------------------
//
// inline FzNum operator-( double opnd1, FzNum opnd2)
//
//    Subtraction between a singleton and a fuzzy number.
//
// Arguments:
//
//    double opnd1
//    FzNum opnd2
//       subtraction operands
//
// Return value:
//
//    the resulting fuzzy number
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 03.03.94)
//
//---------------------------------------------------------------------------

inline FzNum operator-( double opnd1, FzNum opnd2)

{
   return -opnd2 + opnd1;
}


//---------------------------------------------------------------------------
//
// inline FzNum operator*( double opnd1, FzNum opnd2)
//
//    Multiplication between a singleton and a fuzzy number.
//
// Arguments:
//
//    double opnd1
//    FzNum opnd2
//       multiplication operands
//
// Return value:
//
//    the resulting fuzzy number
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 03.03.94)
//
//---------------------------------------------------------------------------

inline FzNum operator*( double opnd1, FzNum opnd2)

{
   return opnd2 * opnd1;
}


#  endif // ___FZNUM_H
