// -*- C++ -*-


// GIST -- Graphical Interactive Simulation result Tool

//   CNCL interface classes

//   1992-94 Martin Horneffer

#ifndef gist_h
#define gist_h

#include <CNCL/Object.h>
#include <CNCL/Coord.h>
 

class GISTObject;
class GISTAttribute;
class GISTGraphic;

class CommImpl;
class GISTWorldImpl;
class GISTObjectImpl;
class GISTAttributeImpl;
class GISTGraphicImpl;
class   GISTPictureImpl;
class   GISTImageImpl;
class   GISTCurveImpl;
class   GISTDiagramImpl;

extern CNClassDesc CN_COMM;
extern CNClassDesc CN_GISTWORLD;
extern CNClassDesc CN_GISTOBJECT;
extern CNClassDesc CN_GISTATTRIBUTE;
extern CNClassDesc   CN_GISTTEXTATTRIBUTE;
extern CNClassDesc   CN_GISTVALUEATTRIBUTE;
extern CNClassDesc   CN_GISTCOORDATTRIBUTE;
extern CNClassDesc   CN_GISTVALUEARRAYATTRIBUTE;
extern CNClassDesc   CN_GISTCOORDARRAYATTRIBUTE;
extern CNClassDesc CN_GISTGRAPHIC;
extern CNClassDesc   CN_GISTPICTURE;
extern CNClassDesc   CN_GISTIMAGE;
extern CNClassDesc   CN_GISTCURVE;
extern CNClassDesc   CN_GISTDIAGRAM;

class Comm: public CNObject {
  CommImpl *impl_;
public:
  inline CommImpl *impl() { return impl_; };

  Comm(const char *host = "localhost", 
       bool abort = TRUE,
       unsigned short port = 2100,
       const char *server = "iv");
  virtual ~Comm();

  /*** Member functions required by CNCL ***/
  static CNObject *new_object(CNParam *param);
  virtual CNClassDesc class_desc() const { return CN_COMM; };
  virtual bool is_a(CNClassDesc desc) const { return desc == CN_COMM ? TRUE : CNObject::is_a(desc); };
  static Comm *cast_from_object(CNObject *obj);
  virtual void print(ostream &strm = cout) const;
  virtual void dump (ostream &strm = cout) const;
};

class GISTWorld: public CNObject {
  GISTWorldImpl *impl_;
public:
  inline GISTWorldImpl *impl() { return impl_; };

  enum Flushmode { always, explicit, automatic };

  GISTWorld(Comm& comm, 
	    const char *name,
	    float xmin, float xmax, float ymin, float ymax, 
	    double starttime = 0.0,
	    Flushmode mode = always);
  virtual ~GISTWorld();

  virtual const char *name() const;
  virtual void Flush(double time);

  /*** Member functions required by CNCL ***/
  static CNObject *new_object(CNParam *param);
  virtual CNClassDesc class_desc() const { return CN_GISTWORLD; };
  virtual bool is_a(CNClassDesc desc) const { return desc == CN_GISTWORLD ? TRUE : CNObject::is_a(desc); };
  static GISTWorld *cast_from_object(CNObject *obj);
  virtual void print(ostream &strm = cout) const;
  virtual void dump (ostream &strm = cout) const;
};


class GISTObject: public CNObject {
  GISTObjectImpl *impl_;
public:
  inline GISTObjectImpl *impl() { return impl_; }

  GISTObject(GISTWorld &default_world, const char *name);
  virtual ~GISTObject();

  virtual const char *name() const;
  virtual GISTAttribute *find_attribute(const char *name);
  virtual GISTGraphic *find_graphic(const char *name);

  /*** Member functions required by CNCL ***/
  static CNObject *new_object(CNParam *param);
  virtual CNClassDesc class_desc() const { return CN_GISTOBJECT; };
  virtual bool is_a(CNClassDesc desc) const { return desc == CN_GISTOBJECT ? TRUE : CNObject::is_a(desc); };
  static GISTObject *cast_from_object(CNObject *obj);
  virtual void print(ostream &strm = cout) const;
  virtual void dump (ostream &strm = cout) const;
};


class GISTAttribute: public CNObject {
  friend class GISTObject;
protected:
  GISTAttribute();
  GISTAttributeImpl *impl_;
public:
  virtual ~GISTAttribute();
  inline GISTAttributeImpl *impl() { return impl_; }

  virtual const char *name() const;
  virtual GISTObject *parent() const;

  /*** Member functions required by CNCL ***/
  static CNObject *new_object(CNParam *param);
  virtual CNClassDesc class_desc() const { return CN_GISTATTRIBUTE; };
  virtual bool is_a(CNClassDesc desc) const { return desc == CN_GISTATTRIBUTE ? TRUE : CNObject::is_a(desc); };
  static GISTAttribute *cast_from_object(CNObject *obj);
  virtual void print(ostream &strm = cout) const;
  virtual void dump (ostream &strm = cout) const;
};

class GISTTextAttribute: public GISTAttribute {
public:
  GISTTextAttribute(GISTObject *, 
		    const char *name, 
		    unsigned short size = 16);
  virtual ~GISTTextAttribute();
  
  virtual void put(double time, const char*data);

  /*** Member functions required by CNCL ***/
  static CNObject *new_object(CNParam *param);
  virtual CNClassDesc class_desc() const { return CN_GISTTEXTATTRIBUTE; };
  virtual bool is_a(CNClassDesc desc) const { return desc == CN_GISTTEXTATTRIBUTE ? TRUE : GISTAttribute::is_a(desc); };
  static GISTTextAttribute *cast_from_object(CNObject *obj);
  virtual void print(ostream &strm = cout) const;
  virtual void dump (ostream &strm = cout) const;
};

class GISTValueAttribute: public GISTAttribute {
public:
  GISTValueAttribute(GISTObject *, const char *name);
  virtual ~GISTValueAttribute();

  virtual void put(double time, float data);

  /*** Member functions required by CNCL ***/
  static CNObject *new_object(CNParam *param);
  virtual CNClassDesc class_desc() const { return CN_GISTVALUEATTRIBUTE; };
  virtual bool is_a(CNClassDesc desc) const { return desc == CN_GISTVALUEATTRIBUTE ? TRUE : GISTAttribute::is_a(desc); };
  static GISTValueAttribute *cast_from_object(CNObject *obj);
  virtual void print(ostream &strm = cout) const;
  virtual void dump (ostream &strm = cout) const;
};

class GISTCoordAttribute: public GISTAttribute {
public:
  GISTCoordAttribute(GISTObject *, const char *name);
  virtual ~GISTCoordAttribute();

  virtual void put(double time, CNCoord data);
  virtual void put(double time, float x, float y);

  /*** Member functions required by CNCL ***/
  static CNObject *new_object(CNParam *param);
  virtual CNClassDesc class_desc() const { return CN_GISTCOORDATTRIBUTE; };
  virtual bool is_a(CNClassDesc desc) const { return desc == CN_GISTCOORDATTRIBUTE ? TRUE : GISTAttribute::is_a(desc); };
  static GISTCoordAttribute *cast_from_object(CNObject *obj);
  virtual void print(ostream &strm = cout) const;
  virtual void dump (ostream &strm = cout) const;
};

class GISTValueArrayAttribute: public GISTAttribute {
public:
  GISTValueArrayAttribute(GISTObject *, 
			  const char *name, 
			  unsigned short dimension);
  virtual ~GISTValueArrayAttribute();

  virtual void put(double time, const float data[]);

  /*** Member functions required by CNCL ***/
  static CNObject *new_object(CNParam *param);
  virtual CNClassDesc class_desc() const { return CN_GISTVALUEARRAYATTRIBUTE; };
  virtual bool is_a(CNClassDesc desc) const { return desc == CN_GISTVALUEARRAYATTRIBUTE ? TRUE : GISTAttribute::is_a(desc); };
  static GISTValueArrayAttribute *cast_from_object(CNObject *obj);
  virtual void print(ostream &strm = cout) const;
  virtual void dump (ostream &strm = cout) const;
};

class GISTCoordArrayAttribute: public GISTAttribute {
public:
  GISTCoordArrayAttribute(GISTObject *, 
			  const char *name, 
			  unsigned short dimension);
  virtual ~GISTCoordArrayAttribute();

  virtual void put(double time, const CNCoord data[]);
  virtual void put(double time, const float data[]);

  /*** Member functions required by CNCL ***/
  static CNObject *new_object(CNParam *param);
  virtual CNClassDesc class_desc() const { return CN_GISTCOORDARRAYATTRIBUTE; };
  virtual bool is_a(CNClassDesc desc) const { return desc == CN_GISTCOORDARRAYATTRIBUTE ? TRUE : GISTAttribute::is_a(desc); };
  static GISTCoordArrayAttribute *cast_from_object(CNObject *obj);
  virtual void print(ostream &strm = cout) const;
  virtual void dump (ostream &strm = cout) const;
};


class GISTGraphic: public CNObject {
  friend class GISTObject;
protected:
  GISTGraphic();
  GISTGraphicImpl *impl_;
public:
  virtual ~GISTGraphic() ;
  inline GISTGraphicImpl *impl() { return impl_; };

  virtual const char *name() const;
  virtual GISTObject *parent() const;
  virtual const char *color_name(float red, float green, float blue);

  /*** Member functions required by CNCL ***/
  static CNObject *new_object(CNParam *param);
  virtual CNClassDesc class_desc() const { return CN_GISTGRAPHIC; };
  virtual bool is_a(CNClassDesc desc) const { return desc == CN_GISTGRAPHIC ? TRUE : CNObject::is_a(desc); };
  static GISTGraphic *cast_from_object(CNObject *obj);
  virtual void print(ostream &strm = cout) const;
  virtual void dump (ostream &strm = cout) const;
};

class GISTPicture: public GISTGraphic {
public:
  GISTPicture(GISTObject *,
	      const char *name, 
	      int priority = 0,
	      GISTTextAttribute *visibility = 0,
	      GISTWorld *world = 0);
  virtual ~GISTPicture();

  virtual void set_color(const char *color);
  virtual void set_color(float red, float green, float blue);
  virtual void set_color(GISTTextAttribute *color);

  virtual void set_brush(float thickness = 0.0, 
			 unsigned short pattern = 0xffff);

  virtual void pen_down();
  virtual void pen_up();
  virtual void pen_up_fill();

  virtual void set_pos(float x, float y);
  virtual void set_pos(CNCoord);
  virtual void set_pos(GISTCoordAttribute *);
  virtual void set_path(GISTCoordArrayAttribute *);
  
  virtual void move_rel(float dx, float dy);
  virtual void move_rel(float dx, GISTValueAttribute *dy);
  virtual void move_rel(GISTValueAttribute *dx, float dy);
  virtual void move_rel(GISTValueAttribute *dx, GISTValueAttribute *dy);

  virtual void move_dir(float r, GISTValueAttribute *a);
  virtual void move_dir(GISTValueAttribute *r, float a);
  virtual void move_dir(GISTValueAttribute *r, GISTValueAttribute *a);
  
  virtual void text(const char *text,
		    float size,
		    const char *font = 0, 
		    const char *style = 0);
  virtual void text(GISTTextAttribute *text,
		    float size,
		    const char *font = 0, 
		    const char *style = 0);

  virtual void draw_image(const char *imagename,
			  float x_align = 0.5, float y_align = 0.5);
  virtual void draw_image(GISTTextAttribute *imagename,
			  float x_align = 0.5, float y_align = 0.5);

  /*** Member functions required by CNCL ***/
  static CNObject *new_object(CNParam *param);
  virtual CNClassDesc class_desc() const { return CN_GISTPICTURE; };
  virtual bool is_a(CNClassDesc desc) const { return desc == CN_GISTPICTURE ? TRUE : GISTGraphic::is_a(desc); };
  static GISTPicture *cast_from_object(CNObject *obj);
  virtual void print(ostream &strm = cout) const;
  virtual void dump (ostream &strm = cout) const;
};

class GISTImage: public GISTGraphic {
public:
  GISTImage(GISTObject *,
	    const char *name, 
	    const char *image, 
	    CNCoord position,
	    float width = 0.0, float height = 0.0,
	    float hcenter = 0.5, float vcenter = 0.5,
	    int priority = 0,
	    GISTTextAttribute *visibility = 0,
	    GISTWorld *world = 0);
  GISTImage(GISTObject *,
	    const char *name, 
	    const char *image, 
	    float x, float y,
	    float width = 0.0, float height = 0.0,
	    float hcenter = 0.5, float vcenter = 0.5,
	    int priority = 0,
	    GISTTextAttribute *visibility = 0,
	    GISTWorld *world = 0);
  GISTImage(GISTObject *,
	    const char *name, 
	    const char *image, 
	    GISTCoordAttribute *position,
	    float width = 0.0, float height = 0.0,
	    float hcenter = 0.5, float vcenter = 0.5,
	    int priority = 0,
	    GISTTextAttribute *visibility = 0,
	    GISTWorld *world = 0);
  GISTImage(GISTObject *,
	    const char *name, 
	    GISTTextAttribute *image, 
	    CNCoord position,
	    float width = 0.0, float height = 0.0,
	    float hcenter = 0.5, float vcenter = 0.5,
	    int priority = 0,
	    GISTTextAttribute *visibility = 0,
	    GISTWorld *world = 0);
  GISTImage(GISTObject *,
	    const char *name, 
	    GISTTextAttribute *image,
	    float x, float y,
	    float width = 0.0, float height = 0.0,
	    float hcenter = 0.5, float vcenter = 0.5,
	    int priority = 0,
	    GISTTextAttribute *visibility = 0,
	    GISTWorld *world = 0);
  GISTImage(GISTObject *,
	    const char *name, 
	    GISTTextAttribute *image,
	    GISTCoordAttribute *position,
	    float width = 0.0, float height = 0.0,
	    float hcenter = 0.5, float vcenter = 0.5,
	    int priority = 0,
	    GISTTextAttribute *visibility = 0,
	    GISTWorld *world = 0);
  virtual ~GISTImage();

  /*** Member functions required by CNCL ***/
  static CNObject *new_object(CNParam *param);
  virtual CNClassDesc class_desc() const { return CN_GISTIMAGE; };
  virtual bool is_a(CNClassDesc desc) const { return desc == CN_GISTIMAGE ? TRUE : GISTGraphic::is_a(desc); };
  static GISTImage *cast_from_object(CNObject *obj);
  virtual void print(ostream &strm = cout) const;
  virtual void dump (ostream &strm = cout) const;
};

class GISTCurve: public GISTGraphic {
public:
  enum Type { simple, minmax, average };

  GISTCurve(GISTObject *,
	    const char *name, 
	    GISTValueAttribute *,
	    CNCoord position,
	    Type type = simple,
	    float width = 0.0, float height = 0.0,
	    float hcenter = 0.5, float vcenter = 0.5,
	    const char *foreground = "foreground",
	    const char *background = "background",
	    int priority = 0,
	    GISTTextAttribute *visibility = 0,
	    GISTWorld *world = 0);
  GISTCurve(GISTObject *,
	    const char *name, 
	    GISTValueAttribute *,
	    float x, float y,
	    Type type = simple,
	    float width = 0.0, float height = 0.0,
	    float hcenter = 0.5, float vcenter = 0.5,
	    const char *foreground = "foreground",
	    const char *background = "background",
	    int priority = 0,
	    GISTTextAttribute *visibility = 0,
	    GISTWorld *world = 0);
  virtual ~GISTCurve();

  virtual void hint(const char *text);

  /*** Member functions required by CNCL ***/
  static CNObject *new_object(CNParam *param);
  virtual CNClassDesc class_desc() const { return CN_GISTCURVE; };
  virtual bool is_a(CNClassDesc desc) const { return desc == CN_GISTCURVE ? TRUE : GISTGraphic::is_a(desc); };
  static GISTCurve *cast_from_object(CNObject *obj);
  virtual void print(ostream &strm = cout) const;
  virtual void dump (ostream &strm = cout) const;
};

class GISTDiagram: public GISTGraphic {
public:
  enum Type { curve, bars, rects, points };

  GISTDiagram(GISTObject *,
	      const char *name, 
	      GISTCoordArrayAttribute *,
	      CNCoord position,
	      Type type = curve,
	      float width = 0.0, float height = 0.0,
	      float hcenter = 0.5, float vcenter = 0.5,
	      const char *foreground = "foreground",
	      const char *background = "background",
	      int priority = 0,
	      GISTTextAttribute *visibility = 0,
	      GISTWorld *world = 0);
  GISTDiagram(GISTObject *,
	      const char *name, 
	      GISTCoordArrayAttribute *,
	      float x, float y,
	      Type type = curve,
	      float width = 0.0, float height = 0.0,
	      float hcenter = 0.5, float vcenter = 0.5,
	      const char *foreground = "foreground",
	      const char *background = "background",
	      int priority = 0,
	      GISTTextAttribute *visibility = 0,
	      GISTWorld *world = 0);
  virtual ~GISTDiagram();

  virtual void hint(const char *text);

  /*** Member functions required by CNCL ***/
  static CNObject *new_object(CNParam *param);
  virtual CNClassDesc class_desc() const { return CN_GISTDIAGRAM; };
  virtual bool is_a(CNClassDesc desc) const { return desc == CN_GISTDIAGRAM ? TRUE : GISTGraphic::is_a(desc); };
  static GISTDiagram *cast_from_object(CNObject *obj);
  virtual void print(ostream &strm = cout) const;
  virtual void dump (ostream &strm = cout) const;
};

#endif 
