//*BHEADER* :ts=8  -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: Pipe.c,v 0.22 1994/07/22 09:21:16 cncl-adm Exp cncl-adm $
 *
 * Class: CNPipe --- UNIX pipe
 *
 *****************************************************************************
 * Copyright (C) 1992/1993   Communication Networks
 *                           Aachen University of Technology
 *                           Kopernikusstr. 16
 *                           W-5100 Aachen
 *                           Germany
 *                           Email: mj@dfv.rwth-aachen.de (Martin Junius)
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 **EHEADER********************************************************************/

#include <unistd.h>
#include <sys/wait.h>

#include "Pipe.h"



/*
 * Constructors
 */
CNPipe::CNPipe()
    : pid(-1), program(""), sin(NIL), sout(NIL)
{
}

CNPipe::CNPipe(CNParam *param)
    : pid(-1), program(""), sin(NIL), sout(NIL)
{
}

CNPipe::CNPipe(const CNString& prog)
    : pid(-1), program(""), sin(NIL), sout(NIL)
{
    open(prog);
}



/*
 * Open pipe to program
 */
int CNPipe::open(const CNString &prog)
{
    int to[2], from[2];

    program = prog;
    if(pid != -1)
	close();

    /*
     * Create pipe via pipe() system call
     *   fd[0] - read
     *   fd[1] - write
     */
    if(pipe(to))
    {
	error(NIL, "CNPipe: ", "can't create pipe to program");
	return -1;
    }
    if(pipe(from))
    {
	error(NIL, "CNPipe: ", "can't create pipe from program");
	return -1;
    }

    /*
     * Fork child and attach pipe to stdin/stdout
     */
    pid = fork();
    if (pid == -1)
    {
	error(NIL, "CNPipe: ", "error in fork()");
	return -1;
    }
    if(pid == 0)
    {
	/* The child process */
	if(dup2(to[0], 0) != 0)
	{
	    error(NIL, "CNPipe: ", "can't attach pipe to stdin");
	    exit(1);
	}
	::close(to[0]);
	::close(to[1]);
	
	if(dup2(from[1], 1) != 1)
	{
	    error(NIL, "CNPipe: ", "can't attach pipe to stdout");
	    exit(1);
	}
	::close(from[0]);
	::close(from[1]);

	execlp(program, program, NIL);	
	
	/* In case execlp() failed */
	error(NIL, "CNPipe: ", "can't exec ", program);
	exit(1);			
    }
    else
    {
	/* The parent process */
	::close(to[0]);
	fdout = to[1];
	sout = new ofstream(to[1]);	// Output stream
	if(sout->fail())
	{
	    error(NIL, "CNPipe: ", "can't open output stream");
	    return -1;
	}
	::close(from[1]);
	fdin = from[0];
	sin  = new ifstream(from[0]);	// Input stream
	if(sin->fail())
	{
	    error(NIL, "CNPipe: ", "can't open input stream");
	    return -1;
	}
    }

    return 0;
}



/*
 * Close pipe
 */
int CNPipe::close()
{
    int status;
    
    if(pid != -1)
    {
	/*
	 * Close pipe
	 */
	::close(fdin);
	::close(fdout);
	delete(sin);
	delete(sout);
	pid = -1;
	program = "";
	
	/*
	 * Wait for child to exit
	 */
	waitpid(pid, &status, 0);
	if(WIFEXITED(status) && WEXITSTATUS(status))
	{
	    error(NIL, "CNPipe: ", "child exited with error status");
	    return -1;
	}
    }
    
    return 0;
}



/***** Default I/O member function for CNCL classes **************************/

// Normal output
void CNPipe::print(ostream &strm) const
{
    strm << "program=" << program << " pid=" << pid
	 << " fdin=" << fdin << " fdout=" << fdout <<endl;
}

// Debug output
void CNPipe::dump(ostream &strm) const
{
    strm << "CNPipe { $Revision: 0.22 $ " << endl;
    print(strm);
    strm << "}" << endl;
}



/***** CNCL stuff for type information ***************************************/

// Describing object for class CNPipe
static CNClass CNPipe_desc("CNPipe", "$Revision: 0.22 $",
			  CNPipe::new_object);

// "Type" for type checking functions
CNClassDesc CN_PIPE = &CNPipe_desc;






