//*BHEADER* :ts=8  -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_            C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_                      N E T W O R K S
 * |_        |_  |_  |_  |_        |_                            C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_                  L I B R A R Y
 *
 * $Id: DLREG.c,v 0.20 1994/04/25 13:27:31 cncl-adm Exp cncl-adm $
 *
 * Class: CNDLREG --- Discrete LRE (LRE III) for compl. distribution function
 *
 *****************************************************************************
 * Copyright (C) 1992/1993   Communication Networks
 *                           Aachen University of Technology
 *                           Kopernikusstr. 16
 *                           W-5100 Aachen
 *                           Germany
 *                           Email: mj@dfv.rwth-aachen.de (Martin Junius)
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 **EHEADER********************************************************************/

/****************************************************************************/
/*                                                                          */
/* letzte Aenderung: 19.04.94                   erstellt am 04.06.93        */
/* adapted to CNCL : 29.10.93                                               */
/****************************************************************************/

#include "DLREG.h"

CNDLREG::CNDLREG( double * xvalues, long level, double eerror, double pre_first, 
    char * llrename, char * ttext, unsigned long mmax_nrv) 
{
    if( level > 1 ) 
        x = new struct result[level+1];// one element more because of get::index
    else
        CNCL::fatal(NIL,"No. of levels <= 1 in CNDLREG constructor !!\n");
    if( xvalues == NIL )              // fill x[].x with default-values 
    {                                 // 0 ... level-1
        for( int i = 0; i<=level; i++)
        {   
            x[i].x = i; x[i].h = 0; x[i].c = 0;
            //printf( "%d %f\n", i, x[i].x);
        }
        index_min = 0;
        xmin      = 0.0; 
        index_max = level;
        xmax      = level - 1;
        int_size  = 1.0;
        equi_dist = TRUE;
    }
    else 
    {
        long i    = 0;
        while( ((xvalues+i) != NIL) && (i<=level) )
        {
            x[i].x = xvalues[i]; x[i].h = 0; x[i].c = 0;
            i++;
        }
        index_min = 0;
        xmin      = (double) x[0].x; 
        index_max = level;
        xmax      = (double) x[level-1].x;
        int_size  = 0.0;
        equi_dist = FALSE;
    }
    if( pre_first < xmin ) pre_index = index_min;
    else if( pre_first > xmax ) pre_index = index_max;
    else 
    {
        pre_index= 0;
        while( pre_index<level ) 
            if( x[pre_index].x == pre_first ) break;
            else pre_index++;
    }
    reler_max       = eerror;
    name            = llrename;
    text            = ttext;
    nrv             = 0;
    base            = 1.0;
    pre_rv          = pre_first;     // state of beginning
    cur_level_index = index_min;
    rv_sum          = 0.0;
    rv_square_sum   = 0.0;
    h               = 0;
    n0              = 2.0/eerror/eerror < 1000 ? long( 2.0/eerror/eerror ):1000;
    max_nrv         = mmax_nrv;
    wasted_right    = wasted_left = 0;
    min_value       = MAXDOUBLE;
    max_value       = -MAXDOUBLE;
    phase           = INITIALIZE;
    end_reached     = FALSE; 
}

CNDLREG::CNDLREG( double xxmin, double xxmax, double iint_size, double eerror, 
    double pre_first, char * lrename, char * ttext, unsigned long mmax_nrv) 
{
    if( iint_size <= 0 ) 
        CNCL::fatal(NIL,"interval width <= 0 in CNDLREG constructor !!\n");
    if( xxmin >= xxmax )
        CNCL::fatal(NIL,"X_min >= X_max in CNDLREG constructor !!\n");
    long level = abs((long)((xxmax - xxmin) / iint_size)) + 1;
    if( level <= 1 ) 
        CNCL::fatal(NIL,"No. of levels <= 1 in CNDLREG constructor !!\n");
    //
    x = new struct result[level+1];
    for( int i = 0; i<=level; i++)
    {   
        x[i].x = (i*iint_size)+xxmin; x[i].h = 0; x[i].c = 0;
        //printf( "%d %f\n", i, x[i].x);
    }
    index_min       = 0; 
    xmin            = xxmin; 
    index_max       = (long) (level);
    xmax            = xxmax;
    int_size        = iint_size;
    equi_dist       = TRUE;
    if( pre_first < xmin ) pre_index = index_min;
    else if( pre_first > xmax ) pre_index = index_max;
    else 
    {
        pre_index= 0;
        while( pre_index<level ) 
            if( x[pre_index].x == pre_first ) break;
            else pre_index++;
    }
    reler_max       = eerror;
    name            = lrename;
    text            = ttext;
    nrv             = 0;
    base            = 1.0;
    pre_rv          = pre_first;     // state of beginning
    cur_level_index = index_min;
    rv_sum          = 0.0;
    rv_square_sum   = 0.0;
    h               = 0;
    n0              = 2.0/eerror/eerror < 1000 ? long( 2.0/eerror/eerror ):1000;
    max_nrv         = mmax_nrv;
    wasted_right    = wasted_left = 0;
    min_value       = MAXDOUBLE;
    max_value       = -MAXDOUBLE;
    phase           = INITIALIZE;
    end_reached     = FALSE;
}

void CNDLREG::put( double cur_rv )
{
    if( nrv+1 < max_nrv )
    {
        cur_index = get_index( cur_rv );
        if( cur_index == NO_INDEX )
        {
            CNCL::warning("Warning: Wrong x value in DLREG::put !");
            return;
        }
        nrv++;
        h++;
        rv_sum        += cur_rv;
        rv_square_sum += cur_rv*cur_rv;
        if (max_value < cur_rv) max_value = cur_rv; 
        else if (min_value > cur_rv) min_value = cur_rv;

        if( cur_index == LOWER ) 
        {
            wasted_left++;
            if( pre_rv > cur_rv ) 
                for(int i=pre_index; i>=index_min ; i--) (x[ i ].c)++;
            pre_rv = xmin-1.0;
            pre_index = index_min;
            return ;
        }
        else if( cur_index == GREATER )
        {
            wasted_right++;
            pre_rv    = xmax+1.0;
            pre_index = index_max;
            return ;
        }
        if( pre_rv > cur_rv ) 
            for(int i=pre_index; i>cur_index; i--) (x[ i ].c)++;
        x[ cur_index ].h++;
        if(( h >= n0 ) && nrv >= 1000 ) 
        {
            phase = CNDLREG::rtc();
            h     = 0;
        }
        //
        // save current values
        pre_rv    = cur_rv;
        pre_index = cur_index;
    }
    else
    {
        phase       = END;
        end_reached = TRUE; 
    }
}

long CNDLREG::get_index( double cur_rv )
{
    if( cur_rv < xmin ) 
        return LOWER;
    else if ( cur_rv > xmax ) 
        return GREATER;
    if( equi_dist ) 
    {
        long index = long( (cur_rv-xmin)/(xmax-xmin)*(index_max-1) + 0.1);
        return cur_rv==x[index].x ? index : (long)NO_INDEX;
    }
    //
    // search of the correct index (Do you a better algorithm ? - Send it !
    struct result * start = x;
    struct result * end   = x+index_max;
    struct result * lauf;
    register int step;
    while( (step = (end - start)) >= 1)
    {
        lauf = start + (step >> 1);
        if( lauf->x == cur_rv )     return step >> 1 ;
        else if( lauf->x > cur_rv ) end = lauf;
        else if( lauf->x < cur_rv ) start = lauf;
    }
    return NO_INDEX;
    //return long(cur_rv - index_min - xmin); 
}

short CNDLREG::fwrite( char * fname )
{
    ofstream output( fname, ios::out );
    CNDLREG::print( output );
    output.close();
    return 1; 
}

void CNDLREG::print( ostream &output ) const
{
    double rho=0, sigrho=0, d=0;
    double uf;
    double nf = double(nrv);
    double vf = wasted_right;
    double G  = vf/nf;
    double cf = 0.0; 
    
    output << setprecision(4);
    output.setf(ios::left, ios::fixed);
    output << "#LRE RESULT (THIS IS A MAGIC LINE)" << endl;
    output << "#---------------------------------------------------------------------------" << endl;
    output << "#    Discrete Lre" << endl;
    output << "#---------------------------------------------------------------------------" << endl;
    output << "#Name: " << name << endl;
    output << "#Text: " << text << endl;
    output << "#Default values: max.rel. error = " << setw(4) << reler_max*100
           << "%  X_min: " << xmin << "  X_max: " << xmax << endl;
    output << "#trials: " << nrv ;
    output << "\tmean: " << mean() << "\tvariance: " << variance() << endl;
    output << "#trials < X_min: " << wasted_left << "\ttrials > X_max: " 
           << wasted_right << endl;
    output << "---------------------------------------------------------------------------" << endl;
    output << "#G(x)           x               rel.error"
           << "       rho             sigrho" << endl;
    //
    // 
    output << setprecision(7);
    for( int i = index_max-1; i > index_min; i--)
    {
        cf = double( x[ i ].c );
        vf += double( x[ i ].h );
        if( nf < 1000.0 || vf < 100.0 || (vf-cf) < 10.0 )
        {
            output << "#";
            output << setiosflags(ios::scientific);
            output << G*base;
            output << " " << setw(15) << x[i].x
                   << endl;
            G = vf/nf;
            output << "#";
            output << G*base;
            output << " " << setw(15) << x[i].x
                   << endl;
        }
        else
        {
            output << setiosflags(ios::scientific);
            output << G*base;
            output << " " << setw(15) << x[i].x;       
            output << setw(16) << d;
            output << setw(16) << rho << endl;
            G = vf/nf;
            rho = 1.0 - cf/vf/( 1.0 - G );
            uf = nf - vf;
            sigrho = sqrt( cf*(((1 - cf/vf)/(vf*vf)) + ((1 - cf/uf)/(uf*uf))));
            output << G*base;
            output << " " << setw(15) << x[i].x;       
            d = sqrt( (1.0 - vf/nf)/vf * (1.0 + rho)/(1.0 - rho));
            output << setw(16) << d;
            output << setw(16) << rho;
            output << setw(16) << sigrho << endl;
            //output << setw(16) << sigrho;
            //output << setw(16) << cf << setw(16) << vf <<endl;
        }
    }
    output << G*base;
    output << " " << setw(15) << x[i].x;       
    output << setw(16) << d;
    output << setw(16) << rho << endl;
    vf += double( x[ i ].h ); G = vf/nf;
    output << G*base;
    output << " " << setw(15) << x[i].x << endl;       
}


CNStatistics::Phase CNDLREG::rtc()
{
    if( nrv < 1000 ) return INITIALIZE;
    else 
    {
        double cf, d, rho;
        double nf = double( nrv );
        double vf = double( nrv - wasted_left);
        double rq = reler_max * reler_max;
        int i     = index_min;
        while( i < index_max-3 )
        {
            vf -= x[i].h; cf = x[i+1].c;
            if( vf<100.0 || (vf-cf)<10 ) 
            { 
                cur_level_index = i; 
                return ITERATE;
            }
            else
            {
                rho = 1.0 - cf/vf/(1.0 - vf/nf);
                d   = ( 1.0 - vf/nf)/vf * (1.0+rho) / (1.0-rho);
                if( d > rq ) 
                { 
                    cur_level_index = i; 
                    return ITERATE;
                }
            }
            i++;
        }
        // i is now index_max-1
        cur_level_index = i; 
        vf -= x[i].h; cf = x[i+1].c;
        if( vf<100.0 ) return ITERATE;
        if( ((vf-cf)==0.0) && (max_value==xmax))
        {
         CNCL::warning("Relative error of last level can`t be calculated: \n",
                        "CNDLREG::status set to END !!\n");
            end_reached     = TRUE; 
            return END;
        }
        else 
        {
            rho = 1.0 - cf/vf/(1.0 - vf/nf);
            d   = ( 1.0 - vf/nf)/vf * (1.0+rho) / (1.0-rho);
            if( d > rq ) return ITERATE;
        }
        end_reached     = TRUE; 
        return END;
    }
}

double CNDLREG::cur_g_lev()          // fuss (19.06.93)
{
    if( nrv < 1000 ) return 1.0;
    else 
    {
        double vf = double( nrv );
        //double cur_x = x[ cur_level_index ].x;
        for( int i=index_min; i<=cur_level_index; i++) 
            vf -= double( x[ i ].h );
        return vf/double( nrv );
    }
}

double CNDLREG::g( double xt )        // fuss (19.06.93)
{
    if( nrv < 1000 ) return 1.0;
    else
    {
        int i;
        double vf = double( nrv ) - double( x[ index_min ].h );
        double nf = double( nrv );
        for( i=index_min; ( i<index_max-1 ) && ( x[i].x != xt ); i++ )
            vf -= x[ i+1 ].h;
        if( x[ i ].x == xt ) return vf/nf;
        else return -1.0;
    }     
}

double CNDLREG::p( double xt )
{
    long index = get_index( xt );
    return (nrv != 0 ? x[index].h/nrv : 0.0);
}

double CNDLREG::mean() const
    {
        return (nrv>0)? double(rv_sum)/nrv: MAXDOUBLE;
    }

double CNDLREG::variance() const
    {
        return (nrv>1)? 
                double(rv_square_sum-mean()*mean()*nrv)/(nrv - 1.0):0.0;
    }

/***** CNCL stuff for type information and exemplar objects ******************/

// Describing object for class CNDLREG
static CNClass CNDLREG_desc("CNDLREG", "$Revision: 0.20 $", NIL);

// "Type" for type checking functions
CNClassDesc CN_DLREG = &CNDLREG_desc;
