//*BHEADER* :ts=8  -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: SDLManager.c,v 0.22 1994/07/22 09:21:42 cncl-adm Exp cncl-adm $
 *
 * Class: CNSDLManager --- ...
 *
 *****************************************************************************
 * Copyright (C) 1992/1993   Communication Networks
 *                           Aachen University of Technology
 *                           Kopernikusstr. 16
 *                           W-5100 Aachen
 *                           Germany
 *                           Email: mj@dfv.rwth-aachen.de (Martin Junius)
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 **EHEADER********************************************************************/

#include "SDLManager.h"

/*
   The constructor has to initialize the pointers (arrays). It is recommended
   that these arrays are initialized correctly in the derived class. The
   SDLManager class accesses array components directly through these poin-
   ters. This is `highly dangerous' but since the derived classes shall be
   produced by an automatic translator it seems to be useable.
*/
CNSDLManager::CNSDLManager(int sys, CNSDLProcessData **p, long ps,
ProcessType *rec, long recs, CNSDLChannel **chl, CNEventScheduler* sim)
{
    system = sys;
    processes = p; pr_size = ps;
    signal_receiver = rec; sr_size = recs;
    channels = chl;
    actual_process = 0;
    sched = sim;
}




// the eventhandling routine called by the scheduler for every signal
/*
   SDLManager receives Signals from the EventScheduler. It transfers them
   to the appropriate process and handles the saved signals. It also reacts
   to procedure calls in sending an INIT signal.
*/
virtual void CNSDLManager::event_handler(const CNEvent *ev)
{
    signal = (CNSDLSignal*) ev;      // save incoming signal
    
    actual_process = signal->receiver();  // prepare manager for 
    thatdata = processes[actual_process]; // current process
    that = thatdata->address();
    
        // first try new signal
    in_save_list = FALSE;
    int old_state = that->state();
    thatdata->sender( signal->sender() );
    
        // deliver signal to process
    int changed_handler = that->signalhandler(signal->type());
       
        // initialize procedure when called
    while( changed_handler ) changed_handler = that->signalhandler(0);
            
    if (thatdata->save_list==NIL || that->state()==old_state) return;
    
        // if state has changed AND save-list exists try saved signals
    in_save_list = TRUE;
    CNDLObject *saved_sig;
    do
    {
        old_state = that->state();
        saved_sig = thatdata->save_list->first(); // at begin of save list
        while( saved_sig )
        {
            signal = (CNSDLSignal*) saved_sig;

            thatdata->sender( signal->sender() );
            has_saved = FALSE;
            
            changed_handler = that->signalhandler(signal->type());
            
                // initialize procedure when called
            while( changed_handler ) changed_handler = that->signalhandler(0);

            if (!has_saved)
            {
                    // remove signal from save list and delete it
                saved_sig = thatdata->save_list->delete_object( saved_sig );
                    // restart at begin of save list if state has changed
                if (that->state() != old_state) break;	
            }
            else saved_sig = thatdata->save_list->next( saved_sig );
        }
    } while( that->state() != old_state );
}


    
             
// save signal
void CNSDLManager::save(int name)
{
    has_saved = TRUE;
    if (!in_save_list) 
    {
            // copy signal (the original will be deleted by the scheduler)
        CNSDLSignal *sig = new CNSDLSignal(signal->type(), signal->scheduled(),
                                          signal->from(), signal->to(),
                                          signal->sender(), signal->receiver(),
                                          signal->system(), signal->object());
        thatdata->save( sig );
    }
}



/*
    SDLManager creates a SDLSignal and delivers it to the EventScheduler.
    The function send is inherited from EventHandler as is scheduler and
    now and some others.
*/
// SDL output functions (first the local ones)
virtual void CNSDLManager::output_x(SignalType name, PId receiver, 
                                    CNObject *data)
{
    // here explicit addressing is used
    if (processes[receiver]->address() == NIL)
        error("SDLManager::output ", "no receiver availlable");
     
    send(new CNSDLSignal(name, now(), actual_process, receiver, system, data));
}

virtual void CNSDLManager::output(SignalType name, CNObject *data)
{
    ProcessType rec = signal_receiver[name];    // implicit addressing

    PId receiver = 2;  // PId 1 means environment 
    
    while (processes[receiver]->p_type != rec) 
        receiver += processes[receiver]->p_number;//look for right process type
    bool can_send = FALSE;
    while (processes[receiver]->p_type == rec) 
    {   
        if (processes[receiver]->address() != NIL) //watch for existing process
        {
            can_send = TRUE;
            break;
        }
        receiver++;
    }
    if (!can_send) error("SDLManager::output ", "no receiver availlable");

    send(new CNSDLSignal(name, now(), actual_process, receiver, system, data));
}

virtual void CNSDLManager::output(SignalType name, ProcessType rec, 
                                  CNObject *data)
{
    PId receiver = 2;  // PId 1 means environment 
    
    while (processes[receiver]->p_type != rec) 
        receiver += processes[receiver]->p_number;//look for right process type
    bool can_send = FALSE;
    while (processes[receiver]->p_type == rec) 
    {
        if (processes[receiver]->address() != NIL) //watch for existing process
        {
            can_send = TRUE;
            break;
        }
        receiver++;
    }
    if (!can_send) error("SDLManager::output ", "no receiver availlable");


    send(new CNSDLSignal(name, now(), actual_process, receiver, system, data));
}


// SDL output functions (now the global one)
virtual void CNSDLManager::output_x(SignalType name, PId receiver, 
                              CNSDLChannel *ch, CNObject *data, int do_not_use)
{
    // signals experience a nondeterministic delay when using a channel
    CNSimTime arrival,
              ref_time = ch->last_arrival; 
    CNRandom *rnd = ch->rnd_delay;
    
    while((arrival = now()+rnd->draw()) < ref_time);  // signals are not 
    ch->last_arrival = arrival;             // allowed to overtake each other

    if (receiver == 1) {
        if (processes[1]==NIL)
            error("SDLManager::output ", "no environment defined");
            
        send(new CNSDLSignal(name, arrival, this, 
                             processes[1]->address()->System, 
                             actual_process, receiver, system, data));
        // addressed to Environment
    }
    else {
        if (processes[receiver]->address() == NIL)
            error("SDLManager::output ", "no receiver availlable");    
    
        send(new CNSDLSignal(name, arrival, actual_process, receiver, 
                             system, data));
    }
}

virtual void CNSDLManager::output(SignalType name, ProcessType rec, 
                              CNSDLChannel *ch, CNObject *data, int do_not_use)
{
    // signals experience a nondeterministic delay when using a channel
    CNSimTime arrival,
              ref_time = ch->last_arrival; 
    CNRandom *rnd = ch->rnd_delay;
    
    while((arrival = now()+rnd->draw()) < ref_time);  // signals are not 
    ch->last_arrival = arrival;             // allowed to overtake each other
    
    PId receiver;
    if (rec == 1) {
        receiver = 1;
        
        if (processes[1]==NIL)
            error("SDLMananger::output ", "no environment defined");
        
        send(new CNSDLSignal(name, arrival, this, 
                             processes[1]->address()->System, 
                             actual_process, receiver, system, data));
        // addressed to Enevironment
    }
    else {
        receiver = 2; 
    
        while (processes[receiver]->p_type != rec) 
            receiver += processes[receiver]->p_number; // look for right
            bool can_send = FALSE;                     // process type
        while (processes[receiver]->p_type == rec) 
        {
            if (processes[receiver]->address() != NIL) // watch for existing 
            {                                          // process
               can_send = TRUE;
               break;
            }
            receiver++;
        }
        if (!can_send) error("SDLManager::output ", "no receiver availlable");

        send(new CNSDLSignal(name, arrival, actual_process, receiver, 
                             system, data));
    }                          
}


    

// timer functions
/*
   SDLManager translates SDL compatible Timer calls into the syntax used
   by the SDLTimer class. These are much more complex due to the actions
   taken by this class. The management of timers is done in that class.
*/
// set timer to an absolute time value
void CNSDLManager::set(CNSimTime time, TimerType timer)
{
    CNSDLSignal *sig = thatdata->p_timer->set(time, timer, thatdata->save_list,
                       scheduler(), actual_process, system); 
    CNEventID id = send( sig );
    thatdata->p_timer->tell_ID( id );
}

// reset timer (and free it)
void CNSDLManager::reset(TimerType timer)
{
    thatdata->p_timer->reset(timer, thatdata->save_list, scheduler());
}

// ask wether timer is used
bool CNSDLManager::active(TimerType timer)
{
    return thatdata->p_timer->active( timer );
}

// free timer (make it inactive). This function is necessary because a timer
// in SDL is active until the timer signal is consumed (not delivered!).
void CNSDLManager::remove(TimerType timer)
{
    thatdata->p_timer->remove( timer );
}



/*
   Dynamic SDL. SDLManager creates process data for the maximum number of
   instances for every process type. So create means to create a process
   and find a free entry in the process data table. Stop means to delete
   the process and free the entry.
*/
// static and dynamic creation of SDL processes
void CNSDLManager::create(CNSDLProcess *proc_ad, PId proc)
{
    if (proc) {
        processes[proc]->address(proc_ad);  // static creation (PId is given)
        processes[proc]->parent(0);
        sched->send_event(new CNSDLSignal(0, 0, this, this, proc, proc,
                                          system, NIL));
             // initialization
    }
    else {
        int ptype = proc_ad->type();  // dynamic creation (no PId is given)
        PId pid = 2;
        while (processes[pid]->p_type != ptype) 
            pid += processes[pid]->p_number;  // look for right process type
        bool could_create = FALSE;
        while (processes[pid]->p_type == ptype) 
        {
            if (processes[pid]->address() == NIL) // watch for a free entry
            {
                processes[pid]->address(proc_ad);
                processes[pid]->parent(actual_process);
                thatdata->offspring(pid);
                output_x(0, pid);             // initialization
                could_create = TRUE;
                break;
            }
            pid++;
        }
        if(!could_create)
        {
            thatdata->offspring(0); // that's SDL's method to tell insucess
            delete proc_ad;         /* delete the process; it shouldn't
                                       have been created anyway  */
        }
    }
}


void CNSDLManager::stop()
{
    thatdata->address(NIL);  // clear from table
    delete that;             // clear from memory
  
    if (thatdata->p_timer)   // delete timer
        thatdata->p_timer->clear(thatdata->save_list, scheduler());
  
    if (thatdata->save_list) // empty savelist
        thatdata->clear_savelist();
  
    PId stopped = thatdata->self(); 
    CNDLObject *next_ev, 
               *event = scheduler()->peek_event();
    while (event) {
        next_ev = event->next();         // look through the list of events
                                         // and delete every signal addressed
        if (event->is_a(CN_SDLSIGNAL)) { // to the stopped process
            CNSDLSignal *sigev = (CNSDLSignal*) event;
            if (sigev->receiver() == stopped)
                scheduler()->delete_event(sigev->id());
        }
        
        event = next_ev; 
    }           
}


// control the channels behavior (set random generator for channel delay)
void CNSDLManager::set_channel_delay(int chnu, CNRandom *rand)
{
    channels[chnu]->rnd_delay = rand;
}

CNRandom *CNSDLManager::get_channel_delay(int chnu)
{
    return channels[chnu]->rnd_delay;
}

                

virtual CNSDLManager *CNSDLManager::S(int num)
{
    error("SDLManager::S ", 
          "this function can be used by the environment only");
    return NIL;
}



/***** Default I/O member function for CNCL classes **************************/

// Normal output
void CNSDLManager::print(ostream &strm) const
{
    strm << "\nSDLManager" << "\nsystem = " << system << endl;
}

// Debug output
void CNSDLManager::dump(ostream &strm) const
{
    strm << "CNSDLManager { $Revision: 0.22 $ ..."
	 << " }" << endl;
    strm << "\nSDLManager" << "\nsystem = " << system << endl;
    strm << "\nactual process:" << endl;
        if (that) that->dump();
        if(thatdata) thatdata->dump();
    strm << "\nlast signal:" << endl; 
        if (signal) signal->dump();
}



/***** CNCL stuff for type information ***************************************/

// Describing object for class CNSDLManager
static CNClass CNSDLManager_desc("CNSDLManager", "$Revision: 0.22 $", NIL);

// "Type" for type checking functions
CNClassDesc CN_SDLMANAGER = &CNSDLManager_desc;

