//*BHEADER* :ts=8  -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: SDLEnvironment.c,v 0.22 1994/07/22 09:21:42 cncl-adm Exp cncl-adm $
 *
 * Class: CNSDLEnvironment --- ...
 *
 *****************************************************************************
 * Copyright (C) 1992/1993   Communication Networks
 *                           Aachen University of Technology
 *                           Kopernikusstr. 16
 *                           W-5100 Aachen
 *                           Germany
 *                           Email: mj@dfv.rwth-aachen.de (Martin Junius)
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 **EHEADER********************************************************************/

#include "SDLEnvironment.h"

/*
   here the environment is created; there is only one! environment process
   with the class name SDLProcess_Environment; only one! instance of this
   class may exist; to ease communication with the environment a global
   pointer to the environment data is used.
*/

CNSDLEnvironment::CNSDLEnvironment(int systems, CNSDLProcess* env_proc, 
CNEventScheduler* sched) : CNSDLManager(1, processes, 2, NIL, 1, NIL, sched)
{
    if (pointer_to_environment_data)
        error("SDLEnvironment ","only one environment allowed");
        
    pointer_to_environment_data = processes[1] = 
                         new CNSDLProcessData(1, 1, 1, TRUE, TRUE);
    create (env_proc, 1); env_proc->System = this;
    
    SDLSystem = new CNSDLManager*[ sys_no = (systems+1) ];
    for (int i=0; i<sys_no; i++) SDLSystem[i] = NIL;
}    



// connect SDLSystem with environment
void CNSDLEnvironment::install(int sys, CNSDLManager *sys_p)
{
    if (sys >= sys_no)
        error("SDLEnvironment ","too many systems installed");
    if (SDLSystem[sys])
        error("SDLEnvironment ","same System(number) installed twice");

    SDLSystem[ sys ] = sys_p;
}



/*
   the modified local output functions. Since the environment "system"
   contains only one process explicit addressing other than to PId 1 is 
   forbidden; implicit addressing always goes to PId 1; there are no 
   internal signalroutes so ProcessType must be 1 in every case.
*/
void CNSDLEnvironment::output_x(SignalType name, PId receiver, 
                                    CNObject *data)
{
    // here explicit addressing is used
    if (receiver != 1)
        error("SDLEnvironment::output ", "wrong signal addressing");
     
    send(new CNSDLSignal(name, now(), 1, 1, 1, data));
}

void CNSDLEnvironment::output(SignalType name, CNObject *data)
{
    // implicit addressing
    send(new CNSDLSignal(name, now(), 1, 1, 1, data));
}

void CNSDLEnvironment::output(SignalType name, ProcessType rec, 
                                  CNObject *data)
{
    if (rec!=1)
        error("SDLEnvironment::output ", "wrong signal addressing");


    send(new CNSDLSignal(name, now(), 1, 1, 1, data));
}

/*
   the modified channel output functions. Since the environment is connected
   to many systems it uses their channels; it also uses internal data of the
   systems to find out the right addresses for the signals, the target
   system has to be set here (unlike in SDL systems)
*/
void CNSDLEnvironment::output_x(SignalType name, PId receiver, 
                       CNSDLChannel *ch, CNObject *data, int target_system)
{
    if (target_system < 2)
        error("SDLEnvironment::output ", "wrong target system");

    // signals experience a nondeterministic delay when using a channel
    CNSimTime arrival,
              ref_time = ch->last_arrival; 
    CNRandom *rnd = ch->rnd_delay;
 
    while((arrival = now()+rnd->draw()) < ref_time); // signals are not
    ch->last_arrival = arrival;          // allowed to overtake each other

    if (receiver == 1)
        error("SDLEnvironment::output ", 
              "no channel to connect environment with environment");
    else {
        if (SDLSystem[target_system]->processes[receiver]->address() == NIL)
            error("SDLEnvironment::output ", "no receiver availlable");    
    
        send(new CNSDLSignal(name, arrival, this, SDLSystem[target_system],
                             1, receiver, 1, data));
    }
}

void CNSDLEnvironment::output(SignalType name, ProcessType rec, 
                         CNSDLChannel *ch, CNObject *data, int target_system)
{
    if (target_system < 2)
        error("SDLEnvironment::output ", "wrong target system");

    // signals experience a nondeterministic delay when using a channel
    CNSimTime arrival,
              ref_time = ch->last_arrival; 
    CNRandom *rnd = ch->rnd_delay;
 
    while((arrival = now()+rnd->draw()) < ref_time);  // signals are not
    ch->last_arrival = arrival;           // allowed to overtake each other
 
    if (rec == 1)
        error("SDLEnvironment::output ", 
              "no channel to connect environment with environment");
    else {
        PId receiver = 2; 
        CNSDLManager *target = SDLSystem[target_system];

        while (target->processes[receiver]->type() != rec) 
            receiver += target->processes[receiver]->max_number();
            bool can_send = FALSE; // look for right process type
        while (target->processes[receiver]->type() == rec) 
        {
            if (target->processes[receiver]->address() != NIL)  
            {                                // watch for existing process
               can_send = TRUE;
               break;
            }
            receiver++;
        }
        if (!can_send) 
            error("SDLEnvironment::output ", "no receiver availlable");

        send(new CNSDLSignal(name, arrival, this, target, 1, 
                                                         receiver, 1, data));
    }                          
}


CNSDLProcessData *CNSDLEnvironment::pointer_to_environment_data = NIL;


/***** Default I/O member function for CNCL classes **************************/

// Normal output
void CNSDLEnvironment::print(ostream &strm) const
{
    strm << "SDLEnvironment:" << endl;
    CNSDLManager::print();
}

// Debug output
void CNSDLEnvironment::dump(ostream &strm) const
{
    strm << "CNSDLEnvironment { $Revision: 0.22 $ ..."
	 << " }" << endl;
    CNSDLManager::dump();
}



/***** CNCL stuff for type information ***************************************/

// Describing object for class CNSDLEnvironment
static CNClass CNSDLEnvironment_desc("CNSDLEnvironment", "$Revision: 0.22 $",
			    CNSDLEnvironment::new_object);

// "Type" for type checking functions
CNClassDesc CN_SDLENVIRONMENT = &CNSDLEnvironment_desc;
