// -*- C++ -*-
/*
 * SocketStream.c
 */

#include "SocketStream.h"

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>


/*
 * Debugging macro
 */
#ifdef DEBUG
# undef DEBUG
# define DEBUG(x)	cerr << x << endl
# include <errno.h>

char *strerror()
{
    extern int errno;
    extern int sys_nerr;
    extern char *sys_errlist[];

    return errno < sys_nerr ? sys_errlist[errno] : "unknown error";
}

#else
# define DEBUG(x)
#endif

#define PROTO_TCP "tcp" // TCP stream sockets

extern "C" 
{
#ifndef __linux__		/* LINUX has prototypes for these functions */
    int accept(int s, struct sockaddr *addr, int *addrlen);
    int bind(int s, struct sockaddr *name, int namelen);
    int close(int s);
    int connect(int s, struct sockaddr *name, int namelen);
    struct hostent *gethostbyname(const char *name);
    int listen(int s, int backlog);
    int setsockopt(int s, int level, int optname, const char *optval,
		   int optlen);
    int socket(int domain, int type, int protocol);
#endif
}

// Constructor: default

SocketStream::SocketStream()
    : type(none), sock(-1), asock(-1)
{
    init(new SocketBuf());
}

// Constructor: open with hostname and port number

SocketStream::SocketStream(const char *host, unsigned short port)
{
    DEBUG( "Constructor SocketStream(" << host << ", " << port << ")" );
    init(new SocketBuf());
    open(host, port);
}

// Constructor: open with hostname and port name

SocketStream::SocketStream(const char *host, const char *port)
{
    DEBUG( "Constructor SocketStream(" << host << ", " << port << ")" );
    init(new SocketBuf());
    open(host, port);
}

// Constructor: open with port number

SocketStream::SocketStream(unsigned short port)
{
    DEBUG( "Constructor SocketStream(" << port << ")" );
    init(new SocketBuf());
    open(port);
}

// Constructor: open with port name

SocketStream::SocketStream(const char *port)
{
    DEBUG( "Constructor SocketStream(" << port << ")" );
    init(new SocketBuf());
    open(port);
}

// Destructor: close

SocketStream::~SocketStream()
{
    DEBUG( "Destructor SocketStream" );
    close();
}

// Client: open with hostname and port number

void SocketStream::open(const char *host, unsigned short port)
{
    if(open_connection(host, port) != -1)
    {
	type = client;
	clear();
	rdbuf()->attach(sock);
    }
    else
	set(ios::badbit);
}

// Client: open with hostname and port name

void SocketStream::open(const char *host, const char *port)
{
    struct servent *service;
    
    service = getservbyname(port, PROTO_TCP);
    if(service)
    {
	DEBUG( "service: " << service->s_name << " "
	       << service->s_port << "/" << service->s_proto );
	open(host, service->s_port);
    }
    else
    {
	DEBUG( "service: " << port << " unknown" );
	set(ios::badbit);
    }
}

// Server: open with port number

void SocketStream::open(unsigned short port)
{
    if(open_listen(port) != -1)
	type = server;
    else
	set(ios::badbit);
}

// Server: open with port name
void SocketStream::open(const char *port)
{
    struct servent *service;
    
    service = getservbyname(port, PROTO_TCP);
    if(service)
    {
	DEBUG( "service: " << service->s_name << " "
	       << service->s_port << "/" << service->s_proto );
	open(service->s_port);
    }
    else
	DEBUG( "service: " << port << " unknown" );
}

// Server: accept connection

void SocketStream::accept()
{
    struct sockaddr_in client;
    int length;

    DEBUG( "waiting for connect from client" );
    length = sizeof(client);
    asock  = ::accept(sock, (struct sockaddr *) &client, &length);
    if(asock == -1)
    {
	set(ios::badbit);
	DEBUG( "error: accept(): " << strerror() );
    }
    else 
    {
	DEBUG( "accept(): new fd = " << asock );
	clear();
	rdbuf()->attach(asock);
    }
}

// Server: close connection from accept()

void SocketStream::accept_close()
{
    if(type == server)
    {
	if(asock != -1)
	{
	    rdbuf()->close();
	    DEBUG( "closing socket (accept) fd=" << asock );
//	    ::shutdown(asock, 2);
	    ::close(asock);
	}
    }
}

// Server/Client: close connection

void SocketStream::close()
{
    if(type == server)
    {
	if(asock != -1)
	{
	    rdbuf()->close();
	    DEBUG( "closing socket (accept) fd=" << asock );
//	    ::shutdown(asock, 2);
	    ::close(asock);
	}
	if(sock != -1)
	{
	    DEBUG( "closing socket fd=" << sock );
//	    ::shutdown(sock, 2);
	    ::close(sock);
	}
    }
    
    if(type == client)
	if(sock != -1)
	{
	    rdbuf()->close();
	    DEBUG( "closing socket fd=" << sock );
//	    ::shutdown(sock, 2);
	    ::close(sock);
	}
}

//////// Private methods /////////////////////////////////////////////////////


// Create socket connection to host and port
    
int SocketStream::open_connection(const char *host, unsigned short port)
{
    struct sockaddr_in server;
    struct hostent *hp, *gethostbyname();
    int opt;

    /*
     * Socket erzeugen
     */
    DEBUG( "creating socket" );
    sock = ::socket(AF_INET, SOCK_STREAM, 0);
    if(sock < 0)
    {
	DEBUG( "error: socket(): " << strerror() );
	return -1;
    }
    DEBUG( "socket fd = " << sock );
    
    /*
     * Socket options
     */
    DEBUG( "Setting socket options SO_REUSEADDR, SO_KEEPALIVE" );
    opt = 1;
    if( setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (char *)&opt, sizeof(int)) )
    {
	DEBUG( " error: setsockopt(): " << strerror() );
	::close(sock);
	return -1;
    }
	
    opt = 1;
    if( setsockopt(sock, SOL_SOCKET, SO_KEEPALIVE, (char *)&opt, sizeof(int)) )
    {
	DEBUG( " error: setsockopt(): " << strerror() );
	::close(sock);
	return -1;
    }
	
    /*
     * Hostadresse holen
     */
    DEBUG( "get host address" );
    hp = gethostbyname(host);
    if(!hp)
    {
	DEBUG( "unknown host " << host );
	return -1;
    }
	
    /*
     * Verbindung herstellen
     */
    server.sin_family = AF_INET;
    memcpy(&server.sin_addr, hp->h_addr, hp->h_length);
    server.sin_port = htons(port);
    DEBUG( "connect to host" );
    if(::connect(sock, (struct sockaddr *) &server, sizeof(server)) < 0)
    {
	DEBUG( "error: connect(): " << strerror() );
	::close(sock);
	sock = -1;
    }
		
    return sock;
}

// Bind socket to port and listen for connections

int SocketStream::open_listen(unsigned short port)
{
    struct sockaddr_in server;
    int opt;
    
    /*
     * Socket erzeugen
     */
    DEBUG( "creating socket" );
    sock = socket(AF_INET, SOCK_STREAM, 0);
    if(sock < 0)
    {
	DEBUG( "error: socket(): " << strerror() );
	return -1;
    }
    DEBUG( "socket fd = " << sock );

    /*
     * Socket options
     */
    DEBUG( "Setting socket options SO_REUSEADDR, SO_KEEPALIVE" );
    opt = 1;
    if( setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (char *)&opt, sizeof(int)) )
    {
	DEBUG( " error: setsockopt(): " << strerror() );
	::close(sock);
	return -1;
    }
	
    opt = 1;
    if( setsockopt(sock, SOL_SOCKET, SO_KEEPALIVE, (char *)&opt, sizeof(int)) )
    {
	DEBUG( " error: setsockopt(): " << strerror() );
	::close(sock);
	return -1;
    }
	
    /*
     * Socket an Adresse/Port binden
     */
    DEBUG( "bind socket to address *, port " << port );
    server.sin_family      = AF_INET;
    server.sin_addr.s_addr = INADDR_ANY;
    server.sin_port        = htons(port);
    if(bind(sock, (struct sockaddr *) &server, sizeof(server)) < 0) {
	DEBUG( "error: bind(): " << strerror() );
	::close(sock);
	return -1;
    }

    /*
     * Listen auf Verbindungen
     */
    DEBUG( "listen to socket" );
    if(listen(sock, 10) < 0) {
	DEBUG( "error: listen(): " << strerror() );
	::close(sock);
	return -1;
    }

    return sock;
}
    
//////////////////////////////////////////////////////////////////////////////
