// -*- C++ -*-

// GIST -- Graphical Interactive Simulation Result Tool

//   CNCL interface classes

//   1992,93 Martin Horneffer

#include "gist_ivimpl.hh"


// handle strings put to an ostream so they can easily be read
// and don't get accidentally splitted

class Quote {
    const char *str_;
  public:
    Quote(const char *str) { str_ = str; };
    ostream& write(ostream& out) {
	const char *s;
	char c;
	for (s = str_; c = *s; s++) {
	    switch (c) {
	      case ' ' :  out << "\\_";  break;
	      case '\n':  out << "\\n";  break;
	      case '\\':  out << "\\\\"; break;
	      default: if (!iscntrl(c))  out << c; break;
	    };
	};
	return out;
    };
};
inline ostream& operator<<(ostream& os, Quote& q) { return q.write(os); };


// interface to InterViews-GIST

CommIVImpl::CommIVImpl(const char *host, unsigned short port)
  : host_(host), port_(port), id(0)
{ 
    socket = new SocketStream(host, port);
    pipe = 0;
    out_ = socket;
};
CommIVImpl::CommIVImpl(const char *host, unsigned short port, long magic)
  : host_(host), port_(port), id(0)
{ 
    if (port) {
	socket = new SocketStream(host, port);
	pipe = 0;
	out_ = socket;
    } else {
	socket = 0;
	pipe = new CNPipe;
	if (pipe->open("igist")) {
	    CNCL::fatal("cannot start iv-gist");
	};
	out_ = &pipe->out();
	in_ = &pipe->in();
	
	char buf[128];
	pipe->in().getline(buf, sizeof(buf));

	char fn[64];
	post_filename(fn, magic);
	ofstream post(fn);
	post << pipe->get_pid() << endl;
	post << "localhost" << endl;
    };
};
CommIVImpl::~CommIVImpl() {
    if (socket) delete socket;
    if (pipe) delete pipe;
};
int CommIVImpl::ok() { 
    return ( out_ && !out_->bad() ); 
};
const char *CommIVImpl::gist_type() { 
    return "iv"; 
};
int CommIVImpl::newid() { 
    return ++id; 
};

WorldIVImpl::WorldIVImpl(CommImpl *comm, const char *name, 
			 float xmin, float xmax, float ymin, float ymax, 
			 double st, int m)
  : GISTWorldImpl(name) 
{
    comm_ = (CommIVImpl*)comm;
    lasttime = st;
    mode = m;
    id = comm_->newid();
    str = new ostrstream;
    comm_->out() <<"GIST2\n"
      <<"create "<< id <<" world "<< st
      <<" "<< xmin <<" "<< xmax <<" "<< ymin <<" "<< ymax 
      <<" "<< Quote(name) <<"\n";
};
WorldIVImpl::~WorldIVImpl() {
    out() << ends;
    comm_->out() << str->str();
    comm_->out() <<"call "<< id <<" flush "<< lasttime <<"\n"; 
    comm_->out() <<"call "<< id <<" die\n";
    comm_->out().flush();
    delete str;
};
void WorldIVImpl::flush() {
    out() << ends;
    comm_->out() << str->str();
    comm_->out().flush();
    delete str;
    str = new ostrstream;
};
void WorldIVImpl::flush(double time, int force) {
    if (mode==1) {
	if (force) {
	    out() <<"call "<< id <<" flush "<< time <<"\n";
	    flush();
	};
    } else if (mode==2) {
	if (time > lasttime) {
	    out() <<"call "<< id <<" flush "<< time <<"\n";
	    flush();
	};
    } else {
	flush();
    };
    lasttime = time;
};
void WorldIVImpl::flush(double time) { 
    flush(time,1); 
};
const char *WorldIVImpl::gist_type() { 
    return "iv"; 
};

ObjectIVImpl::ObjectIVImpl(GISTWorldImpl *world,
			   const char *name)
  : GISTObjectImpl(name) 
{
    world_ = (WorldIVImpl*)world;
    id = world_->comm_->newid();
    world_->out() <<"call "<< world_->id <<" create "<< id 
      <<" object "<< Quote(this->name()) <<"\n";
};
ObjectIVImpl::~ObjectIVImpl() {
    world_->out() <<"call "<< id <<" die\n";
};
const char *ObjectIVImpl::gist_type() { 
    return "iv"; 
};


AttributeIVImpl::AttributeIVImpl(GISTObject *object) {
    obj_ = (ObjectIVImpl*)object->impl();
    world_ = obj_->world_;
    id = world_->comm_->newid();
};
AttributeIVImpl::~AttributeIVImpl() {
    world_->out() <<"call "<< id <<" die\n";
};

TextAttributeIVImpl::TextAttributeIVImpl(GISTObject *object, const char *name)
  : GISTTextAttributeImpl(object,name), AttributeIVImpl(object) 
{
    world_->out() <<"call "<< obj_->id <<" create "<< id 
      <<" attr text "<< Quote(this->name()) <<"\n";
};
void TextAttributeIVImpl::put(double time, const char *data) {
    world_->out() <<"call "<< id <<" put "<< time <<" "<< Quote(data) <<"\n";
    world_->flush(time,0);
};

ValueAttributeIVImpl::ValueAttributeIVImpl(GISTObject *object, const char *name)
  : GISTValueAttributeImpl(object,name), AttributeIVImpl(object) 
{
    world_->out() <<"call "<< obj_->id <<" create "<< id 
      <<" attr value "<< Quote(this->name()) <<"\n";
};
void ValueAttributeIVImpl::put(double time, float data) {
    world_->out() <<"call "<< id <<" put "<< time <<" "<< data <<"\n";
    world_->flush(time,0);
};

CoordAttributeIVImpl::CoordAttributeIVImpl(GISTObject *object, const char *name)
  : GISTCoordAttributeImpl(object,name), AttributeIVImpl(object) 
{
    world_->out() <<"call "<< obj_->id <<" create "<< id 
      <<" attr coord "<< Quote(this->name()) <<"\n";
};
void CoordAttributeIVImpl::put(double time, float x, float y) {
    world_->out() <<"call "<< id <<" put "<< time <<" "<< x <<" "<< y <<"\n";
    world_->flush(time,0);
};

ValueArrayAttributeIVImpl::ValueArrayAttributeIVImpl(GISTObject *object, const char *name, unsigned short arraydim)
  : GISTValueArrayAttributeImpl(object,name), AttributeIVImpl(object) 
{
    dim = arraydim;
    world_->out() <<"call "<< obj_->id <<" create "<< id 
      <<" attr valuearray "<< Quote(this->name()) <<" "<< dim <<"\n";
};
void ValueArrayAttributeIVImpl::put(double time, const float data[]) {
    world_->out() <<"call "<< id <<" put "<< time; 
    unsigned short i = dim;
    while(i--) world_->out() <<" "<< *data++;
    world_->out() <<"\n";
    world_->flush(time,0);
};

CoordArrayAttributeIVImpl::CoordArrayAttributeIVImpl(GISTObject *object, const char *name,unsigned short arraydim)
  : GISTCoordArrayAttributeImpl(object,name,arraydim), AttributeIVImpl(object) 
{
    dim = arraydim;
    world_->out() <<"call "<< obj_->id <<" create "<< id 
      <<" attr coordarray " << Quote(this->name()) <<" "<< dim <<"\n";
};
void CoordArrayAttributeIVImpl::put(double time, const float data[]) {
    world_->out() <<"call "<< id <<" put "<< time; 
    unsigned short i = dim*2;
    while(i--) world_->out() <<" "<< *data++;
    world_->out() <<"\n";
    world_->flush(time,0);
};


GraphicIVImpl::GraphicIVImpl(GISTObject *object) {
    obj_ = (ObjectIVImpl*)object->impl();
    world_ = obj_->world_;
    id = world_->comm_->newid();
};
GraphicIVImpl::~GraphicIVImpl() {
    world_->out() <<"call "<< id <<" die\n";
};
int GraphicIVImpl::byte(float v) {
    int res = int(256*v);
    if (res < 0 ) res = 0;
    if (res > 255) res = 255;
    return res;
};
const char *GraphicIVImpl::xcolor_name(float red, float green, float blue) { 
    ostrstream(colorbuf,80) 
      << form("#%02x%02x%02x", byte(red), byte(green), byte(blue))
      << ends;
    return colorbuf; 
};

PictureIVImpl::PictureIVImpl(GISTObject *parent,
			     const char *name,
			     int prio,
			     GISTTextAttributeImpl *vis,
			     GISTWorldImpl *world)
  : GISTPictureImpl(parent,name), GraphicIVImpl(parent) 
{
    world_->out() <<"call "<< obj_->id <<" create "<< id 
      <<" picture"<<" prio "<< prio;
    if (vis) world_->out() <<" vis "<< ((TextAttributeIVImpl*)vis)->id;
    if (world) world_->out() <<" world "<< ((WorldIVImpl*)world)->id;
    world_->out() <<" name "<< Quote(this->name()) <<"\n"; 
};
const char *PictureIVImpl::color_name(float red, float green, float blue) { 
    return xcolor_name(red, green, blue);
};
void PictureIVImpl::set_color(const char *color) {
    world_->out() <<"call "<< id <<" color "<< Quote(color) <<"\n";
};
void PictureIVImpl::set_color(GISTTextAttributeImpl *attr) {
    world_->out() <<"call "<< id <<" color_a "<< ((TextAttributeIVImpl*)attr)->id <<"\n";
};
void PictureIVImpl::set_brush(float thickness, unsigned short pattern) {
    world_->out() <<"call "<< id <<" brush "<< thickness <<" "<< pattern <<"\n";
};
void PictureIVImpl::pen_down() {
    world_->out() <<"call "<< id <<" pendown\n";
};
void PictureIVImpl::pen_up() {
    world_->out() <<"call "<< id <<" penup\n";
};
void PictureIVImpl::pen_up_fill() {
    world_->out() <<"call "<< id <<" penupfill\n";
};
void PictureIVImpl::set_pos(float x, float y) {
    world_->out() <<"call "<< id <<" setpos "<< x <<" "<< y <<"\n";
};
void PictureIVImpl::set_pos(GISTCoordAttributeImpl *attr) {
    world_->out() <<"call "<< id <<" setpos_c "<< ((CoordAttributeIVImpl*)attr)->id <<"\n";
};
void PictureIVImpl::set_path(GISTCoordArrayAttributeImpl *attr) {
    world_->out() <<"call "<< id <<" setpos_ca "
      << ((CoordArrayAttributeIVImpl*)attr)->id <<"\n";
};
void PictureIVImpl::move_rel(float dx, float dy) {
    world_->out() <<"call "<< id <<" move "
      << dx <<" "<< dy <<"\n";
};
void PictureIVImpl::move_rel(float dx, GISTValueAttributeImpl *dy) {
    world_->out() <<"call "<< id <<" move_y "
      << dx <<" "<< ((ValueAttributeIVImpl*)dy)->id <<"\n";
};
void PictureIVImpl::move_rel(GISTValueAttributeImpl *dx,float dy) {
    world_->out() <<"call "<< id <<" move_x "
      << ((ValueAttributeIVImpl*)dx)->id <<" "<< dy <<"\n";
};
void PictureIVImpl::move_rel(GISTValueAttributeImpl *dx,GISTValueAttributeImpl *dy) {
    world_->out() <<"call "<< id <<" move_xy "
      << ((ValueAttributeIVImpl*)dx)->id <<" "<< ((ValueAttributeIVImpl*)dy)->id <<"\n";
};
void PictureIVImpl::move_dir(float dx, GISTValueAttributeImpl *dy) {
    world_->out() <<"call "<< id <<" move_w "
      << dx <<" "<< ((ValueAttributeIVImpl*)dy)->id <<"\n";
};
void PictureIVImpl::move_dir(GISTValueAttributeImpl *dx,float dy) {
    world_->out() <<"call "<< id <<" move_r "
      << ((ValueAttributeIVImpl*)dx)->id <<" "<< dy <<"\n";
};
void PictureIVImpl::move_dir(GISTValueAttributeImpl *dx,GISTValueAttributeImpl *dy) {
    world_->out() <<"call "<< id <<" move_rw "
      << ((ValueAttributeIVImpl*)dx)->id <<" "<< ((ValueAttributeIVImpl*)dy)->id <<"\n";
};
void PictureIVImpl::text(const char *text,float size,const char *font,const char *style) {
    world_->out() <<"call "<< id <<" text size "<< size;
    if (font) world_->out() <<" font "<< Quote(font);
    if (style) world_->out() <<" style "<< Quote(style);
    world_->out() <<" text "<< Quote(text) <<"\n";
};
void PictureIVImpl::text(GISTTextAttributeImpl *text,float size,const char *font,const char *style) {
    world_->out() <<"call "<< id <<" text size "<< size;
    if (font) world_->out() <<" font "<< Quote(font);
    if (style) world_->out() <<" style "<< Quote(style);
    world_->out() <<" text_a "<< ((TextAttributeIVImpl*)text)->id <<"\n";
};
void PictureIVImpl::draw_image(const char *name,float x_align,float y_align) {
    world_->out() <<"call "<< id <<" image "<< x_align <<" "<< y_align <<" "<< Quote(name) <<"\n";
};
void PictureIVImpl::draw_image(GISTTextAttributeImpl *name,float x_align, float y_align) {
    world_->out() <<"call "<< id <<" image_a "<< x_align <<" "<< y_align <<" "<< ((TextAttributeIVImpl*)name)->id <<"\n";
};

const char *ImageIVImpl::color_name(float red, float green, float blue) { 
    return xcolor_name(red, green, blue);
};
ImageIVImpl::ImageIVImpl(GISTObject *parent,
			 const char *name,
			 const char *image, 
			 float x, float y,
			 float width, float height,
			 float hcenter, float vcenter,
			 int prio,
			 GISTTextAttributeImpl *vis,
			 GISTWorldImpl *world)
  : GISTGraphicImpl(parent,name), GraphicIVImpl(parent) 
{
    world_->out() <<"call "<< obj_->id <<" create "<< id <<" image";
	  world_->out() <<" imagename "<< Quote(image);
    world_->out() <<" position "<< x <<" "<< y;
    world_->out() <<" extension "<< width <<" "<< height;
    world_->out() <<" center "<< hcenter <<" "<< vcenter;
    world_->out() <<" prio "<< prio;
    if (vis) world_->out() <<" vis "<< ((TextAttributeIVImpl*)vis)->id;
    if (world) world_->out() <<" world "<< ((WorldIVImpl*)world)->id;
    world_->out() <<" name "<< Quote(this->name()) <<"\n"; 
};
ImageIVImpl::ImageIVImpl(GISTObject *parent,
			 const char *name,
			 const char *image, 
			 GISTCoordAttributeImpl *position,
			 float width, float height,
			 float hcenter, float vcenter,
			 int prio,
			 GISTTextAttributeImpl *vis,
			 GISTWorldImpl *world)
  : GISTGraphicImpl(parent,name), GraphicIVImpl(parent) 
{
    world_->out() <<"call "<< obj_->id <<" create "<< id <<" image";
    world_->out() <<" imagename "<< Quote(image);
    if (position) 
      world_->out() <<" posattr "<< ((CoordAttributeIVImpl*)position)->id
	<<" extension "<< width <<" "<< height
	<<" center "<< hcenter <<" "<< vcenter
	<<" prio "<< prio;
    if (vis) world_->out() <<" vis "<< ((TextAttributeIVImpl*)vis)->id;
    if (world) world_->out() <<" world "<< ((WorldIVImpl*)world)->id;
    world_->out() <<" name "<< Quote(this->name()) <<"\n"; 
};
ImageIVImpl::ImageIVImpl(GISTObject *parent,
			 const char *name,
			 GISTTextAttributeImpl *image, 
			 float x, float y,
			 float width, float height,
			 float hcenter, float vcenter,
			 int prio,
			 GISTTextAttributeImpl *vis,
			 GISTWorldImpl *world)
  : GISTGraphicImpl(parent,name), GraphicIVImpl(parent) 
{
    world_->out() <<"call "<< obj_->id <<" create "<< id <<" image";
    if (image) 
      world_->out() <<" imageattr "<< ((TextAttributeIVImpl*)image)->id;
    world_->out() <<" position "<< x <<" "<< y;
    world_->out() <<" extension "<< width <<" "<< height;
    world_->out() <<" center "<< hcenter <<" "<< vcenter;
    world_->out() <<" prio "<< prio;
    if (vis) world_->out() <<" vis "<< ((TextAttributeIVImpl*)vis)->id;
    if (world) world_->out() <<" world "<< ((WorldIVImpl*)world)->id;
    world_->out() <<" name "<< Quote(this->name()) <<"\n"; 
};
ImageIVImpl::ImageIVImpl(GISTObject *parent,
			 const char *name,
			 GISTTextAttributeImpl *image, 
			 GISTCoordAttributeImpl *position,
			 float width, float height,
			 float hcenter, float vcenter,
			 int prio,
			 GISTTextAttributeImpl *vis,
			 GISTWorldImpl *world)
  : GISTGraphicImpl(parent,name), GraphicIVImpl(parent) 
{
    world_->out() <<"call "<< obj_->id <<" create "<< id <<" image";
    if (image) 
      world_->out() <<" imageattr "<< ((TextAttributeIVImpl*)image)->id;
    if (position) 
      world_->out() <<" posattr "<< ((CoordAttributeIVImpl*)position)->id;
    world_->out() <<" extension "<< width <<" "<< height;
    world_->out() <<" center "<< hcenter <<" "<< vcenter;
    world_->out() <<" prio "<< prio;
    if (vis) world_->out() <<" vis "<< ((TextAttributeIVImpl*)vis)->id;
    if (world) world_->out() <<" world "<< ((WorldIVImpl*)world)->id;
    world_->out() <<" name "<< Quote(this->name()) <<"\n"; 
};

const char *CurveIVImpl::color_name(float red, float green, float blue) { 
    return xcolor_name(red, green, blue);
};
CurveIVImpl::CurveIVImpl(GISTObject *parent,
			 const char *name,
			 GISTValueAttributeImpl *value, 
			 float x, float y, int type,
			 float width, float height,
			 float hcenter, float vcenter,
			 const char *foreg, const char *backg,
			 int prio,
			 GISTTextAttributeImpl *vis,
			 GISTWorldImpl *world)
  : GISTGraphicImpl(parent,name), CurveIVImpl::GraphicIVImpl(parent) 
{
    world_->out() <<"call "<< obj_->id <<" create "<< id <<" curve ";
    world_->out() << ((ValueAttributeIVImpl*)value)->id <<" type "<< type;
    world_->out() <<" position "<< x <<" "<< y;
    world_->out() <<" extension "<< width <<" "<< height;
    world_->out() <<" center "<< hcenter <<" "<< vcenter;
    world_->out() <<" foreg "<< Quote(foreg);
    world_->out() <<" backg "<< Quote(backg);
    world_->out() <<" prio "<< prio;
    if (vis) world_->out() <<" vis "<< ((TextAttributeIVImpl*)vis)->id;
    if (world) world_->out() <<" world "<< ((WorldIVImpl*)world)->id;
    world_->out() <<" name "<< Quote(this->name()) <<"\n"; 
};
void CurveIVImpl::hint(const char*text) {
    world_->out() <<"call "<< id <<" hint "<< Quote(text) <<"\n";
};

const char *DiagramIVImpl::color_name(float red, float green, float blue) { 
    return xcolor_name(red, green, blue);
};
DiagramIVImpl::DiagramIVImpl(GISTObject *parent,
			     const char *name,
			     GISTCoordArrayAttributeImpl *value, 
			     float x, float y, int type,
			     float width, float height,
			     float hcenter, float vcenter,
			     const char *foreg, const char *backg,
			     int prio,
			     GISTTextAttributeImpl *vis,
			     GISTWorldImpl *world)
  : GISTGraphicImpl(parent,name), GraphicIVImpl(parent) 
{
    world_->out() <<"call "<< obj_->id <<" create "<< id <<" diagram ";
    world_->out() << ((CoordArrayAttributeIVImpl*)value)->id <<" type "<< type;
    world_->out() <<" position "<< x <<" "<< y;
    world_->out() <<" extension "<< width <<" "<< height;
    world_->out() <<" center "<< hcenter <<" "<< vcenter;
    world_->out() <<" foreg "<< Quote(foreg);
    world_->out() <<" backg "<< Quote(backg);
    world_->out() <<" prio "<< prio;
    if (vis) world_->out() <<" vis "<< ((TextAttributeIVImpl*)vis)->id;
    if (world) world_->out() <<" world "<< ((WorldIVImpl*)world)->id;
    world_->out() <<" name "<< Quote(this->name()) <<"\n"; 
};
void DiagramIVImpl::hint(const char*text) {
    world_->out() <<"call "<< id <<" hint "<< Quote(text) <<"\n";
};
