//*BHEADER* :ts=8  -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: FVar.c,v 0.22 1994/07/22 09:21:33 cncl-adm Exp cncl-adm $
 *
 * Class: CNFVar --- Fuzzy variable
 *
 *****************************************************************************
 * Copyright (C) 1992/1993   Communication Networks
 *                           Aachen University of Technology
 *                           Kopernikusstr. 16
 *                           W-5100 Aachen
 *                           Germany
 *                           Email: mj@dfv.rwth-aachen.de (Martin Junius)
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 **EHEADER********************************************************************/

#include "FVar.h"



/*
 * Constructors
 */
CNFVar::CNFVar(CNParam *param)
    : CNNamed(""), varvalue(0), varxmin(0), varxmax(1), varsets(0)
{}

    
CNFVar::CNFVar(double min, double max)
    : CNNamed(""), varvalue(0), varfvalue(NIL), 
      varxmin(min), varxmax(max), varsets(0)
{}


CNFVar::CNFVar(CNStringR xname, double min, double max)
    : CNNamed(xname), varvalue(0), varfvalue(NIL),
      varxmin(min), varxmax(max), varsets(0)
{}



/*
 * Initialize
 */
void CNFVar::init(CNStringR xname, double min, double max)
{
    set_name(xname);
    varxmin = min;
    varxmax = max;
}



/*
 * Add fuzzy set to array
 */
void CNFVar::add_value_set(CNFSet *fset)
{
    int n = varsets.get_size();
    
    varsets.set_size(n + 1);
    varsets[n] = fset;
}

void CNFVar::add_value_set(CNFSet &fset)
{
    add_value_set(&fset);
}


void CNFVar::add_value_set(int i, CNFSet *fset)
{
    int n = varsets.get_size();

    if(i < 0)
	fatal(NIL, "CNFVar ", name(), ": illegal index < 0");
    else
    {
	if(i >= n)
	    varsets.set_size(i + 1);
	varsets[i] = fset;
    }
}

void CNFVar::add_value_set(int i, CNFSet &fset)
{
    add_value_set(i, &fset);
}



/*
 * Get fuzzy set from array
 */
CNFSet *CNFVar::get_value_set(int i)
{
    int n = varsets.get_size();
    
    if(i < 0)
	fatal(NIL, "CNFVar ", name(), ": illegal index < 0");
    else
	return i>=n ? NIL : CNFSet::cast_from_object( varsets[i] );
    /**NOT REACHED**/
    return NIL;
}



/*
 * Get membership value from indexed fuzzy set
 */
double CNFVar::get_membership(int i)
{
    int n = varsets.get_size();
    CNFSet *set;
    
    if(i < 0)
	fatal(NIL, "CNFVar ", name(), ": illegal index < 0");
    else
    {
	set = i>=n ? NIL : CNFSet::cast_from_object( varsets[i] );
	return set ? set->get_membership( value() ) : 0. ;
    }
    /**NOT REACHED**/
    return 0.;
}
    


/*
 * Get/set value
 */
double CNFVar::set_value(double x)
{
    double old = varvalue;
    varvalue = x;
    return old;
}

double CNFVar::value(double x)
{
    double old = varvalue;
    varvalue = x;
    return old;
}


double CNFVar::get_value() const
{
    return varvalue;
}

double CNFVar::value() const
{
    return varvalue;
}


/*
 * Get/set fuzzy value
 */
CNFSet *CNFVar::set_fuzzy_value(CNFSet *x)
{
    CNFSet *old = varfvalue;
    varfvalue = x;
    return old;
}

CNFSet *CNFVar::fuzzy_value(CNFSet *x)
{
    CNFSet *old = varfvalue;
    varfvalue = x;
    return old;
}


CNFSet *CNFVar::get_fuzzy_value() const
{
    return varfvalue;
}

CNFSet *CNFVar::fuzzy_value() const
{
    return varfvalue;
}


/*
 * Get xmin/xmax
 */
double CNFVar::get_xmin() const
{
    return varxmin;
}

double CNFVar::get_xmax() const
{
    return varxmax;
}

double CNFVar::xmin() const
{
    return varxmin;
}

double CNFVar::xmax() const
{
    return varxmax;
}



/*
 * Print fuzzy sets membership with value
 */
void CNFVar::print_membership()
{
    int n = varsets.get_size();
    CNFSet *fset;
    
    cout << "Variable \"" << name() << "\":  " << value() << endl;
    
    for(int i=0; i<n; i++)
    {
	fset = CNFSet::cast_from_object(varsets[i]);
	cout << "  \"" << fset->name()
	     << "\"  " << fset->get_membership(value()) << endl;
    }
}

    

/***** Default I/O member function for CNCL classes **************************/

// CNNormal output
void CNFVar::print(ostream &strm) const
{
    strm << "var " << name() << "(" << xmin() << ", " << xmax() << ")" << endl
	 << "{" << endl;
    int n = varsets.get_size();
    for(int i=0; i<n; i++)
	strm << "\t" << varsets.get(i) << endl;
    strm << "};" << endl;
}

// Debug output
void CNFVar::dump(ostream &strm) const
{
    strm << "CNFVar { $Revision: 0.22 $" << endl
	 << "\"" << name() << "\": value=" << value() 
	 << " xmin=" << xmin() << " xmax=" << xmax() << endl
	 << "fuzzy sets:" << endl;
    int n = varsets.get_size();
    for(int i=0; i<n; i++)
	 varsets.get(i)->dump(strm);
    strm << "}" << endl;
}



/***** CNCL stuff for type information and exemplar objects ******************/

// Describing object for class CNFVar
static CNClass CNFVar_desc("CNFVar", "$Revision: 0.22 $", CNFVar::new_object);

// "Type" for type checking functions
CNClassDesc CN_FVAR = &CNFVar_desc;






