/* -*-C++-*-
 * 
 * Fuzzy Rules Compiler
 *
 * $Id: fuzzyparse.y,v 0.22 1994/07/22 09:24:39 cncl-adm Exp cncl-adm $
 *
 * fuzzyparse.y - fuzzy rules parser (GNU bison)
 */

/* C definitions */
%{
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include "fuzzysubs.h"

int yyerror(char *);
int yylex(void);

extern FILE *hfile, *cfile, *in;
extern char *in_name;

/* Name for this fuzzy rule set */
char *rulebase;

/* Fuzzy sets */
struct Node *currentset;

/* Fuzzy variables */
enum VarUsed { NOTUSED = 0, USED };
struct Var {
	struct Node  node;
	struct List  setlist;
	enum VarUsed inused, outused;
} *currentvar;
struct List varlist = {NULL};

/* Fuzzy clauses */
enum VarType { UNDEF = 0, IN, OUT };
struct Clause {
	struct Node     node;
	struct Var     *var;
	enum   VarType  type;
	struct Node    *set;
} *currentclause;

/* Fuzzy rules */
int   rulenumber = 0;
struct Rule {
	struct List clauselist;
	double      certainty;
} currentrule = { {NULL}, 1.0 };

void initrule(struct Rule *r)
{
	r->clauselist.head = NULL;
	r->certainty = 1.0;
}
%}

/* Return values of the lexical scanner */
%union {
	double  num;
	char   *id;
       }

/* Tokens from lexical scanner */
/* Keywords */
%token Krulebase
%token Kset
%token Ktrapez
%token Ktriangle
%token Kvar
%token Kif
%token Kand
%token Kcertainty
%token Kis
/* Tokens with return value */
%token <id> Identifier
%token <num> Number

/* Fuzzy rules grammar */
%%

Program		: Krulebase RuleBaseName ';' Body
		  {
			struct Node *n = varlist.head;

			while (n)
			{
				freenamelist(&((struct Var *) n)->setlist);
				n = n->next; 
			}
			freenamelist(&varlist);
			free(rulebase);
		  };

RuleBaseName	: Identifier
		  {
		      rulebase = strdup($1);

		      fprintf(hfile, "struct %s\n{\n\t%s();\n", $1, $1);
		      fprintf(hfile, "\t%s(const %s &);\n", $1, $1);
		      fprintf(hfile, "\tCNFRuleBase %sbase;\n", $1);
		      fprintf(cfile, "%s::%s(const %s &)\n{}\n\n", $1, $1, $1);
		      fprintf(cfile, "%s::%s()\n{\n", $1, $1);
		      fprintf(cfile, "\t%sbase.set_name(\"%s\");\n", $1);
		  };

Body		: Body CommandLine |
		  CommandLine;

CommandLine	: Statement ';' |
                  Statement;

Statement	: Definitions;

Definitions	: Kvar VarDefinition |
		  Kif  RuleDefinition;

VarDefinition	: NewVarId '(' Number ',' Number ')' '{' SetList '}'
		  {
		      struct Node *n = currentvar->setlist.head;
		      char *varname = (char *)currentvar->node.data;

		      fprintf(hfile, "\tCNFVar %s;\n", varname);
		      fprintf(cfile, "\t%s.init(\"%s\", %g, %g);\n",
			      varname, varname, $3, $5);

		      while (n)
		      {
			  fprintf(cfile, "\t%s.add_value_set(%s_%s);\n",
				  varname, varname, (char *)n->data);
			  n = n->next;
		      }
		  };

NewVarId	: Identifier
		  {
		      if (!(currentvar = (struct Var *)addnewname(&varlist,
				 sizeof(struct Var), $1)))
			{
				yyerror("redefinition of variable");
				YYERROR;
			}
		  };

SetList		: SetList NewSet  |
		  NewSet;

NewSet		: Kset SetDefinition ';';

SetDefinition	: Ktrapez SetTrapezDef |
		  Ktriangle SetTriangleDef;

SetTrapezDef	: NewSetId '(' Number ',' Number ',' Number ',' Number ')'
		  {
		      fprintf(hfile, "\tCNFSetTrapez %s_%s;\n",
			      (char *)currentvar->node.data,
			      (char *)currentset->data        );
		      fprintf(cfile, "\t%s_%s.init(\"%s\", %g, %g, %g, %g);\n",
			      (char *)currentvar->node.data,
			      (char *)currentset->data,
			      (char *)currentset->data, $3, $5, $7, $9);
		  };

NewSetId	: Identifier
		  {
			if (!(currentset = addnewname(&currentvar->setlist,
						sizeof(struct Node), $1)))
			{
				yyerror("set reused in variable");
				YYERROR;
			}
		  };

SetTriangleDef	: NewSetId '(' Number ',' Number ',' Number ')'
		  {
		      fprintf(hfile, "\tCNFSetTriangle %s_%s;\n",
			      (char *)currentvar->node.data,
			      (char *)currentset->data        );
		      fprintf(cfile, "\t%s_%s.init(\"%s\", %g, %g, %g);\n",
			      (char *)currentvar->node.data,
			      (char *)currentset->data,
			      (char *)currentset->data, $3, $5, $7);
		  };

RuleDefinition	: '(' LeftHandSide ')' '{' RightHandSide '}'
		  {
		      struct Clause *cl = (struct Clause *)
			  currentrule.clauselist.head;
		      char rulename[20];

		      sprintf(rulename,"rule%d",rulenumber++);

		      fprintf(hfile, "\tCNFRule %s;\n", rulename);
		      fprintf(cfile, "\t%s.certainty(%g);\n",
			      rulename, currentrule.certainty);
		      fprintf(cfile, "\t%sbase.add_rule(%s);\n",
			      rulebase, rulename);
		      
		      while (cl)
		      {
			  int         lhs     = (cl->type == IN);
			  struct Var *var     = cl->var;
			  char       *varname = (char *)var->node.data;
			  
			  fprintf(cfile,
				  "\t%s.add_%chs(new CNFClause(%s, %s_%s));\n",
				  rulename, (lhs ? 'l' : 'r'),
				  varname, varname,
				  (char *)cl->set->data);

			  if (lhs)
			  {
			      if (var->inused == NOTUSED)
			      {
				  fprintf(cfile, "\t%sbase.add_in_var(%s);\n",
					 rulebase, varname);
				  var->inused = USED;
			      }
			  }
			  else if (var->outused == NOTUSED)
			  {
			      fprintf(cfile, "\t%sbase.add_out_var(%s);\n",
				      rulebase, varname);
			      var->outused = USED;
			  }
			  
			  cl = (struct Clause *) cl->node.next;
		      }

		      freelist(&currentrule.clauselist);
		      initrule(&currentrule);
		  };

LeftHandSide	: LeftHandSide Kand ConditionP |
		  ConditionP;

ConditionP      : '(' ConditionP ')' |
                  Condition;

Condition	: VarIdentifier Kis SetIdentifier
		  {
			currentclause = (struct Clause *) 
					newnode(sizeof(struct Clause),
						NULL);
			currentclause->var  = currentvar;
			currentclause->type = IN;
			currentclause->set  = currentset;
			add(&currentrule.clauselist,
				(struct Node *) currentclause);
		  };

VarIdentifier	: Identifier
		  {
		      if (!(currentvar = (struct Var *)findname(&varlist,$1)))
		      {
			  yyerror("undefined variable");
			  YYERROR;
		      }
		  };

SetIdentifier	: Identifier
		  {
		      if (!(currentset = findname(&currentvar->setlist,$1)))
		      {
			  yyerror("set not associated with variable");
			  YYERROR;
		      }
		  };

RightHandSide	: RightHandSide AssignP ';' |
                  AssignP ';' |
                  RuleOption ';';

AssignP         : '(' AssignP ')' |
                  Assign;

Assign		: VarIdentifier Kis SetIdentifier
		  {
		      currentclause = (struct Clause *)
			  newnode(sizeof(struct Clause),
				  NULL);
		      currentclause->var  = currentvar;
		      currentclause->type = OUT;
		      currentclause->set  = currentset;
		      add(&currentrule.clauselist,
			  (struct Node *) currentclause);
		  };

RuleOption	: Kcertainty Kis Number
		  {
			currentrule.certainty = $3;
		  };

/* C code */
%%

/* Interface to lexical scanner */
extern int lexline;
extern char *yytext;

/* Error routine */
int yyerror(char *errtxt)
{
    fprintf(stderr, "%s:%d: %s, token '%s'\n",
	    in_name, lexline, errtxt, yytext);
    return 0;
}
