/*
	Module:	Arbitrary-In Sorted-Out (AISO)
	Author:	dick@cs.vu.nl (Dick Grune @ Vrije Universiteit, Amsterdam)

Description:
	This is the body of a module that builds an arbitrary-in
	sorted-out data structure, to be used as a heap, a priority queue, etc.
	See aiso.spc for further info.
*/

#define	PRIVATE	static
	
extern char *AISO_MALLOC();

PRIVATE struct aiso_node *root;		/* root of tree */
#ifdef	AISO_ITER
PRIVATE struct aiso_node *list;		/* start of linked list */
#endif	/* AISO_ITER */

/* the policy */
PRIVATE int aiso_size = 0;
PRIVATE int acc_mark = 1;

#define	add_entry()	(aiso_size++)
#define	rem_entry()	(aiso_size--)
#define	reset_access()	(acc_mark = 1)
#define	count_access()	(acc_mark <<= 1)
#define	must_rotate()	(acc_mark > aiso_size)

int
InsertAiso(v)
	AISO_TYPE v;
{
	register struct aiso_node *new_node;
	register struct aiso_node **hook = &root;
#ifdef	AISO_ITER
	register struct aiso_node **prev = &list;
#endif	/* AISO_ITER */

	new_node = (struct aiso_node *)AISO_MALLOC(sizeof (struct aiso_node));
	if (!new_node) {
		/* avoid modifying the tree */
		return 0;
	}

	while (*hook) {
		register struct aiso_node *an = *hook;

		count_access();
		if (AISO_BEFORE(v, an->an_value)) {
			/* head left */
			if (!an->an_left || !must_rotate()) {
				/* standard action */
				hook = &an->an_left;
			}
			else {
				/* change (l A r) B (C) into (l) A (r B C) */
				register struct aiso_node *anl = an->an_left;

				an->an_left = anl->an_right;
				anl->an_right = an;
				*hook = anl;
				reset_access();
			}
		}
		else {
			/* head right */
			if (!an->an_right || !must_rotate()) {
				/* standard action */
				hook = &an->an_right;
			}
			else {
				/* change (A) B (l C r) into (A B l) C (r) */
				register struct aiso_node *anr = an->an_right;

				an->an_right = anr->an_left;
				anr->an_left = an;
				*hook = anr;
				reset_access();
			}
#ifdef	AISO_ITER
			prev = &an->an_next;
#endif	/* AISO_ITER */
		}
	}

	new_node->an_left = 0;
	new_node->an_right = 0;
#ifdef	AISO_ITER
	new_node->an_next = *prev;
	*prev = new_node;
#endif	/* AISO_ITER */
	new_node->an_value = v;
	*hook = new_node;
	add_entry();
	return 1;
}

#ifdef	AISO_EXTR

int
ExtractAiso(vp)
	AISO_TYPE *vp;
{
	register struct aiso_node **hook = &root;
	register struct aiso_node *an;

	if (!root) return 0;

	while ((an = *hook), an->an_left) {
		/* head left */
		count_access();
		if (!must_rotate()) {
			/* standard action */
			hook = &an->an_left;
		}
		else {
			/* change (l A r) B (C) into (l) A (r B C) */
			register struct aiso_node *anl = an->an_left;

			an->an_left = anl->an_right;
			anl->an_right = an;
			*hook = anl;
			reset_access();
		}
	}
	/* found the first */
	*vp = an->an_value;
	*hook = an->an_right;
#ifdef	AISO_ITER
	list = an->an_next;
#endif	/* AISO_ITER */
	AISO_FREE((char *)an);
	rem_entry();
	return 1;
}

#endif	/* AISO_EXTR */

#ifdef	AISO_ITER

OpenIter(ip)
	AisoIter *ip;
{
	*ip = list;
}

int
GetAisoItem(ip, vp)
	AisoIter *ip;
	AISO_TYPE *vp;
{
	register struct aiso_node *an = *ip;

	if (!an) return 0;

	*vp = an->an_value;
	*ip = an->an_next;
	return 1;
}

CloseIter(ip)
	AisoIter *ip;
{
	*ip = 0;
}

#endif	/* AISO_ITER */

#ifdef	AISO_DEBUG	/* requires AISO_FORMAT */

pr_tree()
{
	pr_inf(0, '-', root);
	printf("================\n");
}

PRIVATE pr_inf(level, ch, an)
	int level;
	char ch;
	struct aiso_node *an;
{
	register int i;

	if (!an) return;

	pr_inf(level+1, '/', an->an_right);
	for (i = 0; i < level; i++) {
		printf("     ");
	}
	printf("%c", ch);
	printf(AISO_FORMAT, an->an_value);
	printf("\n");
	pr_inf(level+1, '\\', an->an_left);
}

#endif	/* AISO_DEBUG */

