/*
 * MS-DOS version of UNIX directory access package.
 * (opendir, readdir, closedir).
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <dos.h>
extern int errno;
#define MAXPATH 64

#include "dir.h"
static char *firstf();
static char *nextf();

typedef int bool;
#define TRUE 1
#define FALSE 0
#define EOS '\0'
#define NULL 0

static char dirpath[MAXPATH+1]= "";
static bool busy= FALSE;

/*
 * Open a directory.
 * This MS-DOS version allows only one directory to be open at a time.
 * The path must exist and be a directory.
 * Then it only saves the file name and returns a pointer to it.
 */

char *
opendir(path)
	char *path;
{
	struct stat buffer;

	if (dirpath[0] != EOS) {
		 errno= EMFILE;
		 return NULL; /* Only one directory can be open at a time */
	}
	if (stat(path, &buffer) == -1)
		return NULL; /* Directory doesn't exist */
	if ((buffer.st_mode & S_IFMT) != S_IFDIR) {
		errno= ENOENT;
		return NULL; /* Not a directory */
	}
	strcpy(dirpath, path);
	path= dirpath + strlen(dirpath);
	if (path > dirpath && path[-1] != ':' && path[-1] != '/' &&
			path[-1] != '\\')
		*path++ = '\\';
	strcpy(path, "*.*");
	busy= FALSE; /* Tell readdir this is the first time */
	return dirpath;
}

/*
 * Close a directory.
 * We must clear the first char of dirpath to signal opendir that
 * no directory is open.
 */

closedir(dirp)
	char *dirp;
{
	dirp[0]= EOS;
}

/*
 * Read the next directory entry.
 */

struct direct *
readdir(dp)
	char *dp;
{
	static struct direct buffer;
	char *p;

	if (!busy)
		p= firstf(dp);
	else
		p= nextf();
	busy= (p != NULL);
	if (!busy)
		return NULL;
	strcpy(buffer.d_name, p);
	buffer.d_namlen= strlen(p);
	return &buffer;
}


/*
 * Low-level implementation -- access to DOS system calls FIRST and NEXT.
 */

static char dta[64]; /* Device Transfer Area */

/* File attributes */
#define READONLY	0x01
#define HIDDEN		0x02
#define SYSTEM		0x04
#define VOLUMEID	0x08
#define DIRECTORY	0x10
#define ARCHIVE		0x20

#define MATCHATTR	(HIDDEN|SYSTEM|DIRECTORY)

static char *
firstf(pat)
    char *pat;
{
    int flags;
    union REGS regs;
    struct SREGS segregs;

    setdta(dta);

    segread(&segregs);
    regs.h.ah = 0x4e;
    regs.x.cx = MATCHATTR;
#ifdef M_I86LM /* Large Model -- far pointers */
    regs.x.dx = FP_OFF(pat);
    segregs.ds = FP_SEG(pat);
#else
    regs.x.dx = (int) pat;
#endif
    flags= intdosx(&regs, &regs, &segregs);
    if (regs.x.cflag)
	return NULL;
    fixfile();
    return dta + 30;
}

static char *
nextf()
{
    int flags;
    union REGS regs;

    setdta(dta);

    regs.h.ah = 0x4f;
    flags= intdos(&regs, &regs);
    if (regs.x.cflag)
	return NULL;
    fixfile();
    return dta + 30;
}

/*
 * Convert the file name in the Device Transfer Area to lower case.
 */

static
fixfile()
{
	char *cp;

	for (cp= dta+30; *cp != EOS; ++cp)
		*cp= tolower(*cp);
}

/*
 * Tell MS-DOS where the Device Transfer Area is.
 */

static
setdta(dta)
    char *dta;
{
    union REGS regs;
    struct SREGS segregs;

    segread(&segregs);
    regs.h.ah = 0x1a;
#ifdef M_I86LM /* Large Model -- far pointers */
    regs.x.dx = FP_OFF(dta);
    segregs.ds = FP_SEG(dta);
#else
    regs.x.dx = (int) dta;
#endif
    intdosx(&regs, &regs, &segregs);
}
