/*-------------------------------------------------------------------------*/
/* Benchmark (Finite Domain)            INRIA Rocquencourt - ChLoE Project */
/*                                                                         */
/* Name           : partit.pl                                              */
/* Title          : integer partitionning                                  */
/* Original Source: Daniel Diaz - INRIA France                             */
/* Adapted by     :                                                        */
/* Date           : September 1993                                         */
/*                                                                         */
/* Partition numbers 1,2,...,N into two groups A and B such that:          */
/*   a) A and B have the same length,                                      */
/*   b) sum of numbers in A = sum of numbers in B,                         */
/*   c) sum of squares of numbers in A = sum of squares of numbers in B.   */
/*                                                                         */
/* This problem admits a solution if N is a multiple of 8.                 */
/*                                                                         */
/* Note: finding a partition of 1,2...,N into 2 groups A and B such that:  */
/*                                                                         */
/*     Sum (k^p) = Sum l^p                                                 */
/*   k in A      l in B                                                    */
/*                                                                         */
/* admits a solution if N mod 2^(p+1) = 0 (N is a multilple of 2^(p+1)).   */
/* Condition a) is a special case where p=0, b) where p=1 and c) where p=2.*/
/*                                                                         */
/* Two redundant constraints are used:                                     */
/*                                                                         */
/*   - in order to avoid duplicate solutions (permutations) we impose      */
/*     A1<A2<....<AN/2, B1<B2<...<BN/2 and A1<B1. This achieves much more  */
/*     pruning than only alldifferents(A) and alldifferents(B).            */
/*                                                                         */
/*   - the half sums are known                                             */
/*                              N                                          */
/*        Sum k^1 = Sum l^1 = (Sum i) / 2 = N*(N+1) / 4                    */
/*       k in A    l in B      i=1                                         */
/*                              N                                          */
/*        Sum k^2 = Sum l^2 = (Sum i^2)/2 = N*(N+1)*(2*N+1) / 12           */
/*       k in A    l in B      i=1                                         */
/*                                                                         */
/* Solution:                                                               */
/* N=8  A=[1,4,6,7]                                                        */
/*      B=[2,3,5,8]                                                        */
/*                                                                         */
/* N=16 A=[1,2,7,8,11,12,13,14]                                            */
/*      B=[3,4,5,6, 9,10,15,16]                                            */
/*                                                                         */
/* N=24 A=[1,2,3,8,13,14,15,16,18,19,20,21]                                */
/*      B=[4,5,6,7, 9,10,11,12,17,22,23,24]                                */
/*                                                                         */
/* N=32 A=[1,3,4, 7, 8, 9,16,20,21,22,23,24,25,26,27,28]                   */
/*      B=[2,5,6,10,11,12,13,14,15,17,18,19,29,30,31,32]                   */
/*-------------------------------------------------------------------------*/

:- main.

q:-	get_labeling(Lab), write('N ?'), read_integer(N),
	statistics(runtime,_),
	partit(N,A,B,Lab), statistics(runtime,[_,Y]),
	write(sol(A,B)), nl,
	write('time : '), write(Y), nl,
	halt_or_else(0,true).




partit(N,A,B,Lab):-
	fd_vector_max(N),
	N2 is N//2,
	length(A,N2),
	domain(A,1,N),
	length(B,N2),
	domain(B,1,N),
	append(A,B,L),
	alldifferent(L),
	no_duplicate(A,B),                           % redundant constraint 1
	half_sums(N,HS1,HS2),                        % redundant constraint 2
	sums(A,HS1,HS2),
	sums(B,HS1,HS2),
	lab(Lab,L).




sums([],0,0).

sums([X|L],S1,S2):-
	sums(L,T1,T2),
	'xx=y'(X,X2),
	S1#=X+T1,
	S2#=X2+T2.




no_duplicate(A,B):-
	ascending_order(A),
	ascending_order(B),
	A=[X1|_],
	B=[X2|_],
	'x>y'(X2,X1).




ascending_order([X|L]):-
	ascending_order(L,X).


ascending_order([],_).

ascending_order([Y|L],X):-
	'x>y'(Y,X),
	ascending_order(L,Y).



half_sums(N,HS1,HS2):-
	S1  is N*(N+1)//2,
	S2  is S1*(2*N+1)//3,
	HS1 is S1//2,           
	HS2 is S2//2.




lab(normal,L):-
	labeling(L).

lab(ff,L):-
	labelingff(L).




get_labeling(Lab):-
	argc(C),
	get_labeling1(C,Lab).


get_labeling1(1,normal).

get_labeling1(2,Lab):-
	argv(1,Lab).




:- q.
