/*-------------------------------------------------------------------------*/
/* Prolog to Wam Compiler               INRIA Rocquencourt - ChLoE Project */
/* Version 1.0  -  C Run-time                           Daniel Diaz - 1992 */
/* Extended to FD Constraints (July 1992 revised May 1994)                 */
/*                                                                         */
/* FD Constraints Implementation - Header file                             */
/*                                                                         */
/* fd_engine.h                                                             */
/*-------------------------------------------------------------------------*/
#include "bool.h"
#include "fd_range.h"



/*---------------------------------*/
/* Constants                       */
/*---------------------------------*/

          /* FD Range/Term Registers */

#define T(t)                       (CPP_CAT(tr,t))
#define R(r)                       (CPP_CAT(rr,r))




          /* FD Variable Frame */

#define FD_VARIABLE_FRAME_SIZE     (OFFSET_RANGE+RANGE_SIZE+CHAINS_SIZE)
#define FD_INT_VARIABLE_FRAME_SIZE (OFFSET_RANGE+RANGE_SIZE)

#define OFFSET_RANGE               4
#define RANGE_SIZE                 (2+(sizeof(Range)/sizeof(WamWord)))

#define OFFSET_CHAINS              (OFFSET_RANGE+RANGE_SIZE)
#define CHAINS_SIZE                7



#define FD_Tag_Value(fdv_adr)      (((WamWord *) fdv_adr)[0])
#define FD_INT_Date(fdv_adr)       (((WamWord *) fdv_adr)[1])

#define PS_Next_Fdv_Adr(fdv_adr)   (((WamWord *) fdv_adr)[2])
#define PS_Propag_Mask(fdv_adr)    (((WamWord *) fdv_adr)[3])

#define Range_Stamp(fdv_adr)       (((WamWord *) fdv_adr)[OFFSET_RANGE])
#define Nb_Elem(fdv_adr)           (((WamWord *) fdv_adr)[OFFSET_RANGE+1])
#define Range(fdv_adr)             ((Range *) ((WamWord *) fdv_adr+OFFSET_RANGE+2))

#define Chains_Stamp(fdv_adr)      (((WamWord *) fdv_adr)[OFFSET_CHAINS])
#define Chains_Mask(fdv_adr)       (((WamWord *) fdv_adr)[OFFSET_CHAINS+1])
#define Chain_Min(fdv_adr)         (((WamWord *) fdv_adr)[OFFSET_CHAINS+2])
#define Chain_Max(fdv_adr)         (((WamWord *) fdv_adr)[OFFSET_CHAINS+3])
#define Chain_Min_Max(fdv_adr)     (((WamWord *) fdv_adr)[OFFSET_CHAINS+4])
#define Chain_Dom(fdv_adr)         (((WamWord *) fdv_adr)[OFFSET_CHAINS+5])
#define Chain_Val(fdv_adr)         (((WamWord *) fdv_adr)[OFFSET_CHAINS+6])




          /* Shorthands for Range(fdv_adr)'s fields */

#define Extra_Cstr(fdv_adr)        (Range(fdv_adr)->extra_cstr)
#define Min(fdv_adr)               (Range(fdv_adr)->min)
#define Max(fdv_adr)               (Range(fdv_adr)->max)
#define Vec(fdv_adr)               (Range(fdv_adr)->vec)




          /* Chain / Propagation Mask */


#define MASK_EMPTY                 0
#define MASK_MIN                   1
#define MASK_MAX                   2
#define MASK_MIN_MAX               4
#define MASK_DOM                   8
#define MASK_VAL                   16


#define Has_Min_Mask(mask)         ((mask) & MASK_MIN)
#define Has_Max_Mask(mask)         ((mask) & MASK_MAX)
#define Has_Min_Max_Mask(mask)     ((mask) & MASK_MIN_MAX)
#define Has_Dom_Mask(mask)         ((mask) & MASK_DOM)
#define Has_Val_Mask(mask)         ((mask) & MASK_VAL)


#define Set_Min_Mask(mask)         ((mask) |= MASK_MIN);
#define Set_Max_Mask(mask)         ((mask) |= MASK_MAX);
#define Set_Min_Max_Mask(mask)     ((mask) |= MASK_MIN_MAX);
#define Set_Dom_Mask(mask)         ((mask) |= MASK_DOM);
#define Set_Val_Mask(mask)         ((mask) |= MASK_VAL);




          /* Chain Record Frame */

#define CHAIN_RECORD_FRAME_SIZE    2

#define CF_Pointer(rec_adr)        (rec_adr[0])
#define Next_Chain(rec_adr)        (rec_adr[1])




          /* Constraint Frame */

#define CONSTRAINT_FRAME_SIZE      4

#define AF_Pointer(cf)             (cf[0])
#define Tell_Fdv_Adr(cf)           (cf[1])
#define Optim2(cf)                 (cf[2])
#define Cstr_Address(cf)           (cf[3])




#define Cstr_Fct_To_WamWord(f)     ((WamWord) (f))
#define WamWord_To_Cstr_Fct(w)     ((Bool (*)()) (w))




          /* Argument Frame */

#define Frame_Variable(fv)         ((WamWord *)(AF[fv]))
#define Frame_Range_Parameter(fp)  ((Range *)  (AF[fp]))
#define Frame_Term_Parameter(fp)   ((int)      (AF[fp]))




          /* Miscellaneous */

#define ENV_VAR_VECTOR_MAX         "VECTORMAX"
#define DEFAULT_VECTOR_MAX         127




#define Decl_Fd_Constraint_Vars                                             \
     WamWord word,*fdv_adr;                                                 \
     WamWord tr0, tr1, tr2, tr3, tr4, tr5, tr6, tr7,                        \
             tr8, tr9, tr10,tr11,tr12,tr13,tr14,tr15,                       \
             tr16,tr17,tr18,tr19,tr20,tr21,tr22,tr23;                       \
     Range   rr0, rr1, rr2, rr3, rr4, rr5, rr6, rr7,                        \
             rr8, rr9, rr10,rr11,rr12,rr13,rr14,rr15,                       \
             rr16,rr17,rr18,rr19,rr20,rr21,rr22,rr23;





#define Decl_Fd_Install_Vars                                                \
     WamWord *fdv_adr;




#define math_min(x,y)              ((x) <= (y) ? (x) : (y))
#define math_max(x,y)              ((x) >= (y) ? (x) : (y))




          /* Errors Messages */

#define ERR_ILLEGAL_RANGE_PARAM    "FD Constraint: Illegal range parameter\n"

#define ERR_ILLEGAL_TERM_PARAM     "FD Constraint: Illegal term parameter\n"

#define MSG_VECTOR_TOO_SMALL       "Warning: Vector too small - maybe lost solutions\n"




/*---------------------------------*/
/* Type Definitions                */
/*---------------------------------*/

/*---------------------------------*/
/* Global Variables                */
/*---------------------------------*/

/*---------------------------------*/
/* Function Prototypes             */
/*---------------------------------*/

void     Init_Fd_Engine         (void);

void     Term_Fd_Engine         (void);

void     Push_Fd_Variable       (void);
void     Push_Fd_Int_Variable  (int n);

Bool     X_Y_Value_In_A_Frame   (WamWord word,WamWord tag,WamWord *adr);

void     Init_Call_Constraint   (void);

Bool     Execute_Propagation    (void);




Bool     Tell_Int_Interval      (int n,int min,int max);

Bool     Tell_Int_Range         (int n,Range *range);

Bool     Tell_Interval_Interval (WamWord *fdv_adr,int min,int max);

Bool     Tell_Range_Range       (WamWord *fdv_adr,Range *range);




Bool     Unify_FDV_And_INT      (WamWord *fdv_adr,int n);

Bool     Unify_FDV_And_FDV      (WamWord *fdv_adr1,WamWord *fdv_adr2);

int      Fd_Variable_Size       (WamWord *fdv_adr);

int      Copy_Fd_Variable       (WamWord *dst_adr,WamWord *fdv_adr);

Bool     Prolog_Range_To_Range  (WamWord *adr,Range *range);

void     Write_Fd_Variable      (FILE *out,WamWord *fdv_adr);




/*---------------------------------*/
/* Auxiliary engine macros         */
/*---------------------------------*/


#define Is_An_FDV_Tagged_Word(word)(Tag_Of(word)==FDV)




#define Fail_In_Constraint                                                  \
     return(FALSE);




#define Trail_Fd_Int_Variable_If_Necessary(fdv_adr)                         \
    {                                                                       \
     if (Word_Needs_Trailing(&FD_Tag_Value(fdv_adr)))                       \
            {                                                               \
             Trail_OV(&FD_Tag_Value(fdv_adr))                               \
             Trail_OV(&FD_INT_Date(fdv_adr))                                \
             Trail_Range_If_Necessary(fdv_adr)                              \
            }                                                               \
    }




#define Trail_Range_If_Necessary(fdv_adr)                                   \
    {                                                                       \
     if (Range_Stamp(fdv_adr)!=STAMP)                                       \
        {                                                                   \
         Trail_MV(fdv_adr+OFFSET_RANGE,RANGE_SIZE)                          \
         if (Is_Sparse(Range(fdv_adr)))                                     \
             Trail_MV((WamWord *) Vec(fdv_adr),vec_size)                    \
         Range_Stamp(fdv_adr)=STAMP;                                        \
        }                                                                   \
    }




#define Trail_Chains_If_Necessary(fdv_adr)                                  \
    {                                                                       \
     if (Chains_Stamp(fdv_adr)!=STAMP)                                      \
        {                                                                   \
         Trail_MV(fdv_adr+OFFSET_CHAINS,CHAINS_SIZE)                        \
         Chains_Stamp(fdv_adr)=STAMP;                                       \
        }                                                                   \
    }




#define Add_Chain(fdv_adr,chain,mask)                                       \
    {                                                                       \
     if (Is_An_FDV_Tagged_Word(FD_Tag_Value(fdv_adr)))                      \
        {                                                                   \
         Trail_Chains_If_Necessary(fdv_adr)                                 \
                                                                            \
         Chains_Mask(fdv_adr) |= mask;                                      \
                                                                            \
         CF_Pointer(H)= (WamWord) CF;                                       \
         Next_Chain(H)= chain;                                              \
                                                                            \
         chain=(WamWord) H;                                                 \
                                                                            \
         H += CHAIN_RECORD_FRAME_SIZE;                                      \
        }                                                                   \
    }




#define Execute_Constraint(fail_code)                                       \
     AF=(WamWord *) AF_Pointer(CF);                                         \
     if (!(*WamWord_To_Cstr_Fct(Cstr_Address(CF)))())                       \
        { fail_code }




/*---------------------------------*/
/* FD engine macros (fd inst->C)   */
/*---------------------------------*/

          /* Interface with Prolog clauses instructions */

#define fd_set_x_AF(nb_arg,x)                                               \
    {                                                                       \
     X(x)=(WamWord) (AF=H);                                                 \
     H += nb_arg;                                                           \
    }




#define fd_set_y_AF(nb_arg,y)                                               \
    {                                                                       \
     Y(E,y)=(WamWord) (AF=H);                                               \
     H += nb_arg;                                                           \
    }




#define fd_x_variable_in_A_frame(x)                                         \
    {                                                                       \
     X(x)=Tag_Value(REF,H);                                                 \
     *AF++ =(WamWord) H;                                                    \
     Push_Fd_Variable();                                                    \
    }




#define fd_x_value_in_A_frame(x)                                            \
    {                                                                       \
     Deref(X(x),word,tag,adr)                                               \
     if (!X_Y_Value_In_A_Frame(word,tag,adr))                               \
         fail                                                               \
    }




#define fd_y_variable_in_A_frame(y)                                         \
    {                                                                       \
     Y(E,y)=Tag_Value(REF,H);                                               \
     *AF++ =(WamWord) H;                                                    \
     Push_Fd_Variable();                                                    \
    }




#define fd_y_value_in_A_frame(y)                                            \
    {                                                                       \
     Deref(Y(E,y),word,tag,adr)                                             \
     if (!X_Y_Value_In_A_Frame(word,tag,adr))                               \
         fail                                                               \
    }




#define fd_x_range_parameter_in_A_frame(x)                                  \
    {                                                                       \
     *AF=(WamWord) H;                                                       \
     H += sizeof(Range) / sizeof(WamWord);                                  \
     if (!Prolog_Range_To_Range(&X(x),(Range *) *AF))                       \
        {                                                                   \
         Lib1(printf,ERR_ILLEGAL_RANGE_PARAM);                              \
         fail                                                               \
        }                                                                   \
     AF++;                                                                  \
    }




#define fd_y_range_parameter_in_A_frame(y)                                  \
    {                                                                       \
     *AF=(WamWord) H;                                                       \
     H += sizeof(Range) / sizeof(WamWord);                                  \
     if (!Prolog_Range_To_Range(&Y(E,y),(Range *) *AF++))                   \
        {                                                                   \
         Lib1(printf,ERR_ILLEGAL_RANGE_PARAM);                              \
         fail                                                               \
        }                                                                   \
    }




#define fd_x_term_parameter_in_A_frame(x)                                   \
    {                                                                       \
     Deref(X(x),word,tag,adr)                                               \
     if (tag!=INT)                                                          \
        {                                                                   \
         Lib1(printf,ERR_ILLEGAL_TERM_PARAM);                               \
         fail                                                               \
        }                                                                   \
     *AF++ =UnTag_INT(word);                                                \
    }




#define fd_y_term_parameter_in_A_frame(y)                                   \
    {                                                                       \
     Deref(Y(E,y),word,tag,adr)                                             \
     if (tag!=INT)                                                          \
        {                                                                   \
         Lib1(printf,ERR_ILLEGAL_TERM_PARAM);                               \
         fail                                                               \
        }                                                                   \
     *AF++ =UnTag_INT(word);                                                \
    }




#define fd_install_constraint_with_x_AF(inst_nb,x)                          \
    {                                                                       \
     Fd_Install_Prototyp(inst_nb);                                          \
                                                                            \
     AF=(WamWord *) X(x);                                                   \
     if (!Install_Fct_Name(inst_nb)())                                      \
         fail                                                               \
    }




#define fd_install_constraint_with_y_AF(inst_nb,y)                          \
    {                                                                       \
     Fd_Install_Prototyp(inst_nb);                                          \
                                                                            \
     AF=(WamWord *) Y(E,y);                                                 \
     if (!Install_Fct_Name(inst_nb)())                                      \
         fail                                                               \
    }




#define fd_call_constraint                                                  \
    {                                                                       \
     Init_Call_Constraint();                                                \
     Execute_Constraint(fail)                                               \
     if (!Execute_Propagation())                                            \
         fail                                                               \
    }




          /* Install instructions */

#define fd_create_C_frame(cstr_nb,tell_fv,opt2_flag)                        \
    {                                                                       \
     Fd_Constraint_Prototyp(cstr_nb);                                       \
                                                                            \
     CF=H;                                                                  \
     AF_Pointer(CF)    =(WamWord) AF;                                       \
     Tell_Fdv_Adr(CF)  =(WamWord) (Frame_Variable(tell_fv));                \
     Optim2(CF)        =opt2_flag;                                          \
     Cstr_Address(CF)  =Cstr_Fct_To_WamWord(Constraint_Fct_Name(cstr_nb));  \
                                                                            \
     H += CONSTRAINT_FRAME_SIZE;                                            \
    }




#define fd_install_ind_min(fv)                                              \
    {                                                                       \
     fdv_adr=Frame_Variable(fv);                                            \
     Add_Chain(fdv_adr,Chain_Min(fdv_adr),MASK_MIN)                         \
    }




#define fd_install_ind_max(fv)                                              \
    {                                                                       \
     fdv_adr=Frame_Variable(fv);                                            \
     Add_Chain(fdv_adr,Chain_Max(fdv_adr),MASK_MAX)                         \
    }




#define fd_install_ind_min_max(fv)                                          \
    {                                                                       \
     fdv_adr=Frame_Variable(fv);                                            \
     Add_Chain(fdv_adr,Chain_Min_Max(fdv_adr),MASK_MIN_MAX)                 \
    }




#define fd_install_ind_dom(fv)                                              \
    {                                                                       \
     fdv_adr=Frame_Variable(fv);                                            \
     Add_Chain(fdv_adr,Chain_Dom(fdv_adr),MASK_DOM)                         \
    }




#define fd_install_dly_val(fv)                                              \
    {                                                                       \
     fdv_adr=Frame_Variable(fv);                                            \
     Add_Chain(fdv_adr,Chain_Val(fdv_adr),MASK_VAL)                         \
    }




          /* Constraint instructions */

#define fd_allocate                                                         \
    {                                                                       \
     save_H=H;                                                              \
     H+=vec_size;                                                           \
    }




#define fd_proceed                                                          \
    {                                                                       \
     return(TRUE);                                                          \
    }





#define fd_tell_interval(tmin,tmax)                                         \
    {                                                                       \
     word=FD_Tag_Value(fdv_adr=(WamWord *) Tell_Fdv_Adr(CF));               \
     if (!Is_An_FDV_Tagged_Word(word))                                      \
        {                                                                   \
         if (!Tell_Int_Interval(Min(fdv_adr),T(tmin),T(tmax)))              \
             Fail_In_Constraint                                             \
        }                                                                   \
      else                                                                  \
         if (Is_Sparse(Range(fdv_adr)))                                     \
            {                                                               \
             save_H=H;                                                      \
             Init_Interval_Range(&R(0),T(tmin),T(tmax));                    \
                                                                            \
             if (!Tell_Range_Range(fdv_adr,&R(0)))                          \
                 Fail_In_Constraint                                         \
            }                                                               \
          else                                                              \
             if (!Tell_Interval_Interval(fdv_adr,T(tmin),T(tmax)))          \
                 Fail_In_Constraint                                         \
    }




#define fd_tell_range(r)                                                    \
    {                                                                       \
     word=FD_Tag_Value(fdv_adr=(WamWord *) Tell_Fdv_Adr(CF));               \
     if (!Is_An_FDV_Tagged_Word(word))                                      \
        {                                                                   \
         if (!Tell_Int_Range(Min(fdv_adr),&R(r)))                           \
             Fail_In_Constraint                                             \
        }                                                                   \
      else                                                                  \
         if (!Tell_Range_Range(fdv_adr,&R(r)))                              \
             Fail_In_Constraint                                             \
    }




          /* Range */

#define fd_interval_range(r,tmin,tmax)                                      \
     Init_Interval_Range(&R(r),T(tmin),T(tmax));




#define fd_range_parameter(r,fp)                                            \
    {                                                                       \
     R(r).vec=NULL;                                                         \
     Range_Copy(&R(r),Frame_Range_Parameter(fp));                           \
    }




#define fd_ind_dom(r,fv)                                                    \
    {                                                                       \
     fdv_adr=Frame_Variable(fv);                                            \
     R(r).vec=NULL;                                                         \
     Range_Copy(&R(r),Range(fdv_adr));                                      \
    }




#define fd_union(r,r1)                                                      \
     Range_Union(&R(r),&R(r1));




#define fd_inter(r,r1)                                                      \
     Range_Inter(&R(r),&R(r1));




#define fd_compl(r)                                                         \
     Range_Compl(&R(r));




#define fd_compl_of_singleton(r,t)                                          \
    {                                                                       \
     R(r).vec=NULL;                                                         \
     Range_Compl_Singleton(&R(r),T(t));                                     \
    }




#define fd_range_add_range(r,r1)                                            \
     Range_Add_Range(&R(r),&R(r1));




#define fd_range_sub_range(r,r1)                                            \
     Range_Sub_Range(&R(r),&R(r1));




#define fd_range_mul_range(r,r1)                                            \
     Range_Mul_Range(&R(r),&R(r1));




#define fd_range_div_range(r,r1)                                            \
     Range_Div_Range(&R(r),&R(r1));




#define fd_range_mod_range(r,r1)                                            \
     Range_Mod_Range(&R(r),&R(r1));




#define fd_range_add_term(r,t)                                              \
     Range_Add_Term(&R(r),T(t));




#define fd_range_sub_term(r,t)                                              \
     Range_Add_Term(&R(r),-T(t));




#define fd_range_mul_term(r,t)                                              \
     Range_Mul_Term(&R(r),T(t));




#define fd_range_div_term(r,t)                                              \
     Range_Div_Term(&R(r),T(t));




#define fd_range_mod_term(r,t)                                              \
     Range_Mod_Term(&R(r),T(t));




#define fd_range_copy(r,r1)                                                 \
    {                                                                       \
     R(r).vec=NULL;                                                         \
     Range_Copy(&R(r),&R(r1));                                              \
    }




#define fd_range_fct(fct_name,r,args)                                       \
    {                                                                       \
     R(r).vec=NULL;                                                         \
     fct_name(&R(r),args);                                                  \
    }




          /* term */

#define fd_integer(t,n)                                                     \
     T(t)= n;




#define fd_term_parameter(t,fp)                                             \
     T(t)=Frame_Term_Parameter(fp);




#define fd_ind_min(t,fv)                                                    \
    {                                                                       \
     fdv_adr=Frame_Variable(fv);                                            \
     T(t)=Min(fdv_adr);                                                     \
    }




#define fd_ind_max(t,fv)                                                    \
    {                                                                       \
     fdv_adr=Frame_Variable(fv);                                            \
     T(t)=Max(fdv_adr);                                                     \
    }




#define fd_ind_min_max(tmin,tmax,fv)                                        \
    {                                                                       \
     fdv_adr=Frame_Variable(fv);                                            \
     T(tmin)=Min(fdv_adr);                                                  \
     T(tmax)=Max(fdv_adr);                                                  \
    }




#define fd_dly_val(t,fv,l_else)                                             \
    {                                                                       \
     word=FD_Tag_Value(fdv_adr=Frame_Variable(fv));                         \
     if (!Is_An_FDV_Tagged_Word(word))                                      \
         T(t)=Min(fdv_adr);                                                 \
      else                           /* tag==FDV */                         \
         goto Label_Name(l_else);                                           \
    }




#define fd_term_add_term(t,t1)                                              \
     T(t) += T(t1);




#define fd_term_sub_term(t,t1)                                              \
     T(t) -= T(t1);




#define fd_term_mul_term(t,t1)                                              \
     T(t)=M_Mul(T(t),T(t1));




#define fd_term_floor_div_term(t,t1)                                        \
     T(t)=M_Div(T(t),T(t1));




#define fd_term_ceil_div_term(t,t1)                                         \
     T(t)=M_Div((T(t) + T(t1) - 1),T(t1));




#define fd_term_mod_term(t,t1)                                              \
     T(t)=M_Mod(T(t),T(t1));




#define fd_term_copy(t,t1)                                                  \
     T(t)=T(t1);




#define fd_term_fct(fct_name,t,args)                                        \
     T(t)=fct_name(args);




#define arg_1(a1)                          a1
#define arg_2(a1,a2)                       a1,a2
#define arg_3(a1,a2,a3)                    a1,a2,a3
#define arg_4(a1,a2,a3,a4)                 a1,a2,a3,a4
#define arg_5(a1,a2,a3,a4,a5)              a1,a2,a3,a4,a5
#define arg_6(a1,a2,a3,a4,a5,a6)           a1,a2,a3,a4,a5,a6
#define arg_7(a1,a2,a3,a4,a5,a6,a7)        a1,a2,a3,a4,a5,a6,a7
#define arg_8(a1,a2,a3,a4,a5,a6,a7,a8)     a1,a2,a3,a4,a5,a6,a7,a8
#define arg_9(a1,a2,a3,a4,a5,a6,a7,a8,a9)  a1,a2,a3,a4,a5,a6,a7,a8,a9




#define rr(r)                     &R(r)                      /* by address */
#define tr(t)                     T(t)                       /* by value   */




/*---------------------------------*/
/* Interface with C files          */
/*---------------------------------*/

#define Label_Name(label)          fd_label##label




#define infinity                   INTERVAL_INFINITY




#define Install_Fct_Name(inst_nb)  Install_##inst_nb




#define Fd_Install_Prototyp(inst_nb)                                        \
    static Bool Install_Fct_Name(inst_nb)(void);




#define Begin_Fd_Install(inst_nb)                                           \
                                                                            \
    static Bool Install_Fct_Name(inst_nb)(void)                             \
    {                                                                       \
     Decl_Fd_Install_Vars




#define End_Fd_Install                                                      \
    }




#define Constraint_Fct_Name(cstr_nb)  Constraint_##cstr_nb




#define Fd_Constraint_Prototyp(cstr_nb)                                     \
    static Bool Constraint_Fct_Name(cstr_nb)(void);




#define Begin_Fd_Constraint(cstr_nb)                                        \
                                                                            \
    static Bool Constraint_Fct_Name(cstr_nb)(void)                          \
    {                                                                       \
     Decl_Fd_Constraint_Vars




#define Fd_Label(label)                                                     \
         Label_Name(label):




#define End_Fd_Constraint                                                   \
    }




