/*-------------------------------------------------------------------------*/
/* Prolog To Wam Compiler                INRIA Rocquencourt - CLoE Project */
/* C Run-time                                           Daniel Diaz - 1994 */
/*                                                                         */
/* Machine Dependent Features                                              */
/*                                                                         */
/* machine.c                                                               */
/*-------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/param.h>

#define MACHINE

#include "wam_engine.h"




/*---------------------------------*/
/* Constants                       */
/*---------------------------------*/

          /* Error Messages */

#define ERR_STACKS_ALLOCATION      "Memory allocation fault"
#define ERR_STACK_OVERFLOW         "%s stack overflow"

#define ERR_CANNOT_OPEN_DEV0       "Cannot open /dev/zero : %s"
#define ERR_CANNOT_UNMAP           "unmap failed : %s"
#define ERR_CANNOT_SHMGET          "shmget failed : %s"
#define ERR_CANNOT_SHMAT           "shmat failed : %s"





/*---------------------------------*/
/* Type Definitions                */
/*---------------------------------*/

/*---------------------------------*/
/* Global Variables                */
/*---------------------------------*/

#ifdef M_USE_MALLOC

static WamWord *check_adr[sizeof(Stacks)/sizeof(InfStack)+1];

#endif

extern char *sys_errlist[];





/*---------------------------------*/
/* Function Prototypes             */
/*---------------------------------*/
static
void      Signal_Handler        (int sig,int code,int scp,WamWord *addr);

#define Round_Up(x,y)           M_Mul(M_Div(((x)+(y)-1),(y)),(y))
#define Round_Down(x,y)         M_Mul(M_Div((x),(y)),(y))






#ifdef M_USE_MALLOC

/*-------------------------------------------------------------------------*/
/* M_ALLOCATE_STACKS                                                       */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void M_Allocate_Stacks(void)

{
 int       len=0;
 WamWord  *addr;
 int       i;

 for(i=0;i<NB_OF_STACKS;i++)
     len+=stk_tbl[i].size;

 addr=(WamWord *) calloc(len,sizeof(WamWord));
 if (addr==NULL)
     Fatal_Error(ERR_STACKS_ALLOCATION);

 for(i=0;i<NB_OF_STACKS;i++)
    {
     stk_tbl[i].stack=addr;
     addr+=stk_tbl[i].size;
     check_adr[i]=addr-M_SECURITY_MARGIN;
     *check_adr[i]=M_MAGIC;
    }

 check_adr[NB_OF_STACKS]=NULL;
}

#endif




#ifdef M_USE_MMAP
#include <sys/mman.h>

/*-------------------------------------------------------------------------*/
/* M_ALLOCATE_STACKS                                                       */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void M_Allocate_Stacks(void)

{
 int      fd;
 int      page_size;
 int      page_word_mask;
 int      len=0;
 WamWord *addr;
 int      i;

 page_size=getpagesize()/sizeof(WamWord);
 fd=open("/dev/zero",0);

 if (fd== -1)
     Fatal_Error(ERR_CANNOT_OPEN_DEV0,sys_errlist[errno]);

 for(i=0;i<NB_OF_STACKS;i++)
    {
     stk_tbl[i].size=Round_Up(stk_tbl[i].size,page_size);
     len+=stk_tbl[i].size+page_size;
    }

 addr=(WamWord *) M_MMAP_HIGH_ADR;
 addr-=len;

 addr=(WamWord *) mmap((caddr_t) addr,len*sizeof(WamWord),
                       PROT_READ|PROT_WRITE,MAP_PRIVATE|MAP_FIXED,fd,0);
 if ((int) addr<0)
     Fatal_Error(ERR_STACKS_ALLOCATION);

 for(i=0;i<NB_OF_STACKS;i++)
    {
     stk_tbl[i].stack=addr;
     addr+=stk_tbl[i].size;
     if (munmap((caddr_t) addr,page_size)== -1)
         Fatal_Error(ERR_CANNOT_UNMAP,sys_errlist[errno]);

     addr+=page_size;
    }

 signal(SIGSEGV,Signal_Handler);
}

#endif




#ifdef M_USE_SHM
#include <sys/ipc.h>
#include <sys/shm.h>

/*-------------------------------------------------------------------------*/
/* M_ALLOCATE_STACKS                                                       */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void M_Allocate_Stacks(void)

{
 int      page_size;
 int      seg_size;
 int      len=0;
 WamWord *addr;
 int      i;
 int      shmid[NB_OF_STACKS];

 page_size=getpagesize()/sizeof(WamWord);
 seg_size =SHMLBA/sizeof(WamWord);

 for(i=0;i<NB_OF_STACKS;i++)
    {
     stk_tbl[i].size=Round_Up(stk_tbl[i].size,page_size);
     len+=Round_Up(stk_tbl[i].size+page_size,seg_size);
    }

 addr=(WamWord *) M_SHM_HIGH_ADR;
 addr=(WamWord *) Round_Down((int) addr,SHMLBA);
 addr-=len;

 for(i=0;i<NB_OF_STACKS;i++)
    {
     shmid[i]=shmget(IPC_PRIVATE,stk_tbl[i].size*sizeof(WamWord),0600);
     if (shmid[i]== -1)
         Fatal_Error(ERR_CANNOT_SHMGET,sys_errlist[errno]);
    }


 for(i=0;i<NB_OF_STACKS;i++)
    {
     stk_tbl[i].stack=addr;
     if ((int) shmat(shmid[i],addr,0)== -1)
         Fatal_Error(ERR_CANNOT_SHMAT,sys_errlist[errno]);

     shmctl(shmid[i],IPC_RMID,0);
     addr+=stk_tbl[i].size;
     addr+=page_size;
    }

 signal(SIGSEGV,Signal_Handler);
}

#endif




/*-------------------------------------------------------------------------*/
/* SIGNAL_HANDLER                                                          */
/*                                                                         */
/*-------------------------------------------------------------------------*/
static void Signal_Handler(int sig,int code,int scp,WamWord *addr)

{
 int i;

 for(i=NB_OF_STACKS;i-- >0;)
     if (addr>=stk_tbl[i].stack+stk_tbl[i].size)
         Fatal_Error(ERR_STACK_OVERFLOW,stk_tbl[i].name);

 Fatal_Error("Segmentation Violation");
}




#ifdef M_USE_MALLOC

/*-------------------------------------------------------------------------*/
/* M_CHECK_STACKS                                                          */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void M_Check_Stacks(void)

{
 WamWord **p=(WamWord **) check_adr;

 for(p=(WamWord **) check_adr;*p;p++)
     if (**p != M_MAGIC)
         Fatal_Error(ERR_STACK_OVERFLOW,
                     stk_tbl[p-(WamWord **) check_adr].name);
}

#endif




#ifdef M_sparc

/*-------------------------------------------------------------------------*/
/* LIB_CALL                                                                */
/*                                                                         */
/*-------------------------------------------------------------------------*/
int Lib_Call(i0,i1,i2,i3,i4,i5)

{
 int res;
 register int g1 asm("g1");
 register int g2 asm("g2");
 register int g3 asm("g3");
 register int g4 asm("g4");
 register int g5 asm("g5");
 register int g6 asm("g6");
 register int g7 asm("g7");

 int save_g1=g1;
 int save_g2=g2;
 int save_g3=g3;
 int save_g4=g4;
 int save_g5=g5;
 int save_g6=g6;
 int save_g7=g7;

 res=(* ((int (*)()) i0))(i1,i2,i3,i4,i5);

 g1=save_g1;
 g2=save_g2;
 g3=save_g3;
 g4=save_g4;
 g5=save_g5;
 g6=save_g6;
 g7=save_g7;
 return res;
}

#endif
