/*-------------------------------------------------------------------------*/
/* Prolog to Wam Compiler               INRIA Rocquencourt - ChLoE Project */
/* C Run-time                                           Daniel Diaz - 1991 */
/*                                                                         */
/* Atoms and Predicates Tables Management                                  */
/*                                                                         */
/* atom_pred.c                                                             */
/*-------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#define ATOM_PRED

#include "wam_engine.h"




/*---------------------------------*/
/* Constants                       */
/*---------------------------------*/

/*---------------------------------*/
/* Type Definitions                */
/*---------------------------------*/

/*---------------------------------*/
/* Global Variables                */
/*---------------------------------*/

int char_type[256]= {

/*  nul soh stx etx eot enq ack bel bs  ht  nl  vt  np  cr  so  si  */
    EOF,LA, LA, LA, LA, LA, LA, LA, LA, LA, LA, LA, LA, LA, LA, LA,

/*  dle dc1 dc2 dc3 dc4 nak syn etb can em sub esc  fs  gs  rs  us  */
    LA, LA, LA, LA, LA, LA, LA, LA, LA, LA, LA, LA, LA, LA, LA, LA,

/*  spc !   "   #   $   %   &   '   (   )   *   +   ,   -   .   /   */
    LA, SC, DQ, SY, SY, CM, SY, QT, PC, PC, SY, SY, PC, SY, SY, SY,

/*  0   1   2   3   4   5   6   7   8   9   :   ;   <   =   >   ?   */
    DI, DI, DI, DI, DI, DI, DI, DI, DI, DI, SY, SC, SY, SY, SY, SY,

/*  @   A   B   C   D   E   F   G   H   I   J   K   L   M   N   O   */
    SY, CL, CL, CL, CL, CL, CL, CL, CL, CL, CL, CL, CL, CL, CL, CL,

/*  P   Q   R   S   T   U   V   W   X   Y   Z   [   \   ]   ^   _   */
    CL, CL, CL, CL, CL, CL, CL, CL, CL, CL, CL, PC, SY, PC, SY, UL,

/*  `   a   b   c   d   e   f   g   h   i   j   k   l   m   n   o    */
    BQ, SL, SL, SL, SL, SL, SL, SL, SL, SL, SL, SL, SL, SL, SL, SL,

/*  p   q   r   s   t   u   v   w   x   y   z   {   |   }   ~   del  */
    SL, SL, SL, SL, SL, SL, SL, SL, SL, SL, SL, PC, PC, PC, SY, LA 

/*  0x80 ... 0xff = 0 (!=EOF since EOF== -1)  */
    };




/*---------------------------------*/
/* Function Prototypes             */
/*---------------------------------*/


/*-------------------------------------------------------------------------*/
/* INIT_ATOM_PRED                                                          */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Init_Atom_Pred(void)

{
 atom_tbl=Hash_Table(MAX_ATOM_TBL_SIZE,sizeof(AtomInf),0);
 if (atom_tbl==NULL)
     Fatal_Error(ERR_ALLOC_TABLE);

 atom_nil           =Create_Atom("[]");
 atom_dot           =Create_Atom(".");
 atom_curly_brackets=Create_Atom("{}");
 atom_fail          =Create_Atom("fail");

 atom_compiler      =Create_Atom(COMPILER);
 atom_wam_version   =Create_Atom(WAM_VERSION);


 pred_tbl=Hash_Table(MAX_PRED_TBL_SIZE,sizeof(PredInf),sizeof(int));
 if (pred_tbl==NULL)
     Fatal_Error(ERR_ALLOC_TABLE);


 oper_tbl=Hash_Table(MAX_OPER_TBL_SIZE,sizeof(OperInf),sizeof(int));
 if (oper_tbl==NULL)
     Fatal_Error(ERR_ALLOC_TABLE);
}




/*-------------------------------------------------------------------------*/
/* CREATE_ALLOCATE_ATOM                                                    */
/*                                                                         */
/*-------------------------------------------------------------------------*/
AtomInf *Create_Allocate_Atom(char *name)

{
 char *name1;

 name1=(char *) Lib1(strdup,name);

 return Create_Atom(name1);
}




/*-------------------------------------------------------------------------*/
/* CREATE_ATOM                                                             */
/*                                                                         */
/*-------------------------------------------------------------------------*/
AtomInf *Create_Atom(char *name)

{
 AtomInf  atom_info;
 AtomInf *atom;
 char    *p;
 int      c_type;
 int      lg;
 Bool     indentifier;
 Bool     symbols;

 if (atom=(AtomInf *) Hash_Lookup(atom_tbl,(char *) (&name),H_FIND))
     return atom;                                        /* already exists */

 atom_info.name       =name;
 atom_info.has_quote  =FALSE;
 atom_info.needs_quote=FALSE;

 indentifier=(*name && char_type[*name]==SL);              /* small letter */
 symbols    =(*name);

 for(lg=0,p=name;*p;p++)
    {
     c_type=char_type[*p];

     if ((c_type & (UL | CL | SL | DI))==0)
         indentifier=FALSE;

     if (c_type!=SY)
         symbols=FALSE;

     if (c_type==QT)
         atom_info.has_quote=TRUE;
    }


 lg=p-name;

 if (indentifier)
     atom_info.type=IDENTIFIER_ATOM;
  else
     if (symbols)
         atom_info.type=SYMBOL_ATOM;
      else
         if (lg==1 && char_type[*name]==SC)
            {
             atom_info.type=SOLO_ATOM;
             atom_info.needs_quote= *name=='.';
            }
          else
            {
             atom_info.type=OTHER_ATOM;
             atom_info.needs_quote=! (lg==2 && 
                                     (name[0]=='[' &&  name[1]==']' ||
                                      name[0]=='{' &&  name[1]=='}')   );
            }

 atom_info.length=lg;
 atom_info.info  =0;

 atom=(AtomInf *) Hash_Lookup(atom_tbl,(char *) &atom_info,H_UPDATE);
 if ((int) atom == -1)
     Fatal_Error(ERR_ATOM_TBL_FULL);

 return atom;
}





/*-------------------------------------------------------------------------*/
/* LOOKUP_ATOM                                                             */
/*                                                                         */
/*-------------------------------------------------------------------------*/
AtomInf *Lookup_Atom(char *name)

{
 return (AtomInf *) Hash_Lookup(atom_tbl,(char *) (&name),H_FIND);
}




/*-------------------------------------------------------------------------*/
/* CREATE_PRED_TABLE                                                       */
/*                                                                         */
/*-------------------------------------------------------------------------*/
PredTbl Create_Pred_Table(int size)

{
 PredTbl tbl;

 if (size==0)
     return NULL;

 tbl=Hash_Table(size,sizeof(PredInf),sizeof(int));
 if (tbl==NULL)
     Fatal_Error(ERR_ALLOC_TABLE);

 return tbl;
}




/*-------------------------------------------------------------------------*/
/* CREATE_PRED                                                             */
/*                                                                         */
/*-------------------------------------------------------------------------*/
PredInf *Create_Pred(AtomInf *atom,int n,Bool public,int module_nb,
                     CodePtr codep)

{
 PredTbl  tbl=(public) ? pred_tbl : module_tbl[module_nb].pred_tbl;
 PredInf  pred_info;
 PredInf *pred;

 pred_info.f_n         =Make_Pred_Key(atom,n);
 pred_info.owner_mod_nb=module_nb;
 pred_info.public      =public;
 pred_info.codep       =codep;

 pred=(PredInf *) Hash_Lookup(tbl,(char *) &pred_info,H_UPDATE);
 if ((int) pred == -1)
     Fatal_Error(ERR_PRED_TBL_FULL);

 return pred;
}





/*-------------------------------------------------------------------------*/
/* LOOKUP_PRED                                                             */
/*                                                                         */
/*-------------------------------------------------------------------------*/
PredInf *Lookup_Pred(AtomInf *atom,int n,Bool public,int module_nb)

{
 PredTbl tbl=(public) ? pred_tbl : module_tbl[module_nb].pred_tbl;

 return (PredInf *) Hash_Fast_Find_Int(tbl,Make_Pred_Key(atom,n));
}




/*-------------------------------------------------------------------------*/
/* CREATE_OPER                                                             */
/*                                                                         */
/*-------------------------------------------------------------------------*/
OperInf *Create_Oper(AtomInf *atom,int type,int prec,int left,int right)

{
 OperInf  oper_info;
 OperInf *oper;

 oper_info.a_t  =Make_Oper_Key(atom,type);
 oper_info.prec =prec;
 oper_info.left =left;
 oper_info.right=right;

 oper=(OperInf *) Hash_Lookup(oper_tbl,(char *) &oper_info,H_UPDATE);
 if ((int) oper == -1)
     Fatal_Error(ERR_OPER_TBL_FULL);

 return oper;
}




/*-------------------------------------------------------------------------*/
/* LOOKUP_OPER                                                             */
/*                                                                         */
/*-------------------------------------------------------------------------*/
OperInf *Lookup_Oper(AtomInf *atom,int type)

{
 return (OperInf *) Hash_Fast_Find_Int(oper_tbl,Make_Oper_Key(atom,type));
}




/*-------------------------------------------------------------------------*/
/* DELETE_OPER                                                             */
/*                                                                         */
/*-------------------------------------------------------------------------*/
OperInf *Delete_Oper(AtomInf *atom,int type)

{
 int key=Make_Oper_Key(atom,type);

 return (OperInf *) Hash_Lookup(oper_tbl,(char *) &key,H_DELETE);
}




/*-------------------------------------------------------------------------*/
/* CREATE_SWT_TABLE                                                        */
/*                                                                         */
/*-------------------------------------------------------------------------*/
SwtTbl Create_Swt_Table(int size)

{
 SwtTbl t;

 if ((t=Hash_Table(size,sizeof(SwtInf),sizeof(int)))==NULL)
     Fatal_Error(ERR_ALLOC_TABLE);

 return t;
}



/*-------------------------------------------------------------------------*/
/* CREATE_SWT_ELEMENT                                                      */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Create_Swt_Element(SwtTbl t,int key,CodePtr codep)

{
 SwtInf swt_info;

 swt_info.key  =key;
 swt_info.codep=codep;

 Hash_Lookup(t,(char *) &swt_info,H_CREATE);
}
