/*
 * vector.c
 *
 * accessing and allocation of vectors
 */

# include	"kalypso.h"
# include	"mem.h"

lispval
iStore (v, i, value)
struct vector	*v;
int		i;
lispval		value;
{
	if (i < 0 || v->size <= i)
		return error ("store: bad index %v", intRet (i));
	v->contents[i] = value;
	return value;
}

lispval
Store (vector, index, value)
lispval	vector, index, value;
{
	struct vector	*v;
	int		i;

	if (!vectorp (vector))
		return error ("store: non-vector %v", vector);
	v = itemtovector (vector);
	if (!nump (index))
		return error ("store: non-numeric %v", index);
	i = itemtonum (index);
	return iStore (v, i, value);
}

lispval
iFetch (v, i)
struct vector	*v;
int		i;
{
	if (i < 0 || v->size <= i)
		return error ("fetch: bad index %v", intRet (i));
	return v->contents[i];
}

lispval
Fetch (vector, index)
lispval	vector, index;
{
	struct vector	*v;
	int		i;

	if (!vectorp (vector))
		return error ("fetch: non-vector %v", vector);
	v = itemtovector (vector);
	if (!nump (index))
		return error ("fetch: non-numeric %v", index);
	i = itemtonum (index);
	return iFetch (v, i);
}

/* this function is not needed anywhere yet, 
	iVectorSize (v)
	struct vector	*v;
	{
		return v->size;
	}
 */

lispval
VectorSize (vector)
lispval	vector;
{
	struct vector	*v;

	if (!vectorp (vector))
		return error ("vector-size: non-vector %v", vector);
	v = itemtovector (vector);
	return numtoitem (v->size);
}

struct dotted *
iVectorToList (v)
struct vector	*v;
{
	int		i;
	struct dotted	*first, *new, *last;
	int		framem;
	
	first = last = nil;
	framem = frameMark ();
	for (i = 0; i < v->size; i++) {
		new = newDotted ();
		new->car = v->contents[i];
		new->cdr = nil;
		if (first)
			last->cdr = new;
		else {
			first = new;
			framePush (listtoitem (first));
		}
		last = new;
	}
	frameReset (framem);
	return first;
}

lispval
VectorToList (vector)
lispval	vector;
{
	struct vector	*v;
	struct dotted	*l;

	if (!vectorp (vector))
		return error ("vector-to-list: non-vector %v", vector);
	v = itemtovector (vector);
	l = iVectorToList (v);
	if (!l)
		return nil;
	return listtoitem (l);
}

struct vector *
iListToVector (l)
register struct dotted	*l;
{
	register int		len;
	register struct vector	*v;
	register int		i;
	struct vector		*iNewVector ();
	int			framem;

	len = iLength (l);
	v = iNewVector (len);
	framem = frameMark ();
	framePush (vectortoitem (v));
	for (i = 0; i < len; i++) {
		v->contents[i] = l->car;
		l = l->cdr;
	}
	frameReset (framem);
	return v;
}

lispval
ListToVector (list)
lispval	list;
{
	struct dotted	*l;
	struct vector	*v;

	if (nilp (list))
		l = nil;
	else {
		if (!listp (list))
			return error ("list-to-vector: non-list %v", list);
		l = itemtolist (list);
	}
	v = iListToVector (l);
	return vectortoitem (v);
}

struct vector *
iNewVector (s)
int	s;
{
	struct vector	*v;
	int		i;
	int		m;

	m = frameMark ();
	v = (struct vector *) newObject (sizeof (struct vector));
	v->contents = 0;
	v->size = 0;
	framePush (vectortoitem (v));
	v->contents = (lispval *) newObject (s * sizeof (lispval));
	v->size = s;
	for (i = 0; i < s; i++)
		v->contents[i] = nil;
	frameReset (m);
	return v;
}

lispval
NewVector (size)
lispval	size;
{
	int		s;
	struct vector	*v;

	if (!nump (size))
		return error ("new-vector: non-numeric %v", size);
	s = itemtonum (size);
	v = iNewVector (s);
	return vectortoitem (v);
}


lispval
SameVector (v1, v2)
lispval	v1, v2;
{
	static char	msg[] = "same-vector: non vector %v";

	if (!vectorp (v1))
		return error (msg, v1);
	if (!vectorp (v2))
		return error (msg, v2);
	if (v1 != v2)
		return nil;
	return symboltoitem (true);
}

/*
 * anyone who wants to save un-referenced vectors better tell us about them
 */

extern struct vector	*signalHandlers;

struct vector	**referencedVectors[] = {
	&signalHandlers,
	0,
};

setVectorRef (v)
struct vector	*v;
{
	int	i;

	if (setObjectRef ((char *) v))
		return;
	if (v->contents)
		(void) setObjectRef ((char *) v->contents);
	for (i = 0; i < v->size; i++)
		setRef (v->contents[i]);
}

struct builtin vectorStuff[] = {
	"store",	Store,		LAMBDA,		3,
	"fetch",	Fetch,		LAMBDA,		2,
	"vector-size",	VectorSize,	LAMBDA,		1,
	"vector-to-list",VectorToList,	LAMBDA,		1,
	"list-to-vector",ListToVector,	LAMBDA,		1,
	"new-vector",	NewVector,	LAMBDA,		1,
	"same-vector",	SameVector,	LAMBDA,		2,
	0,		0,		0,		0,
};
