/*
 * system.c
 *
 * unix dependent primitives
 */

# include	"kalypso.h"

#ifdef SYSTEM_PACKAGE

# include	<sys/types.h>

#ifdef SYSV
# define waitCode(w)	(((w) >> 8) & 0xff)
# define waitSig(w)	((w) & 0xff)
typedef int		waitType;
#else
# include	<sys/wait.h>
# define waitCode(w)	((w).w_T.w_Retcode)
# define waitSig(w)	((w).w_T.w_Termsig)
typedef union wait	waitType;
#endif

lispval
lisp_time ()
{
	int	ret;
	long	time();

	ret = time((long *) 0);
	return numtoitem (ret);
}

#ifndef SYSV
# include	<sys/time.h>
#else
# include	<time.h>
#endif

lispval
lisp_setTime (time)
lispval	time;
{
	long	t;
	int	ret;

	if (nump (time))
		t = itemtonum (time);
	else if (floatpp (time))
		t = (long) *itemtofloatp (time);
	else
		return error ("set-time: non-numeric %v", time);
#ifdef SYSV
	ret = stime (&t);
#else
	{
		struct timeval	date;
		struct timezone	zone;
		gettimeofday (&date, &zone);
		date.tv_sec = t;
		ret = (settimeofday (&date, &zone));
	}
#endif
	if (ret == -1) {
		errorNumber->value = intRet (errno);
		return nil;
	}
	return intRet (ret);
}

lispval timeRet (t)
struct tm	*t;
{
	int		mark;
	lispval		ret;
	struct dotted	*retL;
	lispval		tm_sec, tm_min, tm_hour,
			tm_mday, tm_mon, tm_year,
			tm_wday, tm_yday, tm_isdst;

	ret = nil;
	if (t) {
		mark = frameMark ();
		tm_sec = intRet (t->tm_sec); framePush (tm_sec);
		tm_min = intRet (t->tm_min); framePush (tm_min);
		tm_hour = intRet (t->tm_hour); framePush (tm_hour);
		tm_mday = intRet (t->tm_mday); framePush (tm_mday);
		tm_mon = intRet (t->tm_mon); framePush (tm_mon);
		tm_year = intRet (t->tm_year); framePush (tm_year);
		tm_wday = intRet (t->tm_wday); framePush (tm_wday);
		tm_yday = intRet (t->tm_yday); framePush (tm_yday);
		tm_isdst = intRet (t->tm_isdst); framePush (tm_isdst);
		retL = makeList (9,tm_sec, tm_min, tm_hour,
				  tm_mday, tm_mon, tm_year,
				  tm_wday, tm_yday, tm_isdst);
		frameReset (mark);
		if (retL)
			ret = listtoitem (retL);
	}
	return ret;
}

lispval lisp_local_time (clock)
lispval	clock;
{
	long		c;
	struct tm	*t;

	if (nump (clock))
		c = itemtonum (clock);
	else if (floatpp (clock))
		c = (long) *itemtofloatp (clock);
	else
		return error ("local-time: non-numeric: %v", clock);
	t = localtime (&c);
	return timeRet (t);
}

lispval lisp_gm_time (clock)
lispval	clock;
{
	long		c;
	struct tm	*t;

	if (nump (clock))
		c = itemtonum (clock);
	else if (floatpp (clock))
		c = (long) *itemtofloatp (clock);
	else
		return error ("gm-time: non-numeric: %v", clock);
	t = gmtime (&c);
	return timeRet (t);
}

#ifdef SYSV
# include	<sys/times.h>

# define tms_to_double(tms)	((double) (tms) / 60.0)

#else
# include	<sys/resource.h>

# define tv_to_double(tv)	(((double) (tv).tv_sec) +\
				 (((double) (tv).tv_usec) / 1000000.0))
#endif

lispval
lisp_process_times (self)
lispval	self;
{
	double		utime, stime;
	lispval		uval, sval, ret;
	int		mark;
#ifdef SYSV
	struct tms	usage;

	(void) times (&usage);
	if (nilp (self)) {
		utime = tms_to_double (usage.tms_cutime);
		stime = tms_to_double (usage.tms_cstime);
	} else {
		utime = tms_to_double (usage.tms_utime);
		stime = tms_to_double (usage.tms_stime);
	}
#else
	struct rusage	usage;
	int		who;

	who = (nilp (self) ? RUSAGE_CHILDREN : RUSAGE_SELF);
	getrusage (who, &usage);
	utime = tv_to_double (usage.ru_utime);
	stime = tv_to_double (usage.ru_stime);
#endif
	mark = frameMark ();
	uval = makeFloat (utime);
	framePush (uval);
	sval = makeFloat (stime);
	framePush (sval);
	ret = listtoitem (makeList (2, uval, sval));
	frameReset (mark);
	return ret;
}
  
lispval
lisp_getuid ()
{
	int	ret;

	ret = getuid();
	return numtoitem (ret);
}

lispval
lisp_getgid ()
{
	int	ret;

	ret = getgid();
	return numtoitem (ret);
}

lispval
lisp_getpid ()
{
	int	ret;

	ret = getpid();
	return numtoitem (ret);
}

lispval
lisp_getenv (name)
lispval	name;
{
	char		*n;
	extern char	*getenv();
	character	*value;

	if (!stringp (name))
		return error ("getenv: non-string argument %v", name);
	n = iCstring (itemtostring (name), (int *) 0);
	n = getenv (n);
	if (!n)
		return nil;
	value = iKstring (n, 1, -1);
	if (!value)
		return nil;
	return stringtoitem (value);
}

lispval
lisp_unlink (name)
lispval	name;
{
	char	*f;
	extern int	errno;

	f = printName (name);
	if (!f)
		return error ("unlink: non-string name %v", name);
	if (unlink (f) == -1) {
		errorNumber->value = numtoitem (errno);
		return nil;
	}
	return symboltoitem (true);
}

lispval
lisp_unix_link (name1, name2)
lispval	name1, name2;
{
	char	*f1, *f2;
	extern int	errno;

	f1 = printName (name1);
	if (!f1)
		return error ("unix_link: non-string name %v", name1);
	f2 = printName (name2);
	if (!f2)
		return error ("unix_link: non-string name %v", name2);
	if (link (f1, f2) == -1) {
		errorNumber->value = numtoitem (errno);
		return nil;
	}
	return symboltoitem (true);
}

lispval
lisp_close (fd)
lispval	fd;
{
	int		f;
	extern int	errno;

	if (nump (fd))
		f = itemtonum (fd);
	else if (floatpp (fd))
		f = *itemtofloatp (fd);
	else
		return error ("close: non-numeric %v", fd);
	if (close (f) == -1) {
		errorNumber->value = numtoitem (errno);
		return nil;
	}
	return symboltoitem (true);
}

lispval
lisp_alarm (seconds)
lispval	seconds;
{
	int	s, old;

	if (nump (seconds))
		s = itemtonum (seconds);
	else if (floatpp (seconds))
		s = *itemtofloatp (seconds);
	else
		return error ("alarm: non-numeric %v", seconds);
	if ((old = alarm (s)) == -1) {
		errorNumber->value = intRet (errno);
		return nil;
	}
	return intRet (old);
}

lispval
lisp_fork ()
{
	int	pid;

	switch (pid = fork ()) {
	case -1:
		return nil;
	case 0:
		return symboltoitem (true);
	default:
		return intRet (pid);
	}
}

lispval
lisp_wait ()
{
	waitType	w;
	int		pid;

	pid = wait (&w);
	if (pid == -1)
		return nil;
	return listtoitem (makeList (3, intRet (pid),
					intRet (waitCode (w)),
					intRet (waitSig (w))));
}

lispval
lisp_exec (l, count)
lispval	*l;
int	count;
{
	char	*prog;
	char	**args;
	int	m, i;
	char	*malloc ();
	lispval	arg;

	if (!stringp (*l))
		return error ("exec: non-string program %v", *l);
	m = frameMark ();
	prog = iCstring (itemtostring (*l), (int *) 0);
	++l;
	--count;
	framePush (stringtoitem (prog));
	args = (char **) malloc ((count + 1) * sizeof (char *));
	i = 0;
	while (count--) {
		arg = *l++;
		if (nilp (arg))
			args[i++] = "";
		else if (stringp (arg)) {
			args[i] = iCstring (itemtostring (arg), (int *) 0);
			framePush (stringtoitem (args[i]));
			i++;
		} else {
			free (args);
			frameReset (m);
			return error ("exec: non-string argument %v", arg);
		}
	}
	args[i] = 0;
	execv (prog, args);
	frameReset (m);
	free (args);
	return nil;
}

lispval
lisp_pipe ()
{
	int	piped[2];
	FILE	*in, *out;
	int	m;
	lispval	ret;

	if (pipe (piped) == -1)
		return nil;
	m = frameMark ();
	in = fdopen (piped[0], "r");
	if (!in) {
		close (piped[0]);
		close (piped[1]);
		frameReset (m);
		return nil;
	}
	framePush (filetoitem (in));
	out = fdopen (piped[1], "w");
	if (!out) {
		fclose (in);
		close (piped[1]);
		frameReset (m);
		return nil;
	}
	ret = listtoitem (makeList (2, filetoitem (in),
				       filetoitem (out)));
	frameReset (m);
	return ret;
}

struct builtin systemStuff[] = {
	{ "time", lisp_time, LAMBDA, 0},
	{ "set-time",	lisp_setTime,	LAMBDA,	1 },
	{ "local-time", lisp_local_time, LAMBDA, 1 },
	{ "gm-time", lisp_gm_time, LAMBDA, 1 },
	{ "process-times", lisp_process_times, LAMBDA, 1},
	{ "getuid", lisp_getuid, LAMBDA, 0},
	{ "getgid", lisp_getgid, LAMBDA, 0},
	{ "getpid", lisp_getpid, LAMBDA, 0},
	{ "getenv", lisp_getenv, LAMBDA, 1},
	{ "unlink", lisp_unlink, LAMBDA, 1},
	{ "unix_link", lisp_unix_link, LAMBDA, 2 },
	{ "close", lisp_close, LAMBDA, 1},
	{ "alarm", lisp_alarm, LAMBDA, 1},
	{ "fork", lisp_fork, LAMBDA, 0 },
	{ "wait", lisp_wait, LAMBDA, 0 },
	{ "exec", lisp_exec, LEXPR, 1 },
	{ "pipe", lisp_pipe, LAMBDA, 0 },
	{ 0, 0, 0, 0 },
};
#endif
