/*
 * object.c
 *
 * basic object manipulation
 */

# include "kalypso.h"

lispval
Atom_p (l)
register lispval	l;
{
	if (!listp (l))
		return symboltoitem (true);
	else
		return nil;
}

lispval
Bound_p (l)
register lispval	l;
{
	if (!symbolp (l))
		return error ("bound?: non-symbol %v", l);
	if (!undefp (itemtosymbol(l)->value))	
		return symboltoitem (true);
	else
		return nil;
}

int
iHash (object)
register lispval	object;
{
	if (nilp(object))
		return 0;
	if (jumping)
		return nil;
	switch (TYPE(object)) {
	case NUMTYPE:
		return itemtonum (object);
	case SYMBOLTYPE:
		return iHash (itemtosymbol (object)->name);
	case LISTTYPE:
		return hashList (itemtolist (object));
	case STRINGTYPE:
		return hashString (itemtostring (object));
	case VECTORTYPE:
		return hashVector (itemtovector (object));
	case BUILTINTYPE:
		return (int) itemtobuiltin (object);
	case FILETYPE:
		return (int) itemtofile (object);
	case DICTTYPE:
		return (int) itemtodict (object);
	case FLOATPTYPE:
		return (int) * itemtofloatp (object);
	default:
		return (int) object;
	}
}

lispval
Hash (l)
lispval	l;
{
	register int	ret;

	ret = iHash (l);
	return numtoitem (ret);
}

int
hashList (l)
register struct dotted	*l;
{
	register int	i;

	i = 0;
	while (l) {
		i += iHash (l->car);
		l = l->cdr;
	}
	return i;
}

int
hashString (n)
register character	*n;
{
	register int	h;

	h = 0;
	while (*n) {
		h = (h << 1) ^ *n;
		n = iScdr (n);
	}
	return h;
}

int
hashVector (v)
register struct vector	*v;
{
	return (int) v;
}

struct equalstack {
	lispval	a,b;
};

static struct equalstack	*equalstack, *equalstackp;
static int			equalstacksize;

lispval
Equal_p (a, b)
lispval	a, b;
{
	if (a == b)
		return symboltoitem (true);
	if (jumping)
		return nil;
	if (TYPEM(a) != TYPEM(b))
		return nil;
	if (listp (a)) {
		register struct dotted	*al, *bl;

		al = itemtolist (a);
		bl = itemtolist (b);
		while (al && bl && Equal_p (al->car, bl->car) != nil) {
			al = al->cdr;
			bl = bl->cdr;
		}
		if (!al && !bl)
			return symboltoitem (true);
	} else if (stringp (a)) {
		register character	*as, *bs;

		as = itemtostring (a);
		bs = itemtostring (b);
		while (*as && *bs && iScar (as) == iScar (bs)) {
			as = iScdr (as);
			bs = iScdr (bs);
		}
		if (!*as && !*bs)
			return symboltoitem (true);
	} else if (floatpp (a)) {
		if (*itemtofloatp(a) == *itemtofloatp(b))
			return symboltoitem (true);
	} else if (vectorp (a)) {
		register int		i, size;
		register struct vector	*v1, *v2;

		v1 = itemtovector (a);
		v2 = itemtovector (b);
		if (v1->size == v2->size) {
			switch (checkEstack (a, b)) {
			case 1:
				return symboltoitem (true);
			case 0:
				size = v1->size;
				i = 0;
				while (Equal_p (v1->contents[i], v2->contents[i])) {
					++i;
					if (i == size) {
						popEstack ();
						return symboltoitem (true);
					}
				}
				popEstack ();
			case -1:
				break;
			}
		}
	} else if (dictp (a)) {
		register struct dictionary	*d1, *d2;
		register int			i;
		register struct hashchain	*c1, *c2;
		extern struct hashchain		*iDictionaryBucket();

		switch (checkEstack (a, b)) {
		case 1:
			return symboltoitem (true);
		case 0:
			d1 = itemtodict (a);
			d2 = itemtodict (b);
			for (i = 0; i < d1->hashsize; i++) {
				for (c1 = d1->hashtable[i]; c1; c1 = c1->next) {
					c2 = iDictionaryBucket (d2, c1);
					if (!c2 ||
					    !Equal_p (c1->value, c2->value)) {
						popEstack ();
						return nil;
					}
				}
			}
			for (i = 0; i < d2->hashsize; i++) {
				for (c2 = d2->hashtable[i]; c2; c2 = c2->next) {
					c1 = iDictionaryBucket (d1, c2);
					/*
					 * don't need to test for equality
					 * as the loop above already checked
					 * all common elements
					 */
					if (!c1) {
						popEstack ();
						return nil;
					}
				}
			}
			return symboltoitem (true);
		case -1:
			break;
		}
	}
	return nil;
}

# define INITSIZE	128
# define INCRSIZE	128

checkEstack (l1, l2)
register lispval	l1, l2;
{
	register struct equalstack	*e;
	char				*malloc (), *realloc ();

	if (!equalstack) {
		equalstacksize = INITSIZE;
		equalstack = (struct equalstack *) 
			malloc (equalstacksize * sizeof (*equalstack));
		if (!equalstack)
			panic (0, "out of memory - quiting\n");
		equalstackp = equalstack;
	}
	if (equalstackp == equalstack + equalstacksize) {
		equalstack = (struct equalstack *) realloc (equalstack,
 			equalstacksize + INCRSIZE * sizeof (*equalstack));
		if (!equalstack)
			panic (0, "out of memory - quiting\n");
		equalstackp = equalstack + equalstacksize;
		equalstacksize += INCRSIZE;
	}
	for (e = equalstackp-1; e >= equalstack; e--) {
		/*
		 * identical recursion, data structures are
		 * topologically similar
		 */
		if (e->a == l1 && e->b == l2)
			return 1;
		/*
		 * non identical recursion, data structures are
		 * topologically different
		 */
		if (e->a == l1 || e->b == l2)
			return -1;
	}
	/*
	 * no detected recursion, data structures are
	 * unknown
	 */
	equalstackp->a = l1;
	equalstackp->b = l2;
	++equalstackp;
	return 0;
}

popEstack ()
{
	if (equalstackp == equalstack)
		abort ();
	--equalstackp;
}

lispval
Float_p (l)
lispval l;
{
	if (floatpp (l))
		return symboltoitem (true);
	return nil;
}

lispval
Integer_p (l)
lispval	l;
{
	if (nump (l)) 
		return symboltoitem (true);
	return nil;
}

lispval
Number_p (l)
lispval	l;
{
	if (nump (l) || floatpp (l)) 
		return symboltoitem (true);
	return nil;
}

lispval
List_p (l)
register lispval	l;
{
	if (listp (l) || nilp (l))
		return symboltoitem (true);
	else
		return nil;
}
		
lispval
Member_p (a, l)
register lispval	a;
lispval			l;
{
	register struct dotted	*d;
	if (nilp (l))
		return nil;
	if (!listp (l))
		return error ("member: non-list %v", l);
	for (d = itemtolist (l); d; d = d->cdr) {
		if (Equal_p (a, d->car))
			return listtoitem (d);
		if (jumping)
			break;
	}
	return nil;
}
		
lispval
Nil_p (l)
lispval	l;
{
	if (nilp (l))
		return symboltoitem (true);
	else
		return nil;
}

lispval
String_p (l)
lispval l;
{
	if (stringp (l))
		return symboltoitem (true);
	else
		return nil;
}

lispval
Symbol_p (l)
lispval	l;
{
	if (symbolp (l))
		return symboltoitem (true);
	else
		return nil;
}

lispval
Builtin_p (l)
lispval l;
{
	if (builtinp (l))
		return symboltoitem (true);
	else
		return nil;
}

lispval
Vector_p (l)
lispval	l;
{
	if (vectorp (l))
		return symboltoitem (true);
	else
		return nil;
}

lispval
Dictionary_p (l)
lispval	l;
{
	if (dictp (l))
		return symboltoitem (true);
	else
		return nil;
}

lispval
/*ARGSUSED*/
Quote (l, count)
register lispval	*l;
int			count;
{
	if (!*l)
		return nil;
	return itemtolist (*l)->car;
}

struct builtin objectStuff[] = {
	"atom?",	Atom_p,		LAMBDA,		1,
	"bound?",	Bound_p,	LAMBDA,		1,
	"hash",		Hash,		LAMBDA,		1,
	"equal?",	Equal_p,	LAMBDA,		2,
	"float?",	Float_p,	LAMBDA,		1,
	"integer?",	Integer_p,	LAMBDA,		1,
	"list?",	List_p,		LAMBDA,		1,
	"member?",	Member_p,	LAMBDA,		2,
	"nil?",		Nil_p,		LAMBDA,		1,
	"number?",	Number_p,	LAMBDA,		1,
	"string?",	String_p,	LAMBDA,		1,
	"symbol?",	Symbol_p,	LAMBDA,		1,
	"builtin?",	Builtin_p,	LAMBDA,		1,
	"vector?",	Vector_p,	LAMBDA,		1,
	"dictionary?",	Dictionary_p,	LAMBDA,		1,
	"quote",	Quote,		NLAMBDA,	0,
	0,		0,		0,		0,
};
