/*
 * number.c
 *
 * primitives for all numbers
 */

# include "kalypso.h"

double
iTrunc (dv)
register double dv;
{
	return dv >= 0.0 ? floor (dv) : ceil (dv);
}

lispval
Plus (l, count)
register lispval	*l;
{
	register lispval	i;
	register int		v;
	register double		dv = 0;
	int			isdouble = 0;

	v = 0;
	while (count--) {
		i = *l++;
		if (nump (i)) {
			if (isdouble) {
				dv += itemtonum (i);
			} else {
				v += itemtonum (i);
				if (v > MAXINT || v < MININT) {
					dv = v;
					isdouble = 1;
				}
			}
		} else if (floatpp (i)) {
			if (!isdouble) {
				dv = v;
				isdouble = 1;
			}
			dv += *itemtofloatp (i);
		} else
			return error ("plus: non-numeric %v", i);
	}
	if (isdouble)
		return doubleRet (dv);
	return numtoitem (v);
}

/*
 * this can be made to avoid floating divide, and probably should
 */

lispval
Div (dividend, divisor)
register lispval	dividend, divisor;
{
	double	dv;
	static char	err[] = "div: non-numeric %v";
	double	dq;
	
	if (nump (dividend))
		dv = itemtonum (dividend);
	else if (floatpp (dividend))
		dv = *itemtofloatp (dividend);
	else
		return error (err, dividend);
	if (nump (divisor))
		dq = itemtonum (divisor);
	else if (floatpp (divisor))
		dq = *itemtofloatp (divisor);
	else
		return error (err, divisor);
	if (dq == 0.0)
		return error ("div: by zero");
	dv = iTrunc (dv / dq);
	return doubleRet (dv);
}

/*
 * this can be made to avoid floating divide, and probably should
 */

lispval
Over (dividend, divisor)
register lispval	dividend, divisor;
{
	double	dv;
	static char	err[] = "over: non-numeric %v";
	double	dq;
	
	if (nump (dividend))
		dv = itemtonum (dividend);
	else if (floatpp (dividend))
		dv = *itemtofloatp (dividend);
	else
		return error (err, dividend);
	if (nump (divisor))
		dq = itemtonum (divisor);
	else if (floatpp (divisor))
		dq = *itemtofloatp (divisor);
	else
		return error (err, divisor);
	if (dq == 0.0)
		return error ("over: by zero");
	dv = dv / dq;
	return doubleRet (dv);
}

lispval
Lt_p (l, count)
lispval	*l;
{
	double	v, v0;
	lispval	i;
	static char	err[] = "lt?: non-numeric %v";
	
	i = *l++;
	--count;
	if (nump (i))
		v = (double) itemtonum (i);
	else if (floatpp (i))
		v = *itemtofloatp (i);
	else
		return error (err, i);
	while (count--) {
		i = *l++;
		if (nump (i))
			v0 = (double) itemtonum (i);
		else if (floatpp (i))
			v0 = *itemtofloatp (i);
		else
			return error (err, i);
		if (! (v < v0))
			return nil;
		v = v0;
	}
	return symboltoitem (true);
}

lispval
Gt_p (l, count)
lispval	*l;
{
	double	v, v0;
	lispval	i;
	static char	err[] = "gt?: non-numeric %v";
	
	i = *l++;
	--count;
	if (nump (i))
		v = (double) itemtonum (i);
	else if (floatpp (i))
		v = *itemtofloatp (i);
	else
		return error (err, i);
	while (count--) {
		i = *l++;
		if (nump (i))
			v0 = (double) itemtonum (i);
		else if (floatpp (i))
			v0 = *itemtofloatp (i);
		else
			return error (err, i);
		if (! (v > v0))
			return nil;
		v = v0;
	}
	return symboltoitem (true);
}

lispval
Mod (s, count)
lispval	*s;
{
	static char	err[] = "mod: non-integer %v";
	int	v;
	lispval	i;
	int	q;
	int	sign;
	
	i = *s++;
	--count;
	if (nump (i))
		v = itemtonum (i);
	else
		return error (err, i);
	while (count--) {
		i = *s++;
		if (nump (i)) {
			q = itemtonum (i);
			if (!q)
				return error ("mod: modulus by zero");
			if (q < 0)
				q = -q;
			if (v < 0) {
				sign = -1;
				v = -v;
			} else {
				sign = 1;
			}
			/*
			 * % is only defined for positive numbers
			 * in C
			 */
			v %= q;
			/*
			 * adjust for opposite signs
			 */
			if (sign == -1 && v > 0)
				v = q - v;
 		} else
			return error (err, i);
	}
	return numtoitem (v);
}

lispval
Rem (dividend, divisor)
lispval	dividend, divisor;
{
	double	dv, dq, r;
	static char	err[] = "rem: non-numeric %v";

	if (nump (dividend))
		dv = itemtonum (dividend);
	else if (floatpp (dividend))
		dv = *itemtofloatp (dividend);
	else
		return error (err, dividend);
	if (nump (divisor))
		dq = itemtonum (divisor);
	else if (floatpp (divisor))
		dq = *itemtofloatp (divisor);
	else
		return error (err, divisor);
	if (dq == 0)
		return error ("rem: zero divisor");
	r = iTrunc (dv / dq);
	r = dv - r * dq;
	return doubleRet (r);
}

lispval
Times (l, count)
lispval	*l;
{
	register lispval	i;
	register double		dv;

	dv = 1;
	while (count--) {
		i = *l++;
		if (nump (i))
			dv *= itemtonum (i);
		else if (floatpp (i))
			dv *= *itemtofloatp (i);
		else
			return error ("multiply: non-numeric %v", i);
	}
	return doubleRet (dv);
}

lispval
Minus (s, count)
lispval	*s;
{
	static char	err[] = "minus: non numeric %v";
	int	v;
	double	dv;
	int	isdouble;
	lispval	i;
	double	dq;
	int	q;

	i = *s++;
	--count;
	if (nump (i)) {
		v = itemtonum (i);
		isdouble = 0;
	} else if (floatpp (i)) {
		dv = *itemtofloatp (i);
		isdouble = 1;
	} else
		return error (err, i);
	while (count--) {
		i = *s++;
		if (nump (i)) {
			q = itemtonum (i);
			if (isdouble)
				dv -= q;
			else {
				v -= q;
				if (v < MININT || MAXINT < v) {
					dv = v;
					isdouble = 1;
				}
			}
		} else if (floatpp (i)) {
			dq = *itemtofloatp (i);
			if (!isdouble) {
				dv = v;
				isdouble = 1;
			}
			dv -= dq;
 		} else
			return error (err, i);
	}
	if (isdouble)
		return doubleRet (dv);
	return numtoitem (v);
}

lispval
Floor (l)
lispval l;
{
	double	dv;

	if (nump (l))
		return l;
	else if (floatpp (l)) {
		dv = *itemtofloatp (l);
		dv = floor (dv);
		return doubleRet (dv);
	} else
		return error ("floor: non-numeric %v", l);
}

lispval
Ceiling (l)
lispval l;
{
	double	dv;

	if (nump (l))
		return l;
	else if (floatpp (l)) {
		dv = *itemtofloatp (l);
		dv = ceil (dv);
		return doubleRet (dv);
	} else
		return error ("ceiling: non-numeric %v", l);
}

lispval
Trunc (l)
lispval l;
{
	double	dv;

	if (nump (l))
		return l;
	else if (floatpp (l)) {
		dv = *itemtofloatp (l);
		dv = dv >= 0 ? floor(dv) : ceil (dv);
		return doubleRet (dv);
	} else
		return error ("trunc: non-numeric %v", l);
}

lispval
Round (l)
lispval l;
{
	double	dv;
	if (nump (l))
		return l;
	else if (floatpp (l)) {
		dv = *itemtofloatp (l);
		dv = dv >= 0 ? floor (dv + 0.5) : ceil (dv - 0.5);
		return doubleRet (dv);
	} else
		return error ("round: non-numeric %v", l);
}

struct builtin numberStuff[] = {
	"plus",		Plus,		LEXPR,		0,
	"div",		Div,		LAMBDA,		2,
	"over",		Over,		LAMBDA,		2,
	"lt?",		Lt_p,		LEXPR,		1,
	"gt?",		Gt_p,		LEXPR,		1,
	"mod",		Mod,		LEXPR,		2,
	"rem",		Rem,		LAMBDA,		2,
	"times",	Times,		LEXPR,		0,
	"minus",	Minus,		LEXPR,		2,
	"floor",	Floor,		LAMBDA,		1,
	"ceiling",	Ceiling,	LAMBDA,		1,
	"trunc",	Trunc,		LAMBDA,		1,
	"round",	Round,		LAMBDA,		1,
	0,		0,		0,		0,
};
