/*
 * network.c
 *
 * just enough support to get sockets connected to kalypso
 */

# include	"kalypso.h"
#ifdef NETWORK_PACKAGE
# include	<sys/types.h>
# include	<sys/socket.h>
#define HAS_INET_DOMAIN
#define HAS_UNIX_DOMAIN
#ifdef hpux
# undef HAS_UNIX_DOMAIN
#endif
#ifdef HAS_UNIX_DOMAIN
# include	<sys/un.h>
#endif
#ifdef HAS_INET_DOMAIN
# include	<netinet/in.h>
#endif
# include	<netdb.h>
#ifdef SYSV
# include	<time.h>
#else
# include	<sys/time.h>
#endif

extern int	errno;
extern int	sys_nerr;
extern char	*sys_errlist[];
extern char	*strcpy (), *strcat ();

union addr_union {
	struct sockaddr_in	in_addr;
#ifdef HAS_UNIX_DOMAIN
	struct sockaddr_un	un_addr;
#endif
	struct sockaddr		addr;
};

/*
 * these routines convert between standard lisp addresses and unix addresses
 */

static struct sockaddr *
itemtoaddress (client, address, lenp, defaultHost)
char	*client;
lispval	address;
int	*lenp;
long	defaultHost;
{
#ifdef HAS_INET_DOMAIN
	static struct sockaddr_in	in_addr;
#endif
#ifdef HAS_UNIX_DOMAIN
	static struct sockaddr_un	un_addr;
#endif
	struct sockaddr			*addr;
	lispval				port, host;
	struct dotted			*full_inet;
	char				errorMsg[128];
	struct hostent			*hostent;

#ifdef HAS_INET_DOMAIN
	if (nump (address)) {
		in_addr.sin_family = AF_INET;
		in_addr.sin_port = htons ((short) itemtonum (address));
		in_addr.sin_addr.s_addr = htonl (defaultHost);
		addr = (struct sockaddr *) &in_addr;
		*lenp = sizeof (in_addr);
		return addr;
	}
 	if (floatpp (address)) {
		in_addr.sin_family = AF_INET;
		in_addr.sin_port = htons ((short) *itemtofloatp (address));
		in_addr.sin_addr.s_addr = htonl (defaultHost);
		addr = (struct sockaddr *) &in_addr;
		*lenp = sizeof (in_addr);
		return addr;
	}
 	if (listp (address)) {
		in_addr.sin_family = AF_INET;
		full_inet = itemtolist (address);
		port = full_inet->car;
		if (!full_inet->cdr) {
			(void) strcpy (errorMsg, client);
			(void) strcat (errorMsg, ": invalid argument %v");
			(void) error (errorMsg, address);
			return 0;
		}
		host = full_inet->cdr->car;
		if (nump (port))
			in_addr.sin_port = htons ((short) itemtonum (port));
		else if (floatpp (port))
			in_addr.sin_port = htons ((short) *itemtofloatp (port));
		else {
			(void) strcpy (errorMsg, client);
			(void) strcat (errorMsg, ": non-numeric port %v");
			(void) error (errorMsg, port);
			return 0;
		}
		if (nump (host))
			in_addr.sin_addr.s_addr = htonl (itemtonum (host));
		else if (floatpp (host))
			in_addr.sin_addr.s_addr = htonl (*itemtofloatp (host));
		else if (stringp (host)) {
			hostent = gethostbyname (iCstring (itemtostring (host), (int *) 0));
			if (!hostent) {
				(void) strcpy (errorMsg, client);
				(void) strcat (errorMsg, ": unknown host %v");
				(void) error (errorMsg, host);
				return 0;
			}
			if (hostent->h_addrtype != AF_INET) {
				(void) strcpy (errorMsg, client);
				(void) strcat (errorMsg, ": unexpected address type %v");
				(void) error (errorMsg,
					intRet (hostent->h_addrtype));
				return 0;
			}
			in_addr.sin_addr.s_addr =
				*((long *) hostent->h_addr);
		} else {
			(void) strcpy (errorMsg, client);
			(void) strcat (errorMsg, ": invalid host %v");
			(void) error (errorMsg, host);
			return 0;
		}
		addr = (struct sockaddr *) &in_addr;
		*lenp = sizeof (in_addr);
		return addr;
	}
#endif HAS_INET_DOMAIN
#ifdef HAS_UNIX_DOMAIN
	if (stringp (address)) {
		un_addr.sun_family = AF_UNIX;
		strcpy (un_addr.sun_path, iCstring (itemtostring (address), (int *) 0));
		addr = (struct sockaddr *) &un_addr;
		*lenp = sizeof (un_addr.sun_family) + 
			strlen (un_addr.sun_path) + 1;
		return addr;
	}
#endif HAS_UNIX_DOMAIN
	strcpy (errorMsg, client);
	strcat (errorMsg, ": non socket-address %v");
	(void) error (errorMsg, address);
	return 0;
}

lispval
addresstoitem (addr)
struct sockaddr	*addr;
{
	int	mark;
	struct sockaddr_in	*in_addr;
#ifdef HAS_UNIX_DOMAIN
	struct sockaddr_un	*un_addr;
#endif
	struct dotted	*dotted;
	lispval		ret, num1, num2;
	character	*str;
	int		i;

	mark = frameMark ();
	switch (addr->sa_family) {
#ifdef HAS_UNIX_DOMAIN
	case AF_UNIX:
		un_addr = (struct sockaddr_un *) addr;
		str = iKstring (un_addr->sun_path, 1, -1);
		if (str)
			ret = stringtoitem (str);
		else
			ret = nil;
		break;
#endif
#ifdef HAS_INET_DOMAIN
	case AF_INET:
		in_addr = (struct sockaddr_in *) addr;
		i = (int) ntohs (in_addr->sin_port);
		framePush (num1 = intRet (i));
		i = ntohl (in_addr->sin_addr.s_addr);
		framePush (num2 = intRet (i));
		dotted = makeList (2, num1, num2);
		if (dotted)
			ret = listtoitem (dotted);
		else
			ret = nil;
		break;
#endif
	default:
		ret = nil;
	}
	frameReset (mark);
	return ret;
}

lispval
iSocket (af, type, protocol)
int	af, type, protocol;
{
	int	s;
	int	optval;
	FILE	*f;

	switch (af) {
	default:
		return error ("socket: invalid address family %v", itemtonum (af));
	case AF_UNIX:
	case AF_INET:
	case AF_PUP:
	case AF_IMPLINK:
		break;
	}
	switch (type) {
	default:
		return error ("socket: invalid socket type %v", itemtonum (type));
	case SOCK_STREAM:
	case SOCK_DGRAM:
	case SOCK_RAW:
	case SOCK_SEQPACKET:
	case SOCK_RDM:
		break;
	}
	s = socket (af, type, protocol);
	if (s == -1) {
		errorNumber->value = numtoitem (errno);
		return intRet (-1);
	}
	optval = 1;
	setsockopt (s, SOL_SOCKET, SO_REUSEADDR, &optval, sizeof (optval));
	f = fdopen (s, "r+");
	if (!f)
		return nil;
	return filetoitem (f);
}

lispval
Socket (af, type, protocol)
lispval	af, type, protocol;
{
	int	a, t, p;

	if (nump (af))
		a = itemtonum (af);
	else if (floatpp (af))
		a = *itemtofloatp (af);
	else
		return error ("socket: non-numeric %v", af);
	if (nump (type))
		t = itemtonum (type);
	else if (floatpp (type))
		t = *itemtofloatp (type);
	else
		return error ("socket: non-numeric %v", type);
	if (nump (protocol))
		p = itemtonum (protocol);
	else if (floatpp (protocol))
		p = *itemtofloatp (protocol);
	else
		return error ("socket: non-numeric %v", protocol);
	return iSocket (a, t, p);
}

lispval
iBind (f, name, namelen)
FILE		*f;
struct sockaddr	*name;
int		namelen;
{
	int	ret;

	ret = bind (fileno (f), name, namelen);
	if (ret == -1) {
		errorNumber->value = numtoitem (errno);
		return nil;
	}
	return symboltoitem (true);
}

lispval
Bind (socket, name)
lispval	socket, name;
{
	struct sockaddr		*addr;
	int			addrlen;
	FILE			*f;

	if (!filep (socket))
		return error ("bind: non-file %v", socket);
	f = itemtofile (socket);
	if (!(addr = itemtoaddress ("bind", name, &addrlen, INADDR_ANY)))
		return error ("bind: non socket-address %v", name);
	return iBind (f, addr, addrlen);
}

lispval
iListen (f, backlog)
FILE	*f;
int	backlog;
{
	int	ret;

	if (backlog < 0 || 5 < backlog)
		return error ("listen: invalid backlog %v", itemtonum (backlog));
	ret = listen (fileno (f), backlog);
	if (ret == -1) {
		errorNumber->value = numtoitem (errno);
		return nil;
	}
	return symboltoitem (true);
}

lispval
Listen (socket, backlog)
lispval	socket, backlog;
{
	FILE	*f;
	int	b;

	if (!filep (socket))
		return error ("listen: non-file %v", socket);
	f = itemtofile (socket);
	if (nump (backlog))
		b = itemtonum (backlog);
	else if (floatpp (backlog))
		b = *itemtofloatp (backlog);
	else
		return error ("listen: non-numeric %v", backlog);
	return iListen (f, b);
}

lispval
iAccept (f)
FILE		*f;
{
	int			s;
	FILE			*new;
	int			mark;
	struct dotted		*dotted;
	lispval			ret, address;
	int			addrlen;
	union addr_union	addr;

	++canDoAsyncNow;
	checkAsync ();
	addrlen = sizeof (addr);
	while ((s = accept (fileno (f), &addr.addr, &addrlen)) == -1) {
		if (errno != EINTR)
			break;
		if (jumping) {
			--canDoAsyncNow;
			return nil;
		}
	}
	--canDoAsyncNow;
	if (s == -1) {
		errorNumber->value = numtoitem (errno);
		return nil;
	}
	new = fdopen (s, "r+");
	if (!new) {
		close (s);
		return nil;
	}
	mark = frameMark ();
	framePush (filetoitem (new));
	address = addresstoitem (&addr.addr);
	framePush (address);
	dotted = makeList (2, filetoitem (new), address);
	if (dotted)
		ret = listtoitem (dotted);
	else
		ret = nil;
	frameReset (mark);
	return ret;
}

lispval
Accept (socket)
lispval	socket;
{
	FILE			*original;

	if (!filep (socket))
		return error ("accept: non-file %v", socket);
	original = itemtofile (socket);
	return iAccept (original);
}

iConnect (f, name, namelen)
FILE	*f;
struct sockaddr	*name;
int	namelen;
{
	int	ret;

	++canDoAsyncNow;
	checkAsync ();
	while ((ret = connect (fileno (f), name, namelen)) == -1) {
		if (errno != EINTR)
			break;
		if (jumping) {
			--canDoAsyncNow;
			return -1;
		}
	}
	--canDoAsyncNow;
	if (ret == -1) {
		errorNumber->value = numtoitem (errno);
		return -1;
	}
	return 0;
}

#ifndef INADDR_LOOPBACK
#define INADDR_LOOPBACK	0x7f000001
#endif

lispval
Connect (socket, address)
lispval	socket;
lispval	address;
{
	FILE			*f;
	int			addrlen;
	struct sockaddr		*addr;

	if (!filep (socket))
		return error ("connect: non-file %v", socket);
	f = itemtofile (socket);
	if (!(addr = itemtoaddress ("connect", address, &addrlen, INADDR_LOOPBACK)))
		return nil;
	if (iConnect (f, addr, addrlen) == -1)
		return nil;
	return symboltoitem (true);
}

iSend (file, data, len)
FILE		*file;
char		*data;
{
	int	ret;

	++canDoAsyncNow;
	checkAsync ();
	while ((ret = send (fileno (file), data, len, 0)) == -1) {
		if (errno != EINTR)
			break;
		if (jumping) {
			--canDoAsyncNow;
			return -1;
		}
	}
	--canDoAsyncNow;
	if (ret == -1) {
		errorNumber->value = numtoitem (errno);
	}
	return ret;
}

lispval
Send (socket, data)
lispval	socket;
lispval	data;
{
	FILE		*f;
	char		*d;
	int		ret;
	int		len;

	if (!filep (socket))
		return error ("send: non-file %v", socket);
	f = itemtofile (socket);
	if (!stringp (data))
		return error ("send: non-string: %v", data);
	d = iCstring (itemtostring (data), &len);
	ret = iSend (f, d, len);
	if (ret == -1)
		return nil;
	return intRet (ret);
}

iReceive (f, data, size)
FILE		*f;
char		*data;
int		size;
{
	int	ret;

	++canDoAsyncNow;
	checkAsync ();
	while ((ret = recv (fileno (f), data, size, 0)) == -1) {
		if (errno != EINTR)
			break;
		if (jumping) {
			--canDoAsyncNow;
			return -1;
		}
	}
	if (ret == -1)
		errorNumber->value = numtoitem (errno);
	--canDoAsyncNow;
	return ret;
}

lispval
Receive (socket)
lispval	socket;
{
	char			d[8192];
	int			i;
	FILE			*f;
	character		*str;

	if (!filep (socket))
		return error ("receive: non-file %v", socket);
	f = itemtofile (socket);
	i = iReceive (f, d, sizeof (d)-1);
	if (i == -1)
		return nil;
	str = iKstring (d, 1, i);
	if (!str)
		return nil;
	return stringtoitem (str);
}

iSendTo (file, addr, addrlen, data, len)
FILE		*file;
struct sockaddr	*addr;
int		addrlen;
char		*data;
int		len;
{
	int	ret;

	++canDoAsyncNow;
	checkAsync ();
	while ((ret = sendto (fileno (file), data, len, 0, addr, addrlen)) == -1) {
		if (errno != EINTR)
			break;
		if (jumping) {
			--canDoAsyncNow;
			return -1;
		}
	}
	if (ret == -1) {
		errorNumber->value = numtoitem (errno);
	}
	--canDoAsyncNow;
	return ret;
}

lispval
SendTo (socket, data, address)
lispval	socket;
lispval	data;
lispval	address;
{
	FILE		*f;
	struct sockaddr	*addr;
	int		addrlen;
	char		*d;
	int		ret;
	int		len;

	if (!filep (socket))
		return error ("send-to: non-file %v", socket);
	f = itemtofile (socket);
	if (!stringp (data))
		return error ("send-to: non-string: %v", data);
	d = iCstring (itemtostring (data), &len);
	if (!(addr = itemtoaddress ("send-to", address, &addrlen, INADDR_LOOPBACK)))
		return nil;
	ret = (iSendTo (f, addr, addrlen, d, len));
	if (ret == -1)
		return nil;
	return intRet (ret);
}

iReceiveFrom (f, data, size, addr, addrlen)
FILE		*f;
char		*data;
int		size;
struct sockaddr	*addr;
int		*addrlen;
{
	int	ret;

	++canDoAsyncNow;
	checkAsync ();
	while ((ret = recvfrom (fileno (f), data, size, 0, addr, addrlen)) == -1) {
		if (errno != EINTR)
			break;
		if (jumping) {
			--canDoAsyncNow;
			return -1;
		}
	}
	if (ret == -1)
		errorNumber->value = numtoitem (errno);
	--canDoAsyncNow;
	return ret;
}

lispval
ReceiveFrom (socket)
lispval	socket;
{
	union addr_union	addr;
	int			addrlen;
	char			d[8192];
	lispval			data, address, ret;
	int			mark;
	int			i;
	FILE			*f;
	character		*str;

	if (!filep (socket))
		return error ("receive-from: non-file %v", socket);
	f = itemtofile (socket);
	addrlen = sizeof (addr);
	i = iReceiveFrom (f, d, sizeof (d)-1,
				(struct sockaddr *) &addr, &addrlen);
	if (i == -1)
		return nil;
	str = iKstring (d, 1, i);
	if (str)
		data = stringtoitem (str);
	else
		data = nil;
	mark = frameMark ();
	framePush (data);
	address = addresstoitem (&addr.addr);
	framePush (address);
	ret = listtoitem (makeList (2, data, address));
	frameReset (mark);
	return ret;
}

lispval
GetHostname ()
{
	char		d[8192];
	character	*name;

	gethostname (d, sizeof (d));
	d[sizeof (d) - 1] = '\0';
	name = iKstring (d, 1, -1);
	if (!name)
		return nil;
	return stringtoitem (name);
}

#ifndef FD_ZERO
/* typedef	struct	fd_set { int fds_bits[1]; } fd_set; */
# define FD_ZERO(fdp)	bzero ((fdp), sizeof (*(fdp)))
# define FD_SET(f,fdp)	((fdp)->fds_bits[(f) / (sizeof (int) * 8)] |= (1 << ((f) % (sizeof (int) * 8))))
# define FD_ISSET(f,fdp)	((fdp)->fds_bits[(f) / (sizeof (int) * 8)] & (1 << ((f) % (sizeof (int) * 8))))

#endif

# define max(a,b)	((a)>(b)?(a):(b))

static int
fileNo (object)
lispval	object;
{
	if (filep (object))
		return fileno (itemtofile (object));
	else
		return -1;
}

static int
isready (f)
FILE	*f;
{
	return f->_cnt > 0;
}

static lispval	inError;

static int
fillFd (descriptors, mask, maxfd, readyMask, nreadyp)
lispval	descriptors;
fd_set	*mask;
int	maxfd;
fd_set	*readyMask;
int	*nreadyp;
{
	struct dotted	*d;
	int		i;

	if (!listp (descriptors)) {
		inError = descriptors;
		return -1;
	}
	for (d = itemtolist (descriptors); d; d = d->cdr) {
		i = fileNo (d->car);
		if (i == -1) {
			inError = d->car;
			return -1;
		}
		if (filep (d->car) && isready (itemtofile (d->car))) {
			FD_SET (i, readyMask);
			++ (*nreadyp);
		} else {
			if (i > maxfd)
				maxfd = i;
			FD_SET (i, mask);
		}
	}
	return maxfd;
}

static
struct dotted *
pickSet (descriptors, maskA, maskB, numsetp)
lispval	descriptors;
fd_set	*maskA, *maskB;
int	*numsetp;
{
	struct dotted	*first, *last, *new, *d;
	int		mark;
	int		i;

	if (!listp (descriptors))
		return 0;
	mark = frameMark ();
	first = last = 0;
	for (d = itemtolist (descriptors); d; d = d->cdr) {
		i = fileNo (d->car);
		if (i == -1)
			first = 0;
		if (FD_ISSET (i, maskA) || FD_ISSET (i, maskB)) {
			new = newDotted ();
			new->car = d->car;
			new->cdr = 0;
			if (last)
				last->cdr = new;
			else
				framePush (listtoitem (first = new));
			last = new;
			-- (*numsetp);
		}
	}
	frameReset (mark);
	return first;
}

lispval
Select (reads, writes, timeout)
lispval	reads, writes, timeout;
{
	fd_set	readMask, readReadyMask;
	fd_set	writeMask, writeReadyMask;
	struct timeval	timeo, *tm;
	int	maxfd = -1;
	int	ret, mark;
	int	nreadReady, nwriteReady;
	struct dotted	*retval;

	FD_ZERO (&readMask);
	FD_ZERO (&writeMask);
	FD_ZERO (&readReadyMask);
	FD_ZERO (&writeReadyMask);
	nreadReady = 0;
	nwriteReady = 0;
	if (nilp (reads))
		;
	else if (listp (reads)) {
		maxfd = fillFd (reads, &readMask,
 			maxfd, &readReadyMask, &nreadReady);
		if (maxfd == -1)
			return error ("select: invalid argument %v", inError);
	} else {
		return error ("select: invalid argument %v", reads);
	}
	if (nilp (writes))
		;
	else if (listp (writes)) {
		maxfd = fillFd (writes, &writeMask,
 			maxfd, &writeReadyMask, &nwriteReady);
		if (maxfd == -1)
			return error ("select: invalid argument %v", inError);
	} else
		return error ("select: invalid argument %v", writes);
	if (nilp (timeout))
		tm = 0;
	else if (nump (timeout)) {
		tm = &timeo;
		timeo.tv_sec = itemtonum (timeout);
	} else if (floatpp (timeout)) {
		tm = &timeo;
		timeo.tv_sec = (int) *itemtofloatp (timeout);
		timeo.tv_usec = 1000000 *
			(*itemtofloatp (timeout) -
 				floor (*itemtofloatp (timeout)));
	} else 
		return error ("select: invalid argument %v", timeout);
	if (nreadReady || nwriteReady) {
		tm = &timeo;
		timeo.tv_sec = timeo.tv_usec = 0;
	}
	if (maxfd == -1)
		maxfd = 0;
	ret = select (maxfd+1, &readMask, &writeMask, (fd_set *) 0, tm);
	if (ret == -1) {
		errorNumber->value = numtoitem (errno);
		return nil;
	}
	ret = ret + nreadReady + nwriteReady;
	if (ret == 0) {
		errorNumber->value = nil;
		return nil;
	}
	mark = frameMark ();
	framePush (reads = listtoitem
 		(pickSet (reads, &readMask, &readReadyMask, &ret)));
	if (jumping) {
		frameReset (mark);
		return nil;
	}
	framePush (writes = listtoitem
 		(pickSet (writes, &writeMask, &writeReadyMask, &ret)));
	if (jumping) {
		frameReset (mark);
		return nil;
	}
	retval = makeList (2, reads, writes);
	frameReset (mark);
	return listtoitem (retval);
}

struct builtin networkStuff[] = {
	"socket",	Socket,		LAMBDA,	3,
	"bind", 	Bind,		LAMBDA,	2,
	"listen",	Listen,		LAMBDA,	2,
 	"accept",	Accept,		LAMBDA,	1,
	"connect",	Connect,	LAMBDA, 2,
	"select",	Select,		LAMBDA,	3,
	"send",		Send,		LAMBDA,	2,
	"receive",	Receive,	LAMBDA,	1,
	"send-to",	SendTo,		LAMBDA,	3,
	"receive-from",	ReceiveFrom,	LAMBDA,	1,
	"get-hostname",	GetHostname,	LAMBDA,	0,
	0,		0,		0,	0,
};
#endif
