/*
 * mem.c
 *
 * generic memory allocation routines
 *
 * interface:
 *
 * char *newObject(size)- returns a hunk of at least "size" bytes
 * char *newHunk(index)	- returns a hunk from a particular bucket
 * setObjectRef(h)	- tests and sets the reference bit for a hunk of data
 * setRef(lispval)	- calls the appropriate routine for a lispval
 * iGarbageCollect()	- find unreferenced objects for reuse
 *
 * setRef calls:
 *	setListRef	- for lists
 *	setSymbolRef	- for symbols
 *	setStringRef	- for strings
 *	setDictRef	- for dictionaries
 *	setFileRef	- for files
 *	setVectorRef	- for vectors
 *	setFloatRef	- for floating point numbers
 *	setFileRef	- for files
 *	setCstringRef	- for C strings
 *
 * each of these types are expected to call setObjectRef for themselves
 * and setRef for each object they reference.  Adding new types will almost
 * certainly entail new set.*Ref functions
 *
 * local routines:
 *
 * sizeToIndex(size)	- returns the bucket number for a particular size
 * gimmeBlock()		- calls iGarbageCollect (if it is time) or malloc
 * allocBlock()		- calls malloc
 * disposeBlock()	- calls free
 *
 * noteBlock(b)		- permits address->block translation for these objects
 * unNoteBlock(b)	- removes block from data structure
 * addrToBlock(a,bp,ip)	- translates address (a) to block (*bp) and index (*ip)
 * blockWalk(f)		- calls f with each active block
 *
 * clearRef()		- clears all ref bits in the system
 * clearBlockRef(b)	- clears all ref bits in a particular block
 * tossFree()		- erases the free lists
 * checkRef()		- puts unreferenced hunks on free lists
 * checkBlockRef(b)	- puts unreferenced hunks in a block on free lists
 */

# include	"kalypso.h"
# include	"mem.h"
# include	"avl.h"

extern char	*allocBlock (), *gimmeBlock (), *allocHuge(), *gimmeHuge ();

extern char	*malloc ();

# define MINBLOCKSIZE	(MAXHUNK + MINHUNK + sizeof (struct block))
# define GOODBLOCKSIZE	(0x7fc)
# define BLOCKSIZE	(GOODBLOCKSIZE < MINBLOCKSIZE ? \
				MINBLOCKSIZE : GOODBLOCKSIZE)

# define GARBAGETIME	1000

int		GarbageTime = GARBAGETIME;

#ifndef NDEBUG
extern int	GCdebug;
#endif

# define BITSPERCH		(8)
# define NUMINBLOCK(size) 	(((BLOCKSIZE - sizeof (struct block)) * \
					BITSPERCH) / (1 + BITSPERCH * (size)))
# define BITMAPSIZE(size)	((NUMINBLOCK(size) + (BITSPERCH-1)) / BITSPERCH)

static int sizeMap[NUMSIZES] = {
	MINHUNK,
 	MINHUNK*2,
 	MINHUNK*4,
 	MINHUNK*8,
 	MINHUNK*16,
 	MINHUNK*32,
 	MINHUNK*64,
 	MINHUNK*128,
 	MAXHUNK,
 };

struct bfree *freeList[NUMSIZES];

int	sinceGarbage = 0;

char *
newHunk (sizeIndex)
register sizeIndex;
{
	register char	*new;
	char		*limit;
	struct block	*b;
	int		bsize;

	if (!freeList[sizeIndex]) {
		bsize = sizeMap[sizeIndex];
		b = (struct block *) gimmeBlock ();
		if (!b) {
			if (freeList[sizeIndex])
				goto gotsome;
			b = (struct block *) allocBlock ();
			if (!b)
				panic (0, "out of memory - quiting\n");
		}
		/*
		 * fill in per-block data fields
		 */
		b->sizeIndex = sizeIndex;
		b->bitmap = ((char *) b) + sizeof (struct block);
		b->bitmapsize = BITMAPSIZE(bsize);
		b->data = (char *)
 			((((int) (b->bitmap + b->bitmapsize)) + (MINHUNK-1))
	 			& (~(MINHUNK-1)));
		b->datasize = NUMINBLOCK(bsize) * bsize;
		/*
		 * put this block into the address converter
		 */
		noteBlock (b);
		/*
		 * put it's contents on the free list
		 */
		limit = b->data + (b->datasize - bsize);
		for (new = b->data; new < limit; new += bsize) {
			((struct bfree *) new)->next =
 				(struct bfree *) (new + bsize);
		}
		((struct bfree *) new)->next = freeList[sizeIndex];
		freeList[sizeIndex] = (struct bfree *) b->data;
	}
gotsome:
	new = (char *) freeList[sizeIndex];
	freeList[sizeIndex] = freeList[sizeIndex]->next;
#ifndef NDEBUG
	bzero (new, sizeMap[sizeIndex]);
#endif
	if (((int) new) & 1)
		abort ();
	return new;
}

sizeToIndex (size)
int	size;
{
	int	sizeIndex;
	
	for (sizeIndex = 0; sizeIndex < NUMSIZES; sizeIndex++)
		if (sizeMap[sizeIndex] >= size)
			return sizeIndex;
	return -1;
}

/*
 * take care of giant requests
 */
char *
newHuge (size)
{
	struct block	*huge;

	huge = (struct block *) gimmeHuge
				(sizeof (struct block) + size);
	if (!huge) {
		huge = (struct block *) allocHuge
					(sizeof (struct block) + size);
		if (!huge)
			panic (0, "out of memory - quiting\n");
	}
	huge->sizeIndex = NUMSIZES;
	huge->bitmapsize = 0;
	huge->datasize = size;
	huge->bitmap = 0;
	huge->data = ((char *) huge) + sizeof (*huge);
	noteBlock (huge);
	if (((int) huge) & 01)
		abort ();
	return huge->data;
}

char *
newObject (size)
{
	int	sizeIndex;

	sizeIndex = sizeToIndex (size);
	if (sizeIndex == -1)
		return newHuge (size);
	return newHunk (sizeIndex);
}

char *
newObjectWithSize (size, sizep)
int	size, *sizep;
{
	int	sizeIndex;

	sizeIndex = sizeToIndex (size);
	if (sizeIndex == -1) {
		*sizep = size;
		return newHuge (size);
	}
	*sizep = sizeMap[sizeIndex];
	return newHunk (sizeIndex);
}

char *
gimmeBlock ()
{
	char	*result;

	if (++sinceGarbage >= GarbageTime) {
		iGarbageCollect ();
		sinceGarbage = 0;
		return nil;
	}
	result = allocBlock ();
	if (!result) {
		iGarbageCollect ();
		sinceGarbage = 0;
	}
	return result;
}

char *
allocBlock ()
{
	char	*malloc ();

	return malloc ((unsigned) BLOCKSIZE);
}

disposeBlock (b)
struct block	*b;
{
	free ((char *) b);
}

char *
gimmeHuge (size)
int	size;
{
	char	*result;

	if (++sinceGarbage >= GarbageTime) {
		iGarbageCollect ();
		sinceGarbage = 0;
		return nil;
	}
	result = allocHuge (size);
	if (!result) {
		iGarbageCollect ();
		sinceGarbage = 0;
	}
	return result;
}

char *
allocHuge (size)
int	size;
{
	return malloc ((unsigned) size);
}

/*
 * address->(block,index) translation scheme.  three routines:
 *
 * noteBlock	- install a new block into the database
 * addrToBlock	- translate an address into (block, index)
 * blockWalk	- call a routine with every referenced block
 */

static struct block	*root;

noteBlock (b)
struct block	*b;
{
	(void) tree_insert (&root, b);
#ifndef NDEBUG
	if (GCdebug)
		verifyBlock();
#endif
}

unNoteBlock (b)
struct block	*b;
{
	(void) tree_delete (&root, b);
#ifndef NDEBUG
	if (GCdebug)
		verifyBlock();
#endif
}

#ifndef NDEBUG
verifyBlock ()
{
	if (!tree_verify (root))
		abort ();
}
#endif

#ifdef NOTUSED
addrToBlock (address, blockp, indexp)
register char	*address;
struct block	**blockp;
int		*indexp;
{
	register struct block	*b;
	register int		dist;

	for (b = root; b;) {
		if ((dist = address - b->data) < 0)
			b = b->left;
		else if (dist < b->datasize)
			b = b->right;
		else {
			*blockp = b;
			if (b->sizeIndex < NUMSIZES)
				*indexp = (dist) / sizeMap[b->sizeIndex];
			return 1;
		}
	}
	return 0;
}
#endif

blockTreeWalk (treep, function)
struct block	*treep;
int		(*function)();
{
	if (treep) {
		blockTreeWalk (treep->left, function);
		function (treep);
		blockTreeWalk (treep->right, function);
	}
}

blockWalk (function)
int	(*function)();
{
	blockTreeWalk (root, function);
}

/*
 * garbage collection scheme
 */

/*
 * note that setObjectRef has had addrToBlock inlined for speed
 */

setObjectRef (address)
register char	*address;
{
	register struct block	*b;
	register int		dist;
	register int		byte;
	register int		bit;
	register		index;
	int			old;

	/*
	 * a very simple cache -- this results
	 * in about 16% speed increase in this
	 * routine best case.  Worse case it
	 * results in about a 5% decrease from
	 * wasted computation
	 */
	static struct block	*cache;
	
	if ((b = cache) &&
 	    (dist = ((int) address) - ((int) b->data)) > 0 &&
 	    dist < b->datasize)
		goto cache_hit;

	for (b = root; b;) {
		if ((dist = ((int) address) - ((int) b->data)) < 0)
			b = b->left;
		else if (dist >= b->datasize)
			b = b->right;
		else {
			cache = b;
cache_hit:		;
			if (address = b->bitmap) {
				index = dist / sizeMap[b->sizeIndex];
				byte = index >> 3;
				bit = (1 << (index & 7));
				old = ((unsigned char *) address)[byte] & bit;
				((unsigned char *) address)[byte] |= bit;
			} else
				old = b->ref;
			b->ref = 1;
			return old;
		}
	}
	return 1;
}

/*
 * clearRef: zero's the reference bit for all objects
 */

clearBlockRef (b)
struct block	*b;
{
	int	i;

	if (b->bitmap)
		for (i = 0; i < b->bitmapsize; i++)
			b->bitmap[i] = 0;
	b->ref = 0;
}

clearRef ()
{
	blockWalk (clearBlockRef);
}

/*
 * eliminate any remaining free list
 */

tossFree ()
{
	int	i;

	for (i = 0; i < NUMSIZES; i++)
		freeList[i] = 0;
}

/*
 * checkRef: rebuild the free lists from unused data
 */

struct bfree	*lastFree[NUMSIZES];

struct block	*hughFree;

int	totalBytesFree;
int	totalBytesUsed;

checkBlockRef (b)
struct block	*b;
{
	int			sizeIndex;
	int			size;
	register char		*byte;
	register int		bit;
	register char		*object, *max;
	register struct bfree	*thisLast;

	if (!b->ref) {
#ifndef NDEBUG
		if (GCdebug > 1)
			debug ("unreferenced block at 0x%x\n", b);
#endif
		b->bitmap = (char *) hughFree;
		hughFree = b;
	} else if (b->bitmap) {
		sizeIndex = b->sizeIndex;
		thisLast = lastFree[sizeIndex];
		max = b->data + b->datasize;
		size = sizeMap[sizeIndex];
		byte = b->bitmap;
		bit = 1;
		for (object = b->data; object < max; object += size) {
			if (!(*byte & bit)) {
#ifndef NDEBUG
				if (GCdebug > 2)
					debugUnRefed (object);
#endif
				if (thisLast) {
					thisLast->next = (struct bfree *)
								object;
				} else {
					freeList[sizeIndex] = (struct bfree *)
								object;
				}
				thisLast = (struct bfree *) object;
				totalBytesFree += size;
			} else
				totalBytesUsed += size;
			bit <<= 1;
			if (bit == (1 << BITSPERCH)) {
				bit = 1;
				byte++;
			}
		}
		if (thisLast)
			thisLast->next = nil;
		lastFree[sizeIndex] = thisLast;
	} else
		totalBytesUsed += b->datasize;
}

#ifndef NDEBUG
debugUnRefed (object)
char	*object;
{
	int	first, second;

	first = ((int *) object)[0];
	second = ((int *) object)[1];
	if (TYPEM (first) == 0 && TYPEM (second) != 0)
		debug ("unrefed list %s\n",
			sprint (listtoitem (object), (int *) 0));
}
#endif

checkRef ()
{
	int	i;
	struct block	*n;

	totalBytesFree = 0;
	totalBytesUsed = 0;
	checkFileRef ();
	for (i = 0; i < NUMSIZES; i++)
		lastFree[i] = 0;
	hughFree = 0;
	blockWalk (checkBlockRef);
	while (hughFree) {
		n = (struct block *) hughFree->bitmap;
		unNoteBlock (hughFree);
		disposeBlock (hughFree);
		hughFree = n;
	}
	GarbageTime = 1000 - (totalBytesFree / GOODBLOCKSIZE);
	if (GarbageTime < 10)
		GarbageTime = 10;
#ifndef NDEBUG
	if (GCdebug) {
		debug ("GC: total bytes used: %d\n", totalBytesUsed);
		debug ("GC: total bytes free: %d\n", totalBytesFree);
		debug ("GC: GarbageTime set to %d\n", GarbageTime);
	}
#endif
}

lispval
iSetGarbageTime (frequency)
int	frequency;
{
	if (frequency < 0)
		return error ("set-garbage-time: bad frequency %v",
				intRet (frequency));
	GarbageTime = frequency;
	return intRet (frequency);
}

lispval
SetGarbageTime (frequency)
lispval frequency;
{
	int	f;

	if (nump (frequency))
		f = itemtonum (frequency);
	else if (floatpp (frequency))
		f = (int) *itemtofloatp (frequency);
	else
		return error ("set-garbage-time: non-numeric frequency %v",
			frequency);
	return iSetGarbageTime (f);
}

iGarbageCollect ()
{
	struct symbol		***refedA;
	struct dotted		***refedD;
	struct vector		***refedV;
	extern struct symbol	**referencedSymbols[];
	extern struct dotted	**referencedDotted[];
	extern struct vector	**referencedVectors[];

#ifndef NDEBUG
	if (GCdebug)
		debug ("GC:\n");
#endif
	clearRef ();
	tossFree ();
#ifndef NDEBUG
	if (GCdebug)
		debug ("GC: referenced symbols\n");
#endif
	for (refedA = referencedSymbols; *refedA; ++refedA) {
		if (**refedA) {
#ifndef NDEBUG
			if (GCdebug > 2) {
				debug ("GC: tied-symbol: %s\n",
 				     sprint (symboltoitem (**refedA), (int *) 0));
			}
#endif
			setSymbolRef (**refedA);
		}
	}
#ifndef NDEBUG
	if (GCdebug)
		debug ("GC: referenced lists\n");
#endif
	for (refedD = referencedDotted; *refedD; ++refedD) {
		if (**refedD) {
#ifndef NDEBUG
			if (GCdebug > 2) {
				debug ("GC: tied-list: %s\n",
 					sprint (listtoitem (**refedD), (int *) 0));
			}
#endif
			setListRef (**refedD);
		}
	}
#ifndef NDEBUG
	if (GCdebug)
		debug ("GC: referenced vectors\n");
#endif
	for (refedV = referencedVectors; *refedV; ++refedV) {
		if (**refedV) {
#ifndef NDEBUG
			if (GCdebug > 2) {
				debug ("GC: tied-vector: %s\n",
 					sprint (vectortoitem (**refedV), (int *) 0));
			}
#endif
			setVectorRef (**refedV);
		}
	}
#ifndef NDEBUG
	if (GCdebug)
		debug ("GC: markFrame\n");
#endif
	markFrame ();
	setRef (jumpValue);
#ifndef NDEBUG
	if (GCdebug)
		debug ("GC: checkRef\n");
#endif
	checkRef ();
#ifndef NDEBUG
	if (GCdebug > 2) {
		debug ("GC: debugFreeLists\n");
		debugFreeLists ();
	}
	if (GCdebug) {
		debug ("GC: verifyBlock\n");
		verifyBlock ();
	}
#endif
}

#ifndef NDEBUG
debugFreeLists ()
{
	int	i;
	struct bfree	*b;

	for (i = 0; i < NUMSIZES; i++) {
		debug ("%2d ", i);
		for (b = freeList[i]; b; b = b->next)
			debug ("(0x%x 0x%x) ", b, b->next);
		debug ("\n");
	}
}
#endif

lispval
GarbageCollect ()
{
	iGarbageCollect ();
	return symboltoitem (true);
}

setRef (l)
lispval	l;
{
#ifndef NDEBUG
	if (GCdebug > 2)
		debug ("GC: setRef %s\n", sprint (l), (int *) 0);
#endif
	switch (TYPE(l)) {
	case LISTTYPE:
		setListRef (itemtolist (l));
		break;
	case SYMBOLTYPE:
		setSymbolRef (itemtosymbol(l));
		break;
	case STRINGTYPE:
		setStringRef (itemtostring (l));
		break;
	case FILETYPE:
		setFileRef (itemtofile(l));
		break;
	case DICTTYPE:
		setDictRef (itemtodict(l));
		break;
	case VECTORTYPE:
		setVectorRef (itemtovector(l));
		break;
	case FLOATPTYPE:
		setFloatRef (itemtofloatp(l));
		break;
	case CSTRINGTYPE:
		setCstringRef (itemtocstring(l));
		break;
	}
}

struct builtin memStuff[] = {
	"set-garbage-time",	SetGarbageTime,	LAMBDA,		1,
	"garbage-collect",	GarbageCollect,	LAMBDA,		0,
	0,			0,		0,		0,
};
