/*
 * list.c
 *
 * list manipulation and allocation
 */

# include	"kalypso.h"
# include	"mem.h"

# include	<varargs.h>

lispval
Car (l)
register lispval l;
{

	if (!listp (l))
		return error ("car of non-list %v", l);
	return (itemtolist (l))->car;
}

lispval
Cdr (l)
register lispval	l;
{
	register struct dotted	*cdr;

	if (nilp (l))
		return nil;
	if (!listp (l))
		return error ("cdr of non-list %v", l);
	cdr = itemtolist(l)->cdr;
	if (!cdr)
		return nil;
	return listtoitem (cdr);
}

lispval
Conc (l, count)
register lispval	*l;
int			count;
{
	register struct dotted	*d, *new, *first, *last;
	int			framem;

	first = last = nil;
	framem = frameMark ();
	while (count--) {
		if (nilp (*l)) {
			++l;
			continue;
		}
		if (!listp (*l)) {
			frameReset (framem);
			return error ("conc: non-list argument %v", *l);
		}
		d = itemtolist (*l);
		l++;
		/*
		 * if there are more lists to conc,
		 * new cons cells are needed, else
		 * just tack the current list into
		 * the last
		 */
		if (count >= 1) {
			while (d) {
				new = newDotted ();
				if (!first) {
					first = new;
					framePush (listtoitem (first));
				} else {
					last->cdr = new;
				}
				new->car = d->car;
				new->cdr = nil;
				last = new;
				d = d->cdr;
			}
		} else {
			if (!first)
				first = d;
			else
				last->cdr = d;
		}
	}
	frameReset (framem);
	if (first) {
		return listtoitem (first);
	} else
		return nil;
}

lispval
Cons (car, cdr)
register lispval	car, cdr;
{
	struct dotted	*n;
	struct dotted	*c;

	if (nilp (cdr))
		c = nil;
	else if (!listp (cdr))
		return error ("badly formed cons");
	else
		c = itemtolist (cdr);
	n = newDotted ();
	n->car = car;
	n->cdr = c;
	return listtoitem (n);
}

int
iLength (l)
register struct dotted	*l;
{
	register int	i;

	i = 0;
	while (l) {
		i++;
		l = l->cdr;
	}
	return i;
}

lispval
Length (list)
lispval		list;
{
	register struct dotted	*l;
	register int		c;

	if (nilp (list))
		return numtoitem (0);
	if (!listp (list))
		return error ("length: non-list %v", list);
	l = itemtolist (list);
	c = 0;
	while (l) {
		++c;
		l = l->cdr;
	}
	return numtoitem (c);
}

lispval
Nthcdr (count, list)
lispval	count, list;
{
	register int	c;
	register struct dotted	*l;

	if (!nump (count))
		return error ("nthcdr: non-numeric %v", count);
	if (!listp (list))
		return error ("nthcdr: non-list %v", list);
	c = itemtonum (count);
	l = itemtolist (list);
	if (c < 0)
		return error ("nthcdr: negative argument %v", numtoitem (c));
	while (c--) {
		if (l)
			l = l->cdr;
	}
	if (nilp (l))
		return nil;
	return listtoitem (l);
}

lispval
Reverse (l)
lispval	l;
{
	register struct dotted	*list, *new, *d;
	register lispval	*fp;

	if (nilp (l))
		return nil;
	if (!listp (l))
		return error ("reverse: non-list argument %v", l);
	d = itemtolist (l);
	list = nil;
	checkFrame (nil);
	fp = framep;
	*fp = nil;
	++framep;
	while (d) {
		new = newDotted ();
		*fp = listtoitem (new);
		new->cdr = list;
		new->car = d->car;
		list = new;
		d = d->cdr;
	}
	--framep;
	if (list)
		return listtoitem (list);
	return nil;
}

#ifdef DEBUG
static int	compares;
#endif

/*
 * destructive merge function.  Given two lists in sorted order,
 * mash cdr pointers to create one list in sorted order.
 */

static struct dotted *
merge (lista, listb, func, args)
struct dotted	*lista, *listb;
lispval		func;
struct dotted	*args;		/* must be temporary storage */
{
	lispval		compare;
	struct dotted	*result, *last, *new;

	result = nil;
#ifdef SABER
	last = nil;
#endif
	while (lista && listb) {
		args->car = lista->car;
		args->cdr->car = listb->car;
#ifdef DEBUG
		++compares;
#endif
		compare = Apply (func, listtoitem (args));
		if (jumping)
			return nil;
		if (nilp (compare)) {
			new = listb;
			listb = listb->cdr;
		} else {
			new = lista;
			lista = lista->cdr;
		}
		new->cdr = nil;
		if (result)
			last->cdr = new;
		else
			result = new;
		last = new;
	}
	new = nil;
	/* glue the tail of either list onto the result */
	if (lista) {
		if (result)
			last->cdr = lista;
		else
			result = lista;
	} else if (listb) {
		if (result)
			last->cdr = listb;
		else
			result = listb;
	}
	return result;
}

static struct dotted *
mergesort (list, func, args)
struct dotted	*list;
lispval		func;
struct dotted	*args;
{
	struct dotted	*first, *second;
	int		len, firstLen;
	int		framem;

	/*
	 * split the list into two pieces
	 */
	len = iLength (list);
	if (len <= 1)
		return list;
	firstLen = len/2;
	first = list;
	while (--firstLen)
		list = list->cdr;
	framem = frameMark ();
	second = list->cdr;
	framePush (listtoitem (second));
	list->cdr = nil;
	/*
	 * sort the first half
	 */
	if (first->cdr) {
		first = mergesort (first, func, args);
		if (jumping) {
			frameReset (framem);
			return nil;
		}
		framePush (listtoitem (first));
	}
	/*
	 * sort the second half
	 */
	if (second->cdr) {
		second = mergesort (second, func, args);
		if (jumping) {
			frameReset (framem);
			return nil;
		}
		framePush (listtoitem (second));
	}
	/*
	 * merge the halves together
	 */
	list = merge (first, second, func, args);
	if (jumping) {
		frameReset (framem);
		return nil;
	}
	frameReset (framem);
	return list;
}

lispval
Sort (toSort, func)
lispval	toSort;
lispval	func;
{
	struct dotted	*element;
	int		framem;
	struct dotted	*args, *result;
	struct dotted	*new, *first, *last;

	/*
	 * sort of a trivial case
	 */
	if (nilp (toSort))
		return nil;
	if (!listp (toSort))
		return error ("sort: non-list %v", toSort);
#ifdef DEBUG
	compares = 0;
#endif
	/*
	 * mergesort and merge both destroy lists so a copy is made. This will
	 * cause extra storage to be allocated when the tail of the list is
	 * already in sorted order.  Recognising that case is costly in terms
	 * of comparisons, so the extra allocation is cheaper.
	 */
	element = itemtolist (toSort);
	framem = frameMark ();
	first = nil;
#ifdef SABER
	last = nil;
#endif
	for (; element; element=element->cdr) {
		new = newDotted ();
		if (first)
			last->cdr = new;
		else {
			first = new;
			framePush (listtoitem (new));
		}
		new->car = element->car;
		new->cdr = nil;
		last = new;
	}
	/*
	 * build a two element list that will
	 * be used repeatedly to pass arguments to
	 * Apply
	 */
	args = newDotted ();
	args->car = nil;
	args->cdr = nil;
	framePush (listtoitem (args));
	args->cdr = newDotted ();
	args->cdr->cdr = nil;
	args->cdr->car = nil;
	/* sort */
	result = mergesort (first, func, args);
	frameReset (framem);
#ifdef DEBUG
	printf ("%d\n", compares);
#endif
	if (!result)
		return nil;
	return listtoitem (result);
}

struct dotted *
/*VARARGS1*/
makeList (num, va_alist)
int	num;
va_dcl
{
	va_list	args;
	struct dotted	*first, *last, *new;
	int	mark;
	lispval	v;
	
	va_start (args);
	mark = frameMark ();
	first = last = 0;
	while (num-- > 0) {
		v = va_arg (args, lispval);
		new = newDotted ();
		if (last)
			last->cdr = new;
		else {
			first = new;
			framePush (listtoitem (first));
		}
		last = new;
		new->car = v;
		new->cdr = 0;
	}
	va_end (args);
	frameReset (mark);
	return first;
}

/*
 * anyone who wants to save un-referenced lists better tell us about them
 */

struct dotted	**referencedDotted[] = {
	0,
};

setListRef (l)
struct dotted	*l;
{
	register struct dotted	*list;

	for (list = l; list; list = list->cdr) {
		if (setObjectRef ((char *) list))
			break;
		setRef (list->car);
	}
}

struct builtin listStuff[] = {
	{ "car",		Car,		LAMBDA,	1 },
	{ "cdr",		Cdr,		LAMBDA,	1 },
	{ "conc",		Conc,		LEXPR,	0 },
	{ "cons",		Cons,		LAMBDA,	2 },
	{ "length",		Length,		LAMBDA,	1 },
	{ "nthcdr",		Nthcdr,		LAMBDA,	2 },
	{ "reverse",		Reverse,	LAMBDA,	1 },
	{ "sort",		Sort,		LAMBDA,	2 },
	{ 0,		0,		0,	0 },
};
