/*
 * kalypso.h
 */

# include	<stdio.h>
# include	<errno.h>
# include	"options.h"

extern int	errno;

typedef unsigned	lispval;

struct dotted {
	struct dotted	*cdr;
	lispval		car;
};

struct builtin {
	char		*name;
	lispval		(*function)();
	int		type;
	int		argc;
};

# define UNKNOWN	-1
# define LAMBDA		0
# define NLAMBDA	1
# define LEXPR		2
# define MACRO		3

struct symbol {
	lispval		value;
	lispval		name;
};

# define MAXDICTSIZE	4096
# define MINDICTSIZE	2

struct hashchain {
	lispval			name;
	lispval			value;
	int			hash;		/* full 32 bit hash */
	struct hashchain	*next;
};

struct dictionary {
	struct hashchain	**hashtable;
	int			hashsize;
	int			count;
};

struct vector {
	int		size;		/* number of lispvals */
	lispval		*contents;	/* array of 'size' lispvals */
};

/*
 * generic type mask - useful for most everything except numbers
 */

# define TYPEMASK	(0xf8000000)
# define TYPEM(l)	((l) & TYPEMASK)
# define TYPE(l)	((unsigned) (nump(l) ? NUMTYPE : ((l) & TYPEMASK)))

/*
 * numbers are special, they can invade the
 * type mask as they use only the top bit for recognition
 */

# define NUMTYPE	(0x80000000)	/* 31 bit int */
# define nump(l)	(((l) & NUMTYPE) != 0)
# define numtoitem(a)	((lispval) ((a) | NUMTYPE))
# define itemtonum(a)	((int) (((a) & 0x40000000) ? (a) : ((a) & ~NUMTYPE)))
# define numtonum(a)	(((a) & 0x40000000) ? (a) : ((a) & ~NUMTYPE))

# define nil		0
# define nilp(l)	((l) == nil)

#ifndef itop
#define itop(x)		(x)
#endif

#ifndef ptoi
#define ptoi(x)		(x)
#endif

/*
 * lispval types
 */

# define LISTTYPE	(0x70000000)	/* list */
# define SYMBOLTYPE	(0x08000000)	/* pointer to struct symbol */
# define STRINGTYPE	(0x10000000)	/* literal string */
# define UNDEFTYPE	(0x18000000)	/* undefined */
# define BUILTINTYPE	(0x20000000)	/* builtin function */
# define FILETYPE	(0x28000000)	/* bufio file pointer */
# define DICTTYPE	(0x30000000)	/* dictionary */
# define VECTORTYPE	(0x38000000)	/* vector */
# define FLOATPTYPE	(0x40000000)	/* floating point pointer */
# define CSTRINGTYPE	(0x48000000)	/* C style string (no links) */

# define listp(l)	(TYPEM(l) == LISTTYPE)
# define symbolp(l)	(TYPEM(l) == SYMBOLTYPE)
# define stringp(l)	(TYPEM(l) == STRINGTYPE)
# define undefp(l)	(TYPEM(l) == UNDEFTYPE)
# define builtinp(l)	(TYPEM(l) == BUILTINTYPE)
# define filep(l)	(TYPEM(l) == FILETYPE)
# define dictp(l)	(TYPEM(l) == DICTTYPE)
# define vectorp(l)	(TYPEM(l) == VECTORTYPE)
# define floatpp(l)	(TYPEM(l) == FLOATPTYPE)
# define cstringp(l)	(TYPEM(l) == CSTRINGTYPE)

# define symboltoitem(a)	(ptoi ((lispval) (a)) | SYMBOLTYPE)
# define itemtosymbol(i)	((struct symbol *) itop ((i) & ~TYPEMASK))

# define listtoitem(l)	((lispval) (ptoi ((lispval) (l)) | LISTTYPE))
# define itemtolist(i)	((struct dotted *) itop ((i) & ~TYPEMASK))

# define stringtoitem(s) ((lispval) (ptoi ((lispval) (s)) | STRINGTYPE))
# define itemtostring(i) ((character *) itop ((i) & ~TYPEMASK))

# define undeftoitem(u)	((lispval) (ptoi ((lispval) (u)) | UNDEFTYPE))
# define itemtoundef(i)	itop ((i) & ~TYPEMASK)

# define builtintoitem(b) ((lispval) (ptoi ((lispval) (b)) | BUILTINTYPE))
# define itemtobuiltin(i) ((struct builtin *) itop ((i) & ~TYPEMASK))

# define filetoitem(f) ((lispval) (ptoi ((lispval) (f)) | FILETYPE))
# define itemtofile(i) ((FILE *) itop ((i) & ~TYPEMASK))

# define dicttoitem(f) ((lispval) (ptoi ((lispval) (f)) | DICTTYPE))
# define itemtodict(i) ((struct dictionary *) itop ((i) & ~TYPEMASK))

# define vectortoitem(f) ((lispval) (ptoi ((lispval) (f)) | VECTORTYPE))
# define itemtovector(i) ((struct vector *) itop ((i) & ~TYPEMASK))

# define floatptoitem(f) ((lispval) (ptoi ((lispval) (f)) | FLOATPTYPE))
# define itemtofloatp(i) ((double *) itop ((i) & ~TYPEMASK))

# define cstringtoitem(f) ((lispval) (ptoi ((lispval) (f)) | CSTRINGTYPE))
# define itemtocstring(i) ((char *) itop ((i) & ~TYPEMASK))

/*
 * a macro for functions which return a numeric value.  This
 * checks the argument range and returns an integer if possible
 */

/*
 * these definitions assume 2's compliment arithmetic
 */

# define INTBITS	(~0 & ~NUMTYPE)
# define MAXINT		((int) (INTBITS >> 1))
# define MININT		((int) (-MAXINT - 1))

/*
 * these are the only well-specified machine-dependent functions
 * needed by the arithmetic routines.  Floor must return the
 * largest integer not greater than its argument while ceil must
 * return the smallest integer not less than its argument
 */
 
extern double	floor (), ceil ();
extern lispval	makeFloat ();

# define doubleRet(dv) \
	((MININT <= (dv) && (dv) <= MAXINT && floor ((double) dv) == (dv)) ?\
		numtoitem ((int) (dv)) \
	:\
		makeFloat ((double) dv))

# define intRet(iv) \
	((MININT <= ((int) (iv)) && ((int) (iv)) <= MAXINT) ?\
		numtoitem ((int) iv) \
	:\
		makeFloat ((double) iv))
/*
 * macros for fast memory allocation
 */

struct bfree {
	struct bfree	*next;
};

extern struct bfree	*freeList[];

extern char	*newHunk(), *newObject(), *newObjectWithSize();

struct bfree	*newHunkTemp;

# define newHunkFast(n)		((newHunkTemp = freeList[(n)]) ? \
					(freeList[(n)] = newHunkTemp->next, \
					(char *) newHunkTemp) \
 				 : \
 				 	newHunk (n))

/*
 * these macros *know* how big the structure is - if the structure
 * changes, either replace them with:
   # define newDotted()	((struct dotted *) newObject (sizeof (struct dotted))
   # define newSymbol()	((struct symbol *) newObject (sizeof (struct symbol))
	.
	.
	.
 *
 * or compute the size again.
 */

# define newDotted()		((struct dotted *) newHunkFast(1))
# define newSymbol()		((struct symbol *) newHunkFast(1))
# define newFloat()		((double *) newHunkFast(1))
# define newHashChain()		((struct hashchain *) newHunkFast(2))

/*
 * declarations for global functions
 */

/*
 * string defines.
 *
 * kalypso strings are of type (character *).
 * C strings are of of type (char *).
 *
 * All functions should be type appropriately.
 */
 
typedef unsigned char	character;

/*
 * These characters have special meaning when imbedded in
 * a string.  To actually encode them, use STRQUOTE followed
 * by the entry in quoteMap cooresponding to the character
 */

#define STREND		((character) '\0')
#define STRQUOTE	((character) '\300')
#define STRLNK		((character) '\301')
#define STRBLK		((character) '\302')

#define special(c)	(quoteMap[c] != '\0')
#define quoteChar(c)	(quoteMap[c])
#define unquoteChar(qc)	(unquoteMap[qc])

extern char		quoteMap[256];
extern character	unquoteMap[256];

# define iScdrLnk(s)	((*((s) + 1) == STRLNK) ? *((character **) ((s) + 2)) : (s) + 1)
# define iScdr(s)	((*(s) == STRQUOTE) ? iScdrLnk(iScdrLnk(s)) : iScdrLnk(s))
# define iScar(s)	((*(s) == STRQUOTE) ? unquoteChar(*iScdrLnk(s)) : *(s))

extern char		*sprint(), *spatom();
extern character	*iStrcpy (), *iKstring ();
extern char		*iCstring ();
extern char		*printName ();

/*
 * eval.  This is a macro for speed; not code size!
 */

extern lispval		iEvallist (), error (), Apply ();

#ifndef __GNUC__
lispval EvalTempVariable;
int     EvalTypeVariable;

# define iEval(qq)(							\
	(EvalTypeVariable=TYPE(EvalTempVariable=(qq))),			\
	(EvalTempVariable == nil ? (					\
		nil							\
	) : (EvalTypeVariable==SYMBOLTYPE) ? (				\
		undefp(EvalTempVariable=				\
		    itemtosymbol(EvalTempVariable)->value) ?		\
		(							\
			error ("unbound symbol %v", qq)			\
		) : 							\
			EvalTempVariable				\
	) : (EvalTypeVariable==LISTTYPE) ? (				\
	 	iEvallist(EvalTempVariable)				\
	) : (								\
		EvalTempVariable					\
	)								\
))
#else
static inline lispval
iEval (sexpr)
lispval	sexpr;
{
	lispval	value;
	int	type;

	if (sexpr == nil)
		return nil;
	type = TYPE(sexpr);
	if (type == SYMBOLTYPE) {
		value = itemtosymbol (sexpr)->value;
		if (undefp (value))
			return error ("unbound symbol %v", sexpr);
		return value;
	}
	if (type == LISTTYPE)
		return iEvallist (sexpr);
	return sexpr;
}
#endif

extern struct symbol	*iSymbol ();
extern lispval		symbolCopy ();
extern struct symbol	*true, *quote, *nilName, *kalypsoImage, *expandMacro;
extern struct symbol	*lambda, *nlambda, *lexpr, *macro, *errorNumber;
extern struct symbol	*SystemDictionary;
extern struct bufio	*getFILE ();

extern struct dotted	*makeList ();

extern struct dictionary	*iNewDictionary ();

/*
 * error numbers for errorNumber->value
 */

# define	LISPOK		0
# define	LISPEOF		1
# define	LISPBADFD	2
# define	LISPNOFILE	3
# define	LISPNOCMD	4

/*
 * frame
 */
 
extern int	framesize;
extern lispval	*frame;
extern lispval	*framep;
extern lispval	*frametop;
extern int	jumping;	/* poping frames - return immediately */
extern lispval	jumpValue;	/* value for jump catcher */
extern lispval	throwTag;	/* throw tag value */
extern int	asyncPending;	/* an asychronous event, call async() now */
extern int	canDoAsyncNow;	/* legal to do async at interrupt time */

#define checkAsync()	(asyncPending ? async () : 0)

#define checkFrame(o)	((framep) >= frametop ? growStack (o) : 0)

#define NotJumping	0
#define ReturnJumping	1
#define ErrorJumping	2
#define ThrowJumping	3
#define ContinueJumping	4

/*
 * useful frame macros
 */
 
/* mark current position in frame */
#define frameMark()	(framep - frame)
/* save an item in the frame */
#define framePush(o)	(checkFrame (o), *framep++ = (o))
/* reset the frame to a previous mark */
#define frameReset(m)	(framep = frame + (m))

#define FRAMEMARK	(undeftoitem(0xfffffff))
