/*
 * file.c
 */

# include	"kalypso.h"

# define FILE_TYPE_FILE	0	/* use fclose on this file */
# define FILE_TYPE_PIPE	1	/* use pclose on this file */

static		cleanFclose (), walkFiles (), addFile (), closeFile ();

static FILE	stdin_raw, stdout_raw, stderr_raw;

#ifdef SYSV
static FILE	save_stdin, save_stdout, save_stderr;
#endif

cleanIO ()
{
	
	walkFiles (cleanFclose);
	(void) fflush (stdin);
	(void) fflush (stdout);
	(void) fflush (stderr);
	/*
	 * make the dump image stdio state match the original
	 * state.
	 */
#ifdef SYSV
	save_stdin = *stdin;
	save_stdout = *stdout;
	save_stderr = *stderr;
#else
	if (stdin->_base)
		free (stdin->_base);
	if (stdout->_base)
		free (stdout->_base);
	if (stderr->_base)
		free (stderr->_base);
#endif
	*stdin = stdin_raw;
	*stdout = stdout_raw;
	*stderr = stderr_raw;
}

initIO ()
{
	stdin_raw = *stdin;
	stdout_raw = *stdout;
	stderr_raw = *stderr;
	addFile (stdin, FILE_TYPE_FILE);
	addFile (stdout, FILE_TYPE_FILE);
	addFile (stderr, FILE_TYPE_FILE);
}

/*
 * system V wants the file pointers reset after the dump
 * has finished
 */
 
restoreIO ()
{
#ifdef SYSV
	*stdin = save_stdin;
	*stdout = save_stdout;
	*stderr = save_stderr;
#endif
}

/*
 * return a string suitable for system calls from
 * either a string or a symbol name bound to a string
 */

char *
printName (l)
lispval	l;
{
	register character	*a;
	static char	nullName[] = "";

	if (nilp (l))
		return nullName;
	if (symbolp (l))
		l = itemtosymbol(l)->name;
	if (!stringp (l))
		return nil;
	a = itemtostring (l);
	return iCstring (a, (int *) 0);
}

lispval
Fclose (l)
lispval	l;
{
	if (!filep(l))
		return error ("fclose: non-file %v", l);
	(void) closeFile (itemtofile (l));
	return l;
}

lispval
Fopen (name, mode)
lispval	name, mode;
{
	char	*filename, *filemode;
	FILE	*f;
	int	onetrip;

	if ((filename = printName (name)) == nil)
		return error ("fopen: non-string filename %v", name);
	if ((filemode = printName (mode)) == nil)
		return error ("fopen: non-string filemode %v", mode);
	onetrip = 0;
retry:	;
	f = fopen (filename, filemode);
	if (!f) {
		if (errno == EMFILE && !onetrip) {
			iGarbageCollect ();
			onetrip = 1;
			goto retry;
		}
		errorNumber->value = numtoitem (LISPNOFILE);
		return nil;
	}
	addFile (f, FILE_TYPE_FILE);
	return filetoitem (f);
}

lispval
Fduplicate (file, mode)
lispval	file, mode;
{
	char	*filemode;
	FILE	*original, *new;
	int	onetrip;

	if (!filep (file))
		return error ("fduplicate: non-file %v", file);
	original = itemtofile (file);
	if ((filemode = printName (mode)) == nil)
		return error ("fduplicate: non-string %v", mode);
	onetrip = 0;
retry:	;
	new = fdopen (fileno (original), filemode);
	if (!new) {
		if (errno == EMFILE && !onetrip) {
			iGarbageCollect ();
			onetrip = 1;
			goto retry;
		}
		errorNumber->value = numtoitem (LISPNOFILE);
		return nil;
	}
	addFile (new, FILE_TYPE_FILE);
	return filetoitem (new);
}

/*
 * make references to file0 refer to file1
 */

lispval
Freplace (file0, file1)
lispval	file0, file1;
{
	FILE	*f0, *f1;

	if (!filep (file0))
		return error ("freplace: non-file %v", file0);
	f0 = itemtofile (file0);
	if (!filep (file1))
		return error ("freplace: non-file %v", file1);
	f1 = itemtofile (file1);
	dup2 (fileno (f0), fileno (f1));
	return symboltoitem (true);
}

lispval
Fseek (arg0, arg1, arg2)
lispval arg0, arg1, arg2;
{
	int	ret;
	FILE	*a0;
	long	a1;
	int	a2;

	if (!filep (arg0))
		return error ("fseek: non-file %v", arg0);
	a0 = itemtofile (arg0);
	if (nump (arg1))
		a1 = (long) itemtonum (arg1);
	else if (floatpp (arg1))
		a1 = (long) *itemtofloatp (arg1);
	else
		return error ("fseek: non-numeric %v", arg1);
	if (nump (arg2))
		a2 = itemtonum (arg2);
	else if (floatpp (arg2))
		a2 = (int) *itemtofloatp (arg2);
	else
		return error ("fseek: non-numeric %v", arg2);
	ret = fseek(a0, a1, a2);
	return numtoitem (ret);
}

lispval
Popen (cmd, mode)
lispval	cmd, mode;
{
	char	*command, *filemode;
	FILE	*f, *popen();

	if ((command = printName (cmd)) == nil)
		return error ("popen: non-string command %v", cmd);
	if ((filemode = printName (mode)) == nil)
		return error ("popen: non-string mode %v", mode);
	f = popen (command, filemode);
	if (!f) {
		errorNumber->value = numtoitem (LISPNOCMD);
		return nil;
	}
	addFile (f, FILE_TYPE_PIPE);
	return filetoitem (f);
}

iFgetchar (f)
FILE	*f;
{
	int	c;

	++canDoAsyncNow;
	checkAsync ();
	errno = 0;
	while ((c = getc (f)) == EOF) {
		if (errno != EINTR)
			break;
		if (jumping) {
			clearerr (f);
			--canDoAsyncNow;
			return -1;
		}
	}
	--canDoAsyncNow;
	if (c == -1) {
		errorNumber->value = numtoitem (LISPEOF);
		clearerr (f);
	}
	return c;
}

lispval
Fgetchar (l)
lispval	l;
{
	FILE	*f;
	int	c;

	if (!filep(l))
		return error ("fgetchar: non-file %v", l);
	f = itemtofile (l);
	c = iFgetchar (f);
	if (c == -1)
		return nil;
	return numtoitem (c);
}

iFungetchar (f, c)
FILE	*f;
int	c;
{
	return ungetc (c, f);
}

lispval
Fungetchar (l, c)
lispval	l, c;
{
	FILE	*f;

	if (!filep(l))
		return error ("fungetchar: non-file %v", l);
	f = itemtofile (l);
	if (!nump (c))
		return error ("fungetchar: non-numeric %v", c);
	(void) ungetc (itemtonum (c), f);
	return c;
}

/*
 * ugly but effective fgets routine.  Dynamically allocates
 * additional memory as the input string grows, creating
 * chained strings.
 */

#define FGETS_BSIZE	127
#define FGETS_USEFUL	(FGETS_BSIZE - sizeof (character) - sizeof (character *))

#define checkFgetsBuf()	((s == buf + sizeof (buf)) ?\
				(fixFgetsBuf (), s = buf) : 0)

static character	*firstFgetsBuf, *previousFgetsBuf, *fgetsBuf;
extern character	*stralloc ();

fixFgetsBuf ()
{
	character	*newbuf;

	newbuf = stralloc (FGETS_BSIZE);
	if (previousFgetsBuf) {
		previousFgetsBuf[FGETS_USEFUL] = STRLNK;
		*((character **) (previousFgetsBuf + FGETS_USEFUL+1)) = newbuf;
	} else
		firstFgetsBuf = newbuf;
	previousFgetsBuf = newbuf;
	bcopy (fgetsBuf, newbuf, FGETS_USEFUL);
}

lispval
Fgets (l)
lispval	l;
{
	FILE		*f;
	character	buf[FGETS_USEFUL];
	character	*s, *ret;
	int		c;
	int		remaining;

	if (!filep(l))
		return error ("fgets: non-file %v", l);
	f = itemtofile (l);
	s = buf;
	errno = 0;
	firstFgetsBuf = previousFgetsBuf = 0;
	fgetsBuf = buf;
	do {
		++canDoAsyncNow;
		checkAsync ();
		c = getc (f);
		--canDoAsyncNow;
		if (jumping) {
			clearerr (f);
			return nil;
		}
		if (c == EOF) {
			errorNumber->value = numtoitem (LISPEOF);
			clearerr (f);
			return nil;
		}
		if (special (c)) {
			checkFgetsBuf ();
			*s++ = STRQUOTE;
			c = quoteChar (c);
		}
		checkFgetsBuf ();
		*s++ = c;
	} while (c != '\n');
	checkFgetsBuf ();
	*s++ = '\0';
	remaining = s - buf;
	ret = stralloc (remaining);
	bcopy (buf, ret, s - buf);
	if (previousFgetsBuf) {
		previousFgetsBuf[FGETS_USEFUL] = STRLNK;
		*((character **) (previousFgetsBuf + FGETS_USEFUL+1)) = ret;
		ret = firstFgetsBuf;
	}
	return stringtoitem (ret);
}

lispval
Fputchar (l, count)
lispval	*l;
int	count;
{
	lispval	ret;
	FILE	*f;

	if (!filep(*l))
		return error ("fputchar: non-file %v", *l);
	f = itemtofile (*l);
	l++;
	--count;
	while (count--) {
		ret = *l++;
		if (!nump (ret))
			return error ("fputchar: non-numeric %v", ret);
		(void) putc (itemtonum (ret), f);
	}
	return ret;
}

lispval
Fputs (l, count)
lispval	*l;
int	count;
{
	lispval		ret;
	FILE		*f;
	character	*s;

	if (!filep(*l))
		return error ("fputchar: non-file %v", *l);
	f = itemtofile (*l);
	l++;
	--count;
	while (count--) {
		ret = *l++;
		if (!stringp (ret))
			return error ("fputs: non-string %v", *l);
		s = itemtostring (ret);
		while (*s) {
			(void) putc ((char) iScar (s), f);
			s = iScdr (s);
		}
	}
	return ret;
}

lispval
Fflush (l)
lispval	l;
{
	if (!filep(l))
		return error ("fflush: non-file %v", l);
	(void) fflush (itemtofile (l));
	return l;
}

/*
 * garbage collect file pointers.  This system assumes that
 * files are a scarce resource and that n^2 algorithms during
 * GC will not be performance problems.
 */

static int
cleanFclose (file)
FILE	*file;
{
	if (file != stdin && file != stdout && file != stderr)
		closeFile (file);
}

static struct file_refs {
	struct file_refs	*next;
	short			ref;
	short			type;
	FILE			*file;
} *open_files;

static
walkFiles (function)
int     (*function)();
{
	struct file_refs	*f;

	for (f = open_files; f; f=f->next)
		(void) (*function) (f->file);
}

static
addFile (file, type)
FILE	*file;
{
	struct file_refs	*f;
	extern char		*malloc ();

	f = (struct file_refs *) malloc (sizeof (struct file_refs));
	f->file = file;
	f->next = open_files;
	f->ref = 0;
	f->type = type;
	open_files = f;
}

static
closeFile (file)
FILE	*file;
{
	struct file_refs	*f, *prev;

	prev = 0;
	for (f = open_files; f; f=f->next)
		if (f->file == file) {
			if (prev)
				prev->next = f->next;
			else
				open_files = f->next;
			switch (f->type) {
			case FILE_TYPE_FILE:
				(void) fclose (file);
				break;
			case FILE_TYPE_PIPE:
				(void) pclose (file);
				break;
			}
			free ((char *) f);
			break;
		}
}

setFileRef (file)
FILE *file;
{
	struct file_refs	*f;

	for (f = open_files; f; f=f->next)
		if (f->file == file)
			f->ref = 1;
}

checkFileRef ()
{
	struct file_refs	*f, *next;

	for (f = open_files; f; f = next) {
		next = f->next;
		if (!f->ref)
			closeFile (f->file);
		else
			f->ref = 0;
	}
}

struct builtin fileStuff[] = {
	"fclose",		Fclose,		LAMBDA,		1,
	"fopen",		Fopen,		LAMBDA,		2,
	"fseek",		Fseek,		LAMBDA,		3,
	"popen",		Popen,		LAMBDA,		2,
	"fgetchar",		Fgetchar,	LAMBDA,		1,
	"fungetchar",		Fungetchar,	LAMBDA,		2,
	"fgets",		Fgets,		LAMBDA,		1,
	"fputchar",		Fputchar,	LEXPR,		1,
	"fputs",		Fputs,		LEXPR,		1,
	"fduplicate",		Fduplicate, 	LAMBDA, 	2,
	"freplace",		Freplace,	LAMBDA,		2,
	"fflush",		Fflush,		LAMBDA,		1,
	0,			0,		0,		0,
};
