/*
 * dictionary.c
 *
 * dictionaries are simple associative arrays -- a lispval
 * associated with another lispval.  Don't treat them as
 * anything else.
 */

# include	"kalypso.h"
# include	"mem.h"

static resizeDictionary ();

/*
 * iSymbol -- special function for the reader.
 *
 * look up a name in a dictionary and return the
 * associated value.  If the name is not in the dictionary,
 * insert a symbol, giving it the same name.
 *
 * the copyForInsert is a *very* special case
 * intended to speed the lexer (and not waste a pile
 * of memory).  It only works if the name is a
 * string.
 */

struct symbol *
iSymbol (name, dictionary, copyForInsert)
lispval			name;
struct dictionary	*dictionary;
{
	register struct hashchain	**bucket, *a;
	struct symbol			*iNewSymbol ();
	lispval				i;
	int				hash;
	lispval				Equal_p ();

	hash = iHash (name);
	bucket = &dictionary->hashtable[abs (hash) % dictionary->hashsize];
	for (a = *bucket; a; a = a->next) {
		if (a->hash != hash)
			continue;
		i = Equal_p (name, a->name);
		if (jumping)
			return nil;
		if (!nilp (i)) {
			if (!symbolp (a->value)) {
				i = error ("symbol: found non-symbol %v", a->value);
				if (symbolp (i))
					return itemtosymbol (i);
				return (struct symbol *) 0;
			}
			return itemtosymbol (a->value);
		}
	}
	a = newHashChain ();
	a->next = *bucket;
	*bucket = a;
	a->name = nil;
	a->hash = hash;
 	a->value = symboltoitem (iNewSymbol (name, copyForInsert));
	a->name = itemtosymbol (a->value)->name;
	if (++dictionary->count >= dictionary->hashsize &&
	    dictionary->hashsize < MAXDICTSIZE)
		resizeDictionary (dictionary, dictionary->hashsize*2);
	return itemtosymbol (a->value);
}

lispval
Symbol (name, dictionary)
lispval	name, dictionary;
{
	struct dictionary	*d;
	struct symbol		*s;

	if (!dictp (dictionary))
		return error ("symbol: non dictionary %v", dictionary);
	d = itemtodict (dictionary);
	s = iSymbol (name, d, 0);
	if (jumping)
		return nil;
	return symboltoitem (s);
}

lispval
symbolCopy (name, dictionary)
lispval	name, dictionary;
{
	struct dictionary	*d;
	struct symbol		*s;

	if (!dictp (dictionary))
		return error ("symbol: non-dictionary %v", dictionary);
	d = itemtodict (dictionary);
	s = iSymbol (name, d, 1);
	if (s)
		return symboltoitem (s);
	return nil;
}

lispval
iDictionaryLookup (dictionary, name)
struct dictionary	*dictionary;
lispval			name;
{
	register struct hashchain	**bucket, *a;
	lispval				i;
	int				hash;

	hash = iHash (name);
	bucket = &dictionary->hashtable[abs (hash) % dictionary->hashsize];
	for (a = *bucket; a; a = a->next) {
		if (a->hash != hash)
			continue;
		i = Equal_p (name, a->name);
		if (jumping)
			return nil;
		if (!nilp (i))
			return a->value;
	}
	return nil;
}

struct hashchain *
iDictionaryBucket (dictionary, b)
struct dictionary	*dictionary;
struct hashchain	*b;
{
	register struct hashchain	**bucket, *a;
	lispval				i;
	int				hash;

	hash = b->hash;
	bucket = &dictionary->hashtable[abs (hash) % dictionary->hashsize];
	for (a = *bucket; a; a = a->next) {
		if (a->hash != hash)
			continue;
		i = Equal_p (b->name, a->name);
		if (jumping)
			return nil;
		if (!nilp (i))
			return a;
	}
	return nil;
}

lispval
DictionaryLookup (dictionary, name)
lispval		dictionary;
lispval		name;
{
	struct dictionary	*d;

	if (!dictp (dictionary))
		return error ("dictionary-lookup: non dictionary %v", dictionary);
	d = itemtodict (dictionary);
	return iDictionaryLookup (d, name);
}

lispval
iDictionaryRemove (d, name)
struct dictionary	*d;
lispval			name;
{
	register struct hashchain	**bucket, *a, *prev;
	lispval				i;
	int				hash;

	hash = iHash (name);
	bucket = &d->hashtable[abs (hash) % d->hashsize];
	prev = 0;
	for (a = *bucket; a; a = a->next) {
		if (hash != a->hash)
			continue;
		i = Equal_p (name, a->name);
		if (jumping)
			return nil;
		if (!nilp (i)) {
			if (prev)
				prev->next = a->next;
			else
				*bucket = a->next;
			return a->value;
		}
		prev = a;
	}
	return nil;
}

lispval
DictionaryRemove (dictionary, name)
lispval		dictionary;
lispval		name;
{
	struct dictionary	*d;

	if (!dictp (dictionary))
		return error ("dictionary-remove: non dictionary %v", dictionary);
	d = itemtodict (dictionary);
	return iDictionaryRemove (d, name);
}

lispval
iDictionaryInsert (dictionary, name, value)
struct dictionary	*dictionary;
lispval			name, value;
{
	register struct hashchain	**bucket, *a;
	lispval				i;
	int				hash;

	hash = iHash (name);
	bucket = &dictionary->hashtable[abs (hash) % dictionary->hashsize];
	for (a = *bucket; a; a = a->next) {
		if (a->hash != hash)
			continue;
		i = Equal_p (name, a->name);
		if (jumping)
			return nil;
		if (!nilp (i)) {
			a->value = value;
			return value;
		}
	}
	a = newHashChain ();
	a->name = name;
	a->value = value;
	a->hash = hash;
	a->next = *bucket;
	*bucket = a;
	if (++dictionary->count >= dictionary->hashsize &&
	    dictionary->hashsize < MAXDICTSIZE)
		resizeDictionary (dictionary, dictionary->hashsize*2);
	return value;
}

lispval
DictionaryInsert (dictionary, name, value)
lispval		dictionary;
lispval		name, value;
{
	struct dictionary	*d;

	if (!dictp (dictionary))
		return error ("dictionary-insert: non dictionary %v", dictionary);
	d = itemtodict (dictionary);
	return iDictionaryInsert (d, name, value);
}

struct dictionary *
iNewDictionary ()
{
	struct dictionary	*d;
	int			i;
	int			s;

	s = frameMark ();
	d = (struct dictionary *) newObject (sizeof (struct dictionary));
	d->hashtable = 0;
	d->hashsize = 0;
	d->count = 0;
	framePush (dicttoitem (d));
	d->hashtable = (struct hashchain **)
 	    newObject (MINDICTSIZE * sizeof (struct hashchain *));
	d->hashsize = MINDICTSIZE;
	for (i = 0; i < d->hashsize; i++)
		d->hashtable[i] = 0;
	frameReset (s);
	return d;
}

lispval
NewDictionary ()
{
	struct dictionary	*d;

	d = iNewDictionary ();
	return dicttoitem (d);
}

struct dictionary *
iCopyDictionary (original)
struct dictionary	*original;
{
	struct dictionary		*new;
	register struct hashchain	**new_bucket, **old_bucket;
	register struct hashchain	*new_a, *old_a, *last;
	int				framem;
	
	new = iNewDictionary ();
	framem = frameMark ();
	framePush (dicttoitem (new));
	if (new->hashsize != original->hashsize)
		resizeDictionary (new, original->hashsize);
	new_bucket = new->hashtable;
	for (old_bucket = original->hashtable;
 	     old_bucket < original->hashtable + original->hashsize;
	     old_bucket++)
 	{
		last = 0;
		for (old_a = *old_bucket; old_a; old_a = old_a->next) {
			new_a = newHashChain ();
			new_a->next = 0;
			new_a->name = old_a->name;
			new_a->value = old_a->value;
			new_a->hash = old_a->hash;
			if (last)
				last->next = new_a;
			else
				*new_bucket = new_a;
			last = new_a;
		}
		new_bucket++;
	}
	frameReset (framem);
	return new;
}

lispval
CopyDictionary (original)
lispval	original;
{
	struct dictionary	*orig, *copy;

	if (!dictp (original))
		return error ("copy-dictionary: non-dictionary %v", original);
	orig = itemtodict (original);
	copy = iCopyDictionary (orig);
	return dicttoitem (copy);
}

struct dotted *
iDictionaryToList (d)
struct dictionary	*d;
{
	register struct dotted	*first, *last, *new;
	struct hashchain	**bucket, *a;
	int			mark;
 
	first = last = 0;
	mark = frameMark ();
	for (bucket = d->hashtable;
 	     bucket < d->hashtable + d->hashsize;
	     bucket++)
 	{
		for (a = *bucket; a; a = a->next) {
			new = newDotted ();
			new->car = nil;
			new->cdr = nil;
			if (last)
				last->cdr = new;
			else {
				first = new;
				framePush (listtoitem (first));
			}
			last = new;
			new = newDotted ();
			last->car = listtoitem (new);
			new->cdr = nil;
			new->car = a->name;
			new->cdr = newDotted ();
			new->cdr->cdr = nil;
			new->cdr->car = a->value;
		}
	}
	frameReset (mark);
	return first;
}

lispval
DictionaryToList (dictionary)
lispval	dictionary;
{
	struct dictionary	*d;
	struct dotted		*l;

	if (!dictp (dictionary))
		return error ("dictionary-to-list: non-dictionary %v", dictionary);
	d = itemtodict (dictionary);
	l = iDictionaryToList (d);
	if (l)
		return listtoitem (l);
	return nil;
}

lispval
iListToDictionary (l)
struct dotted	*l;
{
	struct dictionary	*d;
	int			m;
	struct dotted		*namel, *valuel;

	m = frameMark ();
	d = iNewDictionary ();
	framePush (dicttoitem (d));
	while (l) {
		if (!listp (l->car))
			return error ("list-to-dict: non list %v", l->car);
		namel = itemtolist (l->car);
		if (nilp (namel->cdr))
			return error ("list-to-dict: non pair %v", l->car);
		valuel = namel->cdr;
		(void) iDictionaryInsert (d, namel->car, valuel->car);
		l = l->cdr;
	}
	frameReset (m);
	return dicttoitem (d);
}

lispval
ListToDictionary (list)
lispval	list;
{
	if (!listp (list) && !nilp (list))
		return error ("list-to-dict: non list %v", list);
	return iListToDictionary (itemtolist (list));
}

lispval
SameDictionary (d1, d2)
lispval	d1, d2;
{
	static character	msg[] = "same-dictionary: non dictionary %v";

	if (!dictp (d1))
		return error (msg, d1);
	if (!dictp (d2))
		return error (msg, d2);
	if (d1 != d2)
		return nil;
	return symboltoitem (true);
}

static
resizeDictionary (d, newsize)
struct dictionary	*d;
int			newsize;
{
	register struct hashchain	**newtable;
	register struct hashchain	**bucket, *a;
	struct hashchain		*next;
	struct hashchain		**newbucket;
	int				i;

	newtable = (struct hashchain **)
 	    newObject (newsize * sizeof (struct hashchain *));
	for (i = 0; i < newsize; i++)
		newtable[i] = 0;
	for (bucket = d->hashtable; bucket < &d->hashtable[d->hashsize]; bucket++) {
		for (a = *bucket; a; a = next) {
			next = a->next;
			newbucket = &newtable[abs(a->hash) % newsize];
			a->next = *newbucket;
			*newbucket = a;
		}
	}
	d->hashsize = newsize;
	d->hashtable = newtable;
}

setDictRef (d)
struct dictionary *d;
{
	register struct hashchain	**bucket, *a;

	if (setObjectRef ((char *) d))
		return;
	(void) setObjectRef ((char *) d->hashtable);
	for (bucket = d->hashtable; bucket < d->hashtable + d->hashsize; bucket++)
		for (a = *bucket; a; a = a->next) {
			(void) setObjectRef ((char *) a);
			setRef (a->name);
			setRef (a->value);
		}
}

struct builtin dictionaryStuff[] = {
	{ "symbol",		Symbol,			LAMBDA,	2 },
	{ "dictionary-lookup",	DictionaryLookup,	LAMBDA,	2 },
	{ "dictionary-remove",	DictionaryRemove,	LAMBDA,	2 },
	{ "dictionary-insert",	DictionaryInsert,	LAMBDA,	3 },
	{ "new-dictionary",	NewDictionary,		LAMBDA,	0 },
	{ "copy-dictionary",	CopyDictionary,		LAMBDA,	1 },
	{ "dictionary-to-list",	DictionaryToList,	LAMBDA,	1 },
	{ "list-to-dictionary",	ListToDictionary,	LAMBDA, 1 },
	{ "same-dictionary",	SameDictionary,		LAMBDA, 2 },
	{ 0,			0,			0,	0 },
};
