;;; (C) Copyright 1990-1994 by Wade L. Hennessey. All rights reserved.

;;; Functions in this file are needed by both the cross and native compiler.

;;; Return element-type-tag,  element-size, and default initialize-elements
(defun type->element-type-tag (element-type)
  (let* ((exp-type (type-macroexpand element-type))
	 (type (if (atom exp-type) (list exp-type) exp-type)))
    (case (first type)
      (integer (let ((low (second type))
		     (high (third type)))
		 (if (and (numberp low) (numberp high))
		     (if (>= low 0)
			 (cond ((= high 1)
				(values element-type-bit 1 0))
			       ((<= high (1- (expt 2 8)))
				(values element-type-unsigned-8bit 8 0))
			       ((<= high (1- (expt 2 16)))
				(values element-type-unsigned-16bit 16 0))
			       ((<= high (1- (expt 2 32)))
				(values element-type-unsigned-32bit 32 0))
			       (t (values element-type-ptr 32)))
			 (cond ((and (>= low (- (expt 2 7)))
				     (<= high (1- (expt 2 7))))
				(values element-type-signed-8bit 8 0))
			       ((and (>= low (- (expt 2 15)))
				     (<= high (1- (expt 2 15))))
				(values element-type-signed-16bit 16 0))
			       ((and (>= low (- (expt 2 31)))
				     (<= high (1- (expt 2 31))))
				(values element-type-signed-32bit 32 0))
			       (t (values element-type-ptr 32 nil))))
		     (values element-type-ptr 32 0))))
      (float (values element-type-float 64 0.0))
      (character (values element-type-char 8 #\Null))
      ((t array simple-array symbol or and not member satisfies)
       (values element-type-ptr 32 nil))
      (t (error "~A is not a legal element-type" element-type)))))


(defun rewrite-typep (form any-type)
  (complete-delayed-defstructs)
  (let ((mexp-type (type-macroexpand any-type)))
    (cond ((eq mexp-type t) t)
	  ((null mexp-type) nil)
	  (t (let* ((type (if (atom mexp-type) (list mexp-type) mexp-type))
		    (second (if (null (second type)) '* (second type)))
		    (third (if (null (third type)) '* (third type))))
	       (case (first type)
		 (satisfies `(,(second type) ,form))
		 ((simple-array array)
		  (rewrite-array-typep form type second third))
		 (integer
		  (if (null (cdr type))
		      `(integerp ,form)
		      (if (and (fixnump second) (fixnump third))
			  (cond ((and (= second most-negative-fixnum)
				      (= third most-positive-fixnum))
				 `(fixnump ,form))
				((and (= second 0) (= third 1)) `(bitp ,form))
				(t `(integer-range-p ,form ',second ',third)))
			  `(integer-range-p ,form ',second ',third))))
		 (float `(floatp ,form))
		 (character `(characterp ,form))
		 (symbol `(symbolp ,form))
		 (or (cond ((equal type '(or integer float ratoi complex))
			    `(numberp ,form))
			   ((equal type '(or null cons)) `(listp ,form))
			   (t (rewrite-combined-type form type))))
		 ((and not) (rewrite-combined-type form type))
		 (member `(member ,form ',(cdr mexp-type)))
		 (t (let ((info (gethash (first type) *structure-info*)))
		      (if (or (null info)
			      (not (struct-info-named? info)))
			  (multiple-value-bind (new-type rewrite?)
			      (type-macroexpand type)
			    (if rewrite?
				(rewrite-typep form new-type)
				(values nil t))) ; indicate failure
			  `(,(predicate-name info) ,form))))))))))

(defun rewrite-combined-type (form type)
  (let ((tmp (gensym "TMP")))
    `(let ((,tmp ,form))
       ,(cons (first type) (mapcar #'(lambda (x)
				       (rewrite-typep tmp x))
				   (rest type))))))

(defun rewrite-array-typep (form type second third)
  (flet ((pred (name)
	   (if (eq (first type) 'simple-array)
	       (intern (concatenate 'string "SIMPLE-" (string name))
		       *lisp-package*)
	       name)))
    (let ((element-type (if (eq second '*) t second)))
      (if (null (cdr type))
	  `(,(pred 'array-p) ,form)
	  (if (and (listp third) (= (length third) 1))
	      ;; Special case vectors
	      (let ((type-code (type->element-type-tag element-type)))
		(select type-code
			(element-type-char
			 `(,(pred 'string-p) ,form))
			(element-type-float
			 `(,(pred 'float-vector-p) ,form))
			(element-type-bit
			 `(,(pred 'bit-vector-p) ,form))
			(element-type-signed-8bit
			 `(,(pred 'signed-8bit-vector-p) ,form))
			(element-type-unsigned-8bit
			 `(,(pred 'unsigned-8bit-vector-p) ,form))
			(element-type-signed-16bit
			 `(,(pred 'signed-16bit-vector-p) ,form))
			(element-type-unsigned-16bit
			 `(,(pred 'unsigned-16bit-vector-p) ,form))
			(element-type-signed-32bit
			 `(,(pred 'signed-32bit-vector-p) ,form))
			(element-type-unsigned-32bit
			 `(,(pred 'unsigned-32bit-vector-p) ,form))
			(t `(,(pred 'vector-p) ,form))))
	      `(,(pred 'array-type-p) ,form ',element-type ',third))))))

(defun type-macroexpand (form &optional local-macro-env already-expanded?)
  (let ((list-form (if (listp form) form (list form))))
    (multiple-value-bind (new-form expanded?)
	(expand-macro list-form
		      *type-macro-expanders*
		      local-macro-env
		      t
		      nil)
      (if expanded?
	  (type-macroexpand new-form local-macro-env t)
	  (values (expand-inner-types form)
		  (or already-expanded? expanded?))))))
	  
(defun expand-inner-types (type)
  (if (and (listp type)
	   (or (eq (first type) 'array)
	       (eq (first type) 'simple-array))
	   (not (null (second type))))
      `(,(first type)
	,(type-macroexpand (second type))
	,@(or (cddr type) '(*)))
      type))

(defun collect-cases (cases)
  (loop for (case . ignore) in cases
	appending (if (listp case)
		      case
		      (list case))))

(defun processor+os->machine-type (processor os)
  (let ((type (cond ((string-equal os "sunos")
		     (cond ((string-equal processor "sparc") :sun-4)))
		    ((string-equal os "irix")
		     (cond ((string-equal processor "mips") :iris))))))
    (if (null type)
	(warn "Machine type not known for ~A running ~A" processor os)
	type)))

(defun machine-byte-order (machine-type)
  (case machine-type
    (:sun-4 :big-endian)
    (:iris :big-endian)
    (:decstation :little-endian)))

(defun unix->lisp-path-list (string)
  (let ((len (length string)))
    (labels ((doit (start)
	       (if (>= start len)
		   nil
		   (let ((end (or (position #\: string :start start) len)))
		     (cons (subseq string start end)
			   (doit (1+ end)))))))
      (doit 0))))

(defun setf-function-symbol (function-specifier)
  (if (consp function-specifier)
      (let ((accessor (second function-specifier)))
	(or (gethash accessor *setf-methods*)
	    (intern (format nil "SET-~A" accessor)
		    (symbol-package accessor))))
      function-specifier))

(defun set-fdefinition (new-value function-specifier)
  (if (consp function-specifier)
      (progn
        (setf (symbol-function (setf-function-symbol function-specifier))
              new-value)
	;; HEY! fix this to not use EVAL!!! Cheap hack for now...
        (eval `(defsetf ,(cadr function-specifier) 
                        (&rest all-args)
                        (new-value)
                 `(,',(setf-function-symbol function-specifier)
                   ,new-value
                   ,@all-args))))
      (setf (symbol-function function-specifier) new-value)))
