/*
    unixfsys.c  -- Unix file system interface.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/

#include "config.h"
#include <sys/stat.h>
#include <pwd.h>

extern object Kwild;

#ifdef SYSV

#ifdef UNIGRAPH 
#include <sys/dir.h>
#else
#include <dirent.h>
#define DIRECTORY	struct dirent
#endif UNIGRAPH

#ifndef hpux
#define MAXNAMLEN NAME_MAX
#endif hpux

char *
getwd(char *buffer)
{
	char *getcwd();

	return(getcwd(buffer, MAXPATHLEN));
}
#endif SYSV

void
coerce_to_filename(object pathname, char *p)
{
	object namestring; int len;

	namestring = coerce_to_namestring(pathname);
	len = namestring->st.st_fillp;
	if (len >= MAXPATHLEN - 16)
		FEerror("Too long filename: ~S.", 1, namestring);
	memcpy(p, namestring->st.st_self, len+1);
	p[len] = '\0';
}

object
truename(object pathname)
{
	register char *p, *q;
	char filename[MAXPATHLEN];
	char truefilename[MAXPATHLEN];
	char current_directory[MAXPATHLEN];
	char directory[MAXPATHLEN];
	char *getwd();

	coerce_to_filename(pathname, filename);
	for (p = filename, q = 0;  *p != '\0';  p++)
		if (*p == '/')
			q = p;
	if (q == filename) {
		q++;
		getwd(current_directory);
		p = "/";
	} else if (q == 0) {
		q = filename;
		p = getwd(current_directory);
	} else {
		*q++ = '\0';
		getwd(current_directory);
		if (chdir(filename) < 0)
		    FEerror("Cannot get the truename of ~S.", 1, pathname);
		p = getwd(directory);
	}
	if (p[0] == '/' && p[1] == '\0') {
		if (strcmp(q, "..") == 0)
			strcpy(truefilename, "/.");
		else
			sprintf(truefilename, "/%s", q);
	} else if (strcmp(q, ".") == 0)
		strcpy(truefilename, p);
	else if (strcmp(q, "..") == 0) {
		for (q = p + strlen(p);  *--q != '/';) ;
		if (p == q)
			strcpy(truefilename, "/.");
		else {
			*q = '\0';
			strcpy(truefilename, p);
			*q = '/';
		}
	} else
		sprintf(truefilename, "%s/%s", p, q);
	chdir(current_directory);
	pathname = coerce_to_pathname(make_simple_string(truefilename));
	return(pathname);
}

bool
file_exists(object file)
{
	char filename[MAXPATHLEN];
	struct stat filestatus;

	coerce_to_filename(file, filename);
	if (stat(filename, &filestatus) >= 0)
		return(TRUE);
	else
		return(FALSE);
}

FILE *
backup_fopen(char *filename, char *option)
{
	char backupfilename[MAXPATHLEN];
	char command[MAXPATHLEN * 2];

	strcat(strcpy(backupfilename, filename), ".BAK");
	sprintf(command, "mv %s %s", filename, backupfilename);
	system(command);
	return(fopen(filename, option));
}

int
file_len(FILE *fp)
{
	struct stat filestatus;

	fstat(fileno(fp), &filestatus);
	return(filestatus.st_size);
}

Ltruename(int narg, object file)
{
	check_arg(1);
	check_type_or_pathname_string_symbol_stream(&file);
	VALUES(0) = truename(file);
}

Lrename_file(int narg, object old, object new)
{
	char filename[MAXPATHLEN];
	char newfilename[MAXPATHLEN];
	char command[MAXPATHLEN * 2];

	check_arg(2);
	check_type_or_pathname_string_symbol_stream(&old);
	check_type_or_Pathname_string_symbol(&new);
	coerce_to_filename(old, filename);
	old = coerce_to_pathname(old);
	new = coerce_to_pathname(new);
	new = merge_pathnames(new, old, Cnil);
	coerce_to_filename(new, newfilename);
	if (rename(filename, newfilename) < 0)
		FEerror("Cannot rename the file ~S to ~S.",
			2, old, new);
/*	sprintf(command, "mv %s %s", filename, newfilename);
	system(command);
 */
	VALUES(0) = new;
	VALUES(0) = truename(old);
	VALUES(0) = truename(new);
	RETURN(3);
}

Ldelete_file(int narg, object file)
{
	char filename[MAXPATHLEN];

	check_arg(1);
	check_type_or_pathname_string_symbol_stream(&file);
	coerce_to_filename(file, filename);
	if (unlink(filename) < 0)
		FEerror("Cannot delete the file ~S.", 1, file);
	VALUES(0) = Ct;
	RETURN(1);
}

Lprobe_file(int narg, object file)
{
	check_arg(1);

	check_type_or_pathname_string_symbol_stream(&file);
	if (file_exists(file))
		VALUES(0) = truename(file);
	else
		VALUES(0) = Cnil;
	RETURN(1);
}

Lfile_write_date(int narg, object file)
{
	char filename[MAXPATHLEN];
	struct stat filestatus;

	check_arg(1);
	check_type_or_pathname_string_symbol_stream(&file);
	coerce_to_filename(file, filename);
	if (stat(filename, &filestatus) < 0)
	  VALUES(0) = Cnil;
	else
	  VALUES(0) = unix_time_to_universal_time(filestatus.st_mtime);
	RETURN(1);
}

Lfile_author(int narg, object file)
{
	char filename[MAXPATHLEN];
	struct stat filestatus;
	struct passwd *pwent;
	extern struct passwd *getpwuid(uid_t);

	check_arg(1);
	check_type_or_pathname_string_symbol_stream(&file);
	coerce_to_filename(file, filename);
	if (stat(filename, &filestatus) < 0)
		FEerror("Cannot get the file status of ~S.", 1, file);
	pwent = getpwuid(filestatus.st_uid);
	VALUES(0) = make_simple_string(pwent->pw_name);
	RETURN(1);
}

object
homedir_pathname(object user)
{
	int i;
	char *p, filename[MAXPATHLEN];
	struct passwd *pwent;
	extern struct passwd *getpwuid(uid_t), *getpwnam();
	
	if (Null(user))
		pwent = getpwuid(getuid());
	else {
		user = coerce_to_string(user);
		p = user->st.st_self;
		i = user->st.st_fillp;
		if (i > 0 && *p == '~') {
			p++;
			i--;
		}
		if (i == 0)
			pwent = getpwuid(getuid());
		else {
			strncpy(filename, p, i);
			filename[i] = '\0';
			pwent = getpwnam(filename);
		}
	}
	if (pwent == NULL)
		FEerror("Unknown user ~S.", 1, user);
	strcpy(filename, pwent->pw_dir);
	i = strlen(filename);
	if (i == 0 || filename[i-1] != '/') {
		filename[i++] = '/';
		filename[i] = '\0';
	}
	return coerce_to_pathname(make_simple_string(filename));
}

Luser_homedir_pathname(int narg, object host)
{
	if (narg > 1)	/* Ignore optional host argument. */
		FEtoo_many_arguments(&narg);
#ifdef MSDOS
	{ extern char *getenv();
	  char *h = getenv("HOME");
	  if ( h != NULL)
	    VALUES(0) = make_simple_string(h);
	  else
	    VALUES(0) = make_simple_string("/");
	}
#else
	VALUES(0) = homedir_pathname(Cnil);
#endif MSDOS
	RETURN(1);
}

#ifdef BSD
Ldirectory(int narg, object file)
{
	char filename[MAXPATHLEN];
	char command[MAXPATHLEN * 2];
	FILE *fp;
	object *directory;
	register int i, c;
	char iobuffer[BUFSIZ];
	extern FILE *popen();

	check_arg(1);

	check_type_or_pathname_string_symbol_stream(&file);
	file = coerce_to_pathname(file);
	if (file->pn.pn_name==Cnil && file->pn.pn_type==Cnil) {
		coerce_to_filename(file, filename);
		strcat(filename, "*");
	} else if (file->pn.pn_name==Cnil) {
		file->pn.pn_name = Kwild;
		coerce_to_filename(file, filename);
		file->pn.pn_name = Cnil;
	} else if (file->pn.pn_type==Cnil) {
		coerce_to_filename(file, filename);
		strcat(filename, "*");
	} else
		coerce_to_filename(file, filename);
	sprintf(command, "ls -d %s 2> /dev/null", filename);
	fp = popen(command, "r");
	setbuf(fp, iobuffer);
	directory = &VALUES(0);
	*directory = Cnil;
	for (;;) {
		for (i = 0;  c = getc(fp);  i++)
			if (c <= 0)
				goto L;
			else if (c == '\n')
				break;
			else
				filename[i] = c;
		filename[i] = '\0';
		directory = &CDR(*directory = CONS(truename(make_simple_string(filename)),
				       Cnil));
	}
L:
	pclose(fp);
	RETURN(1);
}
#endif BSD

#ifdef SYSV
Ldirectory(int narg, object file)
{
	object name, type;
	char filename[MAXPATHLEN];
	FILE *fp;
	char iobuffer[BUFSIZ];
	DIRECTORY dir;
	int i;

	check_arg(1);

	check_type_or_pathname_string_symbol_stream(&file);
	file = coerce_to_pathname(file);
	name = file->pn.pn_name;
	type = file->pn.pn_type;
	file->pn.pn_name = Cnil;
	file->pn.pn_type = Cnil;
	coerce_to_filename(file, filename);
	file->pn.pn_type = type;
	file->pn.pn_name = name;
	i = strlen(filename);
	if (i > 1 && filename[i-1] == '/')
		filename[i-1] = '\0';
	if (i == 0)
		strcpy(filename, ".");
	fp = fopen(filename, "r");
	if (fp == NULL)
		FEerror("Can't open the directory ~S.", 1,
			make_simple_string(filename));
	setbuf(fp, iobuffer);
	fread(&dir, sizeof(DIRECTORY), 1, fp);
	fread(&dir, sizeof(DIRECTORY), 1, fp);
	filename[MAXNAMLEN] = '\0';
	directory = &VALUES(0);
	*directory = Cnil;
	for (;;) {
		if (fread(&dir, sizeof(DIRECTORY), 1, fp) <= 0)
			break;
		if (dir.d_ino == 0)
			continue;
		strncpy(filename, dir.d_name, MAXNAMLEN);
		pname = coerce_to_pathname(make_simple_string(filename));
		if ((Null(name) || name == Kwild ||
		     equal(name, pname->pn.pn_name)) &&
		    (Null(type) || type == Kwild ||
		     equal(type, pname->pn.pn_type))) {
			pname->pn.pn_directory
			= file->pn.pn_directory;
			directory = &CDR(CONS(truename(pname), Cnil));
		}
	}
	fclose(fp);
	RETURN(1);
}
#endif SYSV

siLchdir(int narg, object directory)
{
	char filename[MAXPATHLEN];

	check_arg(1);
	check_type_or_pathname_string_symbol_stream(&directory);
	coerce_to_filename(directory, filename);
	if (chdir(filename) < 0)
		FEerror("Can't change the current directory to ~S.",
			1, directory);
	VALUES(0) = directory;
	RETURN(1);
}

init_unixfsys()
{
	make_function("TRUENAME", Ltruename);
	make_function("RENAME-FILE", Lrename_file);
	make_function("DELETE-FILE", Ldelete_file);
	make_function("PROBE-FILE", Lprobe_file);
	make_function("FILE-WRITE-DATE", Lfile_write_date);
	make_function("FILE-AUTHOR", Lfile_author);
	make_function("USER-HOMEDIR-PATHNAME", Luser_homedir_pathname);
	make_function("DIRECTORY", Ldirectory);
	make_si_function("CHDIR", siLchdir);
}
