/*
    character.d -- Character routines.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/


#include "config.h"

/******************************* EXPORTS ******************************/

object STreturn;
object STspace;
object STrubout;
object STpage;
object STtab;
object STbackspace;
object STlinefeed;
object STnewline;

/******************************* ------- ******************************/

object Kcontrol;
object Kmeta;
object Ksuper;
object Khyper;

@(defun standard_char_p (c)
	int i;
@
	check_type_character(&c);
	if (char_font(c) != 0 || char_bits(c) != 0)
		@(return Cnil)
	i = char_code(c);
	if (' ' <= i && i < '\177' || i == '\n')
		@(return Ct)
	@(return Cnil)
@)

@(defun graphic_char_p (c)
	int i;
@
	check_type_character(&c);
	if (char_font(c) != 0 || char_bits(c) != 0)
		@(return Cnil)
	i = char_code(c);
	if (' ' <= i && i < '\177')     /* ' ' < '\177'  ??? Beppe*/
		@(return Ct)
	@(return Cnil)
@)

@(defun string_char_p (c)
@
	check_type_character(&c);
	if (char_font(c) != 0 || char_bits(c) != 0)
		@(return Cnil)
	@(return Ct)
@)

@(defun alpha_char_p (c)
	int i;
@
	check_type_character(&c);
	if (char_font(c) != 0 || char_bits(c) != 0)
		@(return Cnil)
	i = char_code(c);
	if (isalpha(i))
		@(return Ct)
	else
		@(return Cnil)
@)

@(defun upper_case_p (c)
@
	check_type_character(&c);
	if (char_font(c) != 0 || char_bits(c) != 0)
		@(return Cnil)
	if (isUpper(char_code(c)))
		@(return Ct)
	@(return Cnil)
@)

@(defun lower_case_p (c)
@
	check_type_character(&c);
	if (char_font(c) != 0 || char_bits(c) != 0)
		@(return Cnil)
	if (isLower(char_code(c)))
		@(return Ct)
	@(return Cnil)
@)

@(defun both_case_p (c)
@
	check_type_character(&c);
	if (char_font(c) != 0 || char_bits(c) != 0)
		@(return Cnil)
	@(return `(isUpper(char_code(c)) || isLower(char_code(c))) ? Ct : Cnil`)
@)

/*
	Digitp(i, r) returns the weight of code i
	as a digit of radix r.
	If r > 36 or i is not a digit, -1 is returned.
*/
digitp(int i, int r)
{
	if ('0' <= i && i <= '9' && 1 < r && i < '0' + r)
		return(i - '0');
	if ('A' <= i && 10 < r && r <= 36 && i < 'A' + (r - 10))
		return(i - 'A' + 10);
	if ('a' <= i && 10 < r && r <= 36 && i < 'a' + (r - 10))
		return(i - 'a' + 10);
	return(-1);
}

@(defun digit_char_p (c &optional (r `MAKE_FIXNUM(10)`))
	int d;
@
	check_type_character(&c);
	check_type_non_negative_integer(&r);
	if (type_of(r) == t_bignum)
		@(return Cnil)
	if (char_font(c) != 0 || char_bits(c) != 0)
		@(return Cnil)
	d = digitp(char_code(c), fix(r));
	if (d < 0)
		@(return Cnil)
	@(return `MAKE_FIXNUM(d)`)
@)

@(defun alphanumericp (c)
	int i;
@
	check_type_character(&c);
	if (char_font(c) != 0 || char_bits(c) != 0)
		@(return Cnil)
	i = char_code(c);
	if (isalnum(i))
		@(return Ct)
	else
		@(return Cnil)
@)

bool
char_eq(object x, object y)
{
	return(char_code(x) == char_code(y)
	    && char_bits(x) == char_bits(y)
	    && char_font(x) == char_font(y));
}

@(defun char_eq (c &rest cs)
	int i;
@
	check_type_character(&c);
	narg--;
	for (i = 0;  i < narg;  i++) {
		check_type_character(&((object *)cs)[i]);
		if (!char_eq(c, ((object *)cs)[i]))
			@(return Cnil)
	}
	@(return Ct)
@)

@(defun char_neq (&rest cs)
	int i, j;
@
	if (narg == 0)
		@(return Ct)
	for (i = 0;  i < narg;  i++) {
		check_type_character(&((object *)cs)[i]);
		for (j = 0;  j < i;  j++)
			if (char_eq(((object *)cs)[j], ((object *)cs)[i]))
				@(return Cnil)
	}
	@(return Ct)
@)

int
char_cmp(object x, object y)
{
	if (char_font(x) < char_font(y))
		return(-1);
	if (char_font(x) > char_font(y))
		return(1);
	if (char_bits(x) < char_bits(y))
		return(-1);
	if (char_bits(x) > char_bits(y))
		return(1);
	if (char_code(x) < char_code(y))
		return(-1);
	if (char_code(x) > char_code(y))
		return(1);
	return(0);
}

Lchar_cmp(int narg, int s, int t, object *args)
{
	int i;

	if (narg == 0)
		FEtoo_few_arguments(&narg);
	for (i = 0; i < narg; i++)
		check_type_character(&args[i]);
	for (i = 1; i < narg; i++)
		if (s*char_cmp(args[i], args[i-1]) < t) {
			VALUES(0) = Cnil;
			return(1);
		}
	VALUES(0) = Ct;
	return(1);
}

Lchar_l(int narg, ...)
{ va_list(args); va_start(args, narg);
  RETURN(Lchar_cmp(narg, 1, 1, (object *)args));}
Lchar_g(int narg, ...)
{ va_list(args); va_start(args, narg);
  RETURN(Lchar_cmp(narg,-1, 1, (object *)args));}
Lchar_le(int narg, ...)
{ va_list(args); va_start(args, narg);
  RETURN(Lchar_cmp(narg, 1, 0, (object *)args));}
Lchar_ge(int narg, ...)
{ va_list(args); va_start(args, narg);
  RETURN(Lchar_cmp(narg,-1, 0, (object *)args));}


bool
char_equal(object x, object y)
{
	int i, j;

	i = char_code(x);
	j = char_code(y);
	if (isLower(i))
		i -= 'a' - 'A';
	if (isLower(j))
		j -= 'a' - 'A';
	return(i == j);
}

@(defun char_equal (c &rest cs)
	int i;
@
	check_type_character(&c);
	narg--;
	for (i = 0;  i < narg;  i++) {
		check_type_character(&((object *)cs)[i]);
		if (!char_equal(c, ((object *)cs)[i]))
			@(return Cnil)
	}
	@(return Ct)
@)

@(defun char_not_equal (&rest cs)
	int i, j;
@
	for (i = 0;  i < narg;  i++) {
		check_type_character(&((object *)cs)[i]);
		for (j = 0;  j < i;  j++)
			if (char_equal(((object *)cs)[j], ((object *)cs)[i]))
				@(return Cnil)
	}
	@(return Ct)
@)

int
char_compare(object x, object y)
{
	int i, j;

	i = char_code(x);
	j = char_code(y);
	if (isLower(i))
		i -= 'a' - 'A';
	if (isLower(j))
		j -= 'a' - 'A';
	if (i < j)
		return(-1);
	else if (i == j)
		return(0);
	else
		return(1);
}

Lchar_compare(int narg, int s, int t, object *args)
{
	int i;

	if (narg == 0)
		FEtoo_few_arguments(&narg);
	for (i = 0; i < narg; i++)
		check_type_character(&args[i]);
	for (i = 1; i < narg; i++)
		if (s*char_compare(args[i], args[i-1]) < t) {
			VALUES(0) = Cnil;
			return(1);
		}
	VALUES(0) = Ct;
	return(1);
}

Lchar_lessp(int narg, ...)
{ va_list(args); va_start(args, narg);
  RETURN(Lchar_compare(narg, 1, 1, (object *)args));}
Lchar_greaterp(int narg, ...)
{ va_list(args); va_start(args, narg);
  RETURN(Lchar_compare(narg,-1, 1, (object *)args));}
Lchar_not_greaterp(int narg, ...)
{ va_list(args); va_start(args, narg);
  RETURN(Lchar_compare(narg, 1, 0, (object *)args));}
Lchar_not_lessp(int narg, ...)
{ va_list(args); va_start(args, narg);
  RETURN(Lchar_compare(narg,-1, 0, (object *)args));}


object
coerce_to_character(object x)
{
	while (TRUE) {
		switch (type_of(x)) {
		case t_fixnum:
			{ int i = fix(x);
			  if (0 <= i && i < CHCODELIM)
				return(code_char(i));
			  break;
			}
		case t_character:
			return(x);

		case t_symbol:
		case t_string:
			if (x->st.st_fillp == 1)
				return(code_char(x->ust.ust_self[0]));
			break;
		}
		x = wrong_type_argument(Scharacter, x);
	}
}

@(defun character (x)
@
	@(return `coerce_to_character(x)`)
@)

@(defun char_code (c)
@
	check_type_character(&c);
	@(return `MAKE_FIXNUM(char_code(c))`)
@)

@(defun char_bits (c)
@
	check_type_character(&c);
	@(return `MAKE_FIXNUM(char_bits(c))`)
@)

@(defun char_font (c)
@
	check_type_character(&c);
	@(return `MAKE_FIXNUM(char_font(c))`)
@)

@(defun code_char (c &o (b `MAKE_FIXNUM(0)`) (f `MAKE_FIXNUM(0)`))
	int fc, fb, ff;
@
	check_type_non_negative_integer(&c);
	check_type_non_negative_integer(&b);
	check_type_non_negative_integer(&f);
	if ((type_of(c) == t_bignum) || (type_of(b) == t_bignum) ||
		(type_of(f) == t_bignum))
		@(return Cnil)
	if ((fc = fix(c)) >= CHCODELIM || (fb = fix(b)) >= CHBITSLIM ||
		(ff = fix(f)) >= CHFONTLIM)
		@(return Cnil)
	@(return `(fb == 0 && ff == 0) ?
				code_char(fc) : MAKE_CHARACTER(fc, fb, ff)`)
@)

@(defun make_char (c &o (b `MAKE_FIXNUM(0)`) (f `MAKE_FIXNUM(0)`))
	int code, fb, ff;
@
	check_type_character(&c);
	code = char_code(c);
	check_type_non_negative_integer(&b);
	check_type_non_negative_integer(&f);
	if ((type_of(b) == t_bignum) || (type_of(f) == t_bignum) ||
		(fb = fix(b)) >= CHBITSLIM || (ff = fix(f)) >= CHFONTLIM)
		@(return Cnil)
	@(return `(fb == 0 && ff == 0) ? code_char(code) :
				MAKE_CHARACTER(code, fb, ff)`)
@)

@(defun char_upcase (c)
@
	check_type_character(&c);
	if (char_font(c) != 0 || char_bits(c) != 0)
		@(return c)
	@(return `isLower(char_code(c)) ?
				code_char(char_code(c) - ('a' - 'A')) :
				c`)
@)

@(defun char_downcase (c)
@
	check_type_character(&c);
	if (char_font(c) != 0 || char_bits(c) != 0)
		@(return Cnil)
	@(return `isUpper(char_code(c)) ?
				code_char(char_code(c) + ('a' - 'A')) :
				c`)
@)

int
digit_weight(int w, int r)
{
	if (r < 2 || r > 36 || w < 0 || w >= r)
		return(-1);
	if (w < 10)
		return(w + '0');
	else
		return(w - 10 + 'A');
}

@(defun digit_char (w
		    &optional
		    (r `MAKE_FIXNUM(10)`)
		    (f `MAKE_FIXNUM(0)`))
	int dw;
@
	check_type_non_negative_integer(&w);
	check_type_non_negative_integer(&r);
	check_type_non_negative_integer(&f);
	if (type_of(w) == t_bignum ||
	    type_of(r) == t_bignum ||
	    type_of(f) == t_bignum)
		@(return Cnil)
	dw = digit_weight(fix(w), fix(r));
	if ((dw < 0) || (fix(f) >= CHFONTLIM))
		@(return Cnil)
	@(return `(fix(f)) == 0 ? code_char(dw) : MAKE_CHARACTER(dw, 0, fix(f))`)
@)

@(defun char_int (c)
@
	check_type_character(&c);
	@(return `MAKE_FIXNUM(char_int(c))`)
@)

@(defun int_char (x)
@
	check_type_non_negative_integer(&x);
	if (type_of(x) == t_bignum)
		@(return Cnil)
	@(return `int_char((int)fix(x))`)
@)

@(defun char_name (c)
@
	check_type_character(&c);
	if (char_bits(c) != 0 || char_font(c) != 0)
		@(return Cnil)
	switch (char_code(c)) {
	case '\r':
		@(return STreturn)
	case ' ':
		@(return STspace)
	case '\177':
		@(return STrubout)
		case '\f':
		@(return STpage)
	case '\t':
		@(return STtab)
	case '\b':
		@(return STbackspace)
	case '\n':
		@(return STnewline)
	}
	@(return Cnil)
@)

@(defun name_char (s)
	char c;
@
	s = coerce_to_string(s);
	if (string_equal(s, STreturn))
		c = '\r'; else
	if (string_equal(s, STspace))
		c = ' '; else
	if (string_equal(s, STrubout))
		c = '\177'; else
	if (string_equal(s, STpage))
		c = '\f'; else
	if (string_equal(s, STtab))
		c = '\t'; else
	if (string_equal(s, STbackspace))
		c = '\b'; else
	if (string_equal(s, STlinefeed) || string_equal(s, STnewline))
		c = '\n'; else
		@(return Cnil)
	@(return `code_char(c)`)
@)

#define char_bit_position(n) \
	( Kcontrol == n ? CONTROL_BIT : \
		 ( Kmeta == n ? META_BIT : \
			 ( Ksuper == n ? SUPER_BIT : \
				 ( Khyper == n ? HYPER_BIT : 0))))

@(defun char_bit (c n)
int i;
@
	check_type_character(&c);
	
	i = char_bit_position(n);
	if (i == 0)
		FEerror("Unknown bit attribute ~S.", 1, n);        
	@(return `(char_bits(c) & i) ? Ct : Cnil`)
@)

@(defun set_char_bit (c n v)
  int i, x;
@
	check_type_character(&c);

	i = char_bit_position(n) << (CHCODEFLEN * 8 + 2);
	if (i == 0)
		FEerror("Unknown bit attribute ~S.", 2, c, n);
	x = (int)c;
	if (Null(v))
		x &= ~i;
	   else
		x |= i;
	@(return `(object)x`)
@)

init_character()
{
	make_constant("CHAR-CODE-LIMIT", MAKE_FIXNUM(CHCODELIM));
	make_constant("CHAR-FONT-LIMIT", MAKE_FIXNUM(CHFONTLIM));
	make_constant("CHAR-BITS-LIMIT", MAKE_FIXNUM(CHBITSLIM));

	STreturn = make_simple_string("RETURN");
	enter_mark_origin(&STreturn);
	STspace = make_simple_string("SPACE");
	enter_mark_origin(&STspace);
	STrubout = make_simple_string("RUBOUT");
	enter_mark_origin(&STrubout);
	STpage = make_simple_string("PAGE");
	enter_mark_origin(&STpage);
	STtab = make_simple_string("TAB");
	enter_mark_origin(&STtab);
	STbackspace = make_simple_string("BACKSPACE");
	enter_mark_origin(&STbackspace);
	STlinefeed = make_simple_string("LINEFEED");
	enter_mark_origin(&STlinefeed);

	STnewline = make_simple_string("NEWLINE");
	enter_mark_origin(&STnewline);

	make_constant("CHAR-CONTROL-BIT", MAKE_FIXNUM(CONTROL_BIT));
	make_constant("CHAR-META-BIT", MAKE_FIXNUM(META_BIT));
	make_constant("CHAR-SUPER-BIT", MAKE_FIXNUM(SUPER_BIT));
	make_constant("CHAR-HYPER-BIT", MAKE_FIXNUM(HYPER_BIT));

	Kcontrol = make_keyword("CONTROL");
	Kmeta = make_keyword("META");
	Ksuper = make_keyword("SUPER");
	Khyper = make_keyword("HYPER");
}

init_character_function()
{
	make_function("STANDARD-CHAR-P", Lstandard_char_p);
	make_function("GRAPHIC-CHAR-P", Lgraphic_char_p);
	make_function("STRING-CHAR-P", Lstring_char_p);
	make_function("ALPHA-CHAR-P", Lalpha_char_p);
	make_function("UPPER-CASE-P", Lupper_case_p);
	make_function("LOWER-CASE-P", Llower_case_p);
	make_function("BOTH-CASE-P", Lboth_case_p);
	make_function("DIGIT-CHAR-P", Ldigit_char_p);
	make_function("ALPHANUMERICP", Lalphanumericp);
	make_function("CHAR=", Lchar_eq);
	make_function("CHAR/=", Lchar_neq);
	make_function("CHAR<", Lchar_l);
	make_function("CHAR>", Lchar_g);
	make_function("CHAR<=", Lchar_le);
	make_function("CHAR>=", Lchar_ge);
	make_function("CHAR-EQUAL", Lchar_equal);
	make_function("CHAR-NOT-EQUAL", Lchar_not_equal);
	make_function("CHAR-LESSP", Lchar_lessp);
	make_function("CHAR-GREATERP", Lchar_greaterp);
	make_function("CHAR-NOT-GREATERP", Lchar_not_greaterp);
	make_function("CHAR-NOT-LESSP", Lchar_not_lessp);
	make_function("CHARACTER", Lcharacter);
	make_function("CHAR-CODE", Lchar_code);
	make_function("CHAR-BITS", Lchar_bits);
	make_function("CHAR-FONT", Lchar_font);
	make_function("CODE-CHAR", Lcode_char);
	make_function("MAKE-CHAR", Lmake_char);
	make_function("CHAR-UPCASE", Lchar_upcase);
	make_function("CHAR-DOWNCASE", Lchar_downcase);
	make_function("DIGIT-CHAR", Ldigit_char);
	make_function("CHAR-INT", Lchar_int);
	make_function("INT-CHAR", Lint_char);
	make_function("CHAR-NAME", Lchar_name);
	make_function("NAME-CHAR", Lname_char);
	make_function("CHAR-BIT", Lchar_bit);
	make_function("SET-CHAR-BIT", Lset_char_bit);
}
