;;; -*- Mode: Lisp; Package: XP; Log: xp.log -*-
;;;
;;; **********************************************************************
;;; This code was written as part of the CMU Common Lisp project at
;;; Carnegie Mellon University, and has been placed in the public domain.
;;; If you want to use this code or any part of CMU Common Lisp, please contact
;;; Scott Fahlman or slisp-group@cs.cmu.edu.
;;;
(ext:file-comment
  "$Header: xptest.lisp,v 1.2 91/02/08 13:37:08 ram Exp $")
;;;
;;; **********************************************************************

;------------------------------------------------------------------------ ;
;                Copyright (c) Richard C. Waters, 1988                    ;
;------------------------------------------------------------------------ ;

;  This is a file of test cases to test XP.  Just load it and run the
;function (DO-TESTS).  It prints out identifying numbers of tests
;as it performs them one after another.  When all of the tests have
;been run, a summary line is printed saying how many tests failed.

;  Whenever a test fails for any reason, an error is signalled.  To
;continue testing, call the function (MORE) either within the break,
;or at top level after aborting the execution of a test.  (The latter
;is useful if a test leads to an infinite loop.)  When all of the
;tests have been completed, the variable FAILED-TESTS contains a list
;of the numbers of the tests that failed.  (You can look at the tests
;themselves by evaluating (NTH N TEST-LIST) for any test number.)

;  After running the tests and fixing problems which arise, you may wish
;to run some or all of the tests again.  Calling (DO-TESTS) runs all
;of the tests again.  Calling (DO-FAILED-TESTS) runs just the tests
;which failed the first time.  (The variable TESTS-FAILED is updated
;to reflect the new state of affairs in either case.)  Calling
;(DO-TEST n) runs just the test with the given number. 

#+:symbolics(use-package "CL")
(in-package "USER")
(eval-when (eval load compile) (xp:install-xp :shadow nil))
(proclaim '(special form test-list failed-tests *dt*))
(defvar in-tester nil)
(defvar tests nil)
(defvar compile-tests T)

(defun do-tests ()
  (format T "~% Running the suit of ~S test cases~%" (length test-list))
  (setq tests (do ((i (1- (length test-list)) (1- i))
		   (r nil (cons i r)))
		  ((minusp i) r))
	failed-tests nil)
  (do-many-tests))

(defun do-failed-tests ()
  (format T "~% Running the ~S failed tests~%" (length failed-tests))
  (setq tests failed-tests failed-tests nil)
  (do-many-tests))

(defun do-many-tests ()
  (loop (when (null tests)
	  (setq failed-tests (nreverse failed-tests))
	  (if (zerop (length failed-tests))
	      (format T "~2% XP passed all tests.")
	      (format T "~2% XP failed ~A tests." (length failed-tests)))
	  (return (values)))
	(format T "~A " (car tests))
        (force-output)
	(do-test (pop tests))))

(defun more ()
  (if in-tester (throw 'in-tester nil) (do-many-tests)))

(defun do-test (n)
  (catch 'in-tester
    (let* ((info (nth n test-list))
	   (*break-on-warnings* T)
	   (tester (if (symbolp (car info)) (pop info) 'test-ordinary))
	   (value (cadr info))
	   (pop-if-no-failure nil)
	   (in-tester T))
      (setq form (car info))
      (when (not (member n failed-tests))
	(push n failed-tests)
	(setq pop-if-no-failure T))
      (let ((result (funcall tester (copy-tree form))))
	(when (not (equal result value))
	  (format t "~%form: ~S~% desired value ~S~% actual value ~S~%"
		  form value result)
	  (error "failed test"))
	(when pop-if-no-failure
	  (pop failed-tests))))))  ;doesn't happen when abort out of error.

;Helper funtions for tests.

(defun test-ordinary (form)
  (setq form `(lambda () ,form))
  (funcall (if compile-tests (compile nil form) form)))

(defun deftest (form) (eval form))

(defun etest (form)
  (let ((xp::*testing-errors* T))
     (catch :testing-errors (eval form))))

;This tests things where format and format-xp must always be identical.

(defmacro formats (f-string &rest args)
  `(if (not (string= (format nil ,(caddr f-string) .,args)
		     (format-xp nil ,f-string .,args)))
       'format-xp-and-format-disagree))

;this compares FORMAT-XP with FORMAT only on the lispm, so that bugs
;in FORMAT on another system will not trigger XP bug reports
(defmacro format-xps (&rest stuff)
  #+symbolics
  (let ((format-stuff
	  (mapcar #'(lambda (a)
		      (if (and (consp a)
			       (eq (car a) 'xp::FS))
			  (caddr a)
			  a))
		  stuff)))
    `(let* ((format-value (format nil .,format-stuff))
	    (xp-value (format-xp nil .,stuff)))
       (if (string= format-value xp-value) xp-value
	   `(format-xp-output ,xp-value and
			      format-output ,format-value disagree))))
  #-symbolics`(format-xp nil .,stuff))

(defmacro plet (width miser &body body)
  `(let ((*PRINT-RIGHT-MARGIN* ,width)
	 (*PRINT-MISER-WIDTH* ,miser)
	 (*PRINT-PRETTY* T)
	 (*PRINT-ARRAY* T)
	 (*PRINT-ESCAPE* T)
	 (*PRINT-CASE* :UPCASE)
	 (*PRINT-CIRCLE* NIL)
	 (*PRINT-GENSYM* T)
	 (*PRINT-LEVEL* NIL)
	 (*PRINT-LENGTH* NIL)
	 (*PRINT-LINES* NIL))
     .,body))

(defmacro prints (thing &rest bindings)
  `(plet 50 0
     (let* (,@ bindings
	    (xp-result (with-output-to-string (sss)
			 (format-xp sss #'xp::do-format-w ,thing)))
	    (normal-result (with-output-to-string (sss)
			     (write ,thing :stream sss))))
       (if (not (string= xp-result normal-result))
	   `(print*-output ,xp-result and print-output
			   ,normal-result disagree)))))

(defmacro print*s (thing &rest bindings)
  `(plet 50 0
     (let* (,@ bindings
	    (xp-result (with-output-to-string (sss)
			 (format-xp sss #'xp::do-format-w ,thing)))
	    (normal-result (with-output-to-string (sss)
			     (write ,thing :stream sss))))
       (if (string= xp-result normal-result) normal-result
	   `(print*-output ,xp-result and print-output
			   ,normal-result disagree)))))

(defmacro print*c (thing &rest bindings)
  (push '(*print-circle* T) bindings)
  `(plet 150 0
     (let* ,bindings
       (format-xp nil #'xp::do-format-w (read-from-string ,thing)))))

(defmacro ftest (width miser form &rest bindings)
  `(plet ,width ,miser (let ,bindings
			 (format-xp nil #'xp::do-format-w ,form))))

(setq test-list '(

;the first bunch of tests test the format directives that xp actually
;calls format to do.  the tests therefore just make sure that xp
;does what format does for these.  if format is wrong in a given lisp,
;xp will be too.
  ((formats #"test ~a." "foo"))
  ((formats #"test ~:a." nil))
  ((formats #"test ~:a." nil))
  ((formats #"test ~+021:a." nil))
  ((formats #"test ~4:@a." nil))
  ((formats #"test ~v:@a." 4 nil))
  ((formats #"test ~8,2,3,'-:a." nil))
  ((formats #"test ~8,,#,'-:a." nil 1 2))
  ((formats #"test ~8,,,'-:a." nil))
  ((formats #"test ~8,2,#,v:a." #\- nil 3))
  ((formats #"test ~8,2,#,v@a." #\- nil 3))
  ((formats #"test ~s." "foo"))
  ((formats #"test ~6s." "foo"))
  ((formats #"~:d." 12345))
  ((formats #"~@d." 12345))
  ((formats #"~:b" 22))
  ((formats #"~@b" 22))
  ((formats #"~:o" 22))
  ((formats #"~@o" 22))
  ((formats #"~:x" 22))
  ((formats #"~@x" 22))
  ((formats #"~7:r" 22))
  ((formats #"~7@r" 22))
  ((formats #"~:r" 4))
  ((formats #"~@r" 4))
  ((formats #"~:c" #\f))
  ((formats #"~@c" #\f))
  ((formats #"~6,2f" 3.14159))
  ((formats #"~6,2,1,'*f" 100.0))
  ((formats #"~9,2,1,,'*e" 3.14159))
  ((formats #"~9,2,1,,'*e" 1.1e13))
  ((formats #"~9,2,1,,'*g" 0.0314159))
  ((formats #"~9,2,1,,'*g" 0.314159))
  ((formats #"$~3$" 3.14))
  ((formats #"~A-~10<~A~;~A~>-~A" 1 'foo 'bar 2))
  ((formats #"~A-~V:<~A~;~A~>-~A" 1 10 'foo 'bar 2))
  ((formats #"~A-~10:<~V~~;~A~>-~A" 1 3 'bar 2))
  ((formats #"~A-~10<~(~A~)~;~?~>-~A" 1 'foo "+~A" '(2) 'bar))

;this next set of tests tests format codes which are supposed
;to be supported exactly the same by xp and format, but are actually
;performed directly by xp.  These are compared with the expected
;results, rather than with format so that bugs in format will
;not get reported as bugs in xp.

  ((format-xps #"~d tr~:@p/~d win~:p" 7 1) "7 tries/1 win")
  ((format-xps #"~d tr~:@p/~d win~:p" 1 0) "1 try/0 wins")
  ((format-xps #"~d tr~@p/~d win~p" 1 1 0 0) "1 try/0 wins")

  ((format-xps #"test~%  test") "test
  test")
  ((format-xps #"test~%~%  test") "test

  test")
  ((format-xps #"test~2%  test") "test

  test")
  ((format-xps #"test~V%  test" nil) "test
  test")
  ((format-xps #"test
  test") "test
  test")
  ((format-xps #"test

  test") "test

  test")

  ((format-xps #"test~&  test") "test
  test")
  ((format-xps #"test~&~&  test") "test
  test")
  ((format-xps #"test~2&  test") "test

  test")

#+symbolics
  ((plet 20 0 (format-xp nil #"~|a~3|")) #.(format nil "~|a~3|"))

  ((format-xps #"test ~~ test") "test ~ test")
  ((format-xps #"test ~#~ test") "test  test")
  ((format-xps #"test ~v~ test" 3) "test ~~~ test")
  ((format-xps #"test ~v~ test" nil) "test ~ test")

  ((format-xps #"test~
  test") "testtest")
  ((format-xps #"test~
  ") "test")
  ((format-xps #"test~
 	test") "testtest") ;contains space tab
  ((format-xps #"test~:
  test") "test  test")
  ((format-xps #"test~@
  test") "test
test")

  ((format-xps #"test~ta")        "test a")
  ((format-xps #"test~,1ta")      "test a")
  ((format-xps #"test2~ta")       "test2 a")
  ((format-xps #"-te~5,2ta")      "-te  a")
  ((format-xps #"-tes~5,2ta")     "-tes a")
  ((format-xps #"-test~5,2ta")    "-test  a")
  ((format-xps #"-teste~5,2ta")   "-teste a")
  ((format-xps #"-tester~5,2ta")  "-tester  a")
  ((format-xps #"-te~5,0ta")      "-te  a")
  ((format-xps #"-tes~5,0ta")     "-tes a")
  ((format-xps #"-test~5,0ta")    "-testa")
  ((format-xps #"-teste~5,0ta")   "-testea")
  ((format-xps #"-tester~5,0ta")  "-testera")
  ((format-xps #"-te~0,2ta")      "-te a")
  ((format-xps #"-tes~0,2ta")     "-tes  a")
  ((format-xps #"-test~0,2ta")    "-test a")
  ((format-xps #"-teste~0,2ta")   "-teste  a")
  ((format-xps #"-tester~0,2ta")  "-tester a")
  ((format-xps #"test~8,3ta")     "test    a")
  ((format-xps #"test~V,Vta" 3 3) "test  a")

  ((format-xps #"test~@ta") "test a")
  ((format-xps #"-te~1,2@ta")      "-te a")
  ((format-xps #"-tes~1,2@ta")     "-tes  a")
  ((format-xps #"-test~1,2@ta")    "-test a")
  ((format-xps #"-teste~1,2@ta")   "-teste  a")
  ((format-xps #"-tester~1,2@ta")  "-tester a")
  ((format-xps #"-te~0,2@ta")      "-te a")
  ((format-xps #"-tes~0,2@ta")     "-tesa")
  ((format-xps #"-test~0,2@ta")    "-test a")
  ((format-xps #"-teste~0,2@ta")   "-testea")
  ((format-xps #"-tester~0,2@ta")  "-tester a")
  ((format-xps #"-te~3,0@ta")      "-te   a")
  ((format-xps #"-tes~3,0@ta")     "-tes   a")
  ((format-xps #"-test~3,0@ta")    "-test   a")
  ((format-xps #"-te~3@ta")        "-te   a")
  ((format-xps #"-tes~3@ta")       "-tes   a")
  ((format-xps #"-test~3@ta")      "-test   a")
  ((format-xps #"-te~0,0@ta")      "-tea")
  ((format-xps #"-tes~0,0@ta")     "-tesa")
  ((format-xps #"-test~0,0@ta")    "-testa")
  ((format-xps #"test~8@ta")       "test        a")
  ((format-xps #"test~8,3@ta")     "test        a")
  ((format-xps #"test~V,V@ta" 8 5) "test           a")

  ((format-xps #"~a~a~*~a" 1 2 3 4 5 6 7) "124")
  ((format-xps #"~a~a~2*~a" 1 2 3 4 5 6 7) "125")
  ((format-xps #"~a~a~V*~a" 1 2 3 4 5 6 7) "127")
  ((format-xps #"~a~a~:*~a" 1 2 3 4 5 6 7) "122")
  ((format-xps #"~a~a~2:*~a" 1 2 3 4 5 6 7) "121")
  ((format-xps #"~a~a~V:*~a" 1 2 3 4 5 6 7) "121")
  ((format-xps #"~a~a~1@*~a" 1 2 3 4 5 6 7) "122")
  ((format-xps #"~a~a~3@*~a" 1 2 3 4 5 6 7) "124")
  ((format-xps #"~a~a~V@*~a" 1 2 3 4 5 6 7) "124")

  ;; ~* in various subcontexts tested below.

  ((format-xps #"test~d ~? test" 2 #"(item ~a)" '(4))
   "test2 (item 4) test")
  ((format-xps #"test~d ~? test" 2 "(item ~a)" '(4))
   "test2 (item 4) test")
  ((plet 20 0 (format-xp nil #"test~d ~? test" 2 #"(item ~a~^)" '(4)))
   "test2 (item 4 test")
  ((plet 20 0 (format-xp nil #"test~d ~? ~D test" 2 #"(item ~a~0^)" '(4 5) 6))
   "test2 (item 4 6 test")

  ((format-xps #"tEst~(tesT ~S~) test" 'one)
   "tEsttest one test")
  ((format-xps #"tEst~:(tesT ~S~) test" 'one)
   "tEstTest One test")
  ((format-xps #"tEst~:(tesT~S~) test" 'one)
   "tEstTestone test")
  ((format-xp nil #"tEst~:( tesT~T~S~) test" 'one)
   "tEst Test One test")
  ((format-xps #"tEst~@( tesT ~S~) test" 'one)
   "tEst Test one test")
  ((format-xps #"tEst~:@( tesT ~S~) test" 'one)
   "tEst TEST ONE test")
  ((plet 44 0 (format-xp nil #"~:(~W~)"
		       '(compiler-let ((a (foo 3)) (b (foo 4)) (c 1))
			  (tuz a b))))
   "(Compiler-Let ((A (Foo 3))
               (B (Foo 4))
               (C 1))
  (Tuz A B))")
  ((plet 50 0 (format-xp nil
     #"foo ~@<aa~@;p~:@_ ~:@(hi ~@<bb ~@;q~(~:@_R~:@_S~)~:>~:@_t~)~:>u"))
   "foo aap
    aa HI BB Q
    aa    BB R
    aa    BB S
    aaTu")

  ((format-xps #"~[a~;b~;c~]" 1) "b")
  ((format-xps #"~2[a~;b~;c~]") "c")
  ((format-xps #"~[foo~]" 1) "")
  ((format-xps #"~[a~;b~:;c~]" 10) "c")
  ((format-xps #"~:[a~;b~]" nil) "a")
  ((format-xps #"~:[a~;b~]" 3) "b")
  ((format-xps #"~@[~A~A~] ~A" 1 2 3) "12 3")
  ((format-xps #"~@[~A~A~] ~A" nil 2 3) " 2")

  ((format-xps #"~{~a~^,~}." '(1 2 3 4)) "1,2,3,4.")
  ((format-xps #"~V{~a~^,~}." 2 '(1 2 3 4)) "1,2,.")  
  ((format-xps #"~2{~a~^,~}." '(1)) "1.")
  ((format-xps #"~{foo~:}." '()) "foo.")
  ((format-xps #"~{~a~#,1^,~}." '(1 2 3 4)) "1,2,3.")
  ((format-xps #"~{~a~3@*~^,~a~^,~}." '(1 2 3 4 5 6 7 8 9)) "1,4,5,8,9.")

  ((format-xps #"~:{~a~^,~}." '((1 2) (3 4))) "1,3,.")
  ((format-xps #"~V:{~a~^,~}." 1 '((1 2) (3 4))) "1,.")
  ((format-xps #"~1:{~a~^,~}."  '()) ".")
  ((format-xps #"~:{foo~:}."  '()) "foo.")
  ((format-xps #"~:{~a~1,1:^,~}." '((1 2) (3 4))) "1.")
  ((format-xps #"~:{~a~#,1:^,~}." '((1 2) (3 4))) "1.")
  ((format-xps #"~:{~a~#:^,~}." '((1) (3 4))) "1.")
  ((format-xps #"~:{~a~3@*~^,~a~^,~}." '((1 2 3 4 5) (6 7 8 9))) "1,4,6,9.")

  ((format-xps #"~@{~a~^,~}." 1 2 3 4) "1,2,3,4.")
  ((format-xps #"~@{~a~1,1^,~}." 1 2 3 4) "1.")
  ((format-xps #"~@{~a~1,1,1^,~}." 1 2 3 4) "1.")
  ((format-xps #"~@{~a~2,1,1^,~}." 1 2 3 4) "1,2,3,4,.")
  ((format-xps #"~V@{~a~^,~}." 2 1 2 3 4) "1,2,.")
  ((format-xps #"~2@{~a~^,~}." 1) "1.")
  ((format-xps #"~@{foo~:}.") "foo.")
  ((plet 20 0 (format-xp nil #"~@{~a~#,1^,~} ~A." 1 2 3 4)) "1,2,3 4.")
  ((format-xps #"~@{~a~3@*~^,~a~^,~}." 1 2 3 4 5 6 7 8 9) "1,4,5,8,9.")

  ((format-xps #"~:@{~a~^,~}." '(1 2) '(3 4)) "1,3,.")
  ((format-xps #"~V:@{~a~^,~}." 1 '(1 2) '(3 4)) "1,.")
  ((format-xps #"~1:@{~a~^,~}.") ".")
  ((format-xps #"~:@{foo~:}.") "foo.")
  ((format-xps #"~:@{foo~}.") ".")
  ((format-xps #"~:@{~a~1,1:^,~}." '(1 2) '(3 4)) "1.")
  ((format-xps #"~:@{~a~#,1:^,~}." '(1 2) '(3 4)) "1.")
  ((format-xps #"~:@{~a~#:^,~}." '(1) '(3 4)) "1.")
  ((format-xps #"~:@{~a~3@*~^,~a~^,~}." '(1 2 3 4 5) '(6 7 8 9)) "1,4,6,9.")

;; ~^ tested in the relevant subcontexts above and below


;the following test extended features of standard format codes.

  ((format-xp nil #"test~:ta") "test a")
  ((format-xp nil #"test~8:ta") "test    a")
  ((format-xp nil #"test~V,V:ta" 8 3) "test    a")
  ((format-xp nil #"test~0,3:ta") "test  a")
  ((format-xp nil #"test~0,4:ta") "test    a")
  ((format-xp nil #"test~0,5:ta") "test a")

  ((format-xp nil #"test~@:ta") "test a")
  ((format-xp nil #"test~8@:ta") "test        a")
  ((format-xp nil #"test~V,V@:ta" 8 3) "test        a")
  ((format-xp nil #"test~8,5@:ta") "test           a")
  ((format-xp nil #"test~0,3@:ta") "test  a")
  ((format-xp nil #"test~0,4@:ta") "testa")

  ((format-xp nil #"fo-~<test~:ta~:>") "fo-test a")
  ((format-xp nil #"fo-~<test~8:ta~:>") "fo-test    a")
  ((format-xp nil #"fo-~<test~8,3:ta~:>") "fo-test    a")
  ((format-xp nil #"fo-~<test~0,3:ta~:>") "fo-test  a")
  ((format-xp nil #"fo-~<test~0,4:ta~:>") "fo-test    a")
  ((format-xp nil #"fo-~<test~0,5:ta~:>") "fo-test a")

  ((format-xp nil #"fo-~<test~:@ta~:>") "fo-test a")
  ((format-xp nil #"fo-~<test~8:@ta~:>") "fo-test        a")
  ((format-xp nil #"fo-~<test~8,3:@ta~:>") "fo-test        a")
  ((format-xp nil #"fo-~<test~8,5:@ta~:>") "fo-test           a")
  ((format-xp nil #"fo-~<test~0,3:@ta~:>") "fo-test  a")
  ((format-xp nil #"fo-~<test~0,4:@ta~:>") "fo-testa")

;the following test the special pretty printing directives.

  ((plet 20 0 (format-xp nil #"--~@<~a~a~:>-~a" 1 2 3 4 5)) "--12-NIL")
  ((plet 20 0 (format-xp nil #"~<~@{~A~^ ~}~:>" '(1 2 . 3))) "1 2 . 3")

  ((plet 20 0 (format-xp nil #"tes~<t~ta~:>")) "test a")
  ((plet 20 0 (format-xp nil #"tes~<t~8ta~:>")) "test    a")
  ((plet 20 0 (format-xp nil #"tes~<t~8,3ta~:>")) "test    a")
  ((plet 20 0 (format-xp nil #"tes~<t~0,3ta~:>")) "test  a")
  ((plet 20 0 (format-xp nil #"tes~<t~0,4ta~:>")) "test    a")
  ((plet 20 0 (format-xp nil #"tes~<t~0,5ta~:>")) "test a")

  ((plet 20 0 (format-xp nil #"tes~<t~@ta~:>")) "test a")
  ((plet 20 0 (format-xp nil #"tes~<t~8@ta~:>")) "test        a")
  ((plet 20 0 (format-xp nil #"tes~<t~8,3@ta~:>")) "test        a")
  ((plet 20 0 (format-xp nil #"tes~<t~8,5@ta~:>")) "test           a")
  ((plet 20 0 (format-xp nil #"tes~<t~0,3@ta~:>")) "test  a")
  ((plet 20 0 (format-xp nil #"tes~<t~0,4@ta~:>")) "testa")

  ((plet 20 0 (format-xp nil #"~a~<~a~*~a~:>~a" 1 '(2 3 4 5 6 7) 0)) "1240")
  ((plet 20 0 (format-xp nil #"~a~<~a~0@*~a~:>~a" 1 '(2 3 4 5 6 7) 0)) "1220")
  ((plet 20 0 (format-xp nil #"~a~@<~a~*~a~:>" 1 2 3 4 5 6 7)) "124")
  ((plet 20 0 (format-xp nil #"~a~@<~a~0@*~a~:>" 1 2 3 4 5 6 7)) "122")

  ((plet 20 0 (format-xp nil #"~a~<~a~^~a~:>~a" 1 '(2) 0)) "120")
  ((plet 20 0 (format-xp nil #"~a~<~a~^~a~:>~a" 1 '(2) 0)) "120")
  ((plet 20 0 (format-xp nil #"~a~@<~a~#,4^~a~:>" 1 2 3 4 5 6)) "12")
  ((plet 16 0
     (let ((*print-length* 2))
       (format-xp nil #"---~<~a~^ ~a~^ ~a~:>--" '(12 3456 789))))
   "---12 3456 ...--")
  ((plet 16 0
     (let ((*print-length* 1))
       (format-xp nil #"---~<~a~^ ~a~^ ~a~:>--" '(12 3456 789))))
   "---12 ...--")
  ((plet 16 0
     (with-output-to-string (s)
       (let ((*print-length* 1))
	 (format-xp s #"---~<~a~^ ~a~^ ~a~:>--" '(12 3456 789)))
       (princ " " s)
       (funcall *last-abbreviated-printing*)))
   "---12 ...-- ---12 3456 789--")

  ((plet 16 0
     (let ((*print-length* 2))
       (format-xp nil #"---~@<~a~^ ~a~^ ~a~:>--" 12 3456 789)))
   "---12 3456 ...--")
  ((plet 16 0
     (let ((*print-length* 2))
       (format-xp nil #"---~:@<~a~^ ~a~^ ~a~:>--" 12 3456 789)))
   "---(12 3456 ...)--")
  ((plet 16 0
     (let ((*print-length* 2))
       (format-xp nil #"---~:@<~a~^ ~a~^ ~a~:>~A--" 12 3456 789)))
   "---(12 3456 ...)NIL--")

  ((plet 20 0 (format-xp nil #"test~<a~2%  test~:>b")) "testa

  testb")
  ((plet 20 0 (format-xp nil #"test~<a
  test~:>b")) "testa
  testb")

  ((plet 20 0 (format-xp nil #"test~<a~&  test~:>b")) "testa
  testb")

  ((plet 20 0 (format-xp nil #"~a~:@<~:>" 1 2 4 5)) "1()")
  ((plet 20 0 (format-xp nil #"~a~:@<~a~a~:>" 1 2 4 5)) "1(24)")
  ((plet 20 0 (format-xp nil #"~a~@<+~;~a~a~:>" 1 2 4 5)) "1+24")
  ((plet 20 0 (format-xp nil #"~a~:@<+~;~a~a~:>" 1 2 4 5)) "1+24)")
  ((plet 20 0 (format-xp nil #"~a~@<+~;~a~a~;*~:>" 1 2 4 5)) "1+24*")
  ((plet 20 0 (format-xp nil #"~a~:@<+~;~a~a~;*~:>" 1 2 4 5)) "1+24*")
   
  ((plet 50 0
     (format-xp nil #"foo ~@<++~@;1~:@_2~:@_3~:>4"))
   "foo ++1
    ++2
    ++34")
  ((plet 50 0
     (format-xp nil #"foo ~@<++~@;1~:@_ hi ~@<##~@;1~:@_2~:@_3~:>~:@_3~:>4"))
   "foo ++1
    ++ hi ##1
    ++    ##2
    ++    ##3
    ++34")
  ((plet 50 0
     (format-xp nil #"foo ~@<++~@;1~:@_2 ~S~:@_3~:>4" "testing
linebreaks"))
"foo ++1
    ++2 \"testing
    ++linebreaks\"
    ++34")

  ((plet 18 0 (format-xp nil #"---~:<~a ~_~a ~_~a~:>--" '(12 3456 789)))
   "---(12 3456 789)--")
  ((plet 17 0 (format-xp nil #"---~:<~a ~_~a ~_~a~:>--" '(12 3456 789)))
   "---(12
    3456
    789)--")
  ((plet 17 0 (format-xp nil #"---~<<~;~a ~_~a ~_~a~;>~:>--" '(12 3456 789)))
   "---<12
    3456
    789>--")
  ((plet 17 0 (format-xp nil #"---~<<~@;~a ~_~a ~_~a~;>~:>--" '(12 3456 789)))
   "---<12
   <3456
   <789>--")
  ((plet 16 0 (format-xp nil #"---~<<~@;~a ~_~a ~_~a~:>--" '(12 3456 789)))
   "---<12
   <3456
   <789--")

  ((plet 15 0
     (format-xp nil #"---~<12	3456 789~:@>--")) ;note tab after "12"
   "---12	3456
   789--")
  ((plet 15 0 (format-xp nil #"---~<~a ~a ~a~:@>--" '(12 3456 789)))
   "---12 3456
   789--")
  ((plet 25 0 (format-xp nil #"---~<~a	~a-~a~@:>--" '(12 3456 789))) ;note tab char
   "---12	3456-789--")

  ((plet 15 0
     (let ((*print-level* 3))
       (format-xp nil #"0~:@<1~:@<2~:@<3~:>~:>~:>")))
   "0(1(2(3)))")
  ((plet 15 0
     (let ((*print-level* 2))
       (format-xp nil #"0~:@<1~:@<2~:@<3~:>~:>~:>")))
   "0(1(2#))")
  ((plet 15 0
     (let ((*print-level* 1))
       (format-xp nil #"0~:@<1~:@<2~:@<3~:>~:>~:>")))
   "0(1#)")  
  ((plet 15 0
     (let ((*print-level* 0))
       (format-xp nil #"0~:@<1~:@<2~:@<3~:>~:>~:>")))
   "0#")
  ((plet 15 0
     (with-output-to-string (s)
       (let ((*print-level* 1))
	 (format-xp s #"0~:@<1~:@<2~:@<3~:>~:>~:>"))
       (format-xp s " ")
       (format-xp s #"0~:@<1~:@<2~:@<3~:>~:>~:>")))
   "0(1#) 0(1(2(3)))")
  ((plet 50 0
     (let ((*print-level* 1))
       (format-xp nil #"~:<~W~:@<~W~:>~:>" '(0 1 2 3 4))))
   "(0#)")

#+symbolics
  (deftest
    (plet 100 0
      (let ((*print-level* 3))
	(defstruct tester2 a b)
	(format-xp nil #"0~:@<1~W~:>" (make-tester2 :A '(1 (2 (3)))))))
   "0(1#S(TESTER2 :A (1 #) :B NIL))")


  ((plet 16 0 (format-xp nil #"---~<~a ~_~a ~_~a~:>--" '(12 3456 789)))
   "---12 3456 789--")
  ((plet 15 0 (format-xp nil #"---~<~a ~_~a ~_~a~:>--" '(12 3456 789)))
   "---12
   3456
   789--")

  ((plet 16 0 (format-xp nil #"---~<~a ~:_~a ~:_~a~:>--" '(12 3456 789)))
   "---12 3456 789--")
  ((plet 11 0 (format-xp nil #"---~<~a ~:_~a ~:_~a~:>--" '(12 3456 789)))
   "---12 3456
   789--")
  ((plet 10 0 (format-xp nil #"---~<~a ~:_~a ~:_~a~:>--" '(12 3456 789)))
   "---12
   3456
   789--")
  ((plet 50 0 (format-xp nil #"---~<~a ~<<<~:@_>>~:>~:_~a~:>--" '(12 () 789)))
   "---12 <<
      >>
   789--")
  ((plet 50 0 (format-xp nil #"---~<~a ~:_~<<<~:@_>>~:>~:_~a~:>--"
		       '(12 () 789)))
   "---12
   <<
   >>
   789--")

  ((plet 16 0 (format-xp nil #"---~<~a ~@_~a ~:_~a~:>--" '(12 3456 789)))
   "---12 3456 789--")
  ((plet 11 0 (format-xp nil #"---~<~a ~@_~a ~:_~a~:>--" '(12 3456 789)))
   "---12 3456
   789--")
  ((plet 11 nil (format-xp nil #"---~<~a ~@_~a ~:_~a~:>--" '(12 3456 789)))
   "---12 3456
   789--")
  ((plet 11 20 (format-xp nil #"---~<~a ~@_~a ~:_~a~:>--" '(12 3456 789)))
   "---12
   3456
   789--")

  ((plet 25 0 (format-xp nil #"---~<~a ~:@_~a ~_~a~:>--" '(12 3456 789)))
   "---12
   3456
   789--")
  ((plet 13 0 (format-xp nil #"---~<~a ~:@_~a ~:_~a~:>--" '(12 3456 789)))
   "---12
   3456 789--")
  ((plet 12 0 (format-xp nil #"---~<~a ~:@_~a ~:_~a~:>--" '(12 3456 789)))
   "---12
   3456
   789--")

  ((plet 15 0 (format-xp nil #"---~<~a~1I ~_~a ~_~a~:>--" '(12 3456 789)))
   "---12
    3456
    789--")
 ((plet 15 0 (format-xp nil #"---~<~a~-2I ~_~a ~_~a~:>--" '(12 3456 789)))
   "---12
 3456
 789--")
 ((plet 15 0 (format-xp nil #"---~<~a~VI ~_~a ~_~a~:>--" '(12 -2 3456 789)))
   "---12
 3456
 789--")
  ((plet 15 0 (format-xp nil #"---~<~a~:I ~_~a ~_~a~:>--" '(12 3456 789)))
   "---12
     3456
     789--")
  ((plet 15 20 (format-xp nil #"---~<~a~:I ~_~a ~_~a~:>--" '(12 3456 789)))
   "---12
   3456
   789--")
  ((plet 15 0 (format-xp nil #"---~<~a ~_~a~-1:I ~_~a~:>--" '(12 3456 789)))
   "---12
   3456
      789--")
  ((plet 15 0 (format-xp nil #"---~<~a ~_~a~V:I ~_~a~:>--" '(12 3456 -1 789)))
   "---12
   3456
      789--")

  ((plet 16 0
     (let ((*print-length* 3))
       (format-xp nil #"---~<~a ~_~a ~_~a~:>--" '(12 3456 789))))
   "---12 3456 789--")

  ((plet 15 0
     (let ((*print-lines* 3))
       (format-xp nil #"---~<~a ~_~a ~_~a~:>--" '(12 3456 789))))
   "---12
   3456
   789--")
  ((plet 15 0
     (let ((*print-lines* 2))
       (format-xp nil #"---~<~a ~_~a ~_~a~:>--" '(12 3456 789))))
   "---12
   3456 ---")
  ((plet 15 0
     (let ((*print-lines* 1))
       (format-xp nil #"---~<~a ~_~a ~_~a~:>--" '(12 3456 789))))
   "---12 ---")
  ((plet 15 0
     (with-output-to-string (s)
       (let ((*print-lines* 1))
	 (format-xp s #"---~<~a ~_~a ~_~a~:>--" '(12 3456 789)))
       (terpri s)
       (funcall *last-abbreviated-printing*)))
   "---12 ---
---12
   3456
   789--")

  ((plet 15 0 (format-xp nil #"---~/fill-style/--" '(12 3456 789)))
   "---12 3456
   789--")
  ((plet 15 0 (format-xp nil #"---~:/fill-style/--" '(12 3456 789)))
   "---(12 3456
    789)--")
  ((plet 15 0 (format-xp nil #"---~:/fill-style/--" '12))
   "---12--")
  ((plet 15 0 (let ((*print-level* 4) (*print-length* 2))
		(format-xp nil #"---~:/fill-style/--" '(12 3456 789))))
   "---(12 3456
    ...)--")
  ((plet 25 0 (let ((*print-level* 4) (*print-length* 2))
		(format-xp nil #"---~/fill-style/--" '(12 3456 789))))
   "---12 3456 ...--")
  ((plet 15 0 (let ((*print-level* 0))
		(format-xp nil #"---~:/fill-style/--" '(12 3456 789))))
   "---#--")
  ((plet 15 0 (let ((*print-level* 0))
		(format-xp nil #"---~/fill-style/--" '(12 3456 789))))
   "---#--")

  ((plet 15 0 (format-xp nil #"---~/linear-style/--" '(12 3456 789)))
   "---12
   3456
   789--")
  ((plet 15 0 (format-xp nil #"---~:/linear-style/--" '(12 3456 789)))
   "---(12
    3456
    789)--")
  ((plet 15 0 (format-xp nil #"---~:/linear-style/--" '12))
   "---12--")
  ((plet 15 0 (let ((*print-level* 4) (*print-length* 2))
		(format-xp nil #"---~:/linear-style/--" '(12 3456 789))))
   "---(12
    3456
    ...)--")
  ((plet 25 0 (let ((*print-level* 4) (*print-length* 2))
		(format-xp nil #"---~/linear-style/--" '(12 3456 789))))
   "---12 3456 ...--")
  ((plet 15 0 (let ((*print-level* 0))
		(format-xp nil #"---~:/linear-style/--" '(12 3456 789))))
   "---#--")
  ((plet 15 0 (let ((*print-level* 0))
		(format-xp nil #"---~/linear-style/--" '(12 3456 789))))
   "---#--")

  ((plet 100 0 (format-xp nil #"---~5/tabular-style/--"
		       '(12 3456 789 22 45656 78)))
   "---12   3456 789  22   45656     78--")
  ((plet 100 0
     (let ((*print-length* 3))
       (format-xp nil #"---~5/tabular-style/--"
		  '(12 3456 789 22 456 78))))
   "---12   3456 789 ...--")
  ((plet 21 0 (format-xp nil #"---~5/tabular-style/--"
		       '(12 3456 789 22 456 78)))
   "---12   3456 789
   22   456  78--")
  ((plet 21 0 (format-xp nil #"---~5:/tabular-style/--"
		       '(12 3456 789 22 456 78)))
   "---(12   3456 789
    22   456  78)--")
  ((plet 100 0
     (let ((*print-length* 3))
       (format-xp nil #"---~5:/tabular-style/--"
		  '(12 3456 789 22 456 78))))
   "---(12   3456 789 ...)--")
  ((plet 41 0 (format-xp nil #"---~V/tabular-style/--"
		       nil '(12 3456 789 22 456 78)))
   "---12              3456
   789             22
   456             78--")
  ((plet 21 0 (format-xp nil #"---~5:/tabular-style/--" ()))
   "---()--")
  ((plet 21 0 (format-xp nil #"---~5:/tabular-style/--" 12))
   "---12--")
  ((plet 21 0 (let ((*print-level* 0)) (format-xp nil #"---~5/tabular-style/--" ())))
   "---#--")
  ((plet 21 0 (let ((*print-level* 0)) (format-xp nil #"---~5:/tabular-style/--" ())))
   "---#--")

  ((plet 90 0
     (let ((*print-escape* nil))
       (format-xp nil #"~W" "foo")))
   "foo")
  ((plet 90 0
     (let ((*print-escape* T))
       (format-xp nil #"~W" "foo")))
   "\"foo\"")

  ((plet 10 0
     (let ((*print-pretty* nil))
       (format-xp nil #"~W" '#(12 3456 789 22 456 78))))
   "#(12 3456 789 22 456 78)")
  ((plet 15 0
     (let ((*print-length* 5))
       (format-xp nil #"~W" '#(12 3456 789 22 456 78))))
   "#(12 3456 789
  22 456 ...)")
  ((plet 15 0
     (let ((*print-length* 0))
       (format-xp nil #"~W" '#(12 3456 789 22 456 78))))
   "#(...)")
  ((plet 15 0
     (let ((*print-level* 0))
       (format-xp nil #"~W" '#(12 3456 789 22 456 78))))
   "##")
  ((plet 10 0
     (let ((*print-pretty* nil))
       (format-xp nil #"~W" '#())))
   "#()")

  ((plet 10 0
     (let ((*print-pretty* nil))
       (format-xp nil #"~W" '#2A((12 3456 789) (22 456 78)))))
   "#2A((12 3456 789) (22 456 78))")
  ((plet 20 0
     (let ((*print-pretty* T))
       (format-xp nil #"~W" '#2A((12 3456 789) (22 456 78)))))
   "#2A((12 3456 789)
    (22 456 78))")
  ((plet 17 0
     (let ((*print-pretty* T))
       (format-xp nil #"~W" '#2A((12 3456 789) (22 456 78))))) 
   "#2A((12 3456
     789)
    (22 456 78))")
  ((plet 10 0
     (let ((*print-pretty* nil))
       (format-xp nil #"~W" '#0Afoo)))
   "#0A FOO")
  ((plet 30 0
     (let ((*print-pretty* T))
       (format-xp nil #"~W" '#3A(((1 12) (1 3456) (1 789))
			       ((1 22) (1 456) (1 78))))))
   "#3A(((1 12) (1 3456) (1 789))
    ((1 22) (1 456) (1 78)))")
  ((plet 30 0
     (let ((*print-pretty* T))
       (format-xp nil #"~W" '#3A((() ()) (() ())))))
   "#3A((() ()) (() ()))")
  ((plet 30 0
     (let ((*print-pretty* T) (*print-level* 2))
       (format-xp nil #"~W" '#3A(((1 12) (1 3456) (1 789))
			       ((1 22) (1 456) (1 78))))))
   "#3A((# # #) (# # #))")
  ((plet 30 0
     (let ((*print-pretty* T) (*print-level* 0))
       (format-xp nil #"~W" '#3A(((1 12) (1 3456) (1 789))
			       ((1 22) (1 456) (1 78))))))
   "#3A#")
  ((plet 30 0
     (let ((*print-pretty* T) (*print-length* 4))
       (format-xp nil #"~W" '#3A(((1 12 1 1 1 1) (1 3456 1 1 1 1))
			       ((1 22 1 1 1 1) (1 456 1 1 1 1))))))
     "#3A(((1 12 1 1 ...)
     (1 3456 1 1 ...))
    ((1 22 1 1 ...)
     (1 456 1 1 ...)))")
  ((plet 30 0
     (let ((*print-pretty* T) (*print-length* 0))
       (format-xp nil #"~W" '#3A(((1 12 1 1 1 1) (1 3456 1 1 1 1))
			       ((1 22 1 1 1 1) (1 456 1 1 1 1))))))
     "#3A(...)")

  ((plet 20 0
     (flet ((foo (xp obj colon atsign &optional (n 3))
	    (if colon (push ":" obj))
	    (if atsign (push "@" obj))
	    (format-xp xp #"~V{~A~}" n obj)))
     (format-xp nil #"-~/foo/-" '(1 2 3 4)))) "-123-")
  ((plet 20 0
     (flet ((foo (xp obj colon atsign &optional (n 3))
	    (if colon (push ":" obj))
	    (if atsign (push "@" obj))
	    (format-xp xp #"~V{~A~}" n obj)))
     (format-xp nil #"-~4/foo/-" '(1 2 3 4)))) "-1234-")
  ((plet 20 0
     (flet ((foo (xp obj colon atsign &optional (n 3))
	    (if colon (push ":" obj))
	    (if atsign (push "@" obj))
	    (format-xp xp #"~V{~A~}" n obj)))
     (format-xp nil #"-~#/foo/-" '(1 2 3 4)))) "-1-")
  ((plet 20 0
     (flet ((foo (xp obj colon atsign &optional (n 3))
	      (if colon (push ":" obj))
	      (if atsign (push "@" obj))
	      (format-xp xp #"~V{~A~}" n obj)))
       (format-xp nil #"-~V/foo/-" 2 '(1 2 3 4)))) "-12-")
  ((plet 20 0
     (flet ((foo (xp obj colon atsign &optional (n 3))
	      (if colon (push ":" obj))
	      (if atsign (push "@" obj))
	      (format-xp xp #"~V{~A~}" n obj)))
       (format-xp nil #"-~:@/foo/-" '(1 2 3 4)))) "-@:1-")
  ((plet 20 0
     (flet ((foo (xp obj colon atsign &optional (n 3))
	      (if colon (push ":" obj))
	      (if atsign (push "@" obj))
	      (format-xp xp #"~V{~A~}" n obj)))
       (let ((*package* (find-package "XP")))
	 (format-xp nil #"-~/foo/-" '(1 2 3 4))))) "-123-")
  ((plet 20 0
     (flet ((foo (xp obj colon atsign &optional (n 3))
	      (if colon (push ":" obj))
	      (if atsign (push "@" obj))
	      (format-xp xp #"~V{~A~}" n obj)))
       (let ((*package* (find-package "XP")))
	 (format-xp nil #"-~/user::foo/-" '(1 2 3 4))))) "-123-")
  ((plet 20 0
     (flet ((bar (xp &rest objects)
	      (format-xp xp #"~{~A~}" objects)))
       (format-xp nil #"-~?-" #'bar '(1 2 3 4)))) "-1234-")

;tests of xp's special printing functions.

  ((plet 10 0
     (let ((*print-pretty* nil))
       (with-output-to-string (s)
	 (write-xp '(setq a 8 c "2") :stream s :base 8 :lines 4))))
   "(SETQ A 10 C \"2\")")
  ((plet 10 0
     (let ((*print-pretty* nil))
       (with-output-to-string (s)
	 (write-xp '(setq a 8 c "2") :stream s :escape nil :pretty T))))
   "(SETQ A 8
      C 2)")
  ((plet 10 0
     (flet ((bar (xp list &rest stuff) (declare (ignore stuff))
	      (write-xp list :stream xp :length 4 :pretty nil)))
       (format-xp nil #"-~/bar/-" '(setq a 8 c "2"))))
   "-(SETQ A 8 C ...)-")
  ((plet 14 0
     (flet ((bar (xp list &rest stuff) (declare (ignore stuff))
	      (write-xp list :stream xp :length 4)))
       (format-xp nil #"-~/bar/-" '(setq a 8 c "2"))))
   "-(SETQ A 8
       C ...)-")

  ((plet 10 0
     (let ((*print-pretty* nil))
       (with-output-to-string (s)
	 (prin1-xp "2" s)
	 (fresh-line-xp s)
	 (write-line-xp "This is a test" s :start 2)
	 (terpri-xp s)
	 (write-string-xp "This is a test" s :end 7)
	 (pprint-xp '(setq a b c d) s) 
         (write-string-xp "more
tests" s)
	 (print-xp "2" s)
	 (write-char-xp #\a s)
	 (write-char-xp #\newline s)
	 (fresh-line-xp s)
	 (princ-xp "2" s))))
   "\"2\"
is is a test

This is
\(SETQ A B
      C D)more
tests
\"2\" a
2")
  ((plet 10 0
     (let ((*print-pretty* t))
       (with-output-to-string (s)
	 (prin1-xp "2" s)
	 (fresh-line-xp s)
	 (write-line-xp "This is a test" s :start 2)
	 (terpri-xp s)
	 (write-string-xp "This is a test" s :end 7)
	 (pprint-xp '(setq a b c d) s)
         (write-string-xp "more
tests" s)
	 (print-xp "2" s)
	 (write-char-xp #\a s)
	 (write-char-xp #\newline s)
	 (fresh-line-xp s)
	 (princ-xp "2" s))))
   "\"2\"
is is a test

This is
\(SETQ A B
      C D)more
tests
\"2\" a
2")
  ((plet 14 0
     (flet ((bar (s item &rest stuff) (declare (ignore stuff))
	      (prin1-xp item s)
	      (fresh-line-xp s)
	      (write-line-xp "This is a test" s :start 2)
	      (terpri-xp s)
	      (write-string-xp "This is a test" s :end 7)
	      (pprint-xp '(setq a b c d) s)
              (write-string-xp "more
tests" s)
	      (print-xp item s)
	      (write-char-xp #\a s)
	      (write-char-xp #\newline s)
	      (fresh-line-xp s)
	      (princ-xp item s)))
       (format-xp nil #"-~/bar/-" "2")))
   "-\"2\"
is is a test

This is
\(SETQ A B
      C D)more
tests
\"2\" a
2-")

  #+symbolics
  ((plet 100 0
     (let ((*print-pretty* t))
       (with-output-to-string (s)
	 (prin1-xp 2 s)
	 (finish-output-xp s)
	 (prin1-xp 2 s)
	 (force-output-xp s)
	 (prin1-xp 2 s)
	 (clear-output-xp s))))
   "222") ;note this is very implementation dependent.	 
  ((plet 100 0
     (flet ((bar (s item &rest stuff) (declare (ignore stuff))
	      (format-xp s #",~_~A" item)
	      (finish-output-xp s)
	      (format-xp s #",~_~A" item)
	      (force-output-xp s)
	      (format-xp s #",~_~A" item)
	      (clear-output-xp s)))
       (format-xp nil #"-~<a ~_aa~:>+~_~:<b~/bar/~:>-" () '(2))))
   "-a aa+
\(b,
 2,
 2)-")

  ((plet 14 0 (progn (setq xp::*format-string-cache* T) (format-xp nil "~A" 4))) "4")
  ((plet 14 0 (format-xp nil "~10<foo~>" 4)) "       foo")
  ((plet 14 0 (format-xp nil "~10<foo~@>" 4)) "       foo")
  ((plet 14 0 (format-xp nil "~@<foo~:>" 4)) "foo")
  ((plet 14 0 (format-xp nil "~@<foo~:@>" 4)) "foo")
  ((plet 14 0 (format-xp nil "~w" 4)) "4")
  ((plet 14 0
     (let ((xp::*format-string-cache* nil))
       (format-xp nil "~w" 4))) "4")
  ((plet 14 0
     (let ((string "~W"))
       (list (format-xp nil string 4) (format-xp nil string 5)))) ("4" "5"))
  ((plet 20 0
     (flet ((bar (xp &rest objects)
	      (format-xp xp "~{~A~}" objects)))
       (format-xp nil #"-~?-" #'bar '(1 2 3 4)))) "-1234-")

  ((with-output-to-string (*standard-output*) (prin1-xp 44)) "44")
  ((with-output-to-string (*standard-output*) (prin1-xp 44 nil)) "44")
  ((with-output-to-string (*terminal-io*) (prin1-xp 44 T)) "44")

  ((plet 100 0 (princ-to-string-xp '(setq a "2" b 8))) "(SETQ A 2 B 8)")
  ((plet 100 0
     (let ((*print-pretty* nil))
       (prin1-to-string-xp '(setq a "2" b 8)))) "(SETQ A \"2\" B 8)")
  ((plet 100 0 (write-to-string-xp '(setq a "2" b 8) :base 8 :right-margin 13))
   "(SETQ A \"2\"
      B 10)")

  (deftest
    (progn (defstruct-xp foo (a 1) (b 2))
	   (plet 10 0 (prin1-to-string-xp (make-foo))))
   "#S(FOO :A 1
       :B 2)")
  (deftest
    (progn (defstruct-xp (foo00 (:conc-name nil)) (a 1) (b 2))
	   (plet 10 0 (prin1-to-string-xp (make-foo00))))
   "#S(FOO00 :A 1
         :B 2)")
  (deftest
    (progn (defstruct-xp (foo0 (:constructor mf)) (a 1) (b 2))
	   (plet 11 0 (prin1-to-string-xp (mf))))
   "#S(FOO0 :A 1
        :B 2)")
  (deftest
    (progn (defstruct-xp (foo1 (:conc-name tuz)) a (b 2))
	   (plet 16 0 (prin1-to-string-xp (make-foo1 :a '(1 2 3)))))
   "#S(FOO1 :A (1 2
            3)
        :B 2)")
  (deftest
    (progn (defun foo2p (ob s d)
	     (if (and *print-level* (not (< d *print-level*))) (princ-xp "#" s)
		 (format-xp s #"#<foo2 ~_is ~A>" (aa ob))))
	   (defstruct-xp (foo2 (:conc-name nil) (:print-function foo2p)) (aa 3))
	   (plet 13 0 (list (let ((*printe-level* 1))
			      (format-xp nil #"~W---" (make-foo2)))
			    (let ((*print-level* 0))
			      (format-xp nil #"~W---" (make-foo2)))
			    (with-output-to-string (s)
			      (prin1 (make-foo2) s)
			      (princ "---" s)))))
   ("#<foo2
is 3>---" "#---" "#<foo2 is 3>---"))
  (deftest
    (progn (defstruct-xp (foo3 (:type list)) (a 1) (b 2))
	   (prints (make-foo3))))
  (deftest
    (progn (defstruct-xp (foo4 (:include foo2)) (e 1))
	   (prints (make-foo4))))

;Tests of things about dispatching

  ((progn (setq *dt* (copy-print-dispatch nil)) nil))
  (deftest
    (let ((*print-dispatch* *dt*))
      (define-print-dispatch null () #"()")
      (plet 20 0 (format-xp nil #"~W" nil)))
   "()")
  (deftest
    (let ((*print-dispatch* *dt*))
      (define-print-dispatch (cons (not (satisfies numberp))
				   (cons (member a b c))) ()
			     #"~{~a+~a~}")
      (plet 20 0 (format-xp nil #"~W" '(a a))))
   "A+A")
  (deftest
    (let ((*print-dispatch* *dt*))
      (define-print-dispatch (cons (cons) (cons (member a))) ((:priority 1))
			     #"~{~a-~a~}")
      (plet 20 0 (format-xp nil #"~W" '((a) a))))
   "(A)-A")
  (deftest
    (let ((*print-dispatch* *dt*))
      (define-print-dispatch hash-table ()
			     #"foof")
      (plet 20 0 (format-xp nil #"~W" (make-hash-table))))
   "foof")
;;; **** Trouble ****
  (deftest
    (let ((*print-dispatch* *dt*))
      (define-print-dispatch (and integer (satisfies evenp))
		       ((:priority 3) (:table *dt*)) #"+~D")
      (plet 20 0 (format-xp nil #"~W" '(1 2 3 4))))
   "(1 +2 3 +4)")
  (deftest
    (let ((*print-dispatch* *dt*))
      (define-print-dispatch (and (member 10 11 12))
		       ((:priority 20) (:table *dt*)) #"**~D")
      (plet 20 0 (format-xp nil #"~W" '(1 12 3 11))))
   "(1 **12 3 **11)")
  (deftest
    (let ((*print-dispatch* *dt*))
      (define-print-dispatch (and (member 10 11 12)) ((:table *dt*)) nil)
      (plet 20 0 (format-xp nil #"~W" '(1 12 3 11))))
   "(1 +12 3 11)")
  (deftest
    (let ((*print-dispatch* *dt*))
      (define-print-dispatch (vector * 4) ((:table *dt*)) (xp obj)
	(format-xp xp #"--~S" (coerce obj 'list)))
      (plet 20 0 (format-xp nil #"~W" '#(k l d a))))
   "--(K L D A)")
  (deftest
    (let ((*print-dispatch* *dt*))
      (define-print-dispatch (cons (member unwind-protect)) ((:table *dt*))
	#'(lambda (xp list)
	    (xp::print-fancy-fn-call xp list '(0 3 1 nil))))
      (plet 20 0 (format-xp nil #"~W" '(unwind-protect (open f) (print errormsg)))))
   "(UNWIND-PROTECT
    (OPEN F)
 (PRINT ERRORMSG))")
  (deftest
    (let ((*print-dispatch* *dt*))
      (define-print-dispatch (cons (member unwind-protect)) ())
      (plet 20 0 (format-xp nil #"~W" '(unwind-protect (open f) (print errormsg)))))
   "(UNWIND-PROTECT
    (OPEN F)
  (PRINT ERRORMSG))")
#-symbolics
  (deftest
    (let ((*print-dispatch* *dt*))
      (define-print-dispatch (cons (member unwind-protect)) () nil)
      (plet 30 1 (format-xp nil #"~W" '(unwind-protect (open f) (print errormsg)))))
   "(UNWIND-PROTECT (OPEN F)
                (PRINT ERRORMSG))") ;note zwei: offset hash table
  (deftest
    (let ((*print-dispatch* *dt*))
      (define-print-dispatch cons ((:priority 1)) #"zot ~{~A ~}")
      (plet 20 0 (format-xp nil #"~W" '(setq a b))))
   "zot SETQ A B ")
  (deftest
    (let ((*print-dispatch* *dt*))
      (define-print-dispatch cons ((:priority 1)))
      (plet 30 0 (format-xp nil #"~W" '(unwind-protect (open f) (print errormsg)))))
   "(UNWIND-PROTECT (OPEN F)
 (PRINT ERRORMSG))")
  (deftest
    (let ((*print-dispatch* *dt*))
      (define-print-dispatch (cons) ((:priority 2)) #"zoz ~{~A ~}")
      (plet 100 0 (format-xp nil #"~W" '(foo bar))))
   "zoz FOO BAR ")
  (deftest
    (let ((*print-dispatch* *dt*))
      (define-print-dispatch (cons integer) ((:priority 3)) #"int ~{~A ~}")
      (plet 100 0 (format-xp nil #"~W" '(3 bar))))
   "int 3 BAR ")
  (deftest
    (let ((*print-dispatch* *dt*))
      (define-print-dispatch (cons (member a b)) ((:priority 3)) #"pip ~{~A ~}")
      (plet 100 0 (format-xp nil #"~W" '(a bar))))
   "pip A BAR ")
  (deftest
    (let ((*print-dispatch* *dt*))
      (define-print-dispatch (cons (member)) ((:priority 4)) #"pop ~{~A~}")
      (plet 100 0 (format-xp nil #"~W" '(a bar))))
   "pip A BAR ")
  (deftest
    (let ((*print-dispatch* *dt*))
      (define-print-dispatch foo () (xp obj)
	(format-xp xp #"foo-~A-~A" (foo-a obj) (foo-b obj)))
      (let ((data (make-foo :a 11 :b 21)))
	(plet 20 0 (format-xp nil #"~W" data))))
    "foo-11-21")
  (deftest
    (let ((*print-dispatch* *dt*))
      (define-print-dispatch foo ())
      (let ((data (make-foo :a 10 :b 20)))
	(plet 20 0 (format-xp nil #"~W" data))))
    "#S(FOO :A +10 :B +20)")

;tests of formats for various special forms.

  ((progn (setq *print-dispatch* (copy-print-dispatch)) nil))
  ((ftest 15 0 '#"~W foo ~A") "#\"~W foo ~A\"")
#+symbolics
  ((ftest 15 0 '(xp::fs "FOO" 2)) "(XP:FS \"FOO\" 2)")
#+symbolics
  ((ftest 15 0 '(xp::fs 1 2)) "(XP:FS 1 2)")

  ((ftest 15 0 '(cons aaaaaa bbbbb))
   "(CONS AAAAAA
      BBBBB)")
  ((ftest 15 0 '(cons aaaaaa (cons a b)) (*print-level* 1))
   "(CONS AAAAAA #)")
  ((ftest 15 0 '(array-has-fill-pointer-p aaaaaa BBBB))
   "(ARRAY-HAS-FILL-POINTER-P AAAAAA
                          BBBB)")

  ((ftest 100 0 '(block foo (if (null y) (return T)) x))
   "(BLOCK FOO (IF (NULL Y) (RETURN T)) X)")
  ((ftest 30 0 '(block foo (if (null y) (return T)) x))
   "(BLOCK FOO
  (IF (NULL Y) (RETURN T))
  X)")
  ((ftest 30 40 '(block foo (if (null y) (return T)) x))
   "(BLOCK
 FOO
 (IF (NULL Y) (RETURN T))
 X)")
  ((ftest 100 0 '(block foo . T))
   "(BLOCK FOO . T)")
  ((ftest 100 0 '(block . T))
   "(BLOCK . T)")
  ((ftest 100 0 '((block . T)) (*print-level* 1))
   "(#)")

  ((ftest 20 0 '(case type (:foo (print 3))))
   "(CASE TYPE
  (:FOO (PRINT 3)))")
  ((ftest 10 0 '(catch 'bar (foo x)))
   "(CATCH 'BAR
  (FOO X))")
  ((ftest 20 0 '(ccase type (:foo (print 3))))
   "(CCASE TYPE
  (:FOO (PRINT 3)))")

  ((ftest 100 0 '(compiler-let ((a (foo 3)) (b (foo 4)) (c 1)) (tuz a b)))
   "(COMPILER-LET ((A (FOO 3)) (B (FOO 4)) (C 1)) (TUZ A B))")
  ((ftest 50 0 '(compiler-let ((a (foo 3)) (b (foo 4)) (c 1)) (tuz a b)))
   "(COMPILER-LET ((A (FOO 3)) (B (FOO 4)) (C 1))
  (TUZ A B))")
  ((ftest 44 0 '(compiler-let ((a (foo 3)) (b (foo 4)) (c 1)) (tuz a b)))
   "(COMPILER-LET ((A (FOO 3))
               (B (FOO 4))
               (C 1))
  (TUZ A B))")
  ((ftest 44 50 '(compiler-let ((a (foo 3)) (b (foo 4)) (c 1)) (tuz a b)))
   "(COMPILER-LET
 ((A (FOO 3)) (B (FOO 4)) (C 1))
 (TUZ A B))")
  ((ftest 30 0 '(compiler-let (bar baz def gack gortch) (tuz a b)))
   "(COMPILER-LET (BAR BAZ DEF
               GACK GORTCH)
  (TUZ A B))")
  ((ftest 40 0 '(compiler-let ((bar baz def gack . gortch))))
   "(COMPILER-LET ((BAR BAZ DEF GACK
                . GORTCH)))")
  ((ftest 100 0 '(compiler-let (bar baz def gack . gortch)))
   "(COMPILER-LET (BAR BAZ DEF GACK . GORTCH))")
  ((ftest 40 0 '(compiler-let foo))
   "(COMPILER-LET FOO)")
  ((ftest 40 0 '(compiler-let ()))
   "(COMPILER-LET ())")
  ((ftest 40 0 '(compiler-let . foo))
   "(COMPILER-LET . FOO)")

  ((ftest 100 0 '(cond ((plusp x) (print x) . 4) (a b) (T (car x))))
   "(COND ((PLUSP X) (PRINT X) . 4) (A B) (T (CAR X)))")
  ((ftest 55 0 '(cond ((plusp x) (print x) (minus x)) (a b) (T (car x))))
   "(COND ((PLUSP X) (PRINT X) (MINUS X))
      (A B)
      (T (CAR X)))")
  ((ftest 36 0 '(cond ((plusp x) (print x) (minus x)) (a b) (T (car x))))
   "(COND ((PLUSP X)
       (PRINT X)
       (MINUS X))
      (A B)
      (T (CAR X)))")
  ((ftest 30 40 '(cond ((plusp x) (print x) (minus x)) (a b) (T (car x))))
   "(COND
 ((PLUSP X)
  (PRINT X)
  (MINUS X))
 (A B)
 (T (CAR X)))")
  ((ftest 10 40 '(cond (a b) . T))
   "(COND
 (A B)
 . T)")
  ((ftest 10 40 '(cond))
"(COND)")

  ((ftest 20 0 '(ctypecase type (:foo (print 3))))
   "(CTYPECASE TYPE
  (:FOO (PRINT 3)))")
  ((ftest 20 0 '(defconstant foo 2 "test"))
   "(DEFCONSTANT FOO 2
  \"test\")")
  ((ftest 30 0 '(defconstant foo 2 (defconstant)) (*print-level* 1))
   "(DEFCONSTANT FOO 2 #)")
  ((ftest 40 0 '(define-setf-method ldb (a b) (make-right body a b)))
   "(DEFINE-SETF-METHOD LDB (A B)
  (MAKE-RIGHT BODY A B))")

  ((ftest 100 0 '(defmacro foo (a (b c) &body d) (car a) (list b c)))
   "(DEFMACRO FOO (A (B C) &BODY D) (CAR A) (LIST B C))")
  ((ftest 40 0 '(defmacro foo (a (b c) &body d) (car a) (list b c)))
   "(DEFMACRO FOO (A (B C) &BODY D)
  (CAR A)
  (LIST B C))")
  ((ftest 25 0 '(defmacro foo (a (b c) &body d) (car a) (list b c)))
   "(DEFMACRO FOO (A (B C)
               &BODY D)
  (CAR A)
  (LIST B C))")
  ((ftest 15 50 '(defmacro foo (a (b c) &body d) (car a) (list b c)))
   "(DEFMACRO
 FOO
 (A
  (B C)
  &BODY
  D)
 (CAR A)
 (LIST B C))")
  ((ftest 100 0 '(defmacro foo () . t))
   "(DEFMACRO FOO () . T)")
  ((ftest 100 0 '(defmacro . foo))
   "(DEFMACRO . FOO)")

  ((ftest 100 0 '(define-modify-macro bar (a b) union "fancy union"))
   "(DEFINE-MODIFY-MACRO BAR (A B) UNION \"fancy union\")")
  ((ftest 40 0 '(define-modify-macro bar (a b) union "fancy union"))
   "(DEFINE-MODIFY-MACRO BAR (A B) UNION
  \"fancy union\")")
  ((ftest 30 0 '(define-modify-macro bar (a b) union "fancy union"))
   "(DEFINE-MODIFY-MACRO BAR
                     (A B)
                     UNION
  \"fancy union\")")
  ((ftest 100 0 '(define-modify-macro bar (a b) union . T))
   "(DEFINE-MODIFY-MACRO BAR (A B) UNION . T)")
  ((ftest 100 0 '(define-modify-macro bar args . T))
   "(DEFINE-MODIFY-MACRO BAR ARGS . T)")
  ((ftest 100 0 '(define-modify-macro bar . T))
   "(DEFINE-MODIFY-MACRO BAR . T)")
  ((ftest 100 0 '(define-modify-macro . T))
   "(DEFINE-MODIFY-MACRO . T)")
  ((print*c "#1=(define-modify-macro foo #1#)")
   "#1=(DEFINE-MODIFY-MACRO FOO #1#)")

  ((ftest 20 0 '(defparameter foo 2 "test"))
   "(DEFPARAMETER FOO 2
  \"test\")")
  ((ftest 40 0 '(defsetf bar (a b) (store) (car x) (make-body a b)))
   "(DEFSETF BAR (A B) (STORE)
  (CAR X)
  (MAKE-BODY A B))")
  ((ftest 20 0 '(defsetf bar (a b) (store) (car x) (make-body a b)))
   "(DEFSETF BAR (A B)
         (STORE)
  (CAR X)
  (MAKE-BODY A B))")
  ((ftest 40 0 '(define-setf-method bar (a b) (car x) (make-body a b)))
   "(DEFINE-SETF-METHOD BAR (A B)
  (CAR X)
  (MAKE-BODY A B))")
  ((ftest 40 0 '(defstruct (foo (:print-fn bar)) acac babab))
   "(DEFSTRUCT (FOO (:PRINT-FN BAR))
  ACAC
  BABAB)")
  ((ftest 30 0 '(deftype bar (a) (satisfies bar-p)))
   "(DEFTYPE BAR (A)
  (SATISFIES BAR-P))")
  ((ftest 30 0 '(defun bar (a) (satisfies bar-p)))
   "(DEFUN BAR (A)
  (SATISFIES BAR-P))")
  ((ftest 20 0 '(defvar foo 2 "test"))
   "(DEFVAR FOO 2
  \"test\")")

  ((ftest 100 0 '(do ((a x (cdr a)) (i 1 (1+ i))) ((plusp a) T) (print a)))
   "(DO ((A X (CDR A)) (I 1 (1+ I))) ((PLUSP A) T)  (PRINT A))")
  ((ftest 55 0 '(do ((a x (cdr a)) (i 1 (1+ i))) ((plusp a) T) (print a)))
   "(DO ((A X (CDR A)) (I 1 (1+ I)))
    ((PLUSP A) T)
  (PRINT A))")
  ((ftest 30 0 '(do ((a x (cdr a)) (i 1 (1+ i))) ((plusp a) T) (print a)))
   "(DO ((A X (CDR A))
     (I 1 (1+ I)))
    ((PLUSP A) T)
  (PRINT A))")
  ((ftest 15 0 '(do ((a x (cdr a)) (i 1 (1+ i))) ((plusp a) T) (print a)))
   "(DO ((A X
      (CDR A))
     (I 1
      (1+ I)))
    ((PLUSP A)
     T)
  (PRINT A))")
  ((ftest 15 20 '(do ((a x (cdr a)) (i 1 (1+ i))) ((plusp a) T) (print a)))
   "(DO
 ((A
   X
   (CDR A))
  (I
   1
   (1+ I)))
 ((PLUSP A)
  T)
 (PRINT A))")
  ((ftest 100 0 '(do () () . T))
   "(DO () ()  . T)")
  ((ftest 100 0 '(do () . T))
   "(DO () . T)")
  ((ftest 100 0 '(do . T))
   "(DO . T)")

  ((ftest 55 0 '(do* ((a x (cdr a)) (i 1 (1+ i))) ((plusp a) T) (print a)))
   "(DO* ((A X (CDR A)) (I 1 (1+ I)))
     ((PLUSP A) T)
  (PRINT A))")
  ((ftest 35 0 '(do-all-symbols (s *package*) (print s)))
   "(DO-ALL-SYMBOLS (S *PACKAGE*)
  (PRINT S))")
  ((ftest 35 0 '(do-external-symbols (s *package*) (print s)))
   "(DO-EXTERNAL-SYMBOLS (S *PACKAGE*)
  (PRINT S))")
  ((ftest 35 0 '(do-symbols (s *package*) (print s)))
   "(DO-SYMBOLS (S *PACKAGE*)
  (PRINT S))")
  ((ftest 25 0 '(dolist (s list) (print s)))
   "(DOLIST (S LIST)
  (PRINT S))")
  ((ftest 25 0 '(dotimes (s list) (print s)))
   "(DOTIMES (S LIST)
  (PRINT S))")

  ((ftest 20 0 '(ecase type (:foo (print 3))))
   "(ECASE TYPE
  (:FOO (PRINT 3)))")
  ((ftest 20 0 '(etypecase type (:foo (print 3))))
   "(ETYPECASE TYPE
  (:FOO (PRINT 3)))")
  ((ftest 20 0 '(eval-when (compile load) (defun foo () (car x))))
   "(EVAL-WHEN (COMPILE LOAD)
  (DEFUN FOO ()
    (CAR X)))")

  ((ftest 100 0 '(flet ((a (a b) (car a) (car b)) (b () t)) (a (b 3))))
   "(FLET ((A (A B) (CAR A) (CAR B)) (B NIL T)) (A (B 3)))")
  ((ftest 50 0 '(flet ((a (a b) (car a) (car b)) (b () t)) (a (b 3))))
   "(FLET ((A (A B) (CAR A) (CAR B)) (B NIL T))
  (A (B 3)))")
  ((ftest 42 0 '(flet ((a (a b) (car a) (car b)) (b () t)) (a (b 3))))
   "(FLET ((A (A B) (CAR A) (CAR B))
       (B NIL T))
  (A (B 3)))")
  ((ftest 30 0 '(flet ((a (a b) (car a) (car b)) (b () t)) (a (b 3))))
   "(FLET ((A (A B)
         (CAR A)
         (CAR B))
       (B NIL T))
  (A (B 3)))")
  ((ftest 35 50 '(flet ((a (a b) (car a) (car b)) (b () t)) (a (b 3))))
   "(FLET
 ((A (A B) (CAR A) (CAR B))
  (B NIL T))
 (A (B 3)))")
  ((ftest 100 0 '(flet (() T . T) . T))
   "(FLET (() T . T) . T)")
  ((ftest 100 0 '(flet T . T))
   "(FLET T . T)")
  ((ftest 100 0 '(flet . T))
   "(FLET . T)")

  ((ftest 100 0 '(function (lambda (a) (car a))))
   "#'(LAMBDA (A) (CAR A))")
  ((ftest 100 0 '(function (lambda (a) (car a))) (*print-pretty* nil))
   "(FUNCTION (LAMBDA (A) (CAR A)))")
  ((ftest 5 20 '(function car))
   "#'CAR")
  ((ftest 100 0 '(function . a))
   "(FUNCTION . A)")
  ((ftest 100 0 '(function))
   "(FUNCTION)")
  ((ftest 100 0 '(function (lambda (a) (car a)) b))
   "(FUNCTION (LAMBDA (A) (CAR A)) B)")

  ((ftest 42 0 '(labels ((a (a b) (car a) (car b)) (b () t)) (a (b 3))))
   "(LABELS ((A (A B) (CAR A) (CAR B))
         (B NIL T))
  (A (B 3)))")
  ((ftest 20 0 '(lambda (a b) (car b) (car b)))
   "(LAMBDA (A B)
  (CAR B)
  (CAR B))")

  ((ftest 34 0 '(let ((a (foo 3)) (b (foo 4)) (c 1)) (tuz a b)))
   "(LET ((A (FOO 3))
      (B (FOO 4))
      (C 1))
  (TUZ A B))")
  ((ftest 34 0 '(let* ((a (foo 3)) (b (foo 4)) (c 1)) (tuz a b)))
   "(LET* ((A (FOO 3))
       (B (FOO 4))
       (C 1))
  (TUZ A B))")
  ((ftest 34 0 '(locally (declar (special x)) (print x)))
   "(LOCALLY (DECLAR (SPECIAL X))
  (PRINT X))")
  ((ftest 42 0 '(macrolet ((a (a b) (car a) (car b)) (b () t)) (a (b 3))))
   "(MACROLET ((A (A B) (CAR A) (CAR B))
           (B NIL T))
  (A (B 3)))")
  ((ftest 42 0 '(multiple-value-bind (a b) (compute-it x) (car a) (car b)))
   "(MULTIPLE-VALUE-BIND (A B)
    (COMPUTE-IT X)
  (CAR A)
  (CAR B))")
  ((ftest 32 0 '(multiple-value-setq (a b) (compute-it x)))
   "(MULTIPLE-VALUE-SETQ (A B)
  (COMPUTE-IT X))")

  ((ftest 100 0
     '(prog (a b c) (print a) L SS (if (null b) c) long0 (car b) long))
   "(PROG (A B C)
      (PRINT A)
 L SS (IF (NULL B) C)
 LONG0 (CAR B)
 LONG)")
  ((ftest 100 100
     '(prog (a b c) (print a) L SS (if (null b) c) long0 (car b) long))
   "(PROG (A B C)
      (PRINT A)
 L SS (IF (NULL B) C)
 LONG0 (CAR B)
 LONG)")
  ((ftest 100 0 '(prog () (print a) nil L . SS))
   "(PROG ()
      (PRINT A)
      NIL
 L . SS)")
  ((ftest 100 0 '(prog T . T))
   "(PROG T . T)")
  ((ftest 100 0 '(prog . T))
   "(PROG . T)")

  ((ftest 100 0 '(prog* ((a 3) b c) L SS (if (null b) c) long0 (car b)))
   "(PROG* ((A 3) B C)
 L SS  (IF (NULL B) C)
 LONG0 (CAR B))")
  ((ftest 25 0 '(progv (a b) (1 2) (car a)))
   "(PROGV (A B) (1 2)
  (CAR A))")

  ((ftest 20 0 '(setq a (car v) b (cdr v)))
   "(SETQ A (CAR V)
      B (CDR V))")
  ((ftest 20 20 '(setq a (car v) b (cdr v)))
   "(SETQ
 A
 (CAR V)
 B
 (CDR V))")
  ((ftest 17 0 '(setq a (car v) b))
   "(SETQ A (CAR V)
      B)")
  ((ftest 17 0 '(setq a (car v) . b))
   "(SETQ A (CAR V)
      . B)")
  ((ftest 100 0 '(setq . a))
   "(SETQ . A)")

  ((ftest 100 0 '(quote (lambda (a) (car a))))
   "'(LAMBDA (A) (CAR A))")
  ((ftest 5 20 '(quote car))
   "'CAR")
  ((ftest 100 0 '(quote . a))
   "(QUOTE . A)")
  ((ftest 100 0 '(quote))
   "(QUOTE)")
  ((ftest 100 0 '(quote (lambda (a) (car a)) b))
   "(QUOTE (LAMBDA (A) (CAR A)) B)")

  ((ftest 20 0 '(return-from foo (computation bar)))
   "(RETURN-FROM FOO
  (COMPUTATION BAR))")

  ((ftest 20 0 '(setf a (car v) b (cdr v)))
   "(SETF A (CAR V)
      B (CDR V))")
  ((ftest 1000 0 '(setf a (car v) b (cdr v)))
   "(SETF A (CAR V) B (CDR V))")
  ((ftest 20 0 '(psetf a (car v) b (cdr v)))
   "(PSETF A (CAR V)
       B (CDR V))")
  ((ftest 20 0 '(psetq a (car v) b (cdr v)))
   "(PSETQ A (CAR V)
       B (CDR V))")

  ((ftest 100 0
     '(tagbody (print a) L SS (if (null b) c) verylong (car b) long))
   "(TAGBODY (PRINT A)
 L SS    (IF (NULL B) C)
 VERYLONG (CAR B)
 LONG)")
  ((ftest 100 0 '(tagbody L SS (if (null b) c) . T))
   "(TAGBODY
 L SS    (IF (NULL B) C) . T)")
  ((ftest 100 0 '(tagbody L . SS))
   "(TAGBODY
 L . SS)")
  ((ftest 100 0 '(tagbody . SS))
   "(TAGBODY . SS)")
  ((ftest 10 0 '(throw 'bar (foo x)))
   "(THROW 'BAR
  (FOO X))")
  ((ftest 20 0 '(typecase type (:foo (print 3))))
   "(TYPECASE TYPE
  (:FOO (PRINT 3)))")
  ((ftest 20 0 '(unless (plusp x) (print x)))
   "(UNLESS (PLUSP X)
  (PRINT X))")
  ((ftest 20 0 '(unwind-protect (open f) (print errormsg)))
   "(UNWIND-PROTECT
    (OPEN F)
  (PRINT ERRORMSG))")
  ((ftest 20 0 '(when (plusp x) (print x)))
   "(WHEN (PLUSP X)
  (PRINT X))")
  ((ftest 35 0 '(with-input-from-string (f string) (read f)))
   "(WITH-INPUT-FROM-STRING (F STRING)
  (READ F))")
  ((ftest 45 0 '(with-open-file (f name :direction :input) (read f)))
   "(WITH-OPEN-FILE (F NAME :DIRECTION :INPUT)
  (READ F))")
  ((ftest 45 0 '(with-open-stream (stream (make-stream)) (read stream)))
   "(WITH-OPEN-STREAM (STREAM (MAKE-STREAM))
  (READ STREAM))")
  ((ftest 35 0 '(with-output-to-string (f string) (print f)))
   "(WITH-OUTPUT-TO-STRING (F STRING)
  (PRINT F))")

;These test the fast printing of simple atoms performed directly by XP

  ((print*s "foo") "\"foo\"")
  ((print*s "fo
o") "\"fo
o\"")
  ((print*s "foo" (*print-escape* nil)) "foo")
  ((print*s "fo\"o") "\"fo\\\"o\"")
  ((print*s "fo\"o" (*print-escape* nil)) "fo\"o")
  ((print*s "fo\\o") "\"fo\\\\o\"")
  ((print*s "fo\\o" (*print-escape* nil)) "fo\\o")

  ((print*s 20) "20")
  ((print*s 20 (*print-base* 8)) "24")
  ((print*s 20 (*print-radix* T)) "20.")
  ((print*s 1/2) "1/2")
  ((print*s 12345678901234567890) "12345678901234567890")
  ((print*s -20) "-20")
  ((print*s 1234567890) "1234567890")

  ((print*s 'foo2) "FOO2")
  ((print*s 'foo-bar (*print-case* :downcase)) "foo-bar")
  ((print*s 'foo-bar (*print-case* :upcase)) "FOO-BAR")
  ((print*s 'foo-bar (*print-case* :capitalize)) "Foo-Bar")
  ((print*s 'foo (*print-escape* nil)) "FOO")
  ((print*s 'install-xp) "INSTALL-XP")
  ((print*s 'install-xp (*print-escape* nil)) "INSTALL-XP")
  ((print*s ':install-xp) ":INSTALL-XP")
  ((print*s ':install-xp (*print-escape* nil)) "INSTALL-XP")
  ((print*s '*<-+->*) "*<-+->*")
  ((print*s '*<-+->*) "*<-+->*")

  ((prints 'xp::print-fixnum))
  ((prints '\fo\o-bar (*print-escape* nil) (*print-case* :capitalize)))
  ((prints '\fo\o-bar (*print-escape* nil) (*print-case* :downcase)))
  ((prints '\fo\o-bar (*print-escape* nil) (*print-case* :upcase)))
  ((prints '||))
  ((prints '|abcdef|))
  ((prints '|4a|))
  ((prints 'A%))
  ((prints '%))
  ((prints '#*10101))
  ((prints '#2A((12 3456 789) (22 456 78)) (*print-array* nil)))

;Tests of circularity printing.

  ((print*c "(A A NIL NIL B B)")
            "(A A NIL NIL B B)")
  ((print*c "(Ac$ A NIL NIL B B)" (*print-right-margin* 15))
   "(AC$ A NIL NIL
 B B)")
  ((print*c "(1 #1=#:FOO #1# #1# . #1#)")
            "(1 #1=#:FOO #1# #1# . #1#)")
  ((print*c "(1 #1=#:FOO #1# . #1#)" (*print-gensym* nil))
            "(1 FOO FOO . FOO)")
  ((print*c "(1 #1=123456789123456789123456789 #1# 123456789123456789123456789)")
            "(1 #1=123456789123456789123456789 #1# 123456789123456789123456789)")
  ((print*c "(1 #1=#:FOO #1# . #1#)" (*print-length* 2))
            "(1 #:FOO ...)")
  ((print*c "(0 (#1=(1 2) B) #1# . #1#)")
            "(0 (#1=(1 2) B) #1# . #1#)")
  ((print*c "#1=#(0 (1 #2=(A B) #1#) #2#)")
            "#1=#(0 (1 #2=(A B) #1#) #2#)")
  ((print*c "#1=#2A((0 0 0) (0 (1 #2=(A A) #1#) #2#))")
            "#1=#2A((0 0 0) (0 (1 #2=(A A) #1#) #2#))")
  ((print*c "(COND #1=((PLUSP X) Y) (X Y) . #1#)")
            "(COND #1=((PLUSP X) Y) (X Y) . #1#)")
  ((print*c "(A (B . #1=(C D)) #1# . #1#)")
            "(A (B . #1=(C D)) #1# . #1#)")
  ((print*c "(A (B . #1=(C . #2=(D E))) #2# #1# F)")
   "(A (B . #2=(C . #1=(D E))) #1# #2# F)")
  ((print*c "(A (B . #1=(C . #2=(D E))) #2# #1# F)" (*print-level* 2))
   "(A (B . #2=#) #1=(D E) #2# F)")
  ((print*c "(setq A #1=(car f) B C D #1#)"
	    (*print-lines* 2) (*PRINT-RIGHT-MARGIN* 20))
   "(SETQ A (CAR F)
      B C ---")
  ((print*c "(setq A #1=(car f) B C D #1#)"
	    (*print-lines* 1) (*PRINT-RIGHT-MARGIN* 20))
   "(SETQ A (CAR F) ---")

;Some tests for particular problems that came up along the way

  ((plet 15 0 (format-xp nil #"aaa~@<bbb~_ccc~:>ddd~_eee"))
   "aaabbbcccdddeee")
  ((plet 14 0 (format-xp nil #"aaa~@<bbb~_ccc~:>ddd~_eee"))
   "aaabbbcccddd
eee")
  ((plet 12 0 (format-xp nil #"aaa~@<bbb~_ccc~:>ddd~_eee"))
   "aaabbbcccddd
eee")
  ((plet 11 0 (format-xp nil #"aaa~@<bbb~_ccc~:>ddd~_eee"))
   "aaabbb
   cccddd
eee")
  ((plet 5 0  (format-xp nil #"aaa~@<bbb~_ccc~:>ddd~_eee"))
   "aaabbb
   cccddd
eee")

  ((plet 15 0 (format-xp nil #"a~@<aa~@<bbb~_ccc~:>ddd~_ee~:>e"))
   "aaabbbcccdddeee")
  ((plet 14 0 (format-xp nil #"a~@<aa~@<bbb~_ccc~:>ddd~_ee~:>e"))
   "aaabbbcccddd
 eee")
  ((plet 12 0 (format-xp nil #"a~@<aa~@<bbb~_ccc~:>ddd~_ee~:>e"))
   "aaabbbcccddd
 eee")
  ((plet 11 0 (format-xp nil #"a~@<aa~@<bbb~_ccc~:>ddd~_ee~:>e"))
   "aaabbb
   cccddd
 eee")
  ((plet 5 0  (format-xp nil #"a~@<aa~@<bbb~_ccc~:>ddd~_ee~:>e"))
   "aaabbb
   cccddd
 eee")

;tests of obscure control paths.

  ((plet 20 0 (with-output-to-string (s)
		(princ "abcde" s)
		(format-xp s #"~%~@<1234~:@_5678~:>")))
   "abcde
1234
5678")

  ((plet 20 0 (format-xp nil #"~@<foo ~4:ia~:@_b~:>"))
   "foo a
        b")
  ((plet 20 0 (format-xp nil #"~@<foo ~4:ia~:@_~:@_b~:>"))
   "foo a

        b")

;tests for going over the length of various queues and stacks.
  ((progn (setq xp::*free-xps* nil)
	  (plet 400 0 (format-xp nil #"~@<foo ~300ia~:@_b~305i~:@_c~:>")))
   #,(format nil "foo a~%~300@Tb~%~305@Tc"))

  ((progn (setq xp::*free-xps* nil)
	  (plet 400 0
	    (format-xp nil #"~@<foo ~250ia~:@_~@<0123456~@;b~@<++++~@;c~:@_d~:>~:>~:>")))
   #,(format nil "foo a~%~250@T0123456b++++c~%~250@T0123456 ++++d"))

  ((progn (setq xp::*free-xps* nil)
	  (plet 400 0 (format-xp nil #"~@<~250@Ta~_~10@Tb~5@Tc~:>")))
   #,(format nil "~250@Ta~10@Tb~5@Tc"))
  ((progn (setq xp::*free-xps* nil)
	  (plet 200 0 (format-xp nil #"~@<~250@Ta~10@Tb~5@Tc~:>")))
   #,(format nil "~250@Ta~10@Tb~5@Tc"))

  ((progn (setq xp::*free-xps* nil)
	  (plet 85 0 (format-xp nil #"~W" 
'((((((((((((((((((((((((((((((((((((((setq a b
					    c d)))))))))))))))))))))))))))))))))))))))))
   ;the next 2 lines must have spaces on them, not tabs
   "((((((((((((((((((((((((((((((((((((((SETQ A B
                                           C D))))))))))))))))))))))))))))))))))))))")

  ((progn (setq xp::*free-xps* nil)
	  (plet 200 0 (format-xp nil #"~W" 
'(1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1
  1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1))))
   "(1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1)")
  ((progn (setq xp::*free-xps* nil)
	  (plet 50 0 (format-xp nil #"~W" 
'(1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1
  1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1))))
   "(1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1
 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1
 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1
 1 1 1 1 1 1 1 1)")

;testing error checking

  (etest (format-xp nil #"ab~1,2") (1 2))
  (etest (format-xp nil #"ab~1,'") (2 5))
  (etest (format-xp nil #"ab~1/foo") (3 4))
  (etest (format-xp nil #"ab~1{foo~(..~{..~} ~}") (4 9))
  (etest (format-xp nil #"ab~!foo") (5 3))
  (etest (format-xp nil #"ab~1,2:@Ifoo") (6 6))
  (etest (format-xp nil #"ab~2:@:Ifoo") (7 6))
  (etest (format-xp nil #"ab~2:@@Ifoo") (8 6))
  (etest (format-xp nil #"ab~2:%foo") (9 5))
  (etest (format-xp nil #"ab~2@%foo") (10 5))
  (etest (format-xp nil #"ab~2@:*foo") (11 6))
  (etest (format-xp nil #"ab~@?foo") (12 4))
  (etest (format-xp nil #"ab~:[foo~]") (13 4))
  (etest (format-xp nil #"ab~@[foo~;bar~]") (14 4))
  (etest (format-xp nil #"ab foo~;bar~]") (15 7))
  (etest (format-xp nil #"ab ~(foo~]bar~)") (16 9))
  (etest (format-xp nil #"ab ~[foo~)bar~]") (17 9))
  (etest (format-xp nil #"ab ~[foo~>bar~]") (18 9))
  (etest (format-xp nil #"ab ~[foo~}bar~]") (19 9))
  (etest (format-xp nil #"ab ~{~}foo") (20 5))
  (etest (format-xp nil #"ab ~#<ff~>foo") (21 4))
  (etest (format-xp nil #"ab ~<f~#%f~>foo") (21 7))
  (etest (format-xp nil #"ab ~<f~#af~>foo") (21 7))
  (etest (format-xp nil #"ab ~22<f~:;f~>foo") (22 10))
  (etest (format-xp nil #"ab ~<f~Wf~>foo") (23 7))
  (etest (format-xp nil #"ab ~<f~;g~;h~;f~:>foo") (24 4))
  (etest (format-xp nil #"ab ~<f~Af~;g~;h~:>foo") (25 5))
  (etest (format-xp nil #"ab ~<f~;g~;h~Ag~:>foo") (26 11))


;tests of things that only work on Symbolics machines.

#+symbolics
  (deftest
   (plet 100 0
     (let ((*print-dispatch* *dt*))
       (define-print-dispatch (member 101) ((:priority 100)) (xp item)
	 (format-xp xp "~W" (car item)))
       (with-output-to-string (s)
         (pprint '#(xx 101 (2)) s))))
   "
#<Error during XP pretty printing of #(XX 101 (2))>")

#+symbolics
  ((ftest 55 0 '(define-print-dispatch (cons (member foo)) ((:table *IPD*)) (xp list)
		  (format-xp xp #"~W" list)))
"(DEFINE-PRINT-DISPATCH (CONS (MEMBER FOO))
                       ((:TABLE *IPD*)) (XP LIST)
  (FORMAT-XP XP #\"~W\" LIST))" )
#+symbolics
  ((ftest 45 0 '(define-print-dispatch (cons (member foo)) ((:table *IPD*)) (xp list)
		  (format-xp xp #"~W" list)))
"(DEFINE-PRINT-DISPATCH (CONS (MEMBER FOO))
                       ((:TABLE *IPD*))
                       (XP LIST)
  (FORMAT-XP XP #\"~W\" LIST))" )

#+symbolics
  ((ftest 55 0 '(zl:do-named foo ((a x (cdr a)) (i 1 (1+ i))) ((plusp a) T) (print a)))
   "(ZL:DO-NAMED FOO
             ((A X (CDR A)) (I 1 (1+ I)))
             ((PLUSP A) T)
  (PRINT A))")

))

;------------------------------------------------------------------------ ;
;                Copyright (c) Richard C. Waters, 1988                    ;
;------------------------------------------------------------------------ ;
