;;; -*- Log: code.log; Package: Mach -*-
;;;
;;; **********************************************************************
;;; This code was written as part of the CMU Common Lisp project at
;;; Carnegie Mellon University, and has been placed in the public domain.
;;; If you want to use this code or any part of CMU Common Lisp, please contact
;;; Scott Fahlman or slisp-group@cs.cmu.edu.
;;;
(ext:file-comment
  "$Header: syscall.lisp,v 1.19 91/09/24 08:50:21 wlott Exp $")
;;;
;;; **********************************************************************
;;;
;;; Interface to Unix system calls under Mach.  Mach system calls are
;;; handled through a matchmaker interface.  Most of the calling syntaxes
;;; of the Unix system calls are similar to those in C.  Refer to section
;;; 2 of the UNIX Programmer's Manual for specific information about the
;;; calls.
;;;
;;; Written by David B. McDonald, July 1986.
;;; Extended by Jim Healy, May 1987.
;;; Modifed by Chris Hoover and William Lott.
;;;
;;; **********************************************************************
;;;

(in-package "MACH")

(export '(mach unix-errno get-unix-error-msg r_ok w_ok x_ok f_ok
	  unix-access timeval timezone unix-chdir setuidexec setgidexec
	  savetext readown writeown execown readgrp writegrp execgrp
	  readoth writeoth execoth unix-chmod unix-close unix-creat
	  unix-link l_set l_incr l_xtnd unix-lseek unix-mkdir o_rdonly
	  o_wronly o_rdwr o_append o_creat o_trunc o_excl unix-creat
	  unix-open unix-read unix-rename unix-rmdir unix-select
	  unix-truncate unix-ftruncate unix-unlink unix-write kbdarg sgtty
	  terminal-speeds winsize kbdarg-scan kbdarg-index kbdarg-text
	  kbdarg-end sgtty-ispeed sgtty-ospeed sgtty-erase sgtty-kill
	  sgtty-flags winsize-ws_row winsize-ws_col winsize-ws_xpixel
	  winsize-ws_ypixel tchars ltchars tty-raw tty-crmod tty-echo tty-lcase
	  tty-cbreak
	  tty-tandem TIOCGETP TIOCSETP TIOCFLUSH TIOCSETC TIOCGETC TIOCSLTC
	  TIOCGLTC TIOCNOTTY TIOCSPGRP TIOCGPGRP TIOCGWINSZ TIOCSWINSZ
	  KBDCGET KBDCSET KBDCRESET
	  KBDCRST KBDCSSTD KBDSGET KBDGCLICK KBDSCLICK FIONREAD unix-ioctl
	  unix-exit stat unix-stat unix-lstat unix-fstat rusage rusage_self
	  rusage_children unix-getrusage unix-gettimeofday
	  unix-settimeofday mach-task_self mach-task_data mach-task_notify
	  direct unix-current-directory unix-getppid unix-getuid
	  unix-isatty unix-ttyname unix-isconsole unix-fork unix-wait
	  wstopped wnohang wuntraced unix-execve unix-symlink
	  unix-getpagesize unix-sync unix-killpg unix-getgid unix-dup
	  unix-fchmod unix-flock unix-fsync unix-getpgrp unix-getegid
	  lock-un lock-nb lock-ex lock-sh prio-min prio-max prio-process
	  prio-pgrp prio-user unix-setpriority unix-getpriority unix-utimes
	  set-utarray unix-chown unix-fchown unix-setpgrp unix-setreuid
	  unix-setregid unix-getdtablesize unix-getrlimit unix-setrlimit
	  unix-getgroups unix-setgroups unix-sethostid unix-gethostid
	  unix-setquota unix-gethostname unix-sethostname unix-dup2
	  unix-setitimer unix-getitimer unix-pipe fdarray fildes
	  unix-readlink unix-fcntl f-dupfd f-getfd f-setfd f-getfl f-setfl
	  f-getown f-setown fndelay fappend fasync fcreat ftrunc fexcl
	  itimer-real itimer-virtual itimer-prof rlimit-cpu rlimit-fsize
	  rlimit-data rlimit-stack rlimit-core rlimit-rss rlimit-nlimits
	  rlimit-infinity msg-send msg-receive msg-option-none send-timeout
	  send-notify rcv-timeout rcv-no-senders send-success
	  send-invalid-memory send-invalid-port send-timed-out
	  send-will-notify-send-notify-in-progress send-kernel-refused
	  send-interrupted send-msg-too-large send-msg-too-small
	  rcv-success rcv-invalid-memory rcv-invalid-port rcv-timed-out
	  rcv-too-large rcv-not-enough-memory rcv-only-sender
	  rcv-interrupted rpc-success kern-success kern-invalid-address
	  kern-protection-failure kern-no-space kern-invalid-argument
	  kern-failure kern-resource-shortage kern-not-receiver
	  kern-no-access typetype notify-port-deleted notify-msg-accepted
	  notify-ownership-rights notify-receive-rights notify-last
	  port-enabled nullport cmu-getaid
	  RFSIOCGETRW RFSIOCSETRW RFSIOCGETS RFSIOCSETS RFSIOCIDENTIFY
	  RFSRW_NONE RFSRW_USER RFSRW_GROUP RFSRW_ACCOUNT RFSRW_PASSWORD
	  RFSRW_LOOKUP rfsRW_t unix-socket unix-bind unix-listen
	  unix-accept unix-connect unix-send unix-sendto unix-recv
	  unix-recvfrom sock-stream sock-dgram sock-raw af-unix af-inet
	  msg-oob msg-peek msg-dontroute rfs-authenticate enoent eexist
	  espipe ewouldblock eio))

(declaim (inline unix-gettimeofday unix-getrusage unix-read unix-write
		 unix-select unix-ioctl))


;;; Constants used in the Mach interface.

(defconstant BADMSGID 1)
(defconstant WRONGARGS 2)
(defconstant BADREPLY 3)
(defconstant NOREPLY 4)
(defconstant UNSPECEXCEPTION 5)

(defconstant msg-option-none 0)
(defconstant send-timeout 1)
(defconstant send-notify 2)
(defconstant rcv-timeout #x100)
(defconstant rcv-no-senders #x200)

(defconstant send-success 0)

(defconstant send-errors-start -100)
(defconstant send-invalid-memory -101)
(defconstant send-invalid-port -102)
(defconstant send-timed-out -103)
(defconstant send-will-notify -105)
(defconstant send-notify-in-progress -106)
(defconstant send-kernel-refused -107)
(defconstant send-interrupted -108)
(defconstant send-msg-too-large -109)
(defconstant send-msg-too-small -110)
(defconstant send-errors-end -111)

(defconstant rcv-success 0)

(defconstant rcv-errors-start -200)
(defconstant rcv-invalid-memory -201)
(defconstant rcv-invalid-port -202)
(defconstant rcv-timed-out -203)
(defconstant rcv-too-large -204)
(defconstant rcv-not-enough-memory -205)
(defconstant rcv-only-sender -206)
(defconstant rcv-interrupted -207)
(defconstant rcv-errors-end -207)

(defconstant rpc-success 0)

(defconstant kern-success 0)
(defconstant kern-invalid-address 1)
(defconstant kern-protection-failure 2)
(defconstant kern-no-space 3)
(defconstant kern-invalid-argument 4)
(defconstant kern-failure 5)
(defconstant kern-resource-shortage 6)
(defconstant kern-not-receiver 7)
(defconstant kern-no-access 8)

(defconstant notify-first #o100)
(defconstant notify-port-deleted (1+ notify-first))
(defconstant notify-msg-accepted (+ notify-first 2))
(defconstant notify-ownership-rights (+ notify-first 3))
(defconstant notify-receive-rights (+ notify-first 4))
(defconstant notify-last (+ notify-first 15))

(defconstant port-enabled -1)
(defconstant nullport 0)


;;;; Types.

(deftype unix-pathname () 'simple-string)
(deftype unix-file-mode () '(unsigned-byte 16))
(deftype unix-fd () `(integer 0 ,most-positive-fixnum))
(deftype unix-pid () '(unsigned-byte 16))
(deftype unix-uid () '(unsigned-byte 16))
(deftype unix-gid () '(unsigned-byte 16))



;;;; System call definitions

(defmacro syscall (name &rest args)
  (let* ((info (get name 'unix-syscall-info))
	 (index (car info))
	 (nargs (cdr info))
	 (n (length args)))
    (if (and (not (eql nargs t)) (/= n nargs))
	(error "Syscall ~S wants ~D arguments, given ~D."
	       name nargs n))
    `(values (ext::call-foreign-function
	      "syscall"
	      '(signed-byte 32)
	      ',(make-list (1+ nargs) :initial-element '(unsigned-byte 32))
	      ,index
	      ,@(mapcar #'(lambda (arg)
			    `(get-useful-value ,arg))
			args))
	     (alien-access (alien-value unix-errno)))))

(proclaim '(inline get-useful-value))

(defun get-useful-value (thing)
  (etypecase thing
    (null 0)
    ((unsigned-byte 32) thing)
    ((signed-byte 32) (logand thing (1- (ash 1 32))))
    (system-area-pointer (sap-int thing))
    ((or simple-string
	 simple-bit-vector
	 (simple-array (unsigned-byte 2) (*))
	 (simple-array (unsigned-byte 4) (*))
	 (simple-array (unsigned-byte 8) (*))
	 (simple-array (unsigned-byte 16) (*))
	 (simple-array (unsigned-byte 32) (*)))
     (sap-int (vector-sap thing)))))


(eval-when (compile eval)
  (defmacro define-syscall (name index nargs)
    `(eval-when (compile load eval)
       (setf (get ',name 'unix-syscall-info) ',(cons index nargs)))))


(define-syscall unix-access 33 2)
(define-syscall unix-chmod 15 2)
(define-syscall unix-fchmod 124 2)
(define-syscall unix-chown 16 3)
(define-syscall unix-fchown 123 3)
(define-syscall unix-close 6 1)
(define-syscall unix-creat 8 2)
(define-syscall unix-dup 41 1)
(define-syscall unix-dup2 90 2)
(define-syscall unix-execv 11 3)
(define-syscall unix-execve 59 3)
(define-syscall unix-exit 1 1)
(define-syscall unix-fcntl 92 3)
(define-syscall unix-flock 131 2)
(define-syscall unix-fork 2 0)
(define-syscall unix-fsync 95 1)
(define-syscall unix-getdtablesize 89 0)
(define-syscall unix-getgid 47 0)
(define-syscall unix-getegid -1 0)
(define-syscall unix-getgroups 79 2)
(define-syscall unix-gethostid 142 0)
(define-syscall unix-sethostid 143 1)
(define-syscall unix-gethostname 87 2)
(define-syscall unix-sethostname 88 2)
(define-syscall unix-getitimer 86 2)
(define-syscall unix-setitimer 83 3)
(define-syscall unix-getpagesize 64 0)
(define-syscall unix-getpeername -1 3)
(define-syscall unix-getpgrp 81 1)
(define-syscall unix-getpid 20 0)
(define-syscall unix-getppid -1 0)
(define-syscall unix-getpriority 100 2)
(define-syscall unix-setpriority 96 3)
(define-syscall unix-getrlimit 144 2)
(define-syscall unix-setrlimit 145 2)
(define-syscall unix-getrusage 117 2)
(define-syscall unix-getsockname 150 3)
(define-syscall unix-getsockopt 118 5)
(define-syscall unix-setsockopt 105 5)
(define-syscall unix-gettimeofday 116 2)
(define-syscall unix-settimeofday 122 2)
(define-syscall unix-getuid 24 0)
(define-syscall unix-ioctl 54 3)
(define-syscall unix-link 9 2)
(define-syscall unix-lseek 19 3)
(define-syscall unix-mkdir 136 2)
(define-syscall unix-open 5 3)
(define-syscall unix-pipe 42 1)
(define-syscall unix-read 3 3)
(define-syscall unix-readlink 58 3)
(define-syscall unix-recv 102 4)
(define-syscall unix-recvfrom 125 6)
(define-syscall unix-recvmsg 113 3)
(define-syscall unix-rename 128 2)
(define-syscall unix-rmdir 137 1)
(define-syscall unix-select 93 5)
(define-syscall unix-send 101 4)
(define-syscall unix-sendto 133 6)
(define-syscall unix-sendmsg 114 3)
(define-syscall unix-setgroups -1 2)
(define-syscall unix-setpgrp 82 2)
(define-syscall unix-setquota 148 2)
(define-syscall unix-setregid 127 2)
(define-syscall unix-setreuid 126 2)
(define-syscall unix-socket 97 3)
(define-syscall unix-socketpair 135 4)
(define-syscall unix-stat 38 2)
(define-syscall unix-lstat 40 2)
(define-syscall unix-fstat 62 2)
(define-syscall unix-symlink 57 2)
(define-syscall unix-sync 36 0)
(define-syscall unix-truncate 129 2)
(define-syscall unix-ftruncate 130 2)
(define-syscall unix-unlink 10 1)
(define-syscall unix-utimes 138 2)
(define-syscall unix-vfork 66 0)
(define-syscall unix-wait 84 1)
(define-syscall unix-write 4 3)

;;; CMU specific syscalls.

(define-syscall cmu-getaid -2 0)

;;; Mach specific syscalls.

#|
(defconstant msg-send -14)
(defconstant msg-receive -15)
(defconstant msg-rpc -16)
|#



;;; Unix error messages.

;;;
;;; This code brought to you by William and Chris.
;;; 
(eval-when (compile eval)

(defvar *compiler-unix-errors* nil)
(setf *compiler-unix-errors* nil)

(defmacro def-unix-error (name number description)
  `(progn
     (eval-when (compile eval)
       (push (cons ,number ,description) *compiler-unix-errors*))
     (defconstant ,name ,number ,description)
     (export ',name)))

(defmacro emit-unix-errors ()
  (let* ((max (apply #'max (mapcar #'car *compiler-unix-errors*)))
	 (array (make-array (1+ max) :initial-element nil)))
    (dolist (error *compiler-unix-errors*)
      (setf (svref array (car error)) (cdr error)))
    `(progn
       (defvar *unix-errors* ',array)
       (proclaim '(simple-vector *unix-errors*)))))

) ;eval-when


;;;; Actual Error Messages

;;; 
;;; From <errno.h>
;;; 
(def-unix-error ESUCCESS 0 "Successful")
(def-unix-error EPERM 1 "Not owner")
(def-unix-error ENOENT 2 "No such file or directory")
(def-unix-error ESRCH 3 "No such process")
(def-unix-error EINTR 4 "Interrupted system call")
(def-unix-error EIO 5 "I/O error")
(def-unix-error ENXIO 6 "No such device or address")
(def-unix-error E2BIG 7 "Arg list too long")
(def-unix-error ENOEXEC 8 "Exec format error")
(def-unix-error EBADF 9 "Bad file number")
(def-unix-error ECHILD 10 "No children")
(def-unix-error EAGAIN 11 "No more processes")
(def-unix-error ENOMEM 12 "Not enough core")
(def-unix-error EACCES 13 "Permission denied")
(def-unix-error EFAULT 14 "Bad address")
(def-unix-error ENOTBLK 15 "Block device required")
(def-unix-error EBUSY 16 "Mount device busy")
(def-unix-error EEXIST 17 "File exists")
(def-unix-error EXDEV 18 "Cross-device link")
(def-unix-error ENODEV 19 "No such device")
(def-unix-error ENOTDIR 20 "Not a director")
(def-unix-error EISDIR 21 "Is a directory")
(def-unix-error EINVAL 22 "Invalid argument")
(def-unix-error ENFILE 23 "File table overflow")
(def-unix-error EMFILE 24 "Too many open files")
(def-unix-error ENOTTY 25 "Not a typewriter")
(def-unix-error ETXTBSY 26 "Text file busy")
(def-unix-error EFBIG 27 "File too large")
(def-unix-error ENOSPC 28 "No space left on device")
(def-unix-error ESPIPE 29 "Illegal seek")
(def-unix-error EROFS 30 "Read-only file system")
(def-unix-error EMLINK 31 "Too many links")
(def-unix-error EPIPE 32 "Broken pipe")
;;; 
;;; Math
(def-unix-error EDOM 33 "Argument too large")
(def-unix-error ERANGE 34 "Result too large")
;;; 
;;; non-blocking and interrupt i/o
(def-unix-error EWOULDBLOCK 35 "Operation would block")
(def-unix-error EDEADLK 35 "Operation would block") ; Ditto
(def-unix-error EINPROGRESS 36 "Operation now in progress")
(def-unix-error EALREADY 37 "Operation already in progress")
;;;
;;; ipc/network software
(def-unix-error ENOTSOCK 38 "Socket operation on non-socket")
(def-unix-error EDESTADDRREQ 39 "Destination address required")
(def-unix-error EMSGSIZE 40 "Message too long")
(def-unix-error EPROTOTYPE 41 "Protocol wrong type for socket")
(def-unix-error ENOPROTOOPT 42 "Protocol not available")
(def-unix-error EPROTONOSUPPORT 43 "Protocol not supported")
(def-unix-error ESOCKTNOSUPPORT 44 "Socket type not supported")
(def-unix-error EOPNOTSUPP 45 "Operation not supported on socket")
(def-unix-error EPFNOSUPPORT 46 "Protocol family not supported")
(def-unix-error EAFNOSUPPORT 47 "Address family not supported by protocol family")
(def-unix-error EADDRINUSE 48 "Address already in use")
(def-unix-error EADDRNOTAVAIL 49 "Can't assign requested address")
;;;
;;; operational errors
(def-unix-error ENETDOWN 50 "Network is down")
(def-unix-error ENETUNREACH 51 "Network is unreachable")
(def-unix-error ENETRESET 52 "Network dropped connection on reset")
(def-unix-error ECONNABORTED 53 "Software caused connection abort")
(def-unix-error ECONNRESET 54 "Connection reset by peer")
(def-unix-error ENOBUFS 55 "No buffer space available")
(def-unix-error EISCONN 56 "Socket is already connected")
(def-unix-error ENOTCONN 57 "Socket is not connected")
(def-unix-error ESHUTDOWN 58 "Can't send after socket shutdown")
(def-unix-error ETOOMANYREFS 59 "Too many references: can't splice")
(def-unix-error ETIMEDOUT 60 "Connection timed out")
(def-unix-error ECONNREFUSED 61 "Connection refused")
;;; 
(def-unix-error ELOOP 62 "Too many levels of symbolic links")
(def-unix-error ENAMETOOLONG 63 "File name too long")
;;; 
(def-unix-error EHOSTDOWN 64 "Host is down")
(def-unix-error EHOSTUNREACH 65 "No route to host")
(def-unix-error ENOTEMPTY 66 "Directory not empty")
;;; 
;;; quotas & resource 
(def-unix-error EPROCLIM 67 "Too many processes")
(def-unix-error EUSERS 68 "Too many users")
(def-unix-error EDQUOT 69 "Disc quota exceeded")
;;;
;;; CMU RFS
(def-unix-error ELOCAL 126 "namei should continue locally")
(def-unix-error EREMOTE 127 "namei was handled remotely")
;;;
;;; VICE
(def-unix-error EVICEERR 70 "Remote file system error ")
(def-unix-error EVICEOP 71 "syscall was handled by Vice")
;;;
;;; Mach Emulation
(def-unix-error ERESTART 72 "Mach Emulation Error (?)")
;;;
;;; And now for something completely different ...
(emit-unix-errors)


;;;; Looking up errors

(def-c-variable ("errno" unix-errno) int)

(defun get-unix-error-msg (&optional
			   (error-number
			    (alien-access (alien-value unix-errno))))
  "Returns a string describing the error number which was returned by a
  UNIX system call."
  (check-type error-number integer)
  (or (and (array-in-bounds-p *unix-errors* error-number)
	   (svref *unix-errors* error-number))
      (format nil "Unknown error [~d]" error-number)))



;;;; msg-send, msg-receive, and msg-rpc

;;; Msg-Send
;;;
;;; Send sends a message to a destination port which is specified in the
;;; message.
;;;  MESSAGE -- an alien-value object (new aliens) which contains a message.
;;;  OPTION  -- an integer constant.
;;;		   send-timeout wait for maxwait milliseconds before aborting.
;;;		   send-notify  allows sender to give exactly one message
;;;				without being suspended should the
;;;				destination port be full.
;;;  MAXWAIT -- an integer.  The number of milliseconds to wait
;;;		if the desitnation port is full.

#+nil
(def-c-routine ("msg_send" msg-send)
	       (int)
  "Sends a message.  OPTION may be one of send-timeout or send-notify.
  If send-timeout is specified, the send will timeout after MAX-WAIT
  milliseconds."
  (message pointer)
  (option int)
  (max-wait int))


;;; Msg-Receive

;;; Receive receives a message
;;;  MESSAGE -- An alien-value object where the message will be placed.
;;;  OPTION  -- An integer one of the following;
;;;		  rcv-timeout	 receive returns when MaxWait time is
;;;				 exceeded.  Otherwise waits until a mesage
;;;				 is received.
;;;		  rcv-no-senders returns if calling task has only access
;;;				 rights to port in message header.
;;;  MAXWAIT -- an integer.  The number of milliseconds to wait
;;;		for a message.
;;;
;;; ***UGLY IMPLEMENTATION DETAIL***
;;;   The kernel trap  can return a code rcv-interrupted when the
;;;   receive is aborted by an interrupt.  When this happens  the receive
;;;   function tries again.
;;;
#+nil
(def-c-routine ("msg_receive" msg-receive)
	       (int)
  "Receives a message.  Option can be one of rcv-timeout or rcv-no-senders.
  If rcv-timeout is specified, the receive will timeout after MaxWait
  milliseconds."
  (message pointer)
  (option int)
  (max-wait int))


;;; Msg-Rpc

;;; Does a send and then a receive.

#+nil
(def-c-routine ("msg_rpc" msg-rpc)
	       (int)
  "Sends message and then immediately does a receive, storing the
  received message into message."
  (message pointer)
  (option int)
  (rcv-size int)
  (send-max-wait int)
  (rcv-max-wait int))


;;;; Some random type definitions:

(defconstant dirblksiz 512 "Size of a directory block.")
(defconstant maxnamelen 255 "Maximum filename size in chars.")

(defrecord unix-block
  (block (unsigned-byte 8) (bytes 512)))

(defrecord direct
  #+sunos (off (unsigned-byte 32) (long-words 1))
  (ino (unsigned-byte 32) (long-words 1))
  (reclen (unsigned-byte 16) (words 1))
  (namelen (unsigned-byte 16) (words 1)))

(def-c-pointer name-pointer (null-terminated-string 1024))



;;; Unix file system calls.

;;; Unix-access accepts a path and a mode.  It returns two values the
;;; first is T if the file is accessible and NIL otherwise.  The second
;;; only has meaning in the second case and is the unix errno value.

(defconstant r_ok 4 "Test for read permission")
(defconstant w_ok 2 "Test for write permission")
(defconstant x_ok 1 "Test for execute permission")
(defconstant f_ok 0 "Test for presence of file")

(def-c-routine ("access" real-unix-access) (int)
  (path null-terminated-string)
  (mode unsigned-int))

(defun unix-access (path mode)
  "Given a file path (a string) and one of four constant modes,
   unix-access returns T if the file is accessible with that
   mode and NIL if not.  It also returns an errno value with
   NIL which determines why the file was not accessible.

   The access modes are:
	r_ok     Read permission.
	w_ok     Write permission.
	x_ok     Execute permission.
	f_ok     Presence of file."
  (declare (type unix-pathname path)
	   (type (mod 8) mode))
  (let ((result (real-unix-access path mode)))
    (if (minusp result)
	(values nil (alien-access (alien-value unix-errno)))
	(values t 0))))

;;; Unix-chdir accepts a directory name and makes that the
;;; current working directory.

(def-c-routine ("chdir" unix-chdir) (int)
  "Given a file path string, unix-chdir changes the current working 
   directory to the one specified."
  (path null-terminated-string))


;;; Unix-chmod accepts a path and a mode and changes the mode to the new mode.

(defconstant setuidexec #o4000 "Set user ID on execution")
(defconstant setgidexec #o2000 "Set group ID on execution")
(defconstant savetext #o1000 "Save text image after execution")
(defconstant readown #o400 "Read by owner")
(defconstant writeown #o200 "Write by owner")
(defconstant execown #o100 "Execute (search directory) by owner")
(defconstant readgrp #o40 "Read by group")
(defconstant writegrp #o20 "Write by group")
(defconstant execgrp #o10 "Execute (search directory) by group")
(defconstant readoth #o4 "Read by others")
(defconstant writeoth #o2 "Write by others")
(defconstant execoth #o1 "Execute (search directory) by others")

(defun unix-chmod (path mode)
  "Given a file path string and a constant mode, unix-chmod changes the
   permission mode for that file to the one specified. The new mode
   can be created by logically OR'ing the following:

      setuidexec        Set user ID on execution.
      setgidexec        Set group ID on execution.
      savetext          Save text image after execution.
      readown           Read by owner.
      writeown          Write by owner.
      execown           Execute (search directory) by owner.
      readgrp           Read by group.
      writegrp          Write by group.
      execgrp           Execute (search directory) by group.
      readoth           Read by others.
      writeoth          Write by others.
      execoth           Execute (search directory) by others.
  
  It returns T on successfully completion; NIL and an error number
  otherwise."
  (declare (type unix-pathname path)
	   (type unix-file-mode mode))
  (multiple-value-bind (value err) (syscall unix-chmod path mode)
    (values (if (eql value 0) T) err)))

;;; Unix-fchmod accepts a file descriptor ("fd") and a file protection mode
;;; ("mode") and changes the protection of the file described by "fd" to 
;;; "mode".

(defun unix-fchmod (fd mode)
  "Given an integer file descriptor and a mode (the same as those
   used for unix-chmod), unix-fchmod changes the permission mode
   for that file to the one specified. T is returned if the call
   was successful."
  (declare (type unix-fd fd)
	   (type unix-file-mode mode))
  (multiple-value-bind (value err) (syscall unix-fchmod fd mode)
    (values (if (eql value 0) T) err)))

(defun unix-chown (path uid gid)
  "Given a file path, an integer user-id, and an integer group-id,
   unix-chown changes the owner of the file and the group of the
   file to those specified.  Either the owner or the group may be
   left unchanged by specifying them as -1.  Note: Permission will
   fail if the caller is not the superuser."
  (declare (type unix-pathname path)
	   (type (or unix-uid (integer -1 -1)) uid)
	   (type (or unix-gid (integer -1 -1)) gid))
  (multiple-value-bind (value err) (syscall unix-chown path uid gid)
    (values (if (eql value 0) T) err)))

;;; Unix-fchown is exactly the same as unix-chown except that the file
;;; is specified by a file-descriptor ("fd") instead of a pathname.

(defun unix-fchown (fd uid gid)
  "Unix-fchown is like unix-chown, except that it accepts an integer
   file descriptor instead of a file path name."
  (declare (type unix-fd fd)
	   (type (or unix-uid (integer -1 -1)) uid)
	   (type (or unix-gid (integer -1 -1)) gid))  
  (multiple-value-bind (value err) (syscall unix-fchown fd uid gid)
    (values (if (eql value 0) T) err)))

;;; Returns the maximum size (i.e. the number of array elements
;;; of the file descriptor table.

(defun unix-getdtablesize ()
  "Unix-getdtablesize returns the maximum size of the file descriptor
   table. (i.e. the maximum number of descriptors that can exist at
   one time.)"
  (multiple-value-bind (value) (syscall unix-getdtablesize)
     value))

;;; Unix-close accepts a file descriptor and attempts to close the file
;;; associated with it.

(defun unix-close (fd)
  "Unix-close takes an integer file descriptor as an argument and
   closes the file associated with it.  T is returned upon successful
   completion, otherwise NIL and an error number."
  (declare (type unix-fd fd))
  (multiple-value-bind (value err) (syscall unix-close fd)
    (values (if (eql value 0) T) err)))

;;; Unix-creat accepts a file name and a mode.  It creates a new file
;;; with name and sets it mode to mode (as for chmod).

(defun unix-creat (name mode)
  "Unix-creat accepts a file name and a mode (same as those for
   unix-chmod) and creates a file by that name with the specified
   permission mode.  It returns T on success, or NIL and an error
   number otherwise."
  (declare (type unix-pathname name)
	   (type unix-file-mode mode))
  (multiple-value-bind (value err) (syscall unix-creat name mode)
    (values (if (< value 0) NIL value) err)))

;;; Unix-dup returns a duplicate copy of the existing file-descriptor
;;; passed as an argument.

(defun unix-dup (fd)
  "Unix-dup duplicates an existing file descriptor (given as the
   argument) and returns T and the value for the duplicate if
   successful."
  (declare (type unix-fd fd))
  (multiple-value-bind (result errcode) (syscall unix-dup fd)
    (if (eql result -1)
	(values NIL errcode)
	(values T result))))

;;; Unix-dup2 makes the second file-descriptor describe the same file
;;; as the first. If the second file-descriptor points to an open
;;; file, it is first closed. In any case, the second should have a 
;;; value which is a valid file-descriptor.

(defun unix-dup2 (fd1 fd2)
  "Unix-dup2 duplicates an existing file descriptor just as unix-dup
   does only the new value of the duplicate descriptor may be requested
   through the second argument.  If a file already exists with the
   requested descriptor number, it will be closed and the number
   assigned to the duplicate."
  (declare (type unix-fd fd1 fd2))
  (multiple-value-bind (result errcode) (syscall unix-dup2 fd1 fd2)
    (if (eql result -1)
	(values NIL errcode)
	(values T result))))

;;; Unix-fcntl takes a file descriptor, an integer command
;;; number, and optional command arguments.  It performs
;;; operations on the associated file and/or returns inform-
;;; ation about the file.

;;; Operations performed on file descriptors:

(defconstant F-DUPFD    0  "Duplicate a file descriptor")
(defconstant F-GETFD    1  "Get file desc. flags")
(defconstant F-SETFD    2  "Set file desc. flags")
(defconstant F-GETFL    3  "Get file flags")
(defconstant F-SETFL    4  "Set file flags")
(defconstant F-GETOWN   5  "Get owner")
(defconstant F-SETOWN   6  "Set owner")


;;; File flags for F-GETFL and F-SETFL:

(defconstant FNDELAY  #o0004   "Non-blocking reads")
(defconstant FAPPEND  #o0010   "Append on each write")
(defconstant FASYNC   #o0100   "Signal pgrp when data ready")
(defconstant FCREAT   #o1000   "Create if nonexistant")
(defconstant FTRUNC   #o2000   "Truncate to zero length")
(defconstant FEXCL    #o4000   "Error if already created")

(defun unix-fcntl (fd cmd arg)
  "Unix-fcntl manipulates file descriptors according to the
   argument CMD which can be one of the following:

   F-DUPFD         Duplicate a file descriptor.
   F-GETFD         Get file descriptor flags.
   F-SETFD         Set file descriptor flags.
   F-GETFL         Get file flags.
   F-SETFL         Set file flags.
   F-GETOWN        Get owner.
   F-SETOWN        Set owner.

   The flags that can be specified for F-SETFL are:

   FNDELAY         Non-blocking reads.
   FAPPEND         Append on each write.
   FASYNC          Signal pgrp when data ready.
   FCREAT          Create if nonexistant.
   FTRUNC          Truncate to zero length.
   FEXCL           Error if already created.
   "
  (declare (type unix-fd fd)
	   (type (integer 0 6) cmd)
	   (type (unsigned-byte 16) arg))
  (multiple-value-bind (result errcode) (syscall unix-fcntl fd cmd arg)
    (if (eql result -1)
	(values NIL errcode)
	(values T result))))

;;; Unix-flock applies an advisory lock to the file described by the
;;; file-descriptor "fd".  Advisory locks can be used by cooperating 
;;; processes but do not guarantee the safety of a file unless heeded
;;; by all processes accessing that file.  The type of lock can be a
;;; combination of lock_ex or lock_sh with lock_nb or to unlock, lock_un.

#|
(defconstant lock-un 8 "Unlock")                   ; Unlock the file.
(defconstant lock-nb 4 "Don't block when locking") ; Don't block attempts by
                                                   ;    other jobs to lock.
(defconstant lock-ex 2 "Exclusive lock")           ; Other jobs can't access.
(defconstant lock-sh 1 "Shared lock")              ; Share access with others.
|#

#+nil
(defun unix-flock (fd lock-mode)
  "Unix-lock applies an advisory lock to the file described by fd
   the one of the following modes:

  lock-un          Unlock the file.
  lock-nb          Don't block when locking.
  lock-ex          Exclusive lock.
  lock-sh          Shared lock.
  "
  (multiple-value-bind (value err) (syscall unix-flock fd lock-mode)
    (values (if (eql value 0) T) err)))

;;; Unix-link creates a hard link from name2 to name1.

(defun unix-link (name1 name2)
  "Unix-link creates a hard link from the file with name1 to the
   file with name2."
  (declare (type unix-pathname name1 name2))
  (multiple-value-bind (value err) (syscall unix-link name1 name2)
    (values (if (eql value 0) T) err)))

;;; Unix-lseek accepts a file descriptor, an offset, and whence value.

(defconstant l_set 0 "set the file pointer")
(defconstant l_incr 1 "increment the file pointer")
(defconstant l_xtnd 2 "extend the file size")

(defun unix-lseek (fd offset whence)
  "Unix-lseek accepts a file descriptor and moves the file pointer ahead
   a certain offset for that file.  Whence can be any of the following:

   l_set        Set the file pointer.
   l_incr       Increment the file pointer.
   l_xtnd       Extend the file size.
  "
  (declare (type unix-fd fd)
	   (type (unsigned-byte 32) offset)
	   (type (integer 0 2) whence))
  (multiple-value-bind (value err) (syscall unix-lseek fd offset whence)
    (values (if (< value 0) NIL value) err)))

;;; Unix-mkdir accepts a name and a mode and attempts to create the
;;; corresponding directory with mode mode.

(defun unix-mkdir (name mode)
  "Unix-mkdir creates a new directory with the specified name and mode.
   (Same as those for unix-fchmod.)  It returns T upon success, otherwise
   NIL and an error number."
  (declare (type unix-pathname name)
	   (type unix-file-mode mode))
  (multiple-value-bind (value err) (syscall unix-mkdir name mode)
    (values (if (eql value 0) T) err)))

;;; Unix-open accepts a pathname (a simple string), flags, and mode and attempts
;;; to open file with name pathname.

(defconstant o_rdonly 0 "Read-only flag.") 
(defconstant o_wronly 1 "Write-only flag.")
(defconstant o_rdwr 2   "Read-write flag.")
(defconstant o_append #o10   "Append flag.")
(defconstant o_creat #o1000  "Create if nonexistant flag.") 
(defconstant o_trunc #o2000  "Truncate flag.")

(defconstant o_excl #o4000  "Error if already exists.")

(defun unix-open (path flags mode)
  "Unix-open opens the file whose pathname is specified by path
   for reading and/or writing as specified by the flags argument.
   The flags argument can be:

     o_rdonly        Read-only flag.
     o_wronly        Write-only flag.
     o_rdwr          Read-and-write flag.
     o_append        Append flag.
     o_creat         Create-if-nonexistant flag.
     o_trunc         Truncate-to-size-0 flag.

   If the o_creat flag is specified, then the file is created with
   a permission of argument mode if the file doesn't exist.  An
   integer file descriptor is returned by unix-open."
  (declare (type unix-pathname path)
	   (type (unsigned-byte 16) flags)
	   (type unix-file-mode mode))
  (multiple-value-bind (fd err) (syscall unix-open path flags mode)
    (values (if (< fd 0) NIL fd) err)))


(def-c-routine ("pipe" real-unix-pipe) (int)
  (fildes system-area-pointer))

(defun unix-pipe ()
  "Unix-pipe sets up a unix-piping mechanism consisting of
  an input pipe and an output pipe.  Unix-Pipe returns two
  values: if no error occurred the first value is the pipe
  to be read from and the second is can be written to.  If
  an error occurred the first value is NIL and the second
  the unix error code."
  (with-stack-alien (fildes unix-fd (long-words 2))
    (let* ((sap (alien-sap (alien-value fildes)))
	   (result (real-unix-pipe sap)))
      (if (minusp result)
	  (values NIL (alien-access (alien-value unix-errno)))
	  (values (sap-ref-32 sap 0)
		  (sap-ref-32 sap 1))))))

;;; Unix-read accepts a file descriptor, a buffer, and the length to read.
;;; It attempts to read len bytes from the device associated with fd
;;; and store them into the buffer.  It returns the actual number of
;;; bytes read.

(defun unix-read (fd buf len)
  "Unix-read attempts to read from the file described by fd into
   the buffer buf until it is full.  Len is the length of the buffer.
   The number of bytes actually read is returned or NIL and an error
   number if an error occured."
  (declare (type unix-fd fd)
	   (type (unsigned-byte 32) len))
  #+sunos
  ;; Note: Under sunos we touch each page before doing the read to give
  ;; the segv handler a chance to fix the permissions.  Otherwise,
  ;; read will return EFAULT.  This also bypasses a bug in 4.1.1 in which
  ;; read fails with EFAULT if the page has never been touched even if
  ;; the permissions are okay.
  (without-gcing
   (let* ((page-size (get-page-size))
	  (1-page-size (1- page-size))
	  (sap (etypecase buf
		 (system-area-pointer buf)
		 (vector (vector-sap buf))))
	  (end (sap+ sap len)))
     (declare (type (and fixnum unsigned-byte) page-size 1-page-size)
	      (type system-area-pointer sap end)
	      (optimize (speed 3) (safety 0)))
     (do ((sap (int-sap (logand (the (unsigned-byte 32)
				     (+ (sap-int sap) 1-page-size))
				(logxor 1-page-size (ldb (byte 32 0) -1))))
	       (sap+ sap page-size)))
	 ((not (pointer< sap end)))
       (declare (type system-area-pointer sap))
       (setf (sap-ref-8 sap 0) (sap-ref-8 sap 0)))))
  (multiple-value-bind (value err) (syscall unix-read fd buf len)
    (values (if (< value 0) NIL value) err)))


(defun unix-readlink (path)
  "Unix-readlink invokes the readlink system call on the file name
  specified by the simple string path.  It returns up to two values:
  the contents of the symbolic link if the call is successful, or
  NIL and the Unix error number."
  (declare (type unix-pathname path))
  (with-trap-arg-block unix-block ub
    (multiple-value-bind (res err)
			 (syscall unix-readlink path
				  (alien-sap (alien-value ub))
				  (/ (record-size 'unix-block) 8))
      (if (<= res 0)
	  (values NIL err)
	  (let ((ts (make-string res)))
	    (%primitive byte-blt
			(alien-sap (alien-value ub)) 0
			(vector-sap ts)
			0 res)
	    ts)))))


;;; Unix-rename accepts two files names and renames the first to the second.

(defun unix-rename (name1 name2)
  "Unix-rename renames the file with string name1 to the string
   name2.  NIL and an error code is returned if an error occured."
  (declare (type unix-pathname name1 name2))
  (multiple-value-bind (value err) (syscall unix-rename name1 name2)
    (values (if (eql value 0) T) err)))

;;; Unix-rmdir accepts a name and removes the associated directory.

(defun unix-rmdir (name)
  "Unix-rmdir attempts to remove the directory name.  NIL and
   an error number is returned if an error occured."
  (declare (type unix-pathname name))
  (multiple-value-bind (value err) (syscall unix-rmdir name)
    (values (if (eql value 0) T) err)))

;;; Unix-select accepts sets of file descriptors and waits for an event
;;; to happen on one of them or to time out.


(defun unix-select (nfds rdfds wrfds xpfds to-secs &optional (to-usecs 0))
  "Unix-select examines the sets of descriptors passed as arguments
   to see if they are ready for reading and writing.  See the UNIX
   Programmers Manual for more information."
  (with-trap-arg-block timeval tv
    (with-stack-alien (rdf (unsigned-byte 32) (long-words 1))
      (with-stack-alien (wrf (unsigned-byte 32) (long-words 1))
	(with-stack-alien (xpf (unsigned-byte 32) (long-words 1))
	  (when to-secs
	    (setf (alien-access (timeval-seconds (alien-value tv))) to-secs)
	    (setf (alien-access (timeval-useconds (alien-value tv))) to-usecs))
	  (setf (alien-access (alien-value rdf)) rdfds)
	  (setf (alien-access (alien-value wrf)) wrfds)
	  (setf (alien-access (alien-value xpf)) xpfds)
	  (multiple-value-bind
	      (value err)
	      (macrolet ((frob (lispvar alienvar)
			   `(if (zerop ,lispvar)
				(int-sap 0)
				(alien-sap (alien-value ,alienvar)))))
		(syscall unix-select nfds
			 (frob rdfds rdf) (frob wrfds wrf) (frob xpfds xpf)
			 (if to-secs (alien-sap (alien-value tv)) (int-sap 0))))
	    (if (eql value -1)
		(values nil err 0 0)
		(values value
			(alien-access (alien-value rdf))
			(alien-access (alien-value wrf))
			(alien-access (alien-value xpf))))))))))

;;; Unix-sync writes all information in core memory which has been modified
;;; to permanent storage (i.e. disk).

(defun unix-sync ()
  "Unix-sync writes all information in core memory which has been
   modified to disk.  It returns NIL and an error code if an error
   occured."
  (multiple-value-bind (value err) (syscall unix-sync)
    (values (if (eql value 0) T) err)))

;;; Unix-fsync writes the core-image of the file described by "fd" to
;;; permanent storage (i.e. disk).

(defun unix-fsync (fd)
  "Unix-fsync writes the core image of the file described by
   fd to disk."
  (declare (type unix-fd fd))
  (multiple-value-bind (value err) (syscall unix-fsync fd)
    (values (if (eql value 0) T) err)))

;;; Unix-truncate accepts a file name and a new length.  The file is
;;; truncated to the new length.

(defun unix-truncate (name len)
  "Unix-truncate truncates the named file to the length (in
   bytes) specified by len.  NIL and an error number is returned
   if the call is unsuccessful."
  (declare (type unix-pathname name)
	   (type (unsigned-byte 32) len))
  (multiple-value-bind (value err) (syscall unix-truncate name len)
    (values (if (eql value 0) T) err)))

(defun unix-ftruncate (fd len)
  "Unix-ftruncate is similar to unix-truncate except that the first
   argument is a file descriptor rather than a file name."
  (declare (type unix-fd fd)
	   (type (unsigned-byte 32) len))
  (multiple-value-bind (value err) (syscall unix-truncate fd len)
    (values (if (eql value 0) T) err)))

(defun unix-symlink (name1 name2)
  "Unix-symlink creates a symbolic link named name2 to the file
   named name1.  NIL and an error number is returned if the call
   is unsuccessful."
  (declare (type unix-pathname name1 name2))
  (multiple-value-bind (value err) (syscall unix-symlink name1 name2)
    (values (if (eql value 0) T) err)))

;;; Unix-unlink accepts a name and deletes the directory entry for that
;;; name and the file if this is the last link.

(defun unix-unlink (name)
  "Unix-unlink removes the directory entry for the named file.
   NIL and an error code is returned if the call fails."
  (declare (type unix-pathname name))
  (multiple-value-bind (value err) (syscall unix-unlink name)
    (values (if (eql value 0) T) err)))

;;; Unix-write accepts a file descriptor, a buffer, an offset, and the
;;; length to write.  It attempts to write len bytes to the device
;;; associated with fd from the the buffer starting at offset.  It returns
;;; the actual number of bytes written.

(proclaim '(inline real-unix-write))

(def-c-routine ("write" real-unix-write)
	       (int)
  (fd int)
  (buffer system-area-pointer)
  (length int))

(defun unix-write (fd buf offset len)
  "Unix-write attempts to write a character buffer (buf) of length
   len to the file described by the file descriptor fd.  NIL and an
   error is returned if the call is unsuccessful."
  (declare (type unix-fd fd)
	   (type (unsigned-byte 32) offset len))
  (let ((value
	 (real-unix-write fd
			  (sap+ (etypecase buf
				  (system-area-pointer buf)
				  ((or simple-string
				       simple-bit-vector
				       (simple-array (unsigned-byte 2) (*))
				       (simple-array (unsigned-byte 4) (*))
				       (simple-array (unsigned-byte 8) (*))
				       (simple-array (unsigned-byte 16) (*))
				       (simple-array (unsigned-byte 32) (*)))
				   (vector-sap buf)))
				offset)
			  len)))
    (values (if (minusp value) nil value)
	    (alien-access (alien-value unix-errno)))))


;;; Unix-ioctl is used to change parameters of devices in a device
;;; dependent way.

(eval-when (compile load eval)
  (defconstant iocparm-mask #x7f)
  (defconstant ioc_void #x20000000)
  (defconstant ioc_out #x40000000)
  (defconstant ioc_in #x80000000)
  (defconstant ioc_inout (logior ioc_in ioc_out))
)

(defrecord kbdarg
  (scan (unsigned-byte 8) (bytes 1))
  (index (unsigned-byte 8) (bytes 1))
  (text (perq-string 32) (bytes 33))
  (end (unsigned-byte 8) (bytes 1)))

(defrecord sgtty
  (ispeed (unsigned-byte 8) (bytes 1))
  (ospeed (unsigned-byte 8) (bytes 1))
  (erase (unsigned-byte 8) (bytes 1))
  (kill (unsigned-byte 8) (bytes 1))
  (flags (unsigned-byte 16) (words 1)))

(defrecord winsize
  (ws_row (unsigned-byte 16) (bytes 2))
  (ws_col (unsigned-byte 16) (bytes 2))
  (ws_xpixel (unsigned-byte 16) (bytes 2))
  (ws_ypixel (unsigned-byte 16) (bytes 2)))

(defconstant terminal-speeds
  '#(nil 50 75 110 nil 150 200 300 600 1200 1800 2400 4800 9600 nil nil))

(defrecord int1
  (int (unsigned-byte 32) (long-words 1)))

(defconstant tty-raw #o40)
(defconstant tty-crmod #o20)
(defconstant tty-echo #o10)
(defconstant tty-lcase #o4)
(defconstant tty-cbreak #o2)
(defconstant tty-tandem #o1)

(eval-when (compile load eval)

(defmacro define-ioctl-command (name dev cmd arg &optional (parm-type :void))
  (declare (fixnum cmd))
  (let* ((rsize (get arg 'record-size))
	 (ptype (case parm-type
		  (:void ioc_void)
		  (:in ioc_in)
		  (:out ioc_out)
		  (:inout ioc_inout)
		  (t (error "Parameter type ~A is illegal." parm-type))))
	 (code (logior (the fixnum (ash (char-code dev) 8)) cmd ptype)))
    (if (null rsize) (setq rsize 0))
    `(eval-when (eval load compile)
       (defconstant ,name ,(logior (ash (logand (truncate rsize 8)
						iocparm-mask) 16) code)))))
)

;;; TTY ioctl commands.

(define-ioctl-command TIOCGETP #\t 8 sgtty :out)
(define-ioctl-command TIOCSETP #\t 9 sgtty :in)
(define-ioctl-command TIOCFLUSH #\t 16 int1 :in)
(define-ioctl-command TIOCSETC #\t 17 tchars :in)
(define-ioctl-command TIOCGETC #\t 18 tchars :out)
(define-ioctl-command TIOCGWINSZ #\t 104 winsize :out)
(define-ioctl-command TIOCSWINSZ #\t 103 winsize :in)

(define-ioctl-command TIOCNOTTY #\t 113 nil :void)
(define-ioctl-command TIOCSLTC #\t 117 ltchars :in)
(define-ioctl-command TIOCGLTC #\t 116 ltchars :out)
(define-ioctl-command TIOCSPGRP #\t 118 int1 :in)
(define-ioctl-command TIOCGPGRP #\t 119 int1 :out)

;;; Keyboard iotctl commands.
(define-ioctl-command KBDCGET #\k 0 kbdarg :inout)
(define-ioctl-command KBDCSET #\k 1 kbdarg :in)
(define-ioctl-command KBDCRESET #\k 2 nil :void)
(define-ioctl-command KBDCRST #\k 3 nil :void)
(define-ioctl-command KBDCSSTD #\k 4 nil :void)
(define-ioctl-command KBDSGET #\k 5 int1 :out)
(define-ioctl-command KBDGCLICK #\k 6 int1 :out)
(define-ioctl-command KBDSCLICK #\k 7 int1 :in)

;;; File ioctl commands.
(define-ioctl-command FIONREAD #\f 127 int1 :out)


(defun unix-ioctl (fd cmd arg)
  "Unix-ioctl performs a variety of operations on open i/o
   descriptors.  See the UNIX Programmer's Manual for more
   information."
  (declare (type unix-fd fd)
	   (type (unsigned-byte 32) cmd))
  (multiple-value-bind (value err) (syscall unix-ioctl fd cmd arg)
    (if (eql value 0)
	(values T 0)
	(values NIL err))))

;;; Unix-exit terminates a program.

(defun unix-exit (&optional (code 0))
  "Unix-exit terminates the current process with an optional
   error code.  If successful, the call doesn't return.  If
   unsuccessful, the call returns NIL and an error number."
  (declare (type (signed-byte 32) code))
  (multiple-value-bind (value err) (syscall unix-exit code)
    (values (if (eql value 0) T) err)))


(defrecord stat
  (dev (signed-byte 16) (words 1))
  (ig1 (signed-byte 16) (words 1))
  (ino (unsigned-byte 32) (long-words 1))
  (mode (unsigned-byte 16) (words 1))
  (nlink (signed-byte 16) (words 1))
  (uid (signed-byte 16) (words 1))
  (gid (signed-byte 16) (words 1))
  (rdev (signed-byte 16) (words 1))
  (ig2 (signed-byte 16) (words 1))
  (size (signed-byte 32) (long-words 1))
  (atime (signed-byte 32) (long-words 1))
  (spare1 (signed-byte 32) (long-words 1))
  (mtime (signed-byte 32) (long-words 1))
  (spare2 (signed-byte 32) (long-words 1))
  (ctime (signed-byte 32) (long-words 1))
  (spare3 (signed-byte 32) (long-words 1))
  (blksize (signed-byte 32) (long-words 1))
  (blocks (signed-byte 32) (long-words 1))
  (spare4 (signed-byte 32) (long-words 1))
  (spare5 (signed-byte 32) (long-words 1)))

(defconstant s_ifmt #o170000)
(defconstant s_ifdir #o40000)
(defconstant s_ifchr #o20000)
(defconstant s_ifblk #o60000)
(defconstant s_ifreg #o100000)
(defconstant s_iflnk #o120000)
(defconstant s_ifsock #o140000)

(defun unix-stat (name)
  "Unix-stat retrieves information about the specified
   file returning them in the form of multiple values.
   See the UNIX Programmer's Manual for a description
   of the values returned.  If the call fails, then NIL
   and an error number is returned instead."
  (declare (type unix-pathname name))
  (with-trap-arg-block stat ab
     (multiple-value-bind (value err)
			  (syscall unix-stat name (alien-sap (alien-value ab)))
       (if (eql value -1)
	   (values NIL err)
	   (values T
		   (alien-access (stat-dev (alien-value ab)))
		   (alien-access (stat-ino (alien-value ab)))
		   (alien-access (stat-mode (alien-value ab)))
		   (alien-access (stat-nlink (alien-value ab)))
		   (alien-access (stat-uid (alien-value ab)))
		   (alien-access (stat-gid (alien-value ab)))
		   (alien-access (stat-rdev (alien-value ab)))
		   (alien-access (stat-size (alien-value ab)))
		   (alien-access (stat-atime (alien-value ab)))
		   (alien-access (stat-mtime (alien-value ab)))
		   (alien-access (stat-ctime (alien-value ab)))
		   (alien-access (stat-blksize (alien-value ab)))
		   (alien-access (stat-blocks (alien-value ab))))))))

(defun unix-lstat (name)
  "Unix-lstat is similar to unix-stat except the specified
   file must be a symbolic link."
  (declare (type unix-pathname name))
  (with-trap-arg-block stat ab
     (multiple-value-bind (value err)
			  (syscall unix-lstat name (alien-sap (alien-value ab)))
       (if (eql value -1)
	   (values NIL err)
	   (values T
		   (alien-access (stat-dev (alien-value ab)))
		   (alien-access (stat-ino (alien-value ab)))
		   (alien-access (stat-mode (alien-value ab)))
		   (alien-access (stat-nlink (alien-value ab)))
		   (alien-access (stat-uid (alien-value ab)))
		   (alien-access (stat-gid (alien-value ab)))
		   (alien-access (stat-rdev (alien-value ab)))
		   (alien-access (stat-size (alien-value ab)))
		   (alien-access (stat-atime (alien-value ab)))
		   (alien-access (stat-mtime (alien-value ab)))
		   (alien-access (stat-ctime (alien-value ab)))
		   (alien-access (stat-blksize (alien-value ab)))
		   (alien-access (stat-blocks (alien-value ab))))))))

(defun unix-fstat (fd)
  "Unix-fstat is similar to unix-stat except the file is specified
   by the file descriptor fd."
  (declare (type unix-fd fd))
  (with-trap-arg-block stat ab
     (multiple-value-bind (value err)
			  (syscall unix-fstat fd (alien-sap (alien-value ab)))
       (if (eql value -1)
	   (values NIL err)
	   (values T
		   (alien-access (stat-dev (alien-value ab)))
		   (alien-access (stat-ino (alien-value ab)))
		   (alien-access (stat-mode (alien-value ab)))
		   (alien-access (stat-nlink (alien-value ab)))
		   (alien-access (stat-uid (alien-value ab)))
		   (alien-access (stat-gid (alien-value ab)))
		   (alien-access (stat-rdev (alien-value ab)))
		   (alien-access (stat-size (alien-value ab)))
		   (alien-access (stat-atime (alien-value ab)))
		   (alien-access (stat-mtime (alien-value ab)))
		   (alien-access (stat-ctime (alien-value ab)))
		   (alien-access (stat-blksize (alien-value ab)))
		   (alien-access (stat-blocks (alien-value ab))))))))

(defrecord rusage
  (utime-seconds (signed-byte 32) (long-words 1))
  (utime-microsc (signed-byte 32) (long-words 1))
  (stime-seconds (signed-byte 32) (long-words 1))
  (stime-microsc (signed-byte 32) (long-words 1))
  (maxrss (signed-byte 32) (long-words 1))
  (ixrss (signed-byte 32) (long-words 1))
  (idrss (signed-byte 32) (long-words 1))
  (isrss (signed-byte 32) (long-words 1))
  (minflt (signed-byte 32) (long-words 1))
  (majflt (signed-byte 32) (long-words 1))
  (nswap (signed-byte 32) (long-words 1))
  (inblock (signed-byte 32) (long-words 1))
  (oublock (signed-byte 32) (long-words 1))
  (msgsnd (signed-byte 32) (long-words 1))
  (msgrcv (signed-byte 32) (long-words 1))
  (nsignals (signed-byte 32) (long-words 1))
  (nvcsw (signed-byte 32) (long-words 1))
  (nivcsw (signed-byte 32) (long-words 1)))

(defconstant rusage_self 0 "The calling process.")
(defconstant rusage_children -1 "Terminated child processes.")

(defun unix-getrusage (who)
  "Unix-getrusage returns information about the resource usage
   of the process specified by who.  Who can be either the
   current process (rusage_self) or all of the terminated
   child processes (rusage_children).  NIL and an error number
   is returned if the call fails."
  (with-trap-arg-block rusage ab
    (multiple-value-bind (value err)
			 (syscall unix-getrusage who
				  (alien-sap (alien-value ab)))
      (when (eql value -1)
	(error "Unix system call getrusage failed: ~A"
	       (mach:get-unix-error-msg err)))
      (values t
	      (+ (* (alien-access (rusage-utime-seconds (alien-value ab)))
		    1000000)
		 (alien-access (rusage-utime-microsc (alien-value ab))))
	      (+ (* (alien-access (rusage-stime-seconds (alien-value ab)))
		    1000000)
		 (alien-access (rusage-stime-microsc (alien-value ab))))
	      (alien-access (rusage-maxrss (alien-value ab)))
	      (alien-access (rusage-ixrss (alien-value ab)))
	      (alien-access (rusage-idrss (alien-value ab)))
	      (alien-access (rusage-isrss (alien-value ab)))
	      (alien-access (rusage-minflt (alien-value ab)))
	      (alien-access (rusage-majflt (alien-value ab)))
	      (alien-access (rusage-nswap (alien-value ab)))
	      (alien-access (rusage-inblock (alien-value ab)))
	      (alien-access (rusage-oublock (alien-value ab)))
	      (alien-access (rusage-msgsnd (alien-value ab)))
	      (alien-access (rusage-msgrcv (alien-value ab)))
	      (alien-access (rusage-nsignals (alien-value ab)))
	      (alien-access (rusage-nvcsw (alien-value ab)))
	      (alien-access (rusage-nivcsw (alien-value ab)))))))

;;; Unix-getrlimit and unix-setrlimit allow setting of the maximum
;;; resource limits for any process (only the super-user can raise
;;; the current limits).  Each limit consists of a soft limit
;;; (rlim-cur) and a hard limit (rlim-max).

#|
(defconstant RLIMIT-CPU     0    "CPU time in seconds.")
(defconstant RLIMIT-FSIZE   1    "Maximum file size.")
(defconstant RLIMIT-DATA    2    "Data segment size.")
(defconstant RLIMIT-STACK   3    "Stack segment size.")
(defconstant RLIMIT-CORE    4    "Core file size.")
(defconstant RLIMIT-RSS     5    "Resident set size.")

(defconstant RLIMIT-NLIMITS 6    "Number of resource limits.")

(defconstant RLIMIT-INFINITY  #x7FFFFFFF   "Infinite value for any limit.")
|#

#+nil
(defrecord rlimit
	(rlim-cur (signed-byte 16) (long-words 1))
	(rlim-max (signed-byte 16) (long-words 1)))

#+nil
(defun unix-getrlimit (resource)
  "Unix-getrlimit returns both the current limit and the
  maximum limit on the resource specified.  The resources
  are:
  
  RLIMIT-CPU        CPU time in seconds.
  RLIMIT-FSIZE      Maximum file size.
  RLIMIT-DATA       Data segment size.
  RLIMIT-STACK      Stack segment size.
  RLIMIT-CORE       Core file size.
  RLIMIT-RSS        Resident set size.
  "
  (with-trap-arg-block rlimit ab
    (multiple-value-bind
	(value err)
	(syscall unix-getrlimit resource
		 (alien-sap (alien-value ab)))
      (if (< value 0)
	  (values NIL err)
	  (values T
		  (alien-access (rlimit-rlim-cur (alien-value ab)))
		  (alien-access (rlimit-rlim-max (alien-value ab))))))))

#+nil
(defun unix-setrlimit (resource soft-lim hard-lim)
  "Unix-setrlimit sets both the current limits (soft-lim)
  and the maximum limits (hard-lim) on the current process
  for the resource specified. (See the description of
  unix-setrlimit.)  NIL and an error number is returned
  if the call fails."
  (with-trap-arg-block rlimit ab
    (setf (alien-access (rlimit-rlim-cur (alien-value ab))) soft-lim)
    (setf (alien-access (rlimit-rlim-max (alien-value ab))) hard-lim)
    (multiple-value-bind
	(value err)
	(syscall unix-setrlimit resource
		 (alien-sap (alien-value ab)))
      (values (eql value 0)
	      err))))


(defun unix-gettimeofday ()
  "Unix-gettimeofday returns 4 values: the seconds and
  microseconds of the current time of day, the timezone
  (in minutes west of Greenwich), and a daylight-savings
  flag.  Signals an error if it is unsuccessful in finding the time."
  (with-trap-arg-block timeval tv
    (with-trap-arg-block timezone tz
      (multiple-value-bind
	  (value err)
	  (syscall unix-gettimeofday
		   (alien-sap (alien-value tv))
		   (alien-sap (alien-value tz)))
	(when (eql value -1)
	  (error "Unix system call gettimeofday failed: ~A"
		 (mach:get-unix-error-msg err)))
	(values T
		(alien-access (timeval-seconds (alien-value tv)))
		(alien-access (timeval-useconds (alien-value tv)))
		(alien-access (timezone-minuteswest (alien-value tz)))
		(alien-access (timezone-dsttime (alien-value tz))))))))

#+nil
(defun unix-settimeofday (seconds useconds minuteswest dsttime)
  "Unix-settimeofday takes four arguments: seconds, microseconds,
   the timezone (in minutes west of Greenwich), and a daylight-
   savings flag.  It sets the current time of day to that time.
   NIL and an error number is returned if an error occurs."
  (with-trap-arg-block timeval tv
    (with-trap-arg-block timezone tz
      (setf (alien-access (timeval-seconds (alien-value tv))) seconds)
      (setf (alien-access (timeval-useconds (alien-value tv))) useconds)
      (setf (alien-access (timezone-minuteswest (alien-value tz))) minuteswest)
      (setf (alien-access (timezone-dsttime (alien-value tz))) dsttime)
      (multiple-value-bind (value err)
			   (syscall unix-settimeofday
				    (alien-sap (alien-value tv))
				    (alien-sap (alien-value tz)))
	(if (eql value -1)
	    (values NIL err)
	    T)))))


;;; Unix-utimes changes the accessed and updated times on UNIX
;;; files.  The first argument is the filename (a string) and
;;; the second argument is a list of the 4 times- accessed and
;;; updated seconds and microseconds.

(def-c-array timevalarr (unsigned-byte 32) 4)

(defun unix-utimes (file utimelist)
  "Unix-utimes sets the 'last-accessed' and 'last-updated'
   times on a specified file.  The argument utimelist is
   a list of 4 times: the seconds and microseconds of the
   accessed date and the seconds and microseconds of the
   updated date in that order.  NIL and an error number is
   returned if the call is unsuccessful."
  (declare (type unix-pathname file) (list utimelist))
  (with-stack-alien (utarray timevalarr (c-sizeof 'timevalarr))
    (setf (alien-access (timevalarr-ref (alien-value utarray) 0))
	  (first utimelist))
    (setf (alien-access (timevalarr-ref (alien-value utarray) 1))
	  (second utimelist))
    (setf (alien-access (timevalarr-ref (alien-value utarray) 2))
	  (third utimelist))
    (setf (alien-access (timevalarr-ref (alien-value utarray) 3))
	  (fourth utimelist))
    (multiple-value-bind (result errcode) 
      (syscall unix-utimes file (alien-sap utarray))
      (values (eql result 0) errcode))))

;;; Some mach specific syscalls.

(def-c-routine ("task_self" mach-task_self)
	       (int))

(def-c-routine ("thread_reply" mach-task_data)
	       (int))

(def-c-routine ("task_notify" mach-task_notify)
	       (int))


;;; Operations on Unix Directories.

(export '(open-dir read-dir close-dir))

(defstruct (directory
	    (:print-function %print-directory))
  name
  (dir-struct (required-argument) :type system-area-pointer))

(defun %print-directory (dir stream depth)
  (declare (ignore depth))
  (format stream "#<Directory ~S>" (directory-name dir)))

(def-c-routine ("opendir" %opendir)
	       (system-area-pointer)
  (name null-terminated-string))

(defun open-dir (pathname)
  (declare (type unix-pathname pathname))
  (let ((kind (unix-file-kind pathname)))
    (case kind
      (:directory
       (let ((dir-struct (%opendir pathname)))
	 (if (zerop (sap-int dir-struct))
	     (values nil (alien-access (alien-value unix-errno)))
	     (make-directory :name pathname :dir-struct dir-struct))))
      ((nil)
       (values nil enoent))
      (t
       (values nil enotdir)))))

(def-c-routine ("readdir" %readdir)
	       (system-area-pointer)
  (dir-struct system-area-pointer))

(defun read-dir (dir)
  (declare (type directory dir))
  (let ((daddr (%readdir (directory-dir-struct dir))))
    (declare (type system-area-pointer daddr))
    (if (zerop (sap-int daddr))
	nil
	(alien-bind ((direct
		      (make-alien 'direct (record-size 'direct) daddr)
		      direct
		      t))
	  (let* ((nlen (alien-access (direct-namelen (alien-value direct))))
		 (ino (alien-access (direct-ino (alien-value direct)))))
	    (declare (type (unsigned-byte 16) nlen))
	    (let ((string (make-string nlen)))
	      (copy-from-system-area
	       daddr (record-size 'direct)
	       string (* vm:vector-data-offset vm:word-bits)
	       (* nlen vm:byte-bits))
	      (values string ino)))))))

(def-c-routine ("closedir" %closedir)
	       (void)
  (dir-struct system-area-pointer))

(defun close-dir (dir)
  (declare (type directory dir))
  (%closedir (directory-dir-struct dir)))


(def-c-routine ("getwd" unix-current-directory)
	       (boolean)
  (name name-pointer :out))



;;;; Support routines for dealing with unix pathnames.

(export '(unix-file-kind unix-maybe-prepend-current-directory
	  unix-resolve-links unix-simplify-pathname))

(defun unix-file-kind (name &optional check-for-links)
  "Returns either :file, :directory, :link, :special, or NIL."
  (declare (simple-string name))
  (multiple-value-bind (res dev ino mode)
		       (if check-for-links
			   (unix-lstat name)
			   (unix-stat name))
    (declare (type (or fixnum null) mode)
	     (ignore dev ino))
    (when res
      (let ((kind (logand mode s_ifmt)))
	(cond ((eql kind s_ifdir) :directory)
	      ((eql kind s_ifreg) :file)
	      ((eql kind s_iflnk) :link)
	      (t :special))))))

(defun unix-maybe-prepend-current-directory (name)
  (declare (simple-string name))
  (if (and (> (length name) 0) (char= (schar name 0) #\/))
      name
      (multiple-value-bind (win dir) (unix-current-directory)
	(if win
	    (concatenate 'simple-string dir "/" name)
	    name))))

(defun unix-resolve-links (pathname)
  "Returns the pathname with all symbolic links resolved."
  (declare (simple-string pathname))
  (let ((len (length pathname))
	(pending pathname))
    (declare (fixnum len) (simple-string pending))
    (if (zerop len)
	pathname
	(let ((result (make-string 1024 :initial-element (code-char 0)))
	      (fill-ptr 0)
	      (name-start 0))
	  (loop
	    (let* ((name-end (or (position #\/ pending :start name-start) len))
		   (new-fill-ptr (+ fill-ptr (- name-end name-start))))
	      (replace result pending
		       :start1 fill-ptr
		       :end1 new-fill-ptr
		       :start2 name-start
		       :end2 name-end)
	      (let ((kind (unix-file-kind (if (zerop name-end) "/" result) t)))
		(unless kind (return nil))
		(cond ((eq kind :link)
		       (multiple-value-bind (link err) (unix-readlink result)
			 (unless link
			   (error "Error reading link ~S: ~S"
				  (subseq result 0 fill-ptr)
				  (get-unix-error-msg err)))
			 (cond ((or (zerop (length link))
				    (char/= (schar link 0) #\/))
				;; It's a relative link
				(fill result (code-char 0)
				      :start fill-ptr
				      :end new-fill-ptr))
			       ((string= result "/../" :end1 4)
				;; It's across the super-root.
				(let ((slash (or (position #\/ result :start 4)
						 0)))
				  (fill result (code-char 0)
					:start slash
					:end new-fill-ptr)
				  (setf fill-ptr slash)))
			       (t
				;; It's absolute.
				(and (> (length link) 0)
				     (char= (schar link 0) #\/))
				(fill result (code-char 0) :end new-fill-ptr)
				(setf fill-ptr 0)))
			 (setf pending
			       (if (= name-end len)
				   link
				   (concatenate 'simple-string
						link
						(subseq pending name-end))))
			 (setf len (length pending))
			 (setf name-start 0)))
		      ((= name-end len)
		       (return (subseq result 0 new-fill-ptr)))
		      ((eq kind :directory)
		       (setf (schar result new-fill-ptr) #\/)
		       (setf fill-ptr (1+ new-fill-ptr))
		       (setf name-start (1+ name-end)))
		      (t
		       (return nil))))))))))

(defun unix-simplify-pathname (src)
  (declare (simple-string src))
  (let* ((src-len (length src))
	 (dst (make-string src-len))
	 (dst-len 0)
	 (dots 0)
	 (last-slash nil))
    (macrolet ((deposit (char)
			`(progn
			   (setf (schar dst dst-len) ,char)
			   (incf dst-len))))
      (dotimes (src-index src-len)
	(let ((char (schar src src-index)))
	  (cond ((char= char #\.)
		 (when dots
		   (incf dots))
		 (deposit char))
		((char= char #\/)
		 (case dots
		   (0
		    ;; Either ``/...' or ``...//...'
		    (unless last-slash
		      (setf last-slash dst-len)
		      (deposit char)))
		   (1
		    ;; Either ``./...'' or ``..././...''
		    (decf dst-len))
		   (2
		    ;; We've found ..
		    (cond
		     ((and last-slash (not (zerop last-slash)))
		      ;; There is something before this ..
		      (let ((prev-prev-slash
			     (position #\/ dst :end last-slash :from-end t)))
			(cond ((and (= (+ (or prev-prev-slash 0) 2)
				       last-slash)
				    (char= (schar dst (- last-slash 2)) #\.)
				    (char= (schar dst (1- last-slash)) #\.))
			       ;; The something before this .. is another ..
			       (deposit char)
			       (setf last-slash dst-len))
			      (t
			       ;; The something is some random dir.
			       (setf dst-len
				     (if prev-prev-slash
					 (1+ prev-prev-slash)
					 0))
			       (setf last-slash prev-prev-slash)))))
		     (t
		      ;; There is nothing before this .., so we need to keep it
		      (setf last-slash dst-len)
		      (deposit char))))
		   (t
		    ;; Something other than a dot between slashes.
		    (setf last-slash dst-len)
		    (deposit char)))
		 (setf dots 0))
		(t
		 (setf dots nil)
		 (setf (schar dst dst-len) char)
		 (incf dst-len))))))
    (when (and last-slash (not (zerop last-slash)))
      (case dots
	(1
	 ;; We've got  ``foobar/.''
	 (decf dst-len))
	(2
	 ;; We've got ``foobar/..''
	 (unless (and (>= last-slash 2)
		      (char= (schar dst (1- last-slash)) #\.)
		      (char= (schar dst (- last-slash 2)) #\.)
		      (or (= last-slash 2)
			  (char= (schar dst (- last-slash 3)) #\/)))
	   (let ((prev-prev-slash
		  (position #\/ dst :end last-slash :from-end t)))
	     (if prev-prev-slash
		 (setf dst-len (1+ prev-prev-slash))
		 (return-from unix-simplify-pathname "./")))))))
    (cond ((zerop dst-len)
	   "./")
	  ((= dst-len src-len)
	   dst)
	  (t
	   (subseq dst 0 dst-len)))))


;;; Unix-setitimer sets the values of one of three interval
;;; timers for the current process.  Which of the three timers
;;; is specified by the constants ITIMER-REAL, -VIRTUAL, and
;;; -PROF as the first argument.  The second argument is a
;;; list of 4 times: the interval seconds, the interval micro-
;;; seconds, the value seconds, and the value microseconds in
;;; that order.

#|
(defconstant ITIMER-REAL    0   "Real time intervals.")
(defconstant ITIMER-VIRTUAL 1   "Virtual time intervals.")
(defconstant ITIMER-PROF    2   "User/system virtual time.")

(def-c-record timeval
	(seconds (signed-byte 32))    ; Seconds.
	(useconds (signed-byte 32)))  ; Microseconds.

(def-c-record itimerval
	(it-interval timeval)    ; Timer interval.
	(it-value timeval))      ; Current values.
|#

#+nil
(defun make-itimer (itimelist)
  (let* ((itimer (make-itimerval))
         (interval (itimerval-it-interval itimer))
         (value (itimerval-it-value itimer)))
     (setf (alien-access (timeval-seconds interval))
		(nth 0 itimelist))
     (setf (alien-access (timeval-useconds interval))
                (nth 1 itimelist))
     (setf (alien-access (timeval-seconds value))
                (nth 2 itimelist))
     (setf (alien-access (timeval-useconds value))
                (nth 3 itimelist))
     itimer))

#+nil
(defun unix-setitimer (which itimelist)
  "Unix-setitimer sets one of three interval timers associated
   with the current process according to the argument which.
   The values for which are ITIMER-REAL, ITIMER-VIRTUAL, and
   ITIMER-PROF.  Itimelist is a list of 4 times: the seconds
   and microseconds of the interval time and the current timer
   value.  NIL and an error number is returned in the event of
   an error."
  (let ((itimer (make-itimer itimelist))
	(ovalue (make-itimerval)))
    (multiple-value-bind (result errcode)
      (syscall unix-setitimer which (alien-sap itimer)
                                    (alien-sap ovalue))
      (if (eql result -1)
	  (values NIL errcode)
 	  (let ((interval (itimerval-it-interval ovalue))
                (value (itimerval-it-value ovalue)))
            (values T
	      (alien-access (timeval-seconds interval))
	      (alien-access (timeval-useconds interval))
              (alien-access (timeval-seconds value))
	      (alien-access (timeval-useconds value))))))))


;;; Unix-getitimer returns the values of one of three
;;; interval timers for the current process.  Which
;;; of the three timer values is returned is specified
;;; by the constants ITIMER-REAL, -VIRTUAL, and -PROF.

#+nil
(defun unix-getitimer (which)
  "Unix-getitimer returns 4 values: the seconds and microseconds
   of the interval time and the current value of the interval
   timer specified by which.  Which can be ITIMER-REAL, ITIMER-
   VIRTUAL, or ITIMER-PROF.  NIL and an error number are returned
   if an error occurs."
  (let ((itimer (make-itimerval)))
    (multiple-value-bind (result errcode)
      (syscall unix-getitimer which (alien-sap itimer))
         (if (eql result -1)
	     (values NIL errcode)
	     (let ((interval (itimerval-it-interval itimer))
                   (value (itimerval-it-value itimer)))
               (values T
		 (alien-access (timeval-seconds interval))
		 (alien-access (timeval-useconds interval))
		 (alien-access (timeval-seconds value))
		 (alien-access (timeval-useconds value)))))))))



;;;; 

;;; Unix-setreuid sets the real and effective user-id's of the current
;;; process to the arguments "ruid" and "euid", respectively.  Usage is
;;; restricted for anyone but the super-user.  Setting either "ruid" or
;;; "euid" to -1 makes the system use the current id instead.

(defun unix-setreuid (ruid euid)
  "Unix-setreuid sets the real and effective user-id's of the current
   process to the specified ones.  NIL and an error number is returned
   if the call fails."
  (multiple-value-bind (value err) (syscall unix-setreuid ruid euid)
    (values (if (eql value 0) T) err)))

;;; Unix-setregid sets the real and effective group-id's of the current
;;; process to the arguments "rgid" and "egid", respectively.  Usage is
;;; restricted for anyone but the super-user.  Setting either "rgid" or
;;; "egid" to -1 makes the system use the current id instead.

(defun unix-setregid (rgid egid)
  "Unix-setregid sets the real and effective group-id's of the current
   process process to the specified ones.  NIL and an error number is
   returned if the call fails."
  (multiple-value-bind (value err) (syscall unix-setregid rgid egid)
    (values (if (eql value 0) T) err)))

(def-c-routine ("getpid" unix-getpid) (int)
  "Unix-getpid returns the process-id of the current process.")

(def-c-routine ("getppid" unix-getppid) (int)
  "Unix-getppid returns the process-id of the parent of the current process.")

(def-c-routine ("getgid" unix-getgid) (int)
  "Unix-getgid returns the real group-id of the current process.")

(def-c-routine ("getegid" unix-getegid) (int)
  "Unix-getegid returns the effective group-id of the current process.")

;;; Unix-getpgrp returns the group-id associated with the
;;; process whose process-id is specified as an argument.
;;; As usual, if the process-id is 0, it refers to the current
;;; process.

(defun unix-getpgrp (pid)
  "Unix-getpgrp returns the group-id of the process associated
   with pid."
  (multiple-value-bind (value err) (syscall unix-getpgrp pid)
    (values (if (< value 0) NIL value) err)))

;;; Unix-setpgrp sets the group-id of the process specified by 
;;; "pid" to the value of "pgrp".  The process must either have
;;; the same effective user-id or be a super-user process.

(defun unix-setpgrp (pid pgrp)
  "Unix-setpgrp sets the process group on the process pid to
   pgrp.  NIL and an error number is returned upon failure."
  (multiple-value-bind (value err) (syscall unix-setpgrp pid pgrp)
    (values (if (eql value 0) T) err)))

(defun unix-getuid ()
  "Unix-getuid returns the real user-id associated with the
   current process."
  (multiple-value-bind (Res err) (syscall unix-getuid)
    (if (eql res -1)
	(values nil err)
	res)))


;;; Unix-getgroups and unix-setgroups make it possible to set
;;; and retrieve the groups for which the current process has
;;; access permission.

#+nil
(eval-when (compile load eval)
  (defconstant ngroups 16))  ; Maximum no of access groups.

#+nil
(def-c-array gidarray (signed-byte 32) ngroups)
			   ; C array of group-access id's.

;;; Make-gidset converts a list of integer group-access id's
;;; to a C array (gidset) of group-access id's which can be
;;; passed to UNIX during unix-setgroups. It returns a C
;;; array of groups.

#+nil
(defun make-gidset (gidlist)
  (do ((gidset (make-gidarray))
       (index 0 (+ index 1))
       (groups gidlist (cdr groups)))
      ((or (null groups) (eql index ngroups))
	gidset)
     (setf (alien-access (gidarray-ref gidset index)) 
		(car groups))))

;;; Make-gidlist converts a C array of group-access id's to
;;; a list of group-access id's which is returned by a call
;;; to unix-getgroups.  It returns a list of groups.

#+nil
(defun make-gidlist (gidset)
  (do* ((gidlist NIL gidlist)
        (index 0 (+ index 1)))
       ((or (eql index ngroups)
	    (eql (alien-access (gidarray-ref gidset index)) 0))
	gidlist)
      (setf gidlist (cons 
	(alien-access (gidarray-ref gidset index)) gidlist))))


;;; Unix-getgroups returns a list of groups for which the
;;; current process has access permission.

#+nil
(defun unix-getgroups ()
  "Unix-getgroups returns a list of groups for which the
   current process has access permission.  NIL and an
   error number is returned if the call fails."
  (let ((gidset (make-gidarray)))
    (multiple-value-bind (result errcode)
      (syscall unix-getgroups ngroups (alien-sap gidset))
      (if (eql result -1)
	  (values NIL errcode)
	  (make-gidlist gidset)))))


;;; Unix-setgroups set the groups for which the current
;;; process has access permission.  It takes a list of
;;; those groups as it's only argument.  Note: Only the
;;; superuser can add access groups.

#+nil
(defun unix-setgroups (gidlist)
  "Unix-setgroups sets the groups for which the current
   process has access to those in gidlist.  NIL and an
   error code is returned if the groups can't be set."
  (multiple-value-bind (result errcode)
    (syscall unix-setgroups (length gidlist)
	(alien-sap (make-gidset gidlist)))
    (values (if (eql result 0) T) errcode)))

;;; CMU-getaid gets the account for the current process.

#+nil
(defun cmu-getaid ()
  "CMU-getaid returns the account number of the current process."
  (multiple-value-bind (result errcode)
		       (syscall cmu-getaid)
    (if (/= result -1)
	result
	(values nil errcode))))

;;; Unix-getpriority returns the current priority of the process,
;;; process-group, or user (determined by the argument "which")
;;; with the id of "who".  The value returned is between -20
;;; (highest priority) and 20 (lowest priority).

#|
(defconstant PRIO-MIN -20 "Minimum (most favorable) priority")
(defconstant PRIO-MAX 20 "Maximum (pleast favorable) priority")

;;; Possibly values for "which".
(defconstant PRIO-PROCESS 0 "Process itself")
(defconstant PRIO-PGRP 1 "Process group")
(defconstant PRIO-USER 2 "User-id")
|#

#+nil
(defun unix-getpriority (which who)
  "Unix-getpriority returns the priority of the current process,
   process-group, or user (determined by the argument which)
   with the id who.  Which can be either PRIO-PROCESS, PRIO-PGRP,
   or PRIO-USER."
  (multiple-value-bind (result errcode)
    (syscall unix-getpriority which who)
    (if (eql result errcode)
	(values T result)
	(values NIL errcode))))

;;; Unix-setpriority sets the priority of the process, process-group,
;;; or user (determined by the argument "which") with the id of "who"
;;; to the new priority of "prio".  The value of "prio" must be
;;; between -20 and 20 or an error will result.  (Only the super-user
;;; may lower the priority value.)

#+nil
(defun unix-setpriority (which who prio)
   "Unix-setpriority sets the priority of the process, process-group,
    or user (determined by the argument "which") with the id of "who"
    to the new priority of "prio".  The value of "prio" must be
    between -20 and 20 or an error will result.  (Only the super-user
    may lower the priority value.)"
  (multiple-value-bind (result errcode) 
    (syscall unix-setpriority which who prio)
    (values (if (eql result 0) T) errcode)))

;;; Unix-setquota allows the superuser to disable/enable quotas
;;; for a file system.  The quotas are taken from the argument
;;; file.

#+nil
(defun unix-setquota (special file)
  "Unix-setquota allows the superuser to disable/enable quotas
   for a file system.  The quotas are taken from the argument
   file."
  (multiple-value-bind (result errcode)
    (syscall unix-setquota special file)
    (values (if (eql result 0) T) errcode)))

;;; Unix-getpagesize returns the number of bytes in the system page.

(defun unix-getpagesize ()
  "Unix-getpagesize returns the number of bytes in a system page."
  (multiple-value-bind (value err) (syscall unix-getpagesize)
    (values (if (< value 0) NIL value) err)))



;;; Unix-sethostname allows the superuser to set the name of the
;;; local machine.



#+nil
(defun unix-sethostname (hostname)
  "Unix-sethostname allows the super-user to set the name of the
   host machine."
  (multiple-value-bind (result errcode)
    (syscall unix-sethostname hostname (length hostname))
    (values (if (eql result 0) T) errcode)))

;;; Returns the standard name of the local host machine.

(def-c-routine ("gethostname" real-unix-gethostname)
	       (int)
  (name name-pointer :out)
  (len int))

(defun unix-gethostname ()
  "Unix-gethostname returns the name of the host machine as 
   a string."
  (multiple-value-bind (result name)
		       (real-unix-gethostname 64)
    (if (eql result -1)
	(values NIL (alien-access (alien-value unix-errno)))
	name)))

;;; Returns the unique 32-bit integer identifier for the
;;; the current machine.

(def-c-routine ("gethostid" unix-gethostid) (unsigned-long)
  "Unix-gethostid returns a 32-bit integer which provides unique
   identification for the host machine.")

;;; Sets the unique 32-bit integer identifier for the
;;; current machine.  (Allowed only to the superuser.)

#+nil
(defun unix-sethostid (hostid)
  "Unix-sethostid sets the 32-bit unique identifying integer
   for the host machine."
  (multiple-value-bind (result errcode)
    (syscall unix-sethostid hostid)
    (values (if (eql result 0) T) errcode)))


;;; Information about the terminal we are attached to.

(def-c-routine ("isatty" unix-isatty) (boolean)
  "Accepts a Unix file descriptor and returns T if the device
  associated with it is a terminal."
  (fd int))

(def-c-routine ("ttyname" unix-ttyname) (null-terminated-string)
  (fd int))

(defun unix-isconsole ()
  "Returns T if the current terminal is the console."
  (multiple-value-bind (cres cdev cino) (unix-stat "/dev/console")
    (when cres
      (multiple-value-bind (res dev ino) (unix-fstat 0)
	(if (and res (eql dev cdev) (eql cino ino)) T NIL)))))

;;; Unix process primitives.

(def-c-routine ("fork" unix-fork) (int)
  "Executes the unix fork system call.  Returns 0 in the child and the pid
  of the child in the parent if it works, or -1 if it doesn't work.")

(defconstant wstopped  #o177)

#+nil
(defun unix-wait ()
  "Executes the unix wait system call, returning status information."
  (multiple-value-bind (res err)
		       (%primitive unix-wait)
    (if (eql res -1)
	(values NIL err)
	(let* ((status (logand err #xFFFF))
	       (retcode (logand (ash status -8) #xFF))
	       (termcode (logand status #xFF)))
	  (declare (fixnum status termcode retcode))
	  (if (eql termcode wstopped)
	      (values res retcode termcode)
	      (values res retcode (ash termcode -7)
		      (logand termcode #x7F)))))))

(defconstant wnohang 1)
(defconstant wuntraced 2)

#|
(defun unix-wait3 (options)
  "Executes Unix wait3 system call, returning status information.
   Currently, the resource usage information is not returned."
  (with-trap-arg-block int1 ws
    (multiple-value-bind (res err) (syscall unix-wait3
					    (lisp::alien-value-sap int1)
					    options 0)
      (if (eql res -1)
	  (values NIL err)
	  (let* ((status (logand (alien-access (int1-int (alien-value ws)))
				 #xFFFF))
		 (termcode (logand (ash status -8) #xFF))
		 (retcode (logand status #xFF)))
	    (declare (fixnum status termcode retcode))
	    (if (eql termcode wstopped)
		(values res retcode termcode)
		(values res retcode (logand (ash status -9) #x7F)
			(logand termcode 1))))))))
|#


;;;; UNIX-EXECVE

(defun unix-execve (program &optional arg-list
			    (environment *environment-list*))
  "Executes the Unix execve system call.  If the system call suceeds, lisp
   will no longer be running in this process.  If the system call fails this
   function returns two values: NIL and an error code.  Arg-list should be a
   list of simple-strings which are passed as arguments to the exec'ed program.
   Environment should be an a-list mapping symbols to simple-strings which this
   function bashes together to form the environment for the exec'ed program."
  (check-type program simple-string)
  (let ((env-list (let ((envlist nil))
		    (dolist (cons environment)
		      (push (if (cdr cons)
				(concatenate 'simple-string
					     (string (car cons)) "="
					     (cdr cons))
				(car cons))
			    envlist))
		    envlist)))
    (sub-unix-execve program arg-list env-list)))


(defmacro round-bytes-to-words (n)
  `(logand (the fixnum (+ (the fixnum ,n) 3)) (lognot 3)))

;;;
;;; STRING-LIST-TO-C-STRVEC	-- Internal
;;; 
;;; STRING-LIST-TO-C-STRVEC is a function which takes a list of
;;; simple-strings and constructs a C-style string vector (strvec) --
;;; a null-terminated array of pointers to null-terminated strings.
;;; This function returns two values: a sap and a byte count.  When the
;;; memory is no longer needed it should be deallocated with
;;; vm_deallocate.
;;; 
(defun string-list-to-c-strvec (string-list)
  ;;
  ;; Make a pass over string-list to calculate the amount of memory
  ;; needed to hold the strvec.
  (let ((string-bytes 0)
	(vec-bytes (* 4 (1+ (length string-list)))))
    (declare (fixnum string-bytes vec-bytes))
    (dolist (s string-list)
      (check-type s simple-string)
      (incf string-bytes (round-bytes-to-words (1+ (length s)))))
    ;;
    ;; Now allocate the memory and fill it in.
    (let* ((total-bytes (+ string-bytes vec-bytes))
	   (vec-sap (system:allocate-system-memory total-bytes))
	   (string-sap (sap+ vec-sap vec-bytes))
	   (i 0))
      (declare (type (and unsigned-byte fixnum) total-bytes i)
	       (type system:system-area-pointer vec-sap string-sap))
      (dolist (s string-list)
	(declare (simple-string s))
	(let ((n (length s)))
	  ;; 
	  ;; Blast the string into place
	  (copy-to-system-area (the simple-string s)
			       (* vm:vector-data-offset vm:word-bits)
			       string-sap 0
			       (* (1+ n) vm:byte-bits))
	  ;; 
	  ;; Blast the pointer to the string into place
	  (setf (sap-ref-sap vec-sap i) string-sap)
	  (setf string-sap (sap+ string-sap (round-bytes-to-words (1+ n))))
	  (incf i)))
      ;; Blast in last null pointer
      (setf (sap-ref-sap vec-sap i) (int-sap 0))
      (values vec-sap total-bytes))))

(defun sub-unix-execve (program arg-list env-list)
  (let ((argv nil)
	(argv-bytes 0)
	(envp nil)
	(envp-bytes 0)
	result error-code)
    (unwind-protect
	(progn
	  ;; Blast the stuff into the proper format
	  (multiple-value-setq
	      (argv argv-bytes)
	    (string-list-to-c-strvec arg-list))
	  (multiple-value-setq
	      (envp envp-bytes)
	    (string-list-to-c-strvec env-list))
	  ;;
	  ;; Now do the system call
	  (multiple-value-setq
	      (result error-code)
	    (syscall unix-execve program
		     (the system:system-area-pointer argv)
		     (the system:system-area-pointer envp))))
      ;; 
      ;; Deallocate memory
      (when argv
	(system:deallocate-system-memory argv argv-bytes))
      (when envp
	(system:deallocate-system-memory envp envp-bytes)))
    (values result error-code)))


;;;; Sockets.

#|
(defconstant sock-stream 1)
(defconstant sock-dgram 2)
(defconstant sock-raw 3)

(defconstant af-unix 1)
(defconstant af-inet 2)

(defconstant msg-oob 1)
(defconstant msg-peek 2)
(defconstant msg-dontroute 4)
|#

#+nil
(defmacro defboolsyscall (syscall args &rest docs)
  `(defun ,syscall ,args
     ,@docs
     (multiple-value-bind (value err)
			  (syscall ,syscall ,@args)
       (values (not (minusp value))
	       err))))
#+nil
(defmacro defintsyscall (syscall args &rest docs)
  `(defun ,syscall ,args
     ,@docs
     (multiple-value-bind (value err)
			  (syscall ,syscall ,@args)
       (values (and (not (minusp value))
		    value)
	       err))))

#+nil
(defintsyscall unix-socket (domain type protocol))

#+nil
(defun unix-bind (sock name)
  (multiple-value-bind (value err)
		       (syscall unix-bind
				sock
				(alien-address name)
				(/ (alien-size name) 8))
    (values (not (minusp value))
	    err)))

#+nil
(defboolsyscall unix-listen (sock backlog))

#+nil
(defun unix-accept (sock &optional from)
  (if (null from)
      (multiple-value-bind (value err)
			   (syscall unix-accept
				    sock
				    0
				    0)
	(values (and (not (minusp value)) value)
		err))
      (with-stack-alien (len (unsigned-byte 32) (long-words 1))
	(setf (alien-access (alien-value len))
	      (/ (alien-size from) 8))
	(multiple-value-bind (value err)
			     (syscall unix-accept
				      sock
				      (alien-address from)
				      (alien-address (alien-value len)))
	  (if (minusp value)
	      (values nil err)
	      (values value
		      (alien-access (alien-value len))))))))

#+nil
(defun unix-connect (sock to)
  (multiple-value-bind (value err)
		       (syscall unix-connect
				sock
				(alien-sap to)
				(/ (alien-size to)
				   8))
    (values (not (minusp value))
	    err)))

#+nil
(defboolsyscall unix-send (fd msg len flags))

#+nil
(defun unix-sendto (fd msg len flags to)
  (multiple-value-bind (value err)
		       (syscall unix-sendto
				fd
				msg
				len
				flags
				to
				(/ (alien-size to)
				   8))
    (values (not (minusp value))
	    err)))

#+nil
(defboolsyscall unix-recv (fd msg len flags))

#+nil
(defun unix-recvfrom (fd msg length flags from)
  (with-stack-alien (len (unsigned-byte 32) (long-words 1))
    (setf (alien-access (alien-value len))
	  (/ (alien-size from) 8))
    (multiple-value-bind (value err)
			 (syscall unix-recvfrom fd msg length flags from
				  (alien-address (alien-value len)))
      (if (minusp value)
	  (values nil err)
	  (values t
		  (alien-access (alien-value len)))))))



;;;; Support For RFS authentication.

;;; rfsRW_t is a c enumeration but is cast to a u_char for storage

(defconstant RFSRW_NONE 0)
(defconstant RFSRW_USER 1)
(defconstant RFSRW_GROUP 2)
(defconstant RFSRW_ACCOUNT 3)
(defconstant RFSRW_PASSWORD 4)
(defconstant RFSRW_LOOKUP 5)

(eval-when (compile load eval)

(defrecord rfsRW_t
  (rfsRW_t (unsigned-byte 8) 8))

) ; eval-when

(define-ioctl-command RFSIOCGETRW #\R 0 rfsRW_t :out)
(define-ioctl-command RFSIOCSETRW #\R 1 rfsRW_t :in)
(define-ioctl-command RFSIOCGETS #\R 2 int1 :out)
(define-ioctl-command RFSIOCSETS #\R 3 int1 :in)
(define-ioctl-command RFSIOCIDENTIFY #\R 4 int1 :void)

;;;; Opening and Closing the Control File

(defconstant rfs-control-file "/../.CONTROL")
(defvar *rfs-control-fd* nil)

(defun open-rfs-control-file ()
  (when *rfs-control-fd*
    (close-rfs-control-file))
  (multiple-value-bind
      (fd err)
      (mach:unix-open rfs-control-file mach:o_rdwr 0)
    (unless fd
      (error "Open failed on RFS Control file: ~A."
	     (mach:get-unix-error-msg err)))
    (setf *rfs-control-fd* fd)))

(defun close-rfs-control-file ()
  (when *rfs-control-fd*
    (multiple-value-bind
	(won err)
	(mach:unix-close *rfs-control-fd*)
      (unless won
	(error "Close failed on RFS Control file: ~A."
	       (mach:get-unix-error-msg err)))
      (setf *rfs-control-fd* nil))))

;;;; IOCTL Calls to Setup Remote RFS Priviledges

(defun set-rfs-default (mode string)
  (with-trap-arg-block rfsRW_t argp
    (setf (alien-access (rfsRW_t-rfsRW_t (alien-value argp))) mode)
    (multiple-value-bind
	(won err)
	(mach:unix-ioctl *rfs-control-fd*
			 mach:RFSIOCSETRW
			 (alien-sap (alien-value argp)))
      (unless won
	(error "IOCTL call failed to set RFS default: ~A."
	       (mach:get-unix-error-msg err)))
      (multiple-value-bind
	  (won err)
	  (mach:unix-write *rfs-control-fd* string 0 (length string))
	(unless won
	  (error "Write failed on RFS Control file: ~A."
		  (mach:get-unix-error-msg err)))))))

(defun rfs-authenticate (user group account password)
  "RFS-Authenticate authenticates a process for remote file system
  access through RFS."
  (unwind-protect
      (progn
	(open-rfs-control-file)
	(when user
	  (set-rfs-default mach:RFSRW_USER user))
	(when group
	  (set-rfs-default mach:RFSRW_GROUP group))
	(when account
	  (set-rfs-default mach:RFSRW_ACCOUNT account))
	(when password
	  (set-rfs-default mach:RFSRW_PASSWORD password)))
    (close-rfs-control-file)))
