;;; -*- Mode: LISP; Syntax: Common-lisp; Package: YY; Base: 10 -*-
;;; yy-clolor.lisp
;;; 
;;;  Copyright (C) 1989,1990,1991 Aoyama Gakuin University
;;;
;;;             All Rights Reserved
;;;
;;; This software is developed for the YY project of Aoyama Gakuin University.
;;; Permission to use, copy, modify, and distribute this software
;;; and its documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notices appear in all copies and that
;;; both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of Aoyama Gakuin
;;; not be used in advertising or publicity pertaining to distribution of
;;; the software without specific, written prior permission.
;;;
;;; This software is made available AS IS, and Aoyama Gakuin makes no
;;; warranty about the software, its performance or its conformity to
;;; any specification.
;;;
;;; To make a contact: Send E-mail to ida@csrl.aoyama.ac.jp for overall
;;; issues. To ask specific questions, send to the individual authors at
;;; csrl.aoyama.ac.jp. To request a mailing list, send E-mail to
;;; yyonx-request@csrl.aoyama.ac.jp.
;;;
;;; Authors:
;;;   version 1.3 91/1/11 by t.kosaka

;;; $B%+%i!<4pK\%/%i%9$NDj5A5Z$S4XO"%a%=%C%I(B


(in-package :yy)

(defvar *top-color-table* nil)
(defvar *near-val* 15)

;;; $B%+%i!<8!:w$N$?$a$N9=B$BN(B
(defstruct color-table
  color                ;;; $B%+%i!<%$%s%9%?%s%9(B
  key                  ;;; $B80(B
  big
  small)

;;; $B?'%/%i%9(B
(defclass color
;;; $B?'$N%/%i%9(B
  ()
  ((red   :initarg :red-no :type integer :accessor red)
   (green :initarg :green-no :type integer :accessor green)
   (blue  :initarg :blue-no :type integer :accessor blue)
   (color-no :initform 0 :initarg :color-no :type integer :accessor color-no)))

;;; color $B%$%s%9%?%s%9I=<(%a%=%C%I(B
(defmethod print-object ((color color) stream)
;;;$B?'$N%$%s%9%?%s%9$rI=<((B
  (format stream "\#<Color RGB [~a ~a ~a]>"
          (red color) (green color) (blue color)))

;;; color $B@8@.4X?t(B
;;; make-color &key (red 0) (green 0) (blue 0)
;;; ARG.  red   =  RGB$B$G$N@V(B
;;;       blue  =  RGB$B$G$N@D(B
;;;       green =  RGB$B$G$NNP(B
;;; RET.
;;;       color instance or NIL

(defun make-color (&key (red 0) (green 0) (blue 0))
;;; $B%G%U%)%k%H$N%+%i!<$O!"(Bred 0 green 0 blue 0$B$G!"9u$K$J$k(B
  (declare (special *all-colors*)
  	   (inline < car push))
  (let ((no (yy-protocol-45 red green blue)))
    (if (< no 0)
	nil
      (let ((color (make-instance  'color :red-no red :color-no no
				 :green-no green :blue-no blue)))
	(set-tree-color color)
	(push color *all-colors*)
	color))))

;;; $B%+%i!<$NLZ9=B$$r:n$k(B
(defun set-tree-color (color)
  (declare (special *top-color-tabel*))
  (let ((top *top-color-tabel*)
	(no (logxor (red color) (green color) (blue color))))
    (if top
	(loop
	  (unless top
	    (return))
	  (cond 
	   ((>= no (color-table-key top))
	    (if (null (color-table-big top))
		(return (setf (color-table-big top)
			  (make-color-table :color color
					    :key no)))
	      (setf top (color-table-big top))))
	   ((< no (color-table-key top))
	    (if (null (color-table-small top))
		(return (setf (color-table-small top)
			  (make-color-table :color color
                                            :key no)))
	    (setf top (color-table-small top))))
	  ))
      (setf *top-color-tabel*
	(make-color-table :color color :key no))))
  )

;;;$B%+%i!<LZ9=B$$+$i$N2rJ|(B
(defun delete-tree-color (color)
  (declare (special *top-color-tabel*))
  (let ((top *top-color-tabel*)
	(no (logxor (red color) (green color) (blue color))))
    (loop
      (unless top
          (return nil))
      (cond
       ((> no (color-table-key top))
	(if (null (color-table-big top))
	    (return nil)
	  (setf top (color-table-big top))))

       ((<  no (color-table-key top))
        (if (null (color-table-small top))
	    (return nil)
	  (setf top (color-table-small top))))
       (t
	(if (null (color-table-color top))
	    (setf top (color-table-big top))
	  (if (eq color (color-table-color top))
	      (return (setf (color-table-color top) nil))
	    (setf top (color-table-big top)))
	  ))
       ))))

;;; RGB$B$K$h$k%+%i!<LZ9=B$$+$i$N8!:w(B
(defun find-from-rgb (red green blue)
  (declare (special *top-color-tabel*))
  (let ((top *top-color-tabel*)
	(no (logxor red green blue)))
    (loop 
      (unless top
          (return nil))
      (cond
       ((> no (color-table-key top))
	(if (null (color-table-big top))
	    (return nil)
	  (setf top (color-table-big top))))

       ((<  no (color-table-key top))
        (if (null (color-table-small top))
	    (return nil)
	  (setf top (color-table-small top))))
       (t
	(if (null (color-table-color top))
	    (setf top (color-table-big top))
	  (if (and (= (red (color-table-color top)) red)
		   (= (green (color-table-color top)) green)
		   (= (blue(color-table-color top)) blue))
	      (return (color-table-color top))
	    (setf top (color-table-big top)))
	  ))
       ))))

;;; RGB$B$K$h$k%+%i!<LZ9=B$$+$i$N8!:w(B
(defun near-from-rgb (red green blue)
  (declare (special *top-color-tabel*))
  (let ((top *top-color-tabel*)
	(no (logxor red green blue)))
    (loop 
      (unless top
          (return nil))
      (cond
       ((> no (color-table-key top))
	(if (null (color-table-big top))
	    (return nil)
	  (setf top (color-table-big top))))

       ((<  no (color-table-key top))
        (if (null (color-table-small top))
	    (return nil)
	  (setf top (color-table-small top))))
       (t
	(if (null (color-table-color top))
	    (setf top (color-table-big top))
	  (if (and (> (+ (red (color-table-color top)) *near-val*) red)
		   (< red (- (red (color-table-color top)) *near-val*))
		   (> (+ (green (color-table-color top)) *near-val*) green)
		   (< green (- (green (color-table-color top)) *near-val*))
		   (> (+ (blue(color-table-color top)) *near-val*) blue)
		   (< blue (- (blue(color-table-color top)) *near-val*)))
	      (return (color-table-color top))
	    (setf top (color-table-big top)))
	  ))
       ))))

;;; $B%+%i!<$NLd$$9g$o$;(B
;;; find-color  &key (red 0) (green 0) (blue0)
;;; ARG.  red   =  RGB$B$G$N@V(B
;;;       blue  =  RGB$B$G$N@D(B
;;;       green =  RGB$B$G$NNP(B
;;; RET.
;;;       color instance or NIL
(defun find-color (&key (red 0) (green 0) (blue 0))
  (declare (special *all-colors*)
  	   (inline = ))
  (find-from-rgb red green blue))


;;; $BL>A0$G%+%i!<$r:n$k(B
;;; make-color-of-name color-name
;;; ARGS. color-name  = $B%+%i!<$NL>A0(B
;;; RET.
;;;      color instance or NIL
(defun make-color-of-name (color-name)
  (declare (special *all-colors*)
  	   (inline < car))
  (let ((list (yy-protocol-46 color-name)))
    (if (< (car list) 0)
        nil
      (let ((color (make-instance 'color 
		     :color-no (car list)
		     :red-no (second list)
		     :green-no (third list) 
		     :blue-no (nth 4 list))))
	(set-tree-color color)
	(push color *all-colors*)
	color))))

;;; $B%+%i!<%;%k$r2rJ|$9$k(B
;;; remove-colors &rest colors
;;;  ARG.  colors   =  $B%+%i!<%$%s%9%?%s%9(B
;;;  RET.  
;;;        NIL
(defun remove-colors (&rest colors)
  (declare (special *all-colors*))
  (dolist (item colors)
    (delete item *all-colors*)
    (delete-tree-color item))

  (yy-protocol-52 (mapcar #'color-no colors)))

;;; $B%+%i!<$N#R#G#B$NCM$rJQ99$9$k(B
;;; replace-color color &key (red (red color))
;;;			     (blue (blue color))
;;;			     (green (green color))
;;; ARG.
;;;     color $B%+%i!<%$%s%9%?%s%9(B
;;;     red   RGB$B$N@V$NCM(B
;;;     blue  RGB$B$N@D$NCM(B
;;;     green RGB$B$NNP$NCM(B
(defmethod replace-color ((color color) &key (red (red color))
					     (blue (blue color))
					     (green (green color)))
  (delete-tree-color color)
  
  (setf (red color) red
	(blue color) blue
	(green color) green)
  (set-tree-color color)
  ;;; Call YY protocol
  (yy-protocol-51 (color-no color) red green blue)
  color)

;;; $B%+%i!<%*%V%8%'%/%H$N%+%i!<HV9f$rJV$9(B
;;; color-code color-object
;;; ARG.
;;;      color-object  $B%+%i!<%$%s%9%?%s%9(B
;;; RET.
;;;;     $B%+%i!<HV9f(B
(defmethod color-code ((color color))
   (color-no color))

;;; $B%+%i!<HV9f$+$i%+%i!<%$%s%9%?%s%9$rJV$9(B
;;; code-color color-code
;;; ARG.  color-code  $B%+%i!<HV9f(B
;;; RET.
;;;       $B%+%i!<%$%s%9%?%s%9(Bor NIL
(defun code-color (color-no)
  (declare (special *all-colors*)
  	   (inline =))
  (dolist (color *all-colors*)
      (when (= (color-no color) color-no)
	(return color)))
	)



