;;; -*- Mode: LISP; Syntax: Common-lisp; Package: YY; Base: 10 -*-
;;; graphic-method.lisp
;;;
;;;  Copyright (C) 1989,1990,1991 Aoyama Gakuin University
;;;
;;;		All Rights Reserved
;;;
;;; This software is developed for the YY project of Aoyama Gakuin University.
;;; Permission to use, copy, modify, and distribute this software
;;; and its documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notices appear in all copies and that
;;; both that copyright notice and this permission notice appear in 
;;; supporting documentation, and that the name of Aoyama Gakuin
;;; not be used in advertising or publicity pertaining to distribution of
;;; the software without specific, written prior permission.
;;;
;;; This software is made available AS IS, and Aoyama Gakuin makes no
;;; warranty about the software, its performance or its conformity to
;;; any specification. 
;;;
;;; To make a contact: Send E-mail to ida@csrl.aoyama.ac.jp for overall
;;; issues. To ask specific questions, send to the individual authors at
;;; csrl.aoyama.ac.jp. To request a mailing list, send E-mail to 
;;; yyonx-request@csrl.aoyama.ac.jp.
;;;
;;; Authors:
;;;   version 1.0 90/06/01 by t.kosaka (kosaka@csrl.aoyama.ac.jp)
;;;   version 1.1 90/07/31 by t.kosaka
;;;   update 1.11 90/09/14 by t.kosaka
;;;   version 1.2 90/11/05 by t.kosaka
;;;   version 1.3 90/12/19 by t.kosaka
;;;   version 1.3 91/01/22 by t.kosaka

;;; Drawing Method
;;; 1990/03/23 written by Yukio Ohta
;;; Update 1990/06/06 written by T.Kosaka
;;; draw-position,draw-position-xy,draw-line,draw-line-xy,
;;; draw-region,draw-region-xy,draw-polygon,draw-polygon-xy,
;;; draw-polyline,draw-poly-line-xy,draw-triange,draw-teriangle-xy,
;;; draw-ellipse,draw-ellipse-xy,draw-circle,draw-circle-xy,
;;; draw-string,draw-string-xy,draw-character,draw-character-xy,
;;; 
;;; Version 1.3 Add after method at drawing primitive method.
;;;             Add method for real-primitive function like a yy-protocol-xx.
;;              Becouse presentaion use grass territory and I can not draw
;;;             wold territory. So, I need some after drawing mehtod 
;;;             for real primitive method.
;;;             Also, Add method for image handling.
(in-package :yy)


;;; $BE@$NIA2h(B 
;;; draw-position-xy graphic-stream x y
;;; ASG. 
;;;         graphic-stream     = $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;         x y                = $B0LCV(B
(defmethod draw-position-xy ((stream graphic-stream) (x integer) (y integer))
  (draw-position-xy-internal stream x y))


;;; draw-position$BIA2h4X?t$N%W%j%_%F%#%V(B
(defun draw-position-xy-internal (stream x y)
  (declare (inline + max min))
  (with-translate-transform-xy ((new-x new-y) stream x  y)
    (with-temp-region-args ((draw-ponit-region) (work-region1 stream)
			   :left new-x :right new-x
			   :top new-y :bottom new-y)
     (let* ((tno (world-territory-no stream))
	    (drawing-region (drawing-region stream))
	    ;; graphic-statement
	    (color (graphic-color stream))
	    (operation (graphic-operation stream)))

        ;;; $B%o!<%k%I%j!<%8%g%s$NJQ99(B
       (setf (world-region stream) draw-ponit-region)

      ;;; $B0LCV$NJQ99(B
      (setf new-x (+ new-x (world-x-start stream))
	    new-y (+ new-y (world-y-start stream)))
      
      ;;; call draw primitive
      (yy-protocol-20 tno new-x new-y operation (color-no color))

      ;;; drawing-region$B$K:#$N%j!<%8%g%s$rDI2C(B
      (set-drawing-region drawing-region draw-ponit-region)
      stream))))

;;; $BE@$NIA2h(B $B%]%8%7%g%sMQ(B
;;; draw-position graphic-stram position
;;; ARG.
;;;             graphci-stream   =  $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;             psosition        =  $B%0%i%U%#%/%9%9%H%j!<%`$G$N0LCV(B
(defmethod draw-position ((stream graphic-stream) (position position))
  (draw-position-xy-internal
       stream (position-x position) (position-y position)))

;;; $B@~$NIA2h(B
;;; draw-line-xy stream x1 y1 x2 y2
;;; ARG.
;;;               stream        =   $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;               x1 y1 x2 y2   =   $B@~$N:BI8(B
(defmethod draw-line-xy ((stream graphic-stream) (x1 integer) (y1 integer)
			 (x2 integer) (y2 integer))
  (draw-line-xy-internal stream x1 y1 x2 y2))

;;; draw-line$BIA2h4X?t$N%W%j%_%#%F%V(B
(defun draw-line-xy-internal (stream x1 y1 x2 y2)
  (declare (inline + max min))
  (with-translate-transform-xy ((new-x1 new-y1) stream x1 y1)
    (with-translate-transform-xy ((new-x2 new-y2) stream x2 y2)
      (with-temp-region-args ((draw-line-region) (work-region1 stream)
			     :left (min new-x1 new-x2)
			     :top (max new-y1 new-y2)
			     :right (max new-x1 new-x2)
			     :bottom (min new-y1 new-y2))
      (let* ((tno (world-territory-no stream))
	     (drawing-region (drawing-region stream))
	     ;; graphic-statement
	     (color (color-no (graphic-color stream)))
	     (operation (graphic-operation stream))
	     (line-width (line-width stream))
	     (line-edge (line-edge stream))
	     (line-dash (line-dashing stream)))


        ;;; $B%o!<%k%I%j!<%8%g%s$NJQ99(B
	(setf (world-region stream) draw-line-region)


      ;;; $B0LCV$NJQ99(B
      (setf new-x1 (+ new-x1 (world-x-start stream))
	    new-y1 (+ new-y1 (world-y-start stream))
	    new-x2 (+ new-x2 (world-x-start stream))
	    new-y2 (+ new-y2 (world-y-start stream)))
      
      ;;; $BIA2h%W%j%_%F%#%V(B
      (yy-protocol-21 tno new-x1 new-y1 new-x2 new-y2
		      line-width operation line-edge color line-dash)

        ;;; drawing-region$B$K:#$N%j!<%8%g%s$rDI2C(B
      (set-drawing-region drawing-region draw-line-region)
      stream)))))


;;; $B@~$NIA2h(B $B%]%8%7%g%sMQ(B
;;; draw-line stream position1 position2
;;; ARG
;;;           stream              =   $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;           position1 position2 =   $B@~$NIA2h$9$k0LCV%$%s%9%?%s%9(B
(defmethod draw-line ((stream graphic-stream) (position1 position)
		      (position2 position))
	(draw-line-xy-internal
	        stream (position-x position1) (position-y position1)
			     (position-x position2) (position-y position2)))


;;; $B6k7A$NIA2h(B
;;; draw-region-xy stream x y width height
;;; ARG.
;;;           stream           =  $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;           x y              =  $B6k7A$rIA2h$9$k3+;O0LCV(B
;;;           width height     =  $B6k7A$NBg$-$5(B
(defmethod draw-region-xy ((stream graphic-stream) (x integer) (y integer)
			 (width integer) (height integer))
  (draw-region-xy-internal stream x y width height))

;;; draw-region-xy $BIA2h%W%j%_%F%#%V(B
(defun draw-region-xy-internal (stream x y width height)
  (declare (inline + min max))
  (with-translate-transform-xy ((new-x new-y) stream x y)
   (with-temp-region-args ((draw-region-region) (work-region1 stream)
                           :left new-x :width width
                           :bottom new-y :height height)
      (let* ((tno (world-territory-no stream))
	     (drawing-region (drawing-region stream))
	     ;; graphic-statement
	     (color (graphic-color stream))
	     (operation (graphic-operation stream))
	     (line-width (line-width stream))
	     (filled-type (filled-type stream))
	     (filled-pattern (if (null (filled-pattern stream))
				 0
			       (filled-pattern stream)))
	     (line-dash (line-dashing stream)))


        ;;; $B%o!<%k%I%j!<%8%g%s$NJQ99(B
	(setf (world-region stream) draw-region-region)


      ;;; $B0LCV$NJQ99(B
      (setf new-x (+ new-x (world-x-start stream))
	    new-y (+ new-y (world-y-start stream)))
      
      ;;; $BIA2h%W%j%_%F%#%V(B
     (normal-draw-region-primitive tno new-x new-y width height line-width 
				operation (color-no color)
				line-dash filled-pattern filled-type)

        ;;; drawing-region$B$K:#$N%j!<%8%g%s$rDI2C(B
     (set-drawing-region drawing-region draw-region-region)
     stream))))

;;; $B6k7A$NIA2h%]%8%7%g%sMQ(B
;;; draw-region stream position width height
;;; ARG.
;;;             stream            =   $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;             position          =   $B6k7A$rIA2h$9$k3+;O0LCV%$%s%9%?%s%9(B
;;;             width height      =   $B6k7A$NBg$-$5(B
(defmethod draw-region ((stream graphic-stream) (pos position)
			 (width integer) (height integer))
  (draw-region-xy-internal stream (position-x pos) (position-y pos)
		  width height))

;;; $B6k7A$NIA2h(B $B%j!<%8%g%sMQ(B
;;; draw-region-region stream region
;;; ARG.
;;;             stream           = $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;             region           = $B6k7A$N0LCV$HBg$-$5$r;}$D%$%s%9%?%s%9(B
(defmethod draw-region-region ((stream graphic-stream) (region region))


  (draw-region-xy-internal stream (region-left region) (region-bottom region)
		  (region-width region) (region-height region)))


;;; draw-region-primitive for world
(defun normal-draw-region-primitive (tno x y width height line-width op cl-no
				line-dash filled-pattern filled-type)
      ;;; $BIA2h%b!<%I$r7hDj(B
  (cond
   ;; draw region without pattern
   ((eql filled-type *fillednon*)
    (yy-protocol-26 tno x y width height line-width
		    op cl-no line-dash))
   ;; draw rectabglw filled color
   ((null filled-pattern)
    (yy-protocol-28 tno x y width height
		    op cl-no 0 ))
   ;; draw rectangle filled pattern
   (t
    (yy-protocol-28 tno x y width height
		    op cl-no filled-pattern)))
  )

;;; $B@^$l@~$NIA2h(B      
;;; draw-polyline-xy stream xy-list
;;; ARG.
;;;          stream            = $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;          xy-list           = $B0LCV(B
(defmethod draw-polyline-xy ((stream graphic-stream) &rest xy-list)
  (draw-polyline-xy-internal stream xy-list))

;;; draw-polyline-xy$BIA2h%W%j%_%F%#%V(B
(defun draw-polyline-xy-internal (stream xy-list)
  (declare (inline + < > man min / car cdr))
 (with-temp-region-args ((draw-polyline-region) (work-region1 stream))
  (let* ((tno (world-territory-no stream))
	 (drawing-region (drawing-region stream))
	 (max-x most-negative-fixnum) (max-y most-negative-fixnum)
	 (min-x most-positive-fixnum) (min-y most-positive-fixnum)
	 (x-start (world-x-start stream))
	 (y-start (world-y-start stream))
	 ;; graphic-statement
	 (color (color-no (graphic-color stream)))
	 (new-xy-list xy-list)
	 (operation (graphic-operation stream))
	 (line-width (line-width stream))
	 (line-edge (line-edge stream))
	 (line-joint-type (line-joint-type stream))
	 (line-dash (line-dashing stream)))

    ;;; $B:BI8%j%9%H$N:BI8CMJQ99(B
    (dotimes (ii (/ (length new-xy-list) 2))
       (multiple-value-bind (nx ny)
	    (translate-transform-xy-valuse stream (car new-xy-list)
				    (car (cdr new-xy-list)))

	    (if (> nx max-x)
		(setf max-x nx))

	    (if (> ny max-y)
		(setf max-y ny))

	    (if (< nx min-x)
		(setf min-x nx))
	    
	    (if (< ny min-y)
		(setf min-y ny))

	    (setf (car new-xy-list) (+ nx x-start)
		  (car (cdr new-xy-list)) (+ ny y-start)
		  new-xy-list (cddr new-xy-list))))
	    
    ;;; $BIA2h%j!<%8%g%s$r@_Dj(B
    (setf (region-left draw-polyline-region) min-x
	  (region-bottom draw-polyline-region) min-y
	  (region-right draw-polyline-region) max-x
	  (region-bottom draw-polyline-region) max-y)

    ;;; $B%o!<%k%I%j!<%8%g%s$NJQ99(B
    (setf (world-region stream) draw-polyline-region)

    ;;; $B@^$l@~$rIA2h(B
    (yy-protocol-23 tno xy-list line-width operation line-edge 
		    line-joint-type
		    color line-dash)


    ;;; drawing-region$B$K:#$N%j!<%8%g%s$rDI2C(B
    (set-drawing-region drawing-region draw-polyline-region)
    stream)))

;;; add-value-for-xy-list
;;; XY$B$N%j%9%H$KBP$7$F!"(Bdx,dy$B$r3F!92C$($k(B
(defun add-value-for-xy-list (xy-list dx dy)
  (declare (inline +))
  (let ((flg t))
    (mapcan #'(lambda (item)
		(if flg
		    (progn 
		      (setf flg nil)
		      (list (+ item dx)))
		  (progn 
		    (setf flg T)
		    (list (+ item dy)))))
	    xy-list)))

	    
;;; $B@^$l@~$NIA2h(B $B%]%8%7%g%sMQ(B
;;; draw-polyline-xy stream position-list
;;; ARG.
;;;          stream           =  $B%0%i%U%#%/%9%9%H%j!<%`(B
;;           position-list    =  $B%]%8%7%g%s%$%s%9%?%s%974(B
(defmethod draw-polyline ((stream graphic-stream) &rest position-list)
  (apply #'draw-polyline-xy-internal stream (reduce #'nconc 
			    (map 'list #'(lambda (x) (list (position-x x)
				 			  (position-y x)))
					position-list))))

;;; $BB?3Q7A$NIA2h(B
;;; draw-polygon-xy stream xy-list
;;; ARG.
;;;         stream            =    $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;         xy-list           =    $B0LCV(B(x y)$B74(B
(defmethod draw-polygon-xy ((stream graphic-stream) &rest xy-list)
  (draw-polygon-xy-internal stream xy-list))

;;; draw-polygon-xy$BIA2h%W%j%_%#%F%V(B
(defun draw-polygon-xy-internal (stream xy-list)
  (declare (inline + < > min max / car cdr))
 (with-temp-region-args ((draw-polygon-region) (work-region1 stream))
  (let* ((tno (world-territory-no stream))
;	 (draw-polygon-region (make-region))
	 (drawing-region (drawing-region stream))
	 (max-x most-negative-fixnum) (max-y most-negative-fixnum)
	 (min-x most-positive-fixnum) (min-y most-positive-fixnum)
	 (x-start (world-x-start stream))
	 (y-start (world-y-start stream))
	 (new-xy-list xy-list)
	 ;; graphic-statement
	 (color (graphic-color stream))
	 (operation (graphic-operation stream))
	 (line-width (line-width stream))
	 (filled-type (filled-type stream))
	 (filled-rule (filled-rule stream))
	 (filled-pattern (filled-pattern stream))
	 (line-joint-type (line-joint-type stream))
	 (line-dash (line-dashing stream)))
    
    ;;; $B:BI8%j%9%H$N:BI8CMJQ99(B
    (dotimes (ii (/ (length new-xy-list) 2))
       (multiple-value-bind (nx ny)
	    (translate-transform-xy-valuse stream (car new-xy-list)
				    (car (cdr new-xy-list)))

	    (if (> nx max-x)
		(setf max-x nx))

	    (if (> ny max-y)
		(setf max-y ny))

	    (if (< nx min-x)
		(setf min-x nx))
	    
	    (if (< ny min-y)
		(setf min-y ny))

	    (setf (car new-xy-list) (+ nx x-start)
		  (car (cdr new-xy-list)) (+ ny y-start)
		  new-xy-list (cddr new-xy-list))))

    ;;; $BIA2h%j!<%8%g%s$r@_Dj(B
    (setf (region-left draw-polygon-region) min-x
	  (region-bottom draw-polygon-region) min-y
	  (region-right draw-polygon-region) max-x
	  (region-bottom draw-polygon-region) max-y)

    ;;; $B%o!<%k%I%j!<%8%g%s$NJQ99(B
    (setf (world-region stream) draw-polygon-region)

    ;;; $BIA2h%W%j%_%F%#%V(B
    (normal-draw-polygon-primitive  tno xy-list line-width operation 
			    line-joint-type (color-no color) line-dash
			    filled-rule filled-pattern filled-type)

    ;;; drawing-region$B$K:#$N%j!<%8%g%s$rDI2C(B
    (set-drawing-region drawing-region draw-polygon-region)
    stream)))

;;; $BB?3Q7A$NIA2h(B $B%]%8%7%g%sMQ(B
;;; draw-polygon stream position-list
;;; ARG.
;;;           stream           =  $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;           position-list    =  $B0LCV%$%s%9%?%s%974(B
(defmethod draw-polygon ((stream graphic-stream) &rest position-list)
  (apply #'draw-polygon-xy-internal stream (reduce #'nconc 
			       (map 'list #'(lambda (x) (list (position-x x)
							  (position-y x)))
				position-list)))
  )

;;; draw-polygon-primitive for world
(defun normal-draw-polygon-primitive (tno xy-list line-width operation 
				   line-joint-type cl-no line-dash filled-rule
				   filled-pattern filled-type)
  (cond
   ;; draw polygon without filled
   ((eql filled-type *fillednon*)
    (yy-protocol-24 tno xy-list
		line-width operation line-joint-type 
		cl-no line-dash))
   
   ;; draw ploygon filled color
   ((null filled-pattern)
    (yy-protocol-27 tno  xy-list
		operation line-joint-type 
		cl-no  filled-rule 0))

   ;; draw polygon filled pattern
   (t
    (yy-protocol-27 tno xy-list
		operation line-joint-type 
		cl-no filled-rule filled-pattern)))
  )

;;; $B;03Q7A$NIA2h(B   ($BFsEyJU;03Q7A(B)
;;; draw-triangle-xy stream x y width height
;;; ARG.
;;;           stream            =   $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;           x y               =   $BDcJU$NCf?40LCV(B 
;;;           width             =   $BDcJU$NI}(B
;;;           height            =   $B9b$5(B
(defmethod draw-triangle-xy ((stream graphic-stream) (x integer) (y integer)
			     (width integer) (height integer))
  (draw-triangle-xy-internal stream x y width height))

;;; draw-triangle-xy $BIA2h%W%j%_%F%#%t(B
(defun draw-triangle-xy-internal (stream x y width height)
  (declare (inline - / + round))
  (draw-polygon-xy-internal stream 
		   (list 
		    (- x (round (/ width 2))) y (+ x (round (/ width 2))) y
		   x (+ y height)))
  )

;;; $B;03Q7A$NIA2h(B $B%]%8%7%g%sMQ(B ($BFsEyJU;03Q7A(B)
;;; draw-triangle stream position width height
;;; ARG.
;;;              stream            = $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;              position          = $BDcJU$NCf?40LCV%$%s%9%?%s%9(B
;;;              width             = $B;03Q7A$NI}(B
;;;              height            = $B9b$5(B
(defmethod draw-triangle ((stream graphic-stream) (position position)
			  (width integer) (height integer))
  (draw-triangle-xy-internal stream (position-x position) (position-y position)
		   width height))


;;; $BBJ1_!"BJ1_8L!"BJ1_$N@p7?$NIA2h(B
;;; draw-ellipse-xy stream center-x center-y x-radius y-radius 
;;;                          &key (start-angle 0)
;;;                               (end-angle 2pi) (clockwize :clockwize)
;;; ARG.
;;;               stream     =  $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;     center-x center-y    =  $BBJ1_$NCf?4(B
;;;               x-radius   =  $B2#<4$K$*$1$kH>7B(B
;;;               y-radius   =  $B=D<4$K$*$1$kH>7B(B
;;;          start-angle     =  $B3+;O3QEY(B($B%i%G%#%"%s(B)
;;;          end-angle       =  $B=*N;3QEY(B($B%i%G%#%"%s(B)
;;;          clockwize       =  $BIA2hJ}8~(B
(defmethod draw-ellipse-xy ((stream graphic-stream) 
			    (center-x integer) (center-y integer)
			    (x-radius integer) (y-radius integer) 
			    &key (start-angle 0)
			    (end-angle (* 2 pi)) (clockwize :clockwize))
  (draw-ellipse-xy-internal stream center-x center-y 
			    x-radius y-radius 
			    :start-angle start-angle
			    :end-angle end-angle
			    :clockwize clockwize))

;;; draw-ellipse-xy$BIA2h%W%j%_%F%#%V(B
(defun draw-ellipse-xy-internal (stream center-x center-y x-radius y-radius
				 &key (start-angle 0)
				 (end-angle (* 2 pi))
				 (clockwize :clockwize))
  (declare (inline * + - max min))
    (case clockwize
      ((:clockwize))
      ((:counter-clockwize)
       (let ((tmp start-angle))
         (setf start-angle end-angle
	       end-angle tmp)))
      (otherwise (error "Draw-ellipse : Unkown clockwize ~s" clockwize)))
    (with-translate-transform-xy ((new-x new-y) stream center-x center-y)
      (with-temp-region-args ((draw-ellipse-region) (work-region1 stream)
                           :left (- new-x x-radius) :right (+ new-x x-radius)
                           :top (+ new-y y-radius) :bottom (- new-y y-radius))
      (let* ((tno (world-territory-no stream))
	     (drawing-region (drawing-region stream))
	     ;; graphic-statement
	     (color (graphic-color stream))
	     (operation (graphic-operation stream))
	     (width (line-width stream))
	     (dash (line-dashing stream))
	     (filled-type (filled-type stream))
	     (filled-pattern (filled-pattern stream))
	     (arc-mode (arc-mode stream)))
        ;; 1/64 $B$r$?$s$$$H$9$k!#(B
        (setf start-angle (floor (* (/ start-angle  pi) 180 64))
  	      end-angle (floor (* (/ end-angle pi) 180 64)))

        ;;; $B%o!<%k%I%j!<%8%g%s$NJQ99(B
	(setf (world-region stream) draw-ellipse-region)

      ;;; $B0LCV$NJQ99(B
      (setf new-x (+ new-x (world-x-start stream))
	    new-y (+ new-y (world-y-start stream)))

      ;;; $BIA2h%W%j%_%F%#%V(B
      (normal-draw-ellipse-primitive tno new-x new-y x-radius y-radius
			      start-angle end-angle width operation
			      (color-no color) dash arc-mode 
			      filled-pattern filled-type)
        ;;; drawing-region$B$K:#$N%j!<%8%g%s$rDI2C(B
      (set-drawing-region drawing-region draw-ellipse-region)
      stream))))

;;; $BBJ1_!"BJ1_8L!"BJ1_$N@p7?$NIA2h(B
;;; draw-ellipse   stream posistion x-radius y-radius 
;;;                         &key (start-angle 0)
;;;                               (end-angle 2pi) (clockwize :clockwize)
;;; ARG.
;;;               stream     =  $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;               position   =  $BBJ1_$NCf?4$N0LCV%$%s%9%?%s%9(B
;;;               x-radius   =  $B2#<4$K$*$1$kH>7B(B
;;;               y-radius   =  $B=D<4$K$*$1$kH>7B(B
;;;          start-angle     =  $B3+;O3QEY(B($B%i%G%#%"%s(B)
;;;          end-angle       =  $B=*N;3QEY(B($B%i%G%#%"%s(B)
;;;          clockwize       =  $BIA2hJ}8~(B
(defmethod draw-ellipse ((stream graphic-stream) (position position)
			    (x-radius integer) (y-radius integer) &key (start-angle 0)
			    (end-angle (* 2 pi)) (clockwize :clockwize))
  (draw-ellipse-xy-internal
                   stream (position-x position) (position-y position)
		   x-radius y-radius :start-angle start-angle
		   :end-angle end-angle :clockwize clockwize))

;;; draw-ellipse-primitive for world
(defun normal-draw-ellipse-primitive (tno new-x new-y x-radius y-radius
				      start-angle end-angle width operation
				      cl-no dash arc-mode filled-pattern 
				      filled-type)
  (cond
   ;; draw ellipse without filled
   ((eql filled-type *fillednon*)
    (yy-protocol-41 tno new-x new-y 
		    (* 2 x-radius) (* 2 y-radius) start-angle end-angle
		    width operation cl-no dash))
   ;; draw ellipse filled color
   ((null filled-pattern)
    (yy-protocol-42 tno new-x new-y
		    (* 2 x-radius) (* 2 y-radius) start-angle end-angle
		    operation cl-no 0 arc-mode))
   ;; draw ellipse filled pattern
   (t
    (yy-protocol-42 tno new-x  new-y
		    (* 2 x-radius) (* 2 y-radius) start-angle end-angle
		    operation cl-no filled-pattern arc-mode)))
)

;;; $B1_!"@p7?!"1_8L$NIA2h(B
;;; draw-circle-xy stream center-x center-y radius &key (start-angle 0)
;;;		(end-angle 2pi) clockwize
;;; ARG.
;;;           stream             =  $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;       center-x center-y      =  $B1_$NCf?4(B
;;;          radius              =  $B1_$NH>7B(B
;;;          start-angle         =  $B3+;O3QEY(B($B%i%G%#%"%s(B)
;;;          end-angle           =  $B=*N;3QEY(B($B%i%G%#%"%s(B)
;;;          clockwize           =  $BIA2hJ}8~(B
(defmethod draw-circle-xy ((stream graphic-stream) (center-x integer)
			   (center-y integer) (radius integer)
			   &key (start-angle 0) (end-angle (* 2 pi))
			   (clockwize :clockwize))
    (case clockwize
      ((:clockwize))
      ((:counter-clockwize)
       (let ((tmp start-angle))
         (setf start-angle end-angle
	       end-angle tmp)))
      (otherwise (error "Draw-circle : Unkown clockwize ~s" clockwize)))

    ;;; draw-circle-intenal
    (draw-circle-xy-internal stream center-x center-y radius start-angle end-angle))


;;; draw-circle-xy-internal stream center-x center-y radius start-angle
;;;				 end-angle
(defun draw-circle-xy-internal (stream center-x center-y radius start-angle
				end-angle)
  (declare (inline + * max min))
  (with-translate-transform-xy ((new-x new-y) stream center-x center-y)
    (with-temp-region-args ((draw-circle-region) (work-region1 stream))
       (setf draw-circle-region (get-draw-circle-region stream new-x new-y
					   radius start-angle end-angle
					   draw-circle-region))
       (let* ((tno (world-territory-no stream))

	     (drawing-region (drawing-region stream))
	     ;; graphic-statement
	     (color (graphic-color stream))
	     (operation (graphic-operation stream))
	     (width (line-width stream))
	     (dash (line-dashing stream))
	     (filled-type (filled-type stream))
	     (filled-pattern (filled-pattern stream))
	     (arc-mode (arc-mode stream)))

        ;; 1/64 $B$r$?$s$$$H$9$k!#(B
        (setf start-angle (floor (* (/ start-angle pi) 180 64))
  	      end-angle (floor (* (/ end-angle pi) 180 64)))
      
        ;;; $B%o!<%k%I%j!<%8%g%s$NJQ99(B
	(setf (world-region stream) draw-circle-region)

      ;;; $B0LCV$NJQ99(B
	(setf new-x (+ new-x (world-x-start stream))
	      new-y (+ new-y (world-y-start stream)))

        ;;; call draw-circle-primitive
	(normal-draw-circle-primitive tno new-x new-y radius 
				      start-angle end-angle
				      width operation (color-no color) dash 
				      arc-mode filled-pattern filled-type)
	
        ;;; drawing-region$B$K:#$N%j!<%8%g%s$rDI2C(B
	(set-drawing-region drawing-region draw-circle-region)
	stream))))


;;; $B1_!"@p7?!"1_8L$NIA2h(B
;;; draw-circle stream position radius &key (start-angle 0)
;;;		(end-angle 2pi) clockwize
;;; ARG.
;;;           stream             =  $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;          position            =  $B1_$NCf?40LCV$N%$%s%9%?%s%9(B
;;;          radius              =  $B1_$NH>7B(B
;;;          start-angle         =  $B3+;O3QEY(B($B%i%G%#%"%s(B)
;;;          end-angle           =  $B=*N;3QEY(B($B%i%G%#%"%s(B)
;;;          clockwize           =  $BIA2hJ}8~(B
(defmethod draw-circle ((stream graphic-stream) (position position)
			(radius integer) &key (start-angle 0)
			(end-angle (* 2 pi)) (clockwize :clockwize))
    (case clockwize
      ((:clockwize))
      ((:counter-clockwize)
       (let ((tmp start-angle))
         (setf start-angle end-angle
	       end-angle tmp)))
      (otherwise (error "Draw-circle : Unkown clockwize ~s" clockwize)))

  (draw-circle-xy-internal
                  stream (position-x position) (position-y position) radius
                  start-angle end-angle))


;;; draw-circle-primitive for wolrd
(defun normal-draw-circle-primitive (tno new-x new-y radius start-angle 
				     end-angle width operation cl-no 
				     dash arc-mode
				     filled-pattern filled-type)
  (cond
   ;; draw circle without filled
   ((eql filled-type *fillednon*)
    (if (and (= start-angle 0) (= end-angle 23040))
	(yy-protocol-22 tno new-x new-y radius width operation
			cl-no dash)
      (yy-protocol-25 tno new-x new-y radius start-angle end-angle
		    width operation cl-no dash)))

   ;; draw circle filled color
   ((null filled-pattern)
    (if (and (= start-angle 0) (= end-angle 23040))
	(yy-protocol-29 tno new-x new-y radius operation 
			cl-no 0)
      (yy-protocol-30 tno new-x new-y radius start-angle end-angle
		      operation cl-no 0 arc-mode)))
   ;; draw circle filled pattern
   (t
    (if (and (= start-angle 0) (= end-angle 23040))
	(yy-protocol-29 tno new-x new-y radius operation
			cl-no filled-pattern)
      (yy-protocol-30 tno new-x new-y radius start-angle end-angle
		    operation cl-no filled-pattern arc-mode))))
)


;;; draw-circle $B$N%j!<%8%g%s$r5a$a$k(B
(defun get-draw-circle-region (stream x y r start end region)
  (declare (inline + / <= max min sin cos round))
  (let ((start-x-offset (round (* (cos start) r)))
  	(start-y-offset (round (* (sin start) r -1)))
	(end-x-offset (round (* (cos end) r)))
	(end-y-offset (round (* (sin end) r -1)))
	(left 0) (top 0) (right 0) (bottom 0))
    (with-translate-transform-xy ((new-x new-y) stream x y)
       (setf left (+ (if (<= start pi end)
    		      (- r)
		    (min start-x-offset end-x-offset))
		  new-x)
          bottom  (+ (if (<= start (+ pi (/ pi 2)) end)
	  	      (- r)
		    (min start-y-offset end-y-offset))
		  new-y)
	  right (+ (if (<= start 0 end)
	  	       r
		     (max start-x-offset end-x-offset))
		   new-x)
	  top (+ (if (<= start (/ pi 2) end)
	  		r
		      (max start-y-offset end-y-offset))
		    new-y))
       (setf (region-left region) left
	     (region-right region) right
	     (region-bottom region) bottom
	     (region-top region) top)
       region
	     
       )
    ))
#|
;;; $B8E:d$,JQ99(B
(defun get-draw-circle-region (stream x y r start end)
  (with-translate-transform-xy ((new-x new-y) stream x y)
    (make-region :left ( - x r) :top (+ y r)
		     :right (+ x r) :bottom (- y r))))
|#



;;; $BJ8;zNs$NIA2h(B 
;;; draw-string-xy stream string x y &key start end direction
;;;            stream         = $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;            string         = $BIA2h$9$kJ8;zNs(B
;;;            x y            = $BIA2h$9$k0LCV(B
;;;            start          = $BJ8;zNs$N3+;O0LCV(B(string$B$J$$$G(B
;;;            end            = $BJ8;zNs$N=*N;0LCV(B(string$B$J$$$G(B)
;;;            direction      = $BIA2h$9$kJ}8~(B
;;; RET.
;;;      (string T or nil)
;;;       T --> $B2~9T$,$"$C$?(B
(defmethod draw-string-xy ((stream graphic-stream) (string string)
			(x integer) (y integer)
			&key (start 0) (end (length string))
			(direction :horizontal))
  (draw-string-xy-internal stream string x y start end direction))

;;; draw-string$BIA2h%W%j%_%F%#%V(B
(defun draw-string-xy-internal (stream string x y start end direction)
  (declare (special *root-window*) (inline =))
  (let* ((position (slot-value stream 'cursor-position))
	 (old-x (position-x  position))
	 (old-y (position-y  position))
	 (old-direction (stream-output-direction stream)))

    (setf (stream-cursor-x-position stream) x
	  (stream-cursor-y-position stream) y
	  (stream-output-direction stream) direction)

    ;;; $B=PNOJ}8~$rD4$Y$k(B
    (prog1 
	(if (eq direction :horizontal)
	    (progn 
	      (drawing-text-internal-yoko stream 
			  (subseq string start end))
	      (values string (if (= old-y (position-y position))
				 NIL
			       T)))
	  (progn 
	    (drawing-text-internal-tate stream (subseq string start end))
	    (values string (if (= old-x (position-x position))
			       NIL
			     T))))
      (setf (position-x position) old-x
	    (position-y position) old-y
	    (stream-output-direction stream) old-direction)
      )))
			       

;;; $BJ8;zNs$NIA2h(B 
;;; draw-string stream string position &key start end direction
;;;            stream         = $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;            string         = $BIA2h$9$kJ8;zNs(B
;;;            position       = $BIA2h$9$k0LCV$N%$%s%9%?%s%9(B
;;;            start          = $BJ8;zNs$N3+;O0LCV(B(string$B$J$$$G(B
;;;            end            = $BJ8;zNs$N=*N;0LCV(B(string$B$J$$$G(B)
;;;            direction      = $BIA2h$9$kJ}8~(B
;;; RET.
;;;      (string T or nil)
;;;       T --> $B2~9T$,$"$C$?(B
(defmethod draw-string ((stream graphic-stream) (string string)
			(position position)
			&key (start 0) (end (length string))
			(direction :horizontal))
  (draw-string-xy-internal stream string (position-x position)
		  (position-y position) start end direction))

;;; draw-string-xy after method 
;;; $BIA2h%7!<%1%s%9$r5-21(B
(defmethod draw-string-xy :after ((stream graphic-stream)  (string string)
			(x integer) (y integer)
			&key (start 0) (end (length string))
			(direction :horizontal))

   (if (presentation-instance stream)
       (push #'(lambda (new-stream)
		  (draw-string-xy new-stream string x y :start start :end end
				  :direction direction))
	     (cdr (last (drawing-sequence (presentation-instance stream)))))
     ))
    

;;; $BJ8;z$NIA2h(B 
;;; draw-char-xy stream char x y &key direction
;;;            stream         = $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;            string         = $BIA2h$9$kJ8;z(B
;;;            x y            = $BIA2h$9$k0LCV(B
;;;            direction      = $BIA2h$9$kJ}8~(B
;;; RET.
;;;      (string T or nil)
;;;       T --> $B2~9T$,$"$C$?(B
(defmethod draw-character-xy ((stream graphic-stream) (char character)
			(x integer) (y integer)
			&key (direction :horizontal))
  (multiple-value-bind (ret1 ret2)
	  (draw-string-xy-internal stream (string char) x y 
				   0 1 direction)
	  (declare (ignore ret1))
    
	  (values char ret2)))

;;; draw-character stream character position
;;; $BJ8;z$NIA2h(B 
;;; Return $BCM(B (string T or nil)
;;; T --> $B2~9T$,$"$C$?(B

;;; $BJ8;z$NIA2h(B 
;;; draw-char stream char position &key direction
;;;            stream         = $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;            string         = $BIA2h$9$kJ8;z(B
;;;            position       = $BIA2h$9$k0LCV$N%$%s%9%?%s%9(B
;;;            direction      = $BIA2h$9$kJ}8~(B
;;; RET.
;;;      (string T or nil)
;;;       T --> $B2~9T$,$"$C$?(B
(defmethod draw-character ((stream graphic-stream) (char character)
			(position position)
			&key (direction :horizontal))
  (draw-character-xy stream char (position-x position)
		     (position-y position) :direction direction))


;;; $B%W%m%s%W%H%&%#%s%I%&$X$NJ8;zNsI=<((B
;;; draw-prompt string
;;; ARG.
;;;             string        = $BI=<($9$kJ8;zNs(B
(defun draw-prompt (string)
  (declare (function clear-window-stream (t) t))
  (clear-window-stream *prompt-window*)
  (draw-string-xy *prompt-window* string 0 (font-kanji-base-line 
				     (stream-font *prompt-window*))
		  ))
