;;; -*- Mode: LISP; Syntax: COMMON-LISP; Base: 10.; Package: XIT -*-
;;;________________________________________________________________________
;;;
;;;                       System: XACT 
;;;                       Module: Palette
;;;                       Version: 1.0
;;;
;;; Copyright (c): Forschungsgruppe DRUID, Juergen Herczeg
;;;                Universitaet Stuttgart
;;;
;;; File: /usr/local/lisp/xit/xact/palette.lisp
;;; File Creation Date: 02/06/92 14:18:01
;;; Last Modification Time: 07/16/93 08:08:44
;;; Last Modification By: Juergen Herczeg
;;;
;;;
;;; Changes (worth to be mentioned):
;;; ================================
;;;
;;;________________________________________________________________________

(in-package :xit)
  
;___________________________________________________________________________
;
;                       palette container window
;___________________________________________________________________________

(defvar *gio-palette*)

(defun gio-palette-exists-p ()
  (declare (special *gio-palette*))
  (and (boundp '*gio-palette*)
       *gio-palette*
       (not (destroyed-p *gio-palette*))))

(defun select-gio-palette ()
  (declare (special *gio-palette*))
  (unless (gio-palette-exists-p)
    (setq *gio-palette* (make-gio-palette)))
  (totop-window *gio-palette*))

(defmethod copy-entry-from-palette ((drag-window basic-contact)
				    (drop-window composite)
				    drop-x drop-y)
  (with-slots ((palette parent)) drag-window
    (unless (eq drop-window palette) ;; entry has been dropped on palette
      (copy-window-to-parent drag-window drop-window drop-x drop-y))))

(defmethod create-new-part-from-palette ((drag-window basic-contact)
					 (drop-window composite)
					 drop-x drop-y)
  (with-slots ((palette parent) view-of) drag-window
    (unless (eq drop-window palette) ;; part has been dropped on palette
      (create-new-part-in-x-y drop-window
			      (getf view-of :class)
			      drop-x drop-y
			      (getf view-of :initargs)))))

(defmethod create-popup-from-palette ((drag-window basic-contact)
				      (drop-window composite)
				      drop-x drop-y)
  (let ((meta-toplevel (meta-toplevel-window drop-window)))
    (if (eq meta-toplevel drop-window)
	(create-new-part-from-palette drag-window drop-window
				      drop-x drop-y)
      (add-shadow-borders-popup-container
       (meta-top-window drop-window)))))

(defmethod create-popup-part-window-icon-container-from-palette
    ((drag-window basic-contact)
     (drop-window composite)
     drop-x drop-y)
  (let ((meta-toplevel (meta-toplevel-window drop-window)))
    (if (eq meta-toplevel drop-window)
	(create-new-part-from-palette drag-window drop-window
				      drop-x drop-y)
      (add-window-icon-popup-part-container
       (meta-top-window drop-window)))))

(defmethod create-popup-part-container-from-palette
    ((drag-window basic-contact)
     (drop-window composite)
     drop-x drop-y)
  (let ((meta-toplevel (meta-toplevel-window drop-window))
	(meta-top-window (meta-top-window drop-window)))
    (if (eq meta-toplevel drop-window)
	(create-new-part-from-palette drag-window drop-window
				      drop-x drop-y)
      (let (pp-connection)
	(if (typep meta-top-window 'popup-part-connection)
	    (setq pp-connection meta-top-window)
	 (setq pp-connection
		(add-window-icon-popup-part-container
		 (meta-top-window drop-window))))
	(identify-popup-part-with-mouse pp-connection)))))

(defmethod create-window-icon-container-from-palette
    ((drag-window basic-contact)
     (drop-window composite)
     drop-x drop-y)
  (let ((meta-toplevel (meta-toplevel-window drop-window))
	(meta-top-window (meta-top-window drop-window)))
    (if (eq meta-toplevel drop-window)
	(create-new-part-from-palette drag-window drop-window
				      drop-x drop-y)
      (let (shrink-window)
	(if (typep meta-top-window 'window-icon-mixin)
	    (setq shrink-window meta-top-window)
	  (setq shrink-window
		(add-window-icon-popup-part-container
		 (meta-top-window drop-window))))
	(identify-window-icon-with-mouse shrink-window)))))

(defmethod drag-and-drop-from-palette (entry)
  (with-slots (view-of) entry
    (let ((drop-type (or (getf view-of :drop-type) 'composite))
	  (action (or (getf view-of :action) :create))
	  (test (getf view-of :test)))
      (setq action
	  (case action
	    (:create #'create-new-part-from-palette)
	    (:copy #'copy-entry-from-palette)
	    (t action)))
      (drag-and-drop-window entry :type drop-type
			          :test test
			          :action action))))
				    
(defun destroy-gio-palette ()
  (declare (special *gio-palette*))
  (destroy-and-make-unbound *gio-palette*))

(defcontact palette-container (margined-window)
  ())

(defmethod client-window ((self palette-container) &key (errorp nil))
  (or (find-part self #'(lambda (child)
			  (and (managed-p child)
			       (not (typep child 'margin)))))
      (and errorp
	   (error "A client-window is missing."))))

(defun make-gio-palette ()
  (while-busy nil
    (let
      ((palette-container
	(make-gio 
	 'paned-window
	 :name :palette
	 :adjust-size? nil
	 :reactivity-entries '((:select) (:move))
	 :parts
	 '((:class text-dispel
		   :name :title
		   :adjust-size? nil
		   :text "Palette"
		   :font (:face :bold)
		   :background "black"
		   :foreground "white"
		   :inside-border 4
		   :display-position :center)

	   ;; window operation menu
	   (:class bitmap-menu
	    :name :window-operations-menu
	    :inside-border 0
	    :layouter (distance-layouter :distance 3 :orientation :right)
	    :action (call :eval (funcall *part-value* (top-window *self*)))
	    :parts
	    ((:view-of refresh-window
	      :bitmap "button-refresh"
	      :action-docu "Refresh Window")
	     (:view-of move-window
	      :bitmap "button-move"
	      :action-docu "Move Window")
	     (:view-of resize-window
	      :bitmap "button-resize"
	      :action-docu "Resize Window")
	     (:view-of totop-window
	      :bitmap "button-totop"
	      :action-docu "Put Window on Top")
	     (:view-of tobottom-window
	      :bitmap "button-tobottom"
	      :action-docu "Put Window to Bottom")
	     (:view-of bury-window
	      :bitmap "button-shrink"
	      :action-docu "Shrink Window")
	     (:view-of destroy
	      :bitmap "button-kill"
	      :cursor "pirate"
	      :action-docu "Remove Window")))

	   (:class single-choice-box-menu
	    :name :palette-menu
	    :border-width 1
	    :selection nil
	    :reactivity-entries
	    ((:read-event (call :eval
				(setf (selection *self*)
				    (selected-palette (part-of *self*)))))
	     (:write-event (call :eval
				 (setf (selected-palette (part-of *self*))
				     (selection *self*))))))
        
	   (:class palette-container
	    :name :palette-container
	    :border-width 0
	    :margins
	    ((standard-margins-with-scroll-bars-without-label
	      :scroll-bar-options (:locations (:right :bottom))))
	    :client-window intel)

	   (:class multi-line-text-dispel
	    :name :documentation
	    :border-width 1))

	 :layouter
	 '(pane-layouter
	   :configuration configuration-1
	   :configurations
	   ((configuration-1
	     ((:title :ask)
	      (empty 3)
	      (menu-strip (:ask :window-operations-menu) :h
			  (empty :rest)
			  (:window-operations-menu :ask)
			  (empty 3))
	      (empty 3)
	      (main :rest :h
		    (:palette-menu 100)
		    (empty 3)
		    (palette-and-docu :rest :v
				      (:palette-container :rest)
				      (empty 3)
				      (:documentation 50)))))))
	 )))
      (create-gio-palettes palette-container)
      palette-container)))

;___________________________________________________________________________
;
;                          gio palette
;___________________________________________________________________________

(defvar *gio-palette-descriptions* nil)

(defclass gio-palette-description ()
  ((name :type symbol :initarg :name
	 :accessor palette-name)
   (text :type string :initarg :text
	 :accessor palette-text)
   (documentation :type string :initarg :documentation
		  :accessor palette-documentation :initform "")
   (class :type symbol :initarg :class
	  :accessor palette-class :initform 'basic-menu)
   (action :initarg :action :accessor palette-action
	   :initform '(call :sender drag-and-drop-from-palette))
   (parts :type list :initarg :parts :accessor palette-parts :initform nil)
   (initargs :type list :initarg :initargs
	     :accessor palette-initargs :initform nil))
  (:documentation "Representation of gio-palettes"))

(defun define-gio-palette (name &key text documentation class action
				     parts initargs)
  (let ((palette (apply #'make-instance 'gio-palette-description
			:name name
			:text (or text (convert-to-sring name))
			:parts parts
			(append
			 (when text `(:text ,text))
			 (when class `(:class ,class))
			 (when documentation `(:documentation ,documentation))
			 (when action `(:action ,action))
			 (when initargs `(:initargs ,initargs))))))
    (install-gio-palette palette)
    palette))

(defmethod install-gio-palette ((palette gio-palette-description))
  (declare (special *gio-palette-descriptions* *gio-palette*))
  (let ((palette-name (palette-name palette)))
    (uninstall-gio-palette palette-name)
    (push palette *gio-palette-descriptions*)
    (when (gio-palette-exists-p)
      (let ((selection (selected-palette *gio-palette*)))
	(create-gio-palette palette *gio-palette*)
	(when (eq selection palette-name)
	  (update-state (contact-display *gio-palette*))
	  (synchronize-display *gio-palette*)
	  (setf (selected-palette *gio-palette*) selection)
	  (read-from-application (part *gio-palette* :palette-menu))
	  )))))

(defmethod create-gio-palette ((palette gio-palette-description)
			       &optional (gio-palette *gio-palette*))
  (declare (special *gio-palette*))
  (let ((palette-menu (part gio-palette :palette-menu))
	(palette-container (part gio-palette :palette-container)))
    (with-slots (name text class action parts initargs) palette
      (prog1
	  (apply #'add-part palette-container
		 :class class
		 :name name
		 :view-of palette
		 :state :withdrawn
		 :background :parent-relative ;; needed to override resources
		 :adjust-size? nil
		 :layouter '(aligning-multiline-distance-layouter
			     :alignment :center
			     :orientation :down
			     :items-per-line 5
			     :line-offset 15
			     :distance 15)
		 :action action
		 ;;:part-mouse-feedback :inverse
		 :parts parts
		 initargs)
	(add-part palette-menu :name name :text text :view-of name)))))

(defun create-gio-palettes (&optional (gio-palette *gio-palette*))
  (declare (special *gio-palette* *gio-palette-descriptions*
		    *default-gio-palette*))
  (dolist (palette (reverse *gio-palette-descriptions*))
    (create-gio-palette palette gio-palette))
  (setf (selected-palette gio-palette) *default-gio-palette*)
  (read-from-application (part gio-palette :palette-menu)))

(defmethod uninstall-gio-palette (palette-name)
  (declare (special *gio-palette* *gio-palette-descriptions*))
  (let ((palette (find palette-name *gio-palette-descriptions*
			    :key #'palette-name)))
    (when palette
      (setq *gio-palette-descriptions*
	  (remove palette *gio-palette-descriptions*)))
    (when (gio-palette-exists-p)
      (remove-gio-palette palette-name *gio-palette*))))

(defmethod uninstall-gio-palette ((palette gio-palette-description))
  (declare (special *gio-palette-descriptions* *gio-palette*))
  (setq *gio-palette-descriptions*
      (remove palette *gio-palette-descriptions*))
  (when (gio-palette-exists-p)
      (remove-gio-palette (palette-name palette) *gio-palette*)))
  
(defun remove-gio-palette (name &optional (gio-palette *gio-palette*))
  (declare (special *gio-palette*))
  (let ((menu-entry (part* gio-palette :palette-menu name))
	(palette-entry (part* gio-palette :palette-container name)))
    (when menu-entry (destroy menu-entry))
    (when palette-entry (destroy palette-entry))))
  
(defun selected-palette (gio-palette)
  (let ((palette (client-window (part gio-palette :palette-container)
				:errorp nil)))
    (when palette
      (contact-name palette))))

(defun (setf selected-palette) (name gio-palette)
  (let* ((palette-container (part gio-palette :palette-container))
	 (new-palette (part palette-container name))
	 (old-palette (client-window palette-container :errorp nil))
	 (doc-window (part gio-palette :documentation)))
    (with-final-layout palette-container
      (when new-palette 
	(setf (contact-state new-palette) :mapped)
	(setf (text doc-window) (palette-documentation (view-of new-palette))))
      (when (and old-palette (not (eq old-palette new-palette)))
	(setf (contact-state old-palette) :withdrawn)))))
	     
;___________________________________________________________________________
;
;                          window creation
;___________________________________________________________________________

(defmethod default-initargs-for (class (parent composite))
  `(:name ,(gentemp class :keyword)
    :parent ,parent
    :reactivity-entries ((:meta-drag-and-drop) (:metasystem))))
  
(defmethod default-initargs-for (class (parent toplevel-window))
  (declare (ignore class))
  (list* :background "white"
	 (call-next-method)))

(defmethod geometry-initargs-for (class (parent composite) x y)
  (declare (ignore class))
  `(:x ,x :y ,y :layouted? t))

(defmethod geometry-initargs-for (class (parent layouted-window) x y)
  (declare (ignore class x y))
  (if (layouter parent)
    ()
    (call-next-method)))

(defmethod geometry-initargs-for (class (parent paned-window) x y)
  (declare (ignore class))
  `(:x ,x :y ,y :layouted? t))

(defmethod create-new-part-in-x-y ((parent composite)
				   class x y initargs)
  (apply #'create-new-part parent class
	 (append initargs
		 (default-initargs-for class parent)
		 (geometry-initargs-for class parent x y))))

(defmethod create-new-part ((parent composite) class
			    &rest initargs)
  (apply #'create-new-part-before-actions parent class initargs)
  (let ((part (apply #'do-create-part parent class initargs)))
    (apply #'create-new-part-after-actions parent part initargs)
    part))

(defmethod resize-created-part? ((self basic-contact))
  t)

(defmethod resize-created-part? ((self adjustable-window))
  (not (adjust-size? self)))

(defmethod create-new-part-before-actions ((parent composite)
					   class &rest initlist)
  (declare (ignore class initlist))
  nil)

(defmethod create-new-part-before-actions :before ((parent container-window)
						   class &rest initlist)
  (declare (ignore class initlist))
  (let ((client-window (client-window parent :errorp nil)))
    (when client-window (destroy client-window))))

(defmethod create-new-part-after-actions ((parent composite)
					  (part basic-contact)
					  &rest initlist)
  (with-slots (display state) part
     (when (and (eq state :mapped)
		(resize-created-part? part)
		;;(not (and (getf initlist :width)
			  ;;(getf initlist :height)))
		)
       (update-state display)
       (process-all-events display)
       (resize-window part))))

(defmethod create-new-part-after-actions ((parent container-window)
					  (part basic-contact)
					  &rest initlist)
  (declare (ignore initlist))
  nil) ;; no resize

(defmethod create-new-part-after-actions :after ((parent composite)
						 (part shadow-borders-popup-container)
						 &rest initlist)
  (declare (ignore initlist))
  (change-layout part)
  (setf (adjust-size? part) t))

(defmethod create-new-part-after-actions :after ((parent composite)
						 (part uniform-part-intel)
						 &rest initlist)
  (declare (ignore initlist))
  (unless (parts part)
    (select-meta-add-part-sheet part)))

(defmethod do-create-part ((parent composite) class &rest initlist)
  (while-busy nil (apply #'make-gio class initlist)))

#||
(defmethod do-create-part ((type (eql 'popup-container)) (parent composite)
			   &optional initlist)
  (apply #'make-gio 'popup-container
	 :client-window :none
	 initlist))

(defmethod do-create-part ((type (eql 'popup-part-container))
			   (parent composite)
			   &optional initlist)
  (apply #'make-gio 'popup-part-container
	 :client-window :none
	 initlist))

(defmethod do-create-part ((type (eql 'window-icon-popup-part-container))
			   (parent composite)
			   &optional initlist)
  (apply #'make-gio 'window-icon-popup-part-container
	 :client-window :none
	 initlist))

(defmethod do-create-part ((type (eql 'window-icon-container))
			   (parent composite)
			   &optional initlist)
  (apply #'make-gio 'window-icon-container
	 :client-window :none
	 initlist))


(defmethod do-create-part ((type (eql 'property-sheet)) (parent composite)
			   &optional initlist)
  (apply #'make-gio 'property-sheet
	 :border-width 1
	 initlist))
||#