;;; -*- Mode: LISP; Syntax: COMMON-LISP; Base: 10.; Package: XIT -*-
;;;_____________________________________________________________________________
;;;
;;;                       System: XIT
;;;                       Module: Object Binder
;;;                       Version: 1.0
;;;
;;; Copyright (c): Forschungsgruppe DRUID, Juergen Herczeg
;;;                Universitaet Stuttgart
;;;
;;; File: /usr/local/lisp/xit/utilities/object-binder-window.lisp
;;; File Creation Date: 03/30/93 16:05:37
;;; Last Modification Time: 07/12/93 11:31:27
;;; Last Modification By: Juergen Herczeg
;;;
;;;
;;; Changes (worth to be mentioned):
;;; ================================
;;;
;;;_____________________________________________________________________________

(in-package :xit)

(defun new-symbol-to-bind (&key (prefix "OBJ-") (package :xit))
  (gentemp prefix package))

(defun symbol-to-bind-from-string (name &key (package :xit))
  (intern (string-upcase name) package))

(defcontact binder-menu-entry (paned-window)
  ((adjust-size? :initform nil)
   (reactivity :initform
	       '((:drop-event
		  (call :eval (set (view-of *self*) *sender*))
		  (call :read)
		  (call :part-event))
		 (:message
		  (call :eval
			(set (view-of *self*)
			     (identify-window *self*
			      :mouse-documentation
			      "Identify object to bind")))
		  (call :read)
		  (call :part-event))
		 (:read-event
		  (call :self broadcast #'read-from-application))))
   (layouter :initform
	     '(pane-layouter
	       :configurations
	       ((configuration-1
		 ((main 1.0 :h
			(:variable 60)
			(empty 3)
			(:value :rest))))))))
  (:resources
   (width :initform 250)
   (height :initform 40)
   (border-width :initform 0)
   (inside-border :initform 0)))

(defcontact binder-variable-dispel (text-dispel)
  ((name :initform :variable)))

(defmethod edit-event-type ((self binder-variable-dispel))
  :double-left-button)

(defmethod initialize-instance :after ((self binder-menu-entry)
				       &rest init-list)
  (declare (ignore init-list))
  (add-part self 
	    :class 'binder-variable-dispel
	    :adjust-size? nil
	    :border-width 2
	    :font '(:face :bold)
	    :mouse-feedback :border
	    :edit-mode :click
	    :reactivity-entries
	    '((:select "Select binding"
	       (call :part-of send-part-event))
	      (:single-right-button "Identify object to bind"
	       (call :message))
	      (:accept-event (call :part-of read-from-application))
	      (:double-right-button "Remove" (call :part-of destroy))
	      (:read-event
	       (call :eval
		(with-slots (text parent) *self*
		  (when (string= text "")
		    (setf (text *self*)
			(format nil "obj-~A"
				(part-position (part-of *self*)))))
		  (with-accessors ((view-of view-of)) parent
		      (let ((old-var view-of))
			(setf view-of (symbol-to-bind-from-string text))
			(when (and old-var (boundp old-var))
			  (set view-of (symbol-value old-var))))))))))

  (add-part self
	    :class 'multi-line-text-dispel
	    :name :value
	    :adjust-size? nil
	    :border-width 1
	    :font '(:size :small)
	    :fill-column :max
	    :separators '(#\Space #\Return #\Newline #\Linefeed)
	    :edit-mode :click
	    :reactivity-entries
	    '((:accept-event
	       (call :eval
		(multiple-value-bind (result error)
		    (ignoring-errors
		     (eval (read-from-string (text *self*))))
		  (if error
		      (flash-window *self*)
		    (set (view-of *self*) result))
		  (read-from-application *self*))))
	      (:single-right-button "Identify object to bind"
	       (call :message))
	      (:read-event
	       (call :eval
		(let ((variable (view-of *self*)))
		  (setf (text *self*)
		      (if (boundp variable)
			  (convert-to-string
			   (symbol-value variable))
			"")))))))

  (read-from-application (part self :variable)))

(defmethod selection-changed ((self binder-menu-entry))
  (setf (inverse? (part self :variable)) (selected? self)))

(defcontact binder-menu (single-choice-menu)
  ((name :initform :binder-menu)
   (inside-border :initform 0)
   (part-class :initform 'binder-menu-entry)
   (part-mouse-feedback :initform :none)
   (reactivity
    :initform
    '((:select "Set focus on bindings"
	       (call :eval (write-to-application (part (part-of *self* 2)
						       :bindings-header))))
      (:part-event (call :write))
      (:write-event
       (call :eval
	(let ((variable (value *self*))
	      (binder (part-of *self* 2)))
	  (setq * (when (boundp variable)
		    (symbol-value variable)))
	  (write-to-application (part binder :bindings-header))
	  (read-from-application (binder-result binder)))))
      ;;(:read-event
      ;; (call :eval
	;;     (when (or (null (parts *self*))
	;;	       (not (string= (text (part* *self*
	;;					  (length (parts *self*))
	;;					  :value))
	;;			     "")))
	;;       (add-part *self*))))
      (:drop-event (call :self add-binding *sender* :selected? t))
      ))))

(defmethod part-event-type ((self binder-menu))
  :double-left-button)

(defmethod add-binding ((self binder-menu) value &key (selected? nil))
  (while-busy ()
    (let* ((new-part (add-part self))
	   (variable (view-of new-part)))
      (set variable value)
      (read-from-application (part new-part :value))
      (when selected? (send-part-event new-part)))))

(defmethod add-all-bindings ((self binder-menu) &rest values)
  (dolist (value values)
    (with-final-layout self
      (add-binding self value))))

(defcontact binder-operation-dispel (abbreviating-text-dispel)
  ((display-position :initform :upper-left)
   (min-width :initform 200)
   (max-width :initform 200)
   (reactivity :initform '((:double-right-button "Remove operation"
			    (call :part-of delete-part *self*))))))

(defun binder-bindings (binder-window)
  (part* binder-window :bindings :binder-menu))

(defun binder-operations (binder-window)
  (part* binder-window :operations :operations-menu))

(defun binder-input (binder-window)
  (part* binder-window :input :text))

(defun binder-history (binder-window)
  (popup-part (part* binder-window :input-history)))

(defun binder-result (binder-window)
  (part* binder-window :result :text))

(defun binder-current-value (binder-window)
  (if (selected? (part binder-window :bindings-header))
      (let ((variable (value (binder-bindings binder-window))))
	(and variable (boundp variable) (symbol-value variable)))
    (view-of (binder-result binder-window))))

(defvar *binder-operations*
        '("*"
	  "(part-of *)"
	  "(parts *)"
	  "(view-of *)"
	  "(layouter *)"
	  "(popup-part *)"
	  "(window-icon *)"
	  ""
	  "(destroy *)"
	  "(flash-window *)"
	  "(copy-window *)"
	  ))

(defvar *binder-history-length* 30)

(defun make-object-binder-window (&optional (parent nil parent-p))
  (apply
   #'make-gio 'window-icon-container
   :name :binder
   :adjust-size? nil
   :window-icon '(text-icon
		  :layouter (aligning-distance-layouter
			     :alignment :center
			     :distance -1)
		  :text-part (:border-width 1
			      :background "white"
			      :text "Object Binder"
			      :font (:size :small))
                  :bitmap-part (:bitmap "object-binder"
				:background "white"
				:border-width 1)
		  :reactivity ((:move)
			       (:drop-event
				(call :view-of expand)
				(call :eval
				 (add-binding
				  (binder-bindings
				   (client-window (view-of *self*)))
				  *sender* :selected? t)))))
   :reactivity-entries
   '((:select)
     (:move)
     ;;(:drop-event
      ;;(call :eval (add-binding (binder-bindings (client-window *self*))
		   ;;*sender* :selected? t)))
     (:configure-notify (call :client change-layout)))
   :client-window
   `(paned-window
     :adjust-size? nil
     :parts
     ((:class text-dispel
       :name :header
       :text "Object Binder"
       :font (:face :bolditalic :size :large)
       :inside-border 4
       :adjust-size? nil
       :background "black"
       :foreground "white"
       :display-position :center)
       
      ;; binder operation menu
      (:class text-menu
       :name :binder-operations-menu
       :layouter (distance-layouter :distance 20 :orientation :right)
       :part-font (:size 14 :face :bold)
       :parts
       ((:text "Bind"
	 :action
	 ((call :self mouse-feedback-off)
	  (call :eval
		(add-binding (binder-bindings (part-of *self* 2))
			     (identify-window *self*
					      :mouse-documentation
					      "Identify object to bind")
			     :selected? t)))
	 :action-docu "Identify and bind object")
	(:text "Reset"
	 :action
	 (call :eval
	       (let* ((binder (part-of *self* 2))
		      (bindings-menu (binder-bindings binder))
		      (ops-menu (binder-operations binder))
		      (input (binder-input binder))
		      (result (binder-result binder)))
		 (setf (parts bindings-menu) nil)
		 (setf (parts ops-menu) nil)
		 (read-from-application ops-menu)
		 (setf (text input) "*")
		 (setf (text result) "")))
	 :action-docu "Reset operations")))

      ;; switch configuration button
      (:class soft-button
       :name :button-switch
       :text-part :none
       :bitmap-part (:bitmap "button-switch")
       :mouse-feedback :border
       :action (call :eval (switch-configuration (layouter (part-of *self*))))
       :action-docu "Refresh Window")

      ;; window operation menu
      (:class bitmap-menu
       :name :window-operations-menu
       :inside-border 0
       :layouter (distance-layouter :distance 3 :orientation :right)
       :action (call :eval (funcall *part-value* (top-window *self*)))
       :parts
       ((:view-of refresh-window
	 :bitmap "button-refresh"
	 :action-docu "Refresh Window")
	(:view-of move-window
	 :bitmap "button-move"
	 :action-docu "Move Window")
	(:view-of resize-window
         :bitmap "button-resize"
	 :action-docu "Resize Window")
	(:view-of totop-window
         :bitmap "button-totop"
	 :action-docu "Put Window on Top")
	(:view-of tobottom-window
	 :bitmap "button-tobottom"
	 :action-docu "Put Window to Bottom")
	(:view-of shrink
	 :bitmap "button-shrink"
	 :action-docu "Shrink Window to Icon")
	(:view-of destroy
	 :bitmap "button-kill"
	 :cursor "pirate"
	 :action-docu "Remove Window")))

      (:class text-dispel
       :name :bindings-header
       :text "Object Bindings"
       :font (:face :bold)
       :inside-border 2
       :border-width 1
       :mouse-feedback :border
       :reactivity-entries
       ((:select "Set focus on bindings" (call :write))
	(:write-event
	 (call :eval
	   (setf (selected? *self*) t
		 (selected? (part (part-of *self*) :result-header)) nil)))))

      (:class soft-button
       :name :remove-bindings-button
       :text-part (:text "Clear")
       :bitmap-part :none
       :mouse-feedback :border
       :action (call :eval
		     (let ((menu (binder-bindings (part-of *self*))))
		       (setf (parts menu) nil)))
       :action-docu "Remove all bindings")

      (:class margined-window
       :name :bindings
       :adjust-size? nil
       :border-width 0
       :margins ((standard-margins-with-scroll-bars-without-label
		  :scroll-bar-options (:locations (:right))))
       :client-window (binder-menu :selection nil))

      (:class text-dispel
       :name :operations-header
       :text "Operations"
       :font (:face :bold)
       :inside-border 2
       :border-width 1)

      (:class soft-button
       :name :remove-operations-button
       :text-part (:text "Clear")
       :bitmap-part :none
       :mouse-feedback :border
       :action (call :eval
		     (let ((menu (binder-operations (part-of *self*))))
		       (setf (parts menu) nil)))
       :action-docu "Remove all operations")

      (:class margined-window
       :name :operations
       :adjust-size? nil
       :border-width 0
       :margins ((standard-margins-with-scroll-bars-without-label
		  :scroll-bar-options (:locations (:right))))
       :client-window
       (text-menu
	:name :operations-menu
	:inside-border 0
	:layouter (distance-layouter :distance 2)
	:part-class binder-operation-dispel
	:part-font (:size :small)
	:action
	(call :eval (let ((input (binder-input (part-of *self* 2))))
		      (setf (text input) *part-value*)
		      (write-to-application input)))
	:reactivity-entries
	((:read-event (call :eval (setf (parts *self*)
				      (mapcar #'(lambda (op)
						  `(:text ,op
							  :view-of ,op))
					      *binder-operations*))))
	 (:map-notify (call :read)))))

      (:class text-dispel
       :name :input-header
       :text "Operation"
       :font (:face :bold)
       :inside-border 2
       :border-width 1)

      (:class soft-button
       :name :repeat-operation-button
       :text-part (:text "Again")
       :bitmap-part :none
       :mouse-feedback :border
       :action
       (call :eval (write-to-application (binder-input (part-of *self*))))
       :action-docu "Repeat current operation")

      (:class soft-button
       :name :save-operation-button
       :text-part (:text "Save")
       :bitmap-part :none
       :mouse-feedback :border
       :action (call :eval (let* ((binder (part-of *self*))
				  (op (text (binder-input binder))))
			     (add-part (binder-operations binder)
				       :text op
				       :view-of op)))
       :action-docu "Save current operation")

      (:class popup-part-container
       :name :input-history
       :adjust-size? t
       :popup-part
       (popup-text-menu
	:background "white"
	:action
	(call :self send-message-event
	            (binder-input (part-of (view-of *self*)))
		    *part-value*)
	:reactivity-entries
	((:message
	  (call :eval
		(let* ((parts (parts *self*))
		       (num-parts (length parts)))
		  (if (< num-parts *binder-history-length*)
		      (add-part *self* :text *value* :view-of *value*)
		    (let ((part (first parts)))
		      (setf (part-position part) num-parts)
		      (setf (text part) *value*
			    (view-of part) *value*))))))))
       :client-window
       (soft-button
	:text-part (:text "History")
	:bitmap-part :none
	:mouse-feedback :border
	:action (call :part-of select-from-popup-part)
	:action-docu "Show operation history"))

      (:class margined-window
       :name :input
       :adjust-size? nil
       :border-width 0
       :margins ((standard-margins-with-scroll-bars-without-label
		  :scroll-bar-options (:locations (:right))))
	    
       :client-window
       (scrollable-multi-line-text-dispel
	:name :text
	:adjust-size? nil
	:text "*"
	:fill-column :wrap
	:separators (#\Space #\Return #\Newline #\Linefeed)
	:edit-mode :click
	:reactivity-entries
	((:accept-event (call :write))
	 (:message "Change text"
	  (call :eval (setf (text *self*) *value*))
	  (call :write))
	 (:write-event
	  (call :eval (let ((binder (part-of *self* 2)))
			(read-from-application (binder-result binder))
			(message-event (binder-history binder)
				       *self* (text *self*))))))
	))
	
      (:class text-dispel
       :name :result-header
       :text "Result"
       :font (:face :bold)
       :inside-border 2
       :border-width 1
       :selected? t
       :mouse-feedback :border
       :reactivity-entries
       ((:select "Set focus on result" (call :write))
	(:write-event
	 (call :eval
	   (setf (selected? *self*) t
		 (selected? (part (part-of *self*) :bindings-header)) nil)))))

      (:class soft-button
       :name :bind-result-button
       :text-part (:text "Bind")
       :bitmap-part :none
       :mouse-feedback :border
       :action (call :eval
		     (let ((binder (part-of *self*)))
		       (add-binding (binder-bindings binder)
				    (view-of (binder-result binder))))))

      (:class soft-button
       :name :bind-all-results-button
       :text-part (:text "Bind All")
       :bitmap-part :none
       :mouse-feedback :border
       :action (call :eval
		     (let ((binder (part-of *self*)))
		       (apply #'add-all-bindings
			      (binder-bindings binder)
			      (view-of (binder-result binder))))))

      (:class margined-window
       :name :result
       :adjust-size? nil
       :border-width 0
       :margins ((standard-margins-with-scroll-bars-without-label
		  :scroll-bar-options (:locations (:right))))
	    
       :client-window
       (scrollable-multi-line-text-dispel
	:name :text
	:adjust-size? nil
	:fill-column :max
	:separators (#\Space #\Return #\Newline #\Linefeed)
	:reactivity-entries
	((:select (call :eval
			(write-to-application
			 (part (part-of *self* 2) :result-header))))
	 (:read-event
	  (call :eval
		(let* ((binder (part-of *self* 2))
		       (input-part (binder-input (part-of *self* 2)))
		       (input-text (text input-part)))
		  (if (string= input-text "")
		      (setf (text *self*) ""
			    (view-of *self*) nil)
		    (multiple-value-bind (result error)
			(ignoring-errors
			 (let ((saved-package *package*)
			       res)
			   (setq *package* (find-package :xit))
			   (setq * (binder-current-value binder)) 
			   (setq res
			       (eval (read-from-string (text input-part))))
			   (setq *package* saved-package)
			   res))
		      (when error (flash-window input-part))
		      (setf (text *self*)
			    (if error "" (convert-to-readable-string result)))
		      (setf (view-of *self*)
			    (unless error result))))))))))
      )
     :layouter
     (pane-layouter
      :configuration configuration-1
      :configurations
      ((configuration-1
	((:header :ask)
	 (empty 3)
	 (menu-strip (:ask :window-operations-menu) :h
		     (empty 3)
		     (:binder-operations-menu :ask)
		     (empty :rest)
		     (:button-switch :ask)
		     (empty 20)
		     (:window-operations-menu :ask)
		     (empty 3))
	 (empty 3)
	 (main :rest :h
	       (bind-field 280 :v
		               (b-heading (:ask :bindings-header) :h
					  (:bindings-header :rest)
					  (empty 3)
					  (:remove-bindings-button :ask))
			       (empty 3)
			       (:bindings :rest))
	       (empty 5)
	       (ops-field :rest :v
			  (ops-heading (:ask :operations-header) :h
				       (:operations-header :rest)
				       (empty 3)
				       (:remove-operations-button :ask))
			  (empty 3)
			  (:operations 0.4)
			  (empty 5)
			  (input-heading (:ask :input-header) :h
					 (:input-header :rest)
					 (empty 3)
					 (:repeat-operation-button :ask)
					 (empty 3)
					 (:save-operation-button :ask)
					 (empty 3)
					 (:input-history :ask))
			  (empty 3)
			  (:input :even)
			  (empty 5)
			  (result-heading (:ask :result-header) :h
					  (:result-header :rest)
					  (empty 3)
					  (:bind-result-button :ask)
					  (empty 3)
					  (:bind-all-results-button :ask))
			  (empty 3)
			  (:result :even)))))
       ))
     )
   (when parent-p `(:parent ,parent))))

(define-resources
  (* binder x) 300
  (* binder y) 50
  (* binder width) 600
  (* binder height) 400
  (* binder paned-window width) 600
  (* binder paned-window height) 400
  (* binder background) "white"
  )