;;; -*- Mode: Lisp;  Package: CLIM-USER; Base: 10; Lowercase: Yes -*-

(in-package :clim-user)

" Written by John Aspinall at Harlequin, Inc.  (jga@harlequin.com)
  Harlequin hereby gives permission for unlimited use of this example code."

;;; This is the earliest version of the Tower of Hanoi application.
;;; It uses no direct manipulation at all - it only has menu commands.

(define-application-frame hanoi ()
    ((towers :initform (make-initial-tower-configuration))
     (tower-x-spacing :initform 120)
     (disc-thickness :initform 16))
  (:panes
    (display :application
	     :display-function 'draw-towers
             :scroll-bars nil
	     :text-cursor nil
	     :width :compute :height :compute))
  (:layouts
    (:default display)))

;;; a disc on a peg is encoded as a single number

(defun make-disc (number peg)
  (+ (* peg 100) number))

(defun disc-number (disc)
  (rem disc 100))

(defun disc-peg (disc)
  (floor disc 100))

(defun make-initial-tower-configuration (&optional (n-rings 10))
  (list (loop for i from 1 to n-rings collect (make-disc i 0))
        nil
        nil))

;;; how to draw the display

(defmethod draw-towers ((frame hanoi) pane &key &allow-other-keys)
  (with-slots (towers tower-x-spacing disc-thickness) frame
    (let ((base-y (* (+ disc-thickness 1) 11)))
      (draw-line* pane 0 base-y (* 3 tower-x-spacing) base-y :line-thickness 2))
    (loop for peg-number from 0
          for tower-discs in towers
          for xc = (floor tower-x-spacing 2) then (+ xc tower-x-spacing)
          do (draw-tower peg-number tower-discs disc-thickness xc pane))))

(defun draw-tower (peg-number tower-discs thickness xc pane)
  (let ((height (* (- 11 (length tower-discs)) (+ thickness 1))))
    (draw-line* pane xc thickness xc (* (+ thickness 1) 11) :line-thickness 2)
    (loop for disc in tower-discs
          as disc-number = (disc-number disc)
          for yd = height then (+ yd thickness 1)
          as width = (* (+ disc-number 2) 10)
          do (draw-disc disc xc yd width thickness pane))))

(defun draw-disc (disc x y width thickness pane)
  (let ((dx (floor width 2)))
    (draw-rectangle* pane (- x dx) y (+ x dx) (+ y thickness))))



(define-hanoi-command (com-move-0-to-1 :menu "0 to 1")
    ()
   (let ((frame *application-frame*))
     (with-slots (towers) frame
       (let* ((tower (nth 0 towers))
              (top-disc (when tower (disc-number (first tower)))))
         (when top-disc (move-disc frame top-disc 0 1))))))

(define-hanoi-command (com-move-1-to-0 :menu "1 to 0")
    ()
   (let ((frame *application-frame*))
     (with-slots (towers) frame
       (let* ((tower (nth 1 towers))
              (top-disc (when tower (disc-number (first tower)))))
         (when top-disc (move-disc frame top-disc 1 0))))))

(define-hanoi-command (com-move-0-to-2 :menu "0 to 2")
    ()
   (let ((frame *application-frame*))
     (with-slots (towers) frame
       (let* ((tower (nth 0 towers))
              (top-disc (when tower (disc-number (first tower)))))
         (when top-disc (move-disc frame top-disc 0 2))))))

(define-hanoi-command (com-move-2-to-0 :menu "2 to 0")
    ()
   (let ((frame *application-frame*))
     (with-slots (towers) frame
       (let* ((tower (nth 2 towers))
              (top-disc (when tower (disc-number (first tower)))))
         (when top-disc (move-disc frame top-disc 2 0))))))

(define-hanoi-command (com-move-1-to-2 :menu "1 to 2")
    ()
   (let ((frame *application-frame*))
     (with-slots (towers) frame
       (let* ((tower (nth 1 towers))
              (top-disc (when tower (disc-number (first tower)))))
         (when top-disc (move-disc frame top-disc 1 2))))))

(define-hanoi-command (com-move-2-to-1 :menu "2 to 1")
    ()
   (let ((frame *application-frame*))
     (with-slots (towers) frame
       (let* ((tower (nth 2 towers))
              (top-disc (when tower (disc-number (first tower)))))
         (when top-disc (move-disc frame top-disc 2 1))))))

#|| 
;;; if you wish to un-define these commands so that you can proceed to
;;; the other Tower-of-Hanoi CLIM demos, evaluate the following forms:

  (remove-menu-item-from-command-table 'hanoi "0 to 1")
  (remove-menu-item-from-command-table 'hanoi "1 to 0")
  (remove-menu-item-from-command-table 'hanoi "0 to 2")
  (remove-menu-item-from-command-table 'hanoi "2 to 0")
  (remove-menu-item-from-command-table 'hanoi "1 to 2")
  (remove-menu-item-from-command-table 'hanoi "2 to 1")
  (remove-command-from-command-table 'com-move-0-to-1 'hanoi)
  (remove-command-from-command-table 'com-move-1-to-0 'hanoi)
  (remove-command-from-command-table 'com-move-0-to-2 'hanoi)
  (remove-command-from-command-table 'com-move-2-to-0 'hanoi)
  (remove-command-from-command-table 'com-move-1-to-2 'hanoi)
  (remove-command-from-command-table 'com-move-2-to-1 'hanoi)

;;; you will also need to make a new instance of the application frame
||#



(defmethod move-disc ((frame hanoi) disc-number source-peg destination-peg)
  (with-slots (towers) frame
    (when (move-possible-p frame disc-number source-peg destination-peg)
      (pop (nth source-peg towers))
      (push (make-disc disc-number destination-peg) (nth destination-peg towers)))))

;;; this method checks that the requested move is possible -
;;; that the disc is on source-peg and the top disc on destination-peg
;;; is larger than the moving disc.

(defmethod move-possible-p ((frame hanoi) disc-number source-peg destination-peg)
  (and (from-move-possible-p frame disc-number source-peg)
       (to-move-possible-p frame disc-number destination-peg)))

(defmethod from-move-possible-p ((frame hanoi) disc-number source-peg)
  (with-slots (towers) frame
    (let ((top-of-source (first (nth source-peg towers))))
      (and top-of-source
	   (= (disc-number top-of-source) disc-number)))))

(defmethod to-move-possible-p ((frame hanoi) disc-number destination-peg)
  (with-slots (towers) frame
    (let ((top-of-destination (first (nth destination-peg towers))))
      (or (null top-of-destination)
	  (> (disc-number top-of-destination) disc-number)))))


(define-hanoi-command (com-exit-hanoi :menu "Exit")
    ()
  (frame-exit *application-frame*))


(defun do-tower-of-hanoi (&optional (frame (make-application-frame 'hanoi)))
  (run-frame-top-level frame)
  frame)

